/*******************************************************************************
 *	TmsDataTest1.cpp		TMS data test to validate operation
 *			T.Barnaby,	BEAM Ltd,	2014-08-07
 *******************************************************************************
 *
 * This test loops once per second fetching cycle information and then fetches
 *  the MeanAll data for this cycle for all cycle periods after injection.
 * The amount of data and the per sample timestamps are validated.
 * On error it will abort with appropriate error message.
 *
 * Works for MeanAll data at the moment.
 */
#include <iostream>
#include <stdio.h>
#include <unistd.h>
#include <TmsD.h>
#include <TmsC.h>

using namespace Tms;
using namespace std;


// Function to read Mean all data, testing
BError tmsTest1(TmsProcess& tmsProcess){
	BError			err;
	DataInfo		dataInfo;
	UInt32			cn = 0;
	BString			ct;
	UInt32			n;
	Data			data;
	UInt32			chan;
	CycleInformation	ci;
	BUInt			c;
	BUInt			s;
	BUInt			p;
	BUInt			numValues;
		
	// Find out the current cycle number and type
	if(err = tmsProcess.getCycleInfo(cn, ct)){
		return err.set(1, BString("Error: Getting Cycle Number: ") + err.getString());
	}
	
	// Get cycle information. Will await the completion of the cycle capture
	if(err = tmsProcess.getCycleInformation(cn, ci)){
		return err.set(1, BString("Error: Getting Cycle Information: ") + err.getString());
	}

	printf("Cycle information: %u %s\n", ci.cycleNumber, ci.cycleType.retStr());
	for(p = 0; p < ci.periods.number(); p++){
		printf("  Period: %2d  Times: %4d - %4d Harm: %2d  NumBunches: %2d BunchMask: 0x%05x NumValues: %8d\n", ci.periods[p].cyclePeriod, ci.periods[p].startTime,
			ci.periods[p].endTime, ci.periods[p].harmonic, ci.periods[p].numBunches, ci.periods[p].bunchMask, ci.periods[p].numValues);
	}

	for(p = 0; p < ci.periods.number(); p++){
		if(ci.periods[p].cyclePeriod == CyclePeriodEvent0)
			break;
	}


//	return err;

	printf("Fetch mean data from all pick-ups for cycle: %u\n", cn);

	// Setup dataInfo
	dataInfo.cycleNumber	= cn;
	dataInfo.channel	= 0;
	dataInfo.cyclePeriod	= CyclePeriodEvent0;
	dataInfo.startTime	= 10;
	dataInfo.orbitNumber	= 0;
	dataInfo.bunchNumber	= 0;
	dataInfo.function	= DataFunctionMeanAll;
//	dataInfo.function	= DataFunctionMean;
	dataInfo.argument	= 0;
	dataInfo.numValues	= 100000;
	dataInfo.beyondPeriod	= 0;

	if(err = tmsProcess.getData(dataInfo, data)){
		for(chan = 0; chan < data.errors.size(); chan++){
			printf("Error: Chan: %d ErrorNo: %d ErrorStr: %s\n", chan + 1, data.errors[chan].getErrorNo(), data.errors[chan].getString().retStr());
		}
		return err.set(1, BString("Error: Getting Data: ") + err.getString());
	}

	printf("MeanAll: Period: %d NumChannels: %d NumSamples: %d\n", dataInfo.cyclePeriod, data.numChannels, data.dataValues.size());
	printf("Data Times: %d - %d\n", data.dataValues[0].time, data.dataValues[data.dataValues.size() - 1].time);

	numValues = (ci.periods[p].endTime - ci.periods[p].startTime);
	printf("NumValues per channel expected: %d\n", numValues);

	// Validate number of samples
	if(data.dataValues.size() != (numValues * data.numChannels)){
		printf("Expected numSamples: %d received numSamples: %d\n", (ci.periods[p].endTime - ci.periods[p].startTime), data.dataValues.size());
//		return err.set(1, "Error: number of samples returned incorrect");
	}

	// Validate time stamps
	for(c = 0; c < data.numChannels; c++){
		for(s = 0; s < numValues; s++){
			if(data.dataValues[(c * numValues) + s].time != ci.periods[p].startTime + s){
				printf("Chan: %d Sample: %d Expected timestamp: %d received timestamp: %d\n", c, s, (ci.periods[p].startTime + s), data.dataValues[s].time);
//				return err.set(1, "Error: timestamp in data incorrect");
			}
		}
	}

	for(s = 0; s < 8; s++){
		printf("Time: %d\n", data.dataValues[s].time);
	}

	return err;
}

// Function to read DataFunctionMeanAll for all cycle periods
BError tmsTest2(TmsProcess& tmsProcess){
	BError			err;
	DataInfo		dataInfo;
	UInt32			cn = 0;
	BString			ct;
	UInt32			n;
	Data			data;
	UInt32			chan;
	CycleInformation	ci;
	BUInt			c;
	BUInt			s;
	BUInt			p;
	BUInt			numValues;
	
	// Find out the current cycle number and type
	if(err = tmsProcess.getCycleInfo(cn, ct)){
		return err.set(1, BString("Error: Getting Cycle Number: ") + err.getString());
	}
	
	// Get cycle information. Will await the completion of the cycle capture
	if(err = tmsProcess.getCycleInformation(cn, ci)){
		return err.set(1, BString("Error: Getting Cycle Information: ") + err.getString());
	}

	printf("Cycle information: %u %s\n", ci.cycleNumber, ci.cycleType.retStr());
	for(p = 0; p < ci.periods.number(); p++){
		printf("  Period: %2d  Times: %4d - %4d Harm: %2d  NumBunches: %2d BunchMask: 0x%05x NumValues: %8d\n", ci.periods[p].cyclePeriod, ci.periods[p].startTime,
			ci.periods[p].endTime, ci.periods[p].harmonic, ci.periods[p].numBunches, ci.periods[p].bunchMask, ci.periods[p].numValues);
	}

	printf("Fetch DataFunctionMeanAll data from all pick-ups and all periods for cycle: %u\n", cn);

	// Setup dataInfo
	dataInfo.cycleNumber	= cn;
	dataInfo.channel	= 0;
	dataInfo.cyclePeriod	= CyclePeriodEvent0;
	dataInfo.startTime	= 0;
	dataInfo.orbitNumber	= 0;
	dataInfo.bunchNumber	= 0;
	dataInfo.function	= DataFunctionMeanAll;
	dataInfo.argument	= 0;
	dataInfo.numValues	= 100000;
	dataInfo.beyondPeriod	= 0;

	for(p = 0; p < ci.periods.number(); p++){

		// Fetch the data for cycle periods after injection
		if(ci.periods[p].cyclePeriod >= CyclePeriodEvent0){
			dataInfo.cyclePeriod = ci.periods[p].cyclePeriod;
			dataInfo.function = DataFunctionMeanAll;

			if(err = tmsProcess.getData(dataInfo, data)){
				for(chan = 0; chan < data.errors.size(); chan++){
					printf("Error: Chan: %d ErrorNo: %d ErrorStr: %s\n", chan + 1, data.errors[chan].getErrorNo(), data.errors[chan].getString().retStr());
				}
				return err.set(1, BString("Error: Getting Data: ") + err.getString());
			}
			
			numValues = (ci.periods[p].endTime - ci.periods[p].startTime);
			printf("MeanAll: Period: %d NumChannels: %d NumSamples: %d NumSamples per channel: %d\n", dataInfo.cyclePeriod, data.numChannels, data.dataValues.size(), numValues);

			// Validate number of samples
			if(data.dataValues.size() != (numValues * data.numChannels)){
				printf("Expected numSamples: %d received numSamples: %d\n", (ci.periods[p].endTime - ci.periods[p].startTime), data.dataValues.size());
				return err.set(1, "Error: number of samples returned incorrect");
			}

			// Validate time stamps
			for(c = 0; c < data.numChannels; c++){
				for(s = 0; s < numValues; s++){
					// Ignore first sample of first period
					if(dataInfo.cyclePeriod == CyclePeriodEvent0)
						continue;
						
					if(data.dataValues[(c * numValues) + s].time != ci.periods[p].startTime + s){
						printf("Chan: %d Sample: %d Expected timestamp: %d received timestamp: %d\n", c, s, (ci.periods[p].startTime + s), data.dataValues[s].time);
						return err.set(1, "Error: timestamp in data incorrect");
					}
				}
			}
		}
	}

	return err;
}

int main(int argc, char** argv){
	BError			err;
	BString			host = "localhost";
	TmsProcess		tmsProcess;

	if(argc == 2)
		host = argv[1];

	// Connect to the Process service
	if(err = tmsProcess.connectService(BString("//") + host + "/tmsProcess1")){
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}
	
	if(err = tmsTest1(tmsProcess)){	
		cerr << "Error: " << err.getString() << "\n";
		return 1;
	}
	printf("\n");
	return 0;

	while(1){
		// Run a normal data gathering cycle as a normal client would.
		if(err = tmsTest1(tmsProcess)){	
			cerr << "Error: " << err.getString() << "\n";
			return 1;
		}
		printf("\n");
		sleep(1);
	}
		
	return 0;
}
