/*
* Title:	PhaseTableWin.cpp 
* Author:	M.Thomas BEAM Ltd
* Date:		2007-02-13
*
* Contents:	State Table Setup
*
* Mod Rec:
*
*/
#include <PhaseTableWin.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qfiledialog.h>
#include <qcheckbox.h>
#include <BFile.h>
#include <BQComboBox.h>
#include <qheaderview.h>

#include <qgroupbox.h>
#include <qvalidator.h>
#include <Globals.h>
#include <math.h>
#include <TmsD.h>
#include <TmsC.h>
#include <TmsS.h>
#include <TmsLib.h>

const unsigned int MAX_CYCLE_STATES = 16;

using namespace Tms;

DetailViewGui::DetailViewGui(){
	QGridLayout*	l;
	
	l = new QGridLayout(this);

	ostateGui = new TmsStateGui(this);
	ostateGui->init();
	ostateGui->setMinimumSize(900,700);
	l->addWidget(ostateGui, 0, 0);
}

void DetailViewGui::init(Tms::CycleParamEdit& cycleParam, int reset){
	ostateGui->view(cycleParam, reset);
}


PhaseTableWin::PhaseTableWin(QWidget* parent,Control& c) : BVBox(parent), ocontrol(c){
	BString		s;
	int		row = 0;
	int		col = 0;
	BHBox*		whb;
	BGroupBoxGrid* 	wbg;
	int		n;

	odetailViewGui = 0;

	oparams.cycleType = "";
	oparams.info = "";
	oparams.pllInitialFrequency = 0;
	oparams.pllInitialFrequencyDelay = 0;
	oparams.pllFrefGain = 0;
	oparams.pllGain = 0;
	oparams.channel = 0;

	for (unsigned int n = 0; n < MAX_CYCLE_STATES;n++) {
		CycleParamState s;
 		oparams.getDefaultState(s);
		ocycleStates.append(s);
	}
	
	QValidator* 		validatorInt = new QIntValidator(-(int)pow(2,31), (int)pow(2,31), this );
	QValidator* 		validatorDouble = new QDoubleValidator(this );
	QRegExpValidator* 	validatorHex = new QRegExpValidator(this);

	QRegExp		rx;
	rx.setPattern("0x[0-9,a-f][0-9,a-f][0-9,a-f][0-9,a-f][0-9,a-f][0-9,a-f][0-9,a-f][0-9,a-f]");
	validatorHex->setRegExp(rx);


	s = "<p><b>Cycle Parameters<b></p>";
	s += "<p>Select Cycle parameters file to be loaded</p>";

	// Load Cycle Parameters
	wbg = new BGroupBoxGrid("Load Cycle Parameters", this);
	ocycleTypes 	= new QListWidget(wbg);
	oloadFromTms 	= new QPushButton("Load from TMS", wbg);
	ofilename 	= new QLineEdit(wbg);
	oselectFile 	= new QPushButton("Select", wbg);
	oloadFromFile 	= new QPushButton("Load from File", wbg);
	
	wbg->addWidget(new QLabel("Load from TMS", wbg), 0, 0);
	wbg->addWidget(ocycleTypes, 0, 1);
	wbg->addWidget(oloadFromTms, 0, 3);
	wbg->addWidget(new QLabel("Load from File", wbg), 1, 0);
	wbg->addWidget(ofilename, 1, 1);
	wbg->addWidget(oselectFile, 1, 2);
	wbg->addWidget(oloadFromFile, 1, 3);

	whb = new BHBox(this);

	// Base Paramters
	wbg = new BGroupBoxGrid("Base Parameters", whb);
	ocycleType 			= new QLineEdit(wbg);
	oname 				= new QLineEdit(wbg);
	ochannel 			= new QSpinBox(wbg); ochannel->setMinimum(0); ochannel->setMaximum(999);
	oring	 			= new QSpinBox(wbg); oring->setMinimum(0); oring->setMaximum(4);
	oinfo 				= new QLineEdit(wbg);
	opllCycleStartFrequency 	= new QLineEdit(wbg);opllCycleStartFrequency->setValidator(validatorInt);
	opllInitialFrequency 		= new QLineEdit(wbg);opllInitialFrequency->setValidator(validatorInt);
	opllInitialFrequencyDelay	= new QLineEdit(wbg);opllInitialFrequencyDelay->setValidator(validatorInt);
	opllFrefGain			= new QLineEdit(wbg);opllFrefGain->setValidator(validatorInt);
	opllGain			= new QLineEdit(wbg);opllGain->setValidator(validatorInt);
	opllDdsMinimum			= new QLineEdit(wbg);opllDdsMinimum->setValidator(validatorInt);
	opllDdsMaximum			= new QLineEdit(wbg);opllDdsMaximum->setValidator(validatorInt);
	ostateDelay			= new QLineEdit(wbg);ostateDelay->setValidator(validatorInt);

	row = 0;col = 0;
	addEditorToGrid(wbg, "Cycle Type",			ocycleType, row, col);
	addEditorToGrid(wbg, "Name",				oname, row, col + 2); row++;
	addEditorToGrid(wbg, "Ring",				oring, row, col);
	addEditorToGrid(wbg, "Channel",				ochannel, row, col + 2); row++;
	addEditorToGrid(wbg, "Info",				oinfo, row, col, 3); row++;
	addEditorToGrid(wbg, "Pll CycleStart Frequency",	opllCycleStartFrequency, row, col); row++;
	addEditorToGrid(wbg, "Pll Init Frequency",		opllInitialFrequency, row, col);
	addEditorToGrid(wbg, "Pll Init Frequency Delay(ms)",	opllInitialFrequencyDelay, row, col + 2); row++;
	addEditorToGrid(wbg, "Pll Fref Gain",			opllFrefGain, row, col);
	addEditorToGrid(wbg, "Pll Gain (shift right)",		opllGain, row, col + 2); row++;
	addEditorToGrid(wbg, "Pll Dds Max",			opllDdsMinimum, row, col);
	addEditorToGrid(wbg, "Pll Dds Min",			opllDdsMaximum, row, col + 2); row++;
	addEditorToGrid(wbg, "State Delay (Fref)",		ostateDelay, row, col);

	// Positions
	wbg = new BGroupBoxGrid("Positions", whb);
	ophaseDelays 	= new BWTable(wbg);
	ooffset 	= new QSpinBox(wbg);
	oaddOffsets 	= new QPushButton("Add Offset to all channels", wbg);
	oloadDefaults 	= new QPushButton("Load Defaults", wbg);

	ophaseDelays->setColumns(bstringToList("Phase Offset"));
	//ophaseDelays->table()->setColumnStretchable(0, true);
	//ophaseDelays->table()->setSectionResizeMode(QHeaderView::Stretch);
	ophaseDelays->table()->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);

	ooffset->setMaximum(512);
	ooffset->setMinimum(-512);
	ooffset->setWrapping(true);

	row = 0; col = 0;
	addEditorToGrid(wbg, "Positions", ophaseDelays, 0, 0);
	addEditorToGrid(wbg, "Offset", ooffset, 1, 0);
	
	wbg->addWidget(oloadDefaults, 2, 0);
	wbg->addWidget(oaddOffsets, 2, 1);
	
	// State Setup
	ostateSetups = new BGroupBoxGrid("State Setup", this);
	
	oenableModifications	= new QCheckBox("Enable Modifications", ostateSetups);
	ostateNum	= new QSpinBox(ostateSetups);
	ostateNum->setMinimum(0);
	ostateNum->setMaximum(15);
	ostateNum->setWrapping(0);
	ophaseNames	= new QComboBox(ostateSetups);
	oaddNextState	= new QPushButton("Append state", ostateSetups);
	odelLastState	= new QPushButton("Delete Last State", ostateSetups);

	olo1Harmonic 	= new QLineEdit(ostateSetups); olo1Harmonic->setValidator(validatorInt);
	olo1Phase 	= new QLineEdit(ostateSetups); olo1Phase->setValidator(validatorDouble);
	of1Ref	 	= new QCheckBox("F1RefSigma", ostateSetups);
	of1Lo	 	= new QCheckBox("F1LoMsb", ostateSetups);
	olo2Harmonic 	= new QLineEdit(ostateSetups); olo2Harmonic->setValidator(validatorInt);
	olo2Phase 	= new QLineEdit(ostateSetups); olo2Phase->setValidator(validatorDouble);
	of2Ref	 	= new QCheckBox("F2RefSigma", ostateSetups);
	of2Lo	 	= new QCheckBox("F2LoMsb", ostateSetups);

	obit6 		= new QCheckBox("StateBit6", ostateSetups);
	obit7 		= new QCheckBox("StateBit7", ostateSetups);
	opllF2 		= new QCheckBox("Pll Use F2", ostateSetups);
	oacquireData 	= new QCheckBox("Acquire Data", ostateSetups);

	obunchMask = new QLineEdit(ostateSetups); obunchMask->setValidator(validatorHex);
	omean1Mask = new QLineEdit(ostateSetups); omean1Mask->setValidator(validatorHex);
	omean2Mask = new QLineEdit(ostateSetups); omean2Mask->setValidator(validatorHex);
	
	ogateWidth 	= new QLineEdit(ostateSetups); ogateWidth->setValidator(validatorDouble);
	ogatePhase 	= new QLineEdit(ostateSetups); ogatePhase->setValidator(validatorDouble);
	oblrWidth 	= new QLineEdit(ostateSetups); oblrWidth->setValidator(validatorDouble);
	oblrPhase 	= new QLineEdit(ostateSetups); oblrPhase->setValidator(validatorDouble);

	oenableModifications->setMaximumWidth(200);
	oaddNextState->setMaximumWidth(200);
	odelLastState->setMaximumWidth(200);
	obunchMask->setMaximumWidth(200);
	omean1Mask->setMaximumWidth(200);
	omean2Mask->setMaximumWidth(200);
	olo1Harmonic->setMaximumWidth(200);
	olo1Phase->setMaximumWidth(200);
	olo2Harmonic->setMaximumWidth(200);
	olo2Phase->setMaximumWidth(200);
	oacquireData->setMaximumWidth(200);
	ogateWidth->setMaximumWidth(200);
	ogatePhase->setMaximumWidth(200);
	oblrWidth->setMaximumWidth(200);
	oblrPhase->setMaximumWidth(200);

	row = 0;col = 0;
	ostateSetups->addWidget(oenableModifications, row, 0);
	ostateSetups->addWidget(ostateNum, row, col+1);
	addEditorToGrid(ostateSetups,"Period", ophaseNames, row, col+2);  
	ostateSetups->addWidget(oaddNextState, row, col+4);
	ostateSetups->addWidget(odelLastState, row, col+5); row++;

	addEditorToGrid(ostateSetups, "Lo1 Harmonic",	olo1Harmonic, row, col); 
	addEditorToGrid(ostateSetups, "Lo1 Phase",	olo1Phase, row, col + 2);
	ostateSetups->addWidget(of1Ref, row, col+4);
	ostateSetups->addWidget(of1Lo, row, col+5); row++;
	addEditorToGrid(ostateSetups, "Lo2 Harmonic",	olo2Harmonic, row, col); 
	addEditorToGrid(ostateSetups, "Lo2 Phase",	olo2Phase, row, col + 2);
	ostateSetups->addWidget(of2Ref, row, col+4);
	ostateSetups->addWidget(of2Lo, row, col+5); row++;
	ostateSetups->addWidget(obit6, row, col+1);
	ostateSetups->addWidget(obit7, row, col+3);
	ostateSetups->addWidget(opllF2, row, col+4);
	ostateSetups->addWidget(oacquireData, row, col+5); row++;

	addEditorToGrid(ostateSetups, "Bunch Mask",	obunchMask, row, col); 
	addEditorToGrid(ostateSetups, "Mean1Mask",	omean1Mask, row, col + 2);
	addEditorToGrid(ostateSetups, "Mean2Mask",	omean2Mask, row, col + 4); row++;
	addEditorToGrid(ostateSetups, "Gate Width",	ogateWidth, row, col);
	addEditorToGrid(ostateSetups, "Gate Phase",	ogatePhase, row, col + 2); row++;
	addEditorToGrid(ostateSetups, "Blr Width",	oblrWidth, row, col); 
	addEditorToGrid(ostateSetups, "Blr Phase",	oblrPhase, row, col + 2); row++;

	BHBox* 	nextStatesLabels = new BHBox(ostateSetups);
	new QLabel("", nextStatesLabels);
	new QLabel("Delay", nextStatesLabels);
	new QLabel("HChange", nextStatesLabels);
	new QLabel("Injection", nextStatesLabels);
	new QLabel("CalStart", nextStatesLabels);
	new QLabel("CalStop", nextStatesLabels);
	new QLabel("CycleStop", nextStatesLabels);
	ostateSetups->addWidget(nextStatesLabels, row, 0, 1, 5);
	row++;

	BHBox* 	nextStates = new BHBox(ostateSetups);
	new QLabel("NextStates", nextStates);
	for(n = 0; n < 6; n++){
		onextStates[n] = new QSpinBox(nextStates);
		onextStates[n]->setMinimum(0);
		onextStates[n]->setMaximum(15);
		connect(onextStates[n], SIGNAL(valueChanged(int)), this, SLOT(setCyclePhaseParams()));
	}
	ostateSetups->addWidget(nextStates, row, 0, 1, 5);

	// Control Buttons
	BHBox* controls	= new BHBox(this);
	oupload 	= new QPushButton("Upload",controls);
	osaveFile 	= new QPushButton("Save to file",controls);
	odelete 	= new QPushButton("Delete from server",controls);
	oupdate 	= new QPushButton("Update State", controls);
	oviewDetails 	= new QPushButton("View State Details", controls);

#ifdef ZAP_QT3
	controls->setSpacing(20);
#endif
	oupload->setMaximumWidth(200);
	osaveFile->setMaximumWidth(200);
	odelete->setMaximumWidth(200);
	oviewDetails->setMaximumWidth(200);
	oupdate->setMaximumWidth(200);

	ophaseNames->addItem("All");
	ophaseNames->addItem("Calibration");
	ophaseNames->addItem("Event0");
	ophaseNames->addItem("Event1");
	ophaseNames->addItem("Event2");
	ophaseNames->addItem("Event3");
	ophaseNames->addItem("Event4");
	ophaseNames->addItem("Event5");
	ophaseNames->addItem("Event6");
	ophaseNames->addItem("Event7");
	ophaseNames->addItem("Event8");
	ophaseNames->addItem("Event9");

	connect(oloadFromFile,SIGNAL(clicked()),this,SLOT(loadFromFile()));
	connect(oloadFromTms,SIGNAL(clicked()),this,SLOT(loadFromTms()));
	connect(oselectFile,SIGNAL(clicked()),this,SLOT(selectFile()));
	connect(oaddOffsets,SIGNAL(clicked()),this,SLOT(addOffsets()));
	connect(oloadDefaults,SIGNAL(clicked()),this,SLOT(loadDefaults()));
	connect(osaveFile,SIGNAL(clicked()),this,SLOT(saveFile()));
	connect(odelete,SIGNAL(clicked()),this,SLOT(deleteTms()));
	connect(oupdate,SIGNAL(clicked()),this,SLOT(updateDetails()));
	connect(oviewDetails,SIGNAL(clicked()),this,SLOT(viewDetails()));
	connect(oupload,SIGNAL(clicked()),this,SLOT(saveTms()));


	connect(ostateNum,SIGNAL(valueChanged(int)),this,SLOT(cycleStateChanged()));
	connect(ophaseNames,SIGNAL(activated(int)),this,SLOT(setCyclePhaseParams()));

	connect(obunchMask,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(omean1Mask,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(omean2Mask,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(olo1Harmonic,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(olo1Phase,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(olo2Harmonic,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(olo2Phase,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(ogateWidth,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(ogatePhase,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(oblrWidth,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));
	connect(oblrPhase,SIGNAL(textChanged(const QString &)),this,SLOT(setCyclePhaseParams()));

	connect(of1Ref,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(of1Lo,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(of2Ref,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(of2Lo,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(obit6,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(obit7,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(opllF2,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));
	connect(oacquireData,SIGNAL(toggled(bool)),this,SLOT(setCyclePhaseParams()));

	connect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));
	connect(oaddNextState,SIGNAL(clicked()),this,SLOT(addNextState()));
	connect(odelLastState,SIGNAL(clicked()),this,SLOT(delLastState()));
	connect(&ocontrol,SIGNAL(newConnection()),this,SLOT(update()));
}
	
PhaseTableWin::~PhaseTableWin() {}


void	PhaseTableWin::showEvent(QShowEvent* event){
	BVBox::show();
	update();
}


void 	PhaseTableWin::update() {
	if (! isVisible())
		return;
	updateCycleTypes();			// Cycle Types list is retrieved from server
	updateCycleInfo();
	updateCycleParamState();
}

void	PhaseTableWin::updateCycleParamState() {
	ocycleStates.clear();
	oparams.getStates(ocycleStates);

	ostateNum->setValue(0);
	refreshParamState(ocycleStates[0]);
}


/* updateCycleType() - Get the list on known cycle types from server and populate selection list.
*
*/

BError PhaseTableWin::updateCycleTypes() {
	BError			err;
	BIter			i;
	BList<CycleParamItem> 	l;
	
	if (err = ocontrol.getControlList(l)) {
		return err;
	}
	
	ocycleTypes->clear();
	for (l.start(i);! l.isEnd(i);l.next(i)) {
		ocycleTypes->addItem((l[i].cycleType + " " +  l[i].ring + " " + l[i].channel).retStr());
	}
	return err;
}


/* updateCycleInfo() - 	Given cycle params (oparams) update the Gui to
*			should the cycle information.
*/

void PhaseTableWin::updateCycleInfo() {

	ocycleType->setText(oparams.cycleType.retStr());
	oname->setText(oparams.name.retStr());
	oring->setValue(oparams.ring);
	ochannel->setValue(oparams.channel);
	oinfo->setText(oparams.info.retStr());
	oinfo->setCursorPosition(0);
	
	opllCycleStartFrequency->setText(uIntToStr(oparams.pllCycleStartFrequency));
	opllInitialFrequency->setText(uIntToStr(oparams.pllInitialFrequency));
	opllInitialFrequencyDelay->setText(uIntToStr(oparams.pllInitialFrequencyDelay));
	opllFrefGain->setText(uIntToStr(oparams.pllFrefGain));
	opllGain->setText(uIntToStr(oparams.pllGain));
	opllDdsMinimum->setText(uIntToStr(oparams.pllDdsMinimum));
	opllDdsMaximum->setText(uIntToStr(oparams.pllDdsMaximum));
	ostateDelay->setText(uIntToStr(oparams.stateDelay));

	ophaseDelays->setRowCount(oparams.frefPhaseDelay.size());

	 for(unsigned int i = 0; i < oparams.frefPhaseDelay.size(); i++ ) {
		BString	s;
		s.printf("%d",oparams.frefPhaseDelay[i]);
		ophaseDelays->setText(i,0,s.retStr());
	}
}


void	PhaseTableWin::setParams() {
	bool		ok;
	BIter		i;
	
	oparams.cycleType 			= ocycleType->text();
	oparams.name 				= oname->text();
	oparams.ring 				= oring->value();
	oparams.channel 			= ochannel->value();
	oparams.info 				= oinfo->text();
	oparams.pllCycleStartFrequency 		= opllCycleStartFrequency->text().toUInt(&ok);
	oparams.pllInitialFrequency 		= opllInitialFrequency->text().toUInt(&ok);
	oparams.pllInitialFrequencyDelay 	= opllInitialFrequencyDelay->text().toUInt(&ok);
	oparams.pllFrefGain 			= opllFrefGain->text().toUInt(&ok);
	oparams.pllGain 			= opllGain->text().toUInt(&ok);
	oparams.pllDdsMinimum			= opllDdsMinimum->text().toUInt(&ok);
	oparams.pllDdsMaximum			= opllDdsMaximum->text().toUInt(&ok);
	oparams.stateDelay			= ostateDelay->text().toUInt(&ok);

	for(unsigned int i = 0; i < oparams.frefPhaseDelay.size(); i++ ) {
		oparams.frefPhaseDelay[i] = ophaseDelays->getText(i,0).retInt();
	}
	setCyclePhaseParams();
	
	if (oenableModifications->isChecked()){
		// Validate
		for(ocycleStates.start(i); !ocycleStates.isEnd(i); ocycleStates.next(i)){
			if((ocycleStates[i].lo1Harmonic < 1) || (ocycleStates[i].lo1Harmonic > 31)){
				warningDialog("Parameter Error", BError(1, "Lo1Harmonic out of range"));
				return;
			}
			if((ocycleStates[i].lo2Harmonic < 1) || (ocycleStates[i].lo2Harmonic > 31)){
				warningDialog("Parameter Error", BError(1, "Lo2Harmonic out of range"));
				return;
			}
		}

		oparams.setStates(ocycleStates);
	}
}

void	PhaseTableWin::setCyclePhaseParams() {
	bool	ok;
	BUInt 	n;

	n = ostateNum->value();
	if(ocycleStates.number() > n){
		ocycleStates[n].num		= n;
		ocycleStates[n].period		= ophaseNames->currentIndex();

		ocycleStates[n].lo1Harmonic 	= olo1Harmonic->text().toUInt(&ok);
		ocycleStates[n].lo1Phase 	= olo1Phase->text().toDouble(&ok);
		ocycleStates[n].lo2Harmonic 	= olo2Harmonic->text().toUInt(&ok);
		ocycleStates[n].lo2Phase 	= olo2Phase->text().toDouble(&ok);

		ocycleStates[n].state.delay = onextStates[0]->value();
		ocycleStates[n].state.hchange = onextStates[1]->value();
		ocycleStates[n].state.injection = onextStates[2]->value();
		ocycleStates[n].state.calStart = onextStates[3]->value();
		ocycleStates[n].state.calStop = onextStates[4]->value();
		ocycleStates[n].state.cycleStop = onextStates[5]->value();
		
		ocycleStates[n].state.pllReference1 	= of1Ref->isChecked();
		ocycleStates[n].state.pllLO1FromAddress = of1Lo->isChecked();
		ocycleStates[n].state.pllReference2 	= of2Ref->isChecked();
		ocycleStates[n].state.pllLO2FromAddress = of2Lo->isChecked();
		ocycleStates[n].state.pllFeedbackSelect = opllF2->isChecked();
		ocycleStates[n].state.acquireData 	= oacquireData->isChecked();
		ocycleStates[n].state.bit6 		= obit6->isChecked();
		ocycleStates[n].state.bit7 		= obit7->isChecked();

		ocycleStates[n].bunchMask 	= obunchMask->text().toUInt(&ok,16);
		ocycleStates[n].mean1Mask 	= omean1Mask->text().toUInt(&ok,16);
		ocycleStates[n].mean2Mask 	= omean2Mask->text().toUInt(&ok,16);
		ocycleStates[n].gateWidth 	= ogateWidth->text().toDouble(&ok);
		ocycleStates[n].gatePhase 	= ogatePhase->text().toDouble(&ok);
		ocycleStates[n].blrWidth 	= oblrWidth->text().toDouble(&ok);
		ocycleStates[n].blrPhase 	= oblrPhase->text().toDouble(&ok);
	}
}


void	PhaseTableWin::cycleStateChanged(int num){
	ostateNum->setValue(num);
}

void	PhaseTableWin::cycleStateChanged() {
	BUInt n = ostateNum->value();
	
	if(n >= ocycleStates.number())
		n = ocycleStates.number() - 1;
		
	refreshParamState(ocycleStates[n]);
	if(odetailViewGui)
		odetailViewGui->ostateGui->setState(n);
}

void	PhaseTableWin::refreshParamState(CycleParamState s) {
	int	n;
	bool	allowEdit = oenableModifications->isChecked();

	ostateNum->setValue(s.num);
	ophaseNames->setCurrentIndex(s.period);

	olo1Harmonic->setText(uIntToStr(s.lo1Harmonic));
	olo1Phase->setText(doubleToStr(s.lo1Phase));;
	olo2Harmonic->setText(uIntToStr(s.lo2Harmonic));
	olo2Phase->setText(doubleToStr(s.lo2Phase));;

	of1Ref->setChecked(s.state.pllReference1);
	of1Lo->setChecked(s.state.pllLO1FromAddress);
	of2Ref->setChecked(s.state.pllReference2);
	of2Lo->setChecked(s.state.pllLO2FromAddress);
	opllF2->setChecked(s.state.pllFeedbackSelect);
	oacquireData->setChecked(s.state.acquireData);
	obit6->setChecked(s.state.bit6);
	obit7->setChecked(s.state.bit7);

	obunchMask->setText(uIntToHexStr(s.bunchMask));
	omean1Mask->setText(uIntToHexStr(s.mean1Mask));
	omean2Mask->setText(uIntToHexStr(s.mean2Mask));
	ogateWidth->setText(doubleToStr(s.gateWidth));
	ogatePhase->setText(doubleToStr(s.gatePhase));
	oblrWidth->setText(doubleToStr(s.blrWidth));
	oblrPhase->setText(doubleToStr(s.blrPhase));

	for(n = 0; n < 6; n++){
		onextStates[n]->setValue((s.state.value >> (28 - (4 * n))) & 0xF);
		onextStates[n]->setEnabled(allowEdit);
	}
	
	oaddNextState->setEnabled(allowEdit);
	odelLastState->setEnabled(allowEdit);
	obunchMask->setEnabled(allowEdit);
	omean1Mask->setEnabled(allowEdit);
	omean2Mask->setEnabled(allowEdit);
	olo1Harmonic->setEnabled(allowEdit);
	olo1Phase->setEnabled(allowEdit);
	olo2Harmonic->setEnabled(allowEdit);
	olo2Phase->setEnabled(allowEdit);
	of1Ref->setEnabled(allowEdit);
	of1Lo->setEnabled(allowEdit);
	of2Ref->setEnabled(allowEdit);
	of2Lo->setEnabled(allowEdit);
	opllF2->setEnabled(allowEdit);
	oacquireData->setEnabled(allowEdit);
	obit6->setEnabled(allowEdit);
	obit7->setEnabled(allowEdit);
	ogateWidth->setEnabled(allowEdit);
	ogatePhase->setEnabled(allowEdit);
	oblrWidth->setEnabled(allowEdit);
	oblrPhase->setEnabled(allowEdit);
}



void PhaseTableWin::enableStateEdit() {
	BString msg;
	if (oenableModifications->isChecked()) {
		msg = "<p>Enabling the state setup will overwrite all state/phase table information</p>";
		msg += "<p>Are you sure you wish to continue ?</p>";
		if (! confirmDialog("BEWARE",msg)) {
			oenableModifications->disconnect();
			oenableModifications->setChecked(false);
			connect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));
			return;
		}	
	}
	update();
}

void	PhaseTableWin::addNextState() {
	CycleParamState 	s;

	if (ocycleStates.number() >= MAX_CYCLE_STATES) {
		warningDialog("Adding Additional State",BError(1,"Maximum number of state exceeded"));
		return;
	}

	oparams.getDefaultState(s);
	s.num = ocycleStates.number();
	ocycleStates.append(s);
	ostateNum->setValue(s.num);
	refreshParamState(s);
}

void	PhaseTableWin::delLastState() {
	
	if (ocycleStates.number() < 3) {
		warningDialog("Deleteing State",BError(1,"A minimum of two states are required"));
		return;
	}	

	ocycleStates.deleteLast();
	refreshParamState(ocycleStates.rear());
}

void PhaseTableWin::loadDefaults() {
	CycleParamEdit::getdefaultPickupPositions(oparams.frefPhaseDelay);
	update();
}


void PhaseTableWin::saveFile() {
	BError			err;
	BFile			f;
	BString			fname;


	setParams();

	if(oparams.cycleType == ""){
		warningDialog("Error", BError(1, "Cycle Type not set"));
		return;
	}
	
	fname = ofilename->text();
        QString s = QFileDialog::getSaveFileName(this, "Tms Cycle Parameters File (*.spt)", "/usr/tms/stateTables");
	fname = s;
	if (fname.len() == 0)
		return;

	if (QFile::exists(fname.retStr())){
		BString msg = "The specified file already exists are you ";
		msg += BString("positive you wish to over write the file");
		if (! confirmDialog("Confirm Action",msg)) {
			return;
		}	
	}
	if(err = oparams.writeToFile(fname)) {
		warningDialog("Cycle Parameters - Save",err);
		return;
	}
	ofilename->setText(fname.retStr());
	gstatusbar->showMessage("Cycle Parameters Saved",2000);
}

void PhaseTableWin::loadFromFile() {
	BError		err;
	BFile		f;
	BString		fname;

	disconnect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));
	oenableModifications->setChecked(false);
	connect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));
	
	fname = ofilename->text();
	if (err = f.open(fname,"r")) {
		warningDialog("Cycle Parameters - Reading",err);
		return;
	}
	if(err = oparams.readFromFile(fname)) {
		warningDialog("Cycle Parameters - Read Parsing",err);
		return;
	}	
	update();
	gstatusbar->showMessage("Cycle Parameters Read",2000);
}


void PhaseTableWin::saveTms() {
	BError	err;
	
	setParams();

	if(oparams.cycleType == ""){
		warningDialog("Error", BError(1, "Cycle Type not set"));
		return;
	}
	
	if (err = ocontrol.setControlInfo(oparams)) {
		warningDialog("Upload of Cycle Params",err);
		return;
	}
	update();
	gstatusbar->showMessage("Cycle Parameters Uploaded",2000);
}

void PhaseTableWin::deleteTms(){
	BError	err;

	setParams();
	
	if(oparams.cycleType == ""){
		warningDialog("Error", BError(1, "Cycle Type not set"));
		return;
	}
	
	// Check with user
	if(confirmDialog("BEWARE", BString("Are you sure you wish to delete Cycle Type: ") + oparams.cycleType + " from the TMS system ?")){
		if (err = ocontrol.delControlInfo(oparams.cycleType, 0, 0)){
			warningDialog("Delete of Cycle Params",err);
			return;
		}
		oparams.clear();
		update();
		gstatusbar->showMessage("Cycle Parameters Deleted", 2000);
	}
}

void PhaseTableWin::updateDetails(){
	setParams();
	if(odetailViewGui)
		odetailViewGui->init(oparams, 0);
}

void PhaseTableWin::viewDetails() {
	if(!odetailViewGui){
		odetailViewGui = new DetailViewGui();
		connect(odetailViewGui->ostateGui, SIGNAL(stateChanged(int)), this, SLOT(cycleStateChanged(int)));
	}

	odetailViewGui->show();
	setParams();
	odetailViewGui->init(oparams, 1);
}

void PhaseTableWin::loadFromTms() {
	BError err;
	BString	str = ocycleTypes->currentItem()->text();
	BString	type;
	BUInt	ring;
	BUInt	channel;
	

	type = str.pullWord();
	ring = str.pullWord().retInt();
	channel = str.pullWord().retInt();
	
	disconnect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));
	oenableModifications->setChecked(false);
	connect(oenableModifications,SIGNAL(toggled(bool)),this,SLOT(enableStateEdit()));


	if (err = ocontrol.getControlInfo(type, ring, channel, oparams)) {
		BString msg;
		msg.printf("Attempting to access cycle type (%s) ",type.retStr());
		msg += err.getString();
		err.set(err.getErrorNo(),msg);			
		warningDialog("Load from Tms",msg);
		return;
	}
	update();
}

void PhaseTableWin::selectFile() {
       QString s = QFileDialog::getOpenFileName(this, "Tms Cycle Parameters File (*.spt)", "/usr/tms/stateTables");
	ofilename->setText(s);
}

void	PhaseTableWin::warningDialog(BString title, BError err){
	BString		m;
	QMessageBox	b;
	
	m = BString("<h5>") + title + "</h5><p>" + err.getString() + "</p>";
	b.setMinimumWidth(300);
	b.setWindowTitle("tmsControlGui - Warning");
	b.setIcon(QMessageBox::Warning);
	b.setText(m.retStr());
	b.exec();
}

int	PhaseTableWin::confirmDialog(BString title,BString msg){
	BString		s;
	
	s = BString("<h5>") + title + "</h5><p>" + msg + "</p>";

	QMessageBox mb( "tmsControlGui - Question",
   	     s.retStr(),
        	QMessageBox::Information,
        	QMessageBox::Yes | QMessageBox::Default,
        	QMessageBox::Cancel | QMessageBox::Escape,
		QMessageBox::NoButton );
	switch( mb.exec() ) {
		case QMessageBox::Yes:
        		return 1;
	        	break;
  		case QMessageBox::No:
        		return 0;
		        break;
		    case QMessageBox::Cancel:
		        return 0;
		        break;
	}
	return 0;
}

void PhaseTableWin::addOffsets() {
	 int val = ooffset->value();

	 for(unsigned int i = 0; i < oparams.frefPhaseDelay.size(); i++ ) {
		int	a;
		a = oparams.frefPhaseDelay[i] + val;
		a &= 0x1ff;
		oparams.frefPhaseDelay[i] = a;
	}
	update();
}

void PhaseTableWin::addEditorToGrid(BGroupBoxGrid* wg, BString label, QWidget* editor, int row, int col, int colSpan){
	wg->addWidget(new QLabel(label.retStr(), wg), row, col);
	wg->addWidget(editor, row, col + 1, 1, colSpan);		
}

QString PhaseTableWin::uIntToStr(BUInt32 val) {
	QString	s;
	s.sprintf("%u",val);
	return s;
}

QString PhaseTableWin::doubleToStr(double val) {
	QString	s;
	s.sprintf("%f",val);
	return s;
}

QString PhaseTableWin::uIntToHexStr(BUInt32 val) {
	QString	s;
	s.sprintf("0x%x",val);
	return s;
}
