/*******************************************************************************
 *	BWidgets.h	BWidgets BEAM QT Widgets
 *	T.Barnaby,	BEAM Ltd,	2009-07-29
 *	Copyright (c) 2022 All Right Reserved, Beam Ltd, https://www.beam.ltd.uk
 *	For license see LICENSE.txt at the root of the beamlib source tree.
 *******************************************************************************
 */
#ifndef BWidgets_H
#define BWidgets_H	1

#include <qboxlayout.h>
#include <qtablewidget.h>
#include <qcheckbox.h>

#include <qdatetimeedit.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qgroupbox.h>
#include <qlineedit.h>
#include <qspinbox.h>

#include <BString.h>
#include <BArray.h>
#include <BDate.h>
#include <BTimeStamp.h>

// Handle default spacing on QLayout objects
extern int blayoutSpacing;					///< Default internal margins and spacing of layout widgets

inline void blayoutSetSpacing(QLayout& layout, int s){		///< Sets  internal margins and spacing of layout widgets
	layout.setSpacing(s);
	layout.setContentsMargins(s, s, s, s);
}

void setBackgroundColour(QWidget* widget, QColor col);		///< Sets the background colour of a widget

BString justify(BString str);

class BHBox : public QFrame {
public:
			BHBox(QWidget* parent);

	QHBoxLayout&	layout();
protected:
	void		childEvent(QChildEvent* event);
private:
	QHBoxLayout	olayout;

};

class BVBox : public QFrame {
public:
			BVBox(QWidget* parent);

	QVBoxLayout&	layout();
protected:
	void		childEvent(QChildEvent* event);
private:
	QVBoxLayout	olayout;

};

class BGrid : public QFrame {
public:
			BGrid(BUInt size, Qt::Orientation orientation, QWidget* parent);

	QGridLayout&	layout();
protected:
	void		childEvent(QChildEvent* event);
private:
	QGridLayout	olayout;
	Qt::Orientation	oorientation;
	BUInt		osize;
	BUInt		orow;
	BUInt		ocolumn;
};

class BGroupBox : public QGroupBox {
public:
			BGroupBox(QString title, BUInt size, Qt::Orientation orientation, QWidget* parent);

	QGridLayout&	layout();
protected:
	void		childEvent(QChildEvent* event);
private:
	QGridLayout	olayout;
	Qt::Orientation	oorientation;
	BUInt		osize;
	BUInt		orow;
	BUInt		ocolumn;
};

class BGroupBoxGrid : public QGroupBox {
public:
			BGroupBoxGrid(QString title, QWidget* parent);

	void		addWidget(QWidget* w, int row, int col, int rowSpan = 1, int colSpan = 1, Qt::Alignment alignment = Qt::Alignment());

	QGridLayout&	layout();

private:
	QGridLayout	olayout;
};

class BWTableWidgetItem : public QTableWidgetItem {
public:
				BWTableWidgetItem(const BWTableWidgetItem& other);
				BWTableWidgetItem(int row, const QString &text);
				BWTableWidgetItem(int row = -1);

	QTableWidgetItem*	clone() const;
	int			dataRow() const;		///< Returns the data row

private:
	int		orow;				///< Data row
};

class BWTableView : public QTableWidget {
        Q_OBJECT
public:
			BWTableView(QWidget* parent);
			~BWTableView();
	
	void		setColumns(BStringList columns);
	void		setColumnWidths(BStringList columnWidths);
	void		setColumnAlignment(int col,  int alignment);

	void		setText(int row, int col, BString value);
	BString		getText(int row, int col);

	void		set(int row, int col, BString value);
	void		set(int row, int col, int value);
	void		set(int row, int col, BUInt value);
	void		set(int row, int col, double value);
	void		setEditable(int on);
	
	BList<int>	selectedDataRows();				///< Return list of selected data row indexes
	int		currentDataRow();				///< Single selected data row
	Bool		isDataRowSelected(int r);			///< Check if given data row is selected
	
protected:
	int		oeditable;
};

class BWTable : public BVBox {
        Q_OBJECT
public:
			BWTable(QWidget* parent);
			~BWTable();
	
	void		setColumns(BStringList columns);
	void		setRowCount(int num);
	void		setText(int row, int col, BString value);
	BString		getText(int row, int col);
	void		setEditable(int on);
	void		addButton(BString name, BString memberFunc);

	BList<int>	selectedDataRows();				///< Return list of selected data row indexes
	Bool		isDataRowSelected(int r);			///< Check if given data row is selected

	BWTableView*	table();



public slots:
	void		appendRow();
	void		deleteRow();

protected:
	int			oeditable;
	BWTableView*		owTable;
	BHBox*			owButtons;
	BList<QPushButton*>	owButtonList;
};

class BWEditListSelect : public BHBox {
        Q_OBJECT
public:
			BWEditListSelect(QWidget* parent, BStringList selectionList = BStringList());
			~BWEditListSelect();

	void		setReadOnly(int on);
	void		setSelection(BStringList selectionList);
	void		setString(BString str);
	BString		getString();

private:
	BStringList		oselectionList;
	BList<QCheckBox*>	oboxes;
};


class BWEditComboBox : public BHBox {
        Q_OBJECT
public:
			BWEditComboBox(QWidget* parent, BStringList options = BStringList(), int editable = 0);
			~BWEditComboBox();
	
	void		setReadOnly(int on);
	void		clear();
	void		addItem(BString str);
	void		setSelection(BStringList selectionList);
	void		setCurrentIndex(int item);

	void		setString(BString str);
	BString		getString();
	int		getIndex();

signals:
	void		valueChanged(BString str);

private slots:
	void		update();

protected:
	QComboBox	owComboBox;
	BArray<BString>	ostringList;
};

class BWEditMoney : public BHBox {
        Q_OBJECT
public:
			BWEditMoney(QWidget* parent, BTimeStamp timeStamp = BTimeStamp());
			~BWEditMoney();

	void		setReadOnly(int on);
	void		clear();
	void		setPaletteBackgroundColor (const QColor& colour);
	
	void		setMoney(double value, double vat);
	void		getMoney(double& value, double& vat);

signals:
	void		valueChanged();

private slots:
	void		updateValue();
	void		updateVat();
	void		updateTotal();

private:
	double		round(double value);
	void		setValue(QLineEdit*, double v);
	double		getValue(QLineEdit* w);
	QLineEdit*	owValue;
	QLineEdit*	owVat;
	QLineEdit*	owTotal;
};

class BWMessageBox : public QMessageBox {
	Q_OBJECT
public:
	static int	warning(QWidget* parent, const QString& caption, const QString& text, const QString& button0Text = QString(), const QString& button1Text = QString(), const QString& button2Text = QString(), int defaultButtonNumber = 0, int escapeButtonNumber = -1);
};

class BField {
public:
		BField(int pos = 0, int size = 0, BString post = "", int min = 0, int max = 0, int value = 0) : opos(pos), osize(size), opost(post), omin(min), omax(max), ovalue(value) {};

	int	opos;
	int	osize;
	BString	opost;
	int	omin;
	int	omax;
	int	ovalue;
};

class BFieldEdit : public QLineEdit {
        Q_OBJECT
public:
			BFieldEdit(QWidget* parent);
	void		setSize();
	virtual int	validate(int v = 1);
	virtual void	setBlank(Bool on);
	
signals:
	void		changed();

protected:
	void		keyPressEvent(QKeyEvent* event);
	void		focusInEvent(QFocusEvent* event);
	void		focusOutEvent(QFocusEvent* event);
	void		mouseReleaseEvent(QMouseEvent* event);
	void		wheelEvent(QWheelEvent* event);
	void		contextMenuEvent(QContextMenuEvent* event);
	
	void		updatePos();
	void		updateFromText();
	void		updateFromFields();

	unsigned int		ofield;
	BArray<BField>		ofields;
	Bool			oblank;
};


class BWEditDate : public BFieldEdit {
        Q_OBJECT
public:
	enum Format 	{ ISO, UK, ISO_BLANK, UK_BLANK };

			BWEditDate(QWidget* parent, Format format = ISO);

	void		setReadOnly(int on);
	void		clear();
	
	void		setDate(BDate date);
	BDate		getDate();

protected:
	void		keyPressEvent(QKeyEvent* event);
private:
	void		clearWhenNotSet();
	int		validate(int v = 1);
	Format		oformat;
	BDate		odate;
	static BDate	gdate;
};

class BWEditTimeStamp : public BFieldEdit {
        Q_OBJECT
public:
	enum Format 	{ DATE_TIME_MICROSEC, DATE_TIME,  DATE_TIME_MICROSEC_BLANK, DATE_TIME_BLANK};

			BWEditTimeStamp(QWidget* parent, Format format = DATE_TIME_MICROSEC);

	void		setReadOnly(int on);
	
	void		setTimeStamp(BTimeStamp timeStamp);
	BTimeStamp	getTimeStamp();

protected:
	void		keyPressEvent(QKeyEvent* event);
private:
	void		clearWhenNotSet();
	int		validate(int v = 1);
	Format		oformat;
	BTimeStamp	otimeStamp;
	static BTimeStamp	gtimeStamp;
};

#endif
