/*******************************************************************************
 *	PupeMemTest.c	Simpel Pupe Memtest program
 *			T.Barnaby,	BEAM Ltd,	2007-05-18
 *******************************************************************************
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <admxrc2.h>

#ifdef linux
#include <unistd.h>
#endif

// FPGA Control Registers
const int	IMEM_REG	= 0x0000 >> 2;	// Block RAM Index
const int	LOCKED		= 0x0008 >> 2;	// DCM Locked Status
const int	ADDR_REG	= 0x0010 >> 2;	// PAGE Register for SDRAM access
const int	MEM_REG		= 0x0018 >> 2;	// SDRAM Bank Select Register and BlockRam enable
const int	IER		= 0x0020 >> 2;	// Interrupt Enable Register
const int	ISR		= 0x0028 >> 2;	// Interrupt Status Register
const int	MEM_RAS		= 0x0030 >> 2;	// SDRAM Read Address Scaling
const int	MEM_RAO		= 0x0038 >> 2;	// SDRAM Read Address Offset
const int	MEM_GNT0	= 0x0400 >> 2;	// Memory Grant Registers
const int	MEM_GNT1	= 0x0408 >> 2;	// Memory Grant Registers
const int	MEM_GNT2	= 0x0410 >> 2;	// Memory Grant Registers
const int	MEM_GNT3	= 0x0408 >> 2;	// Memory Grant Registers

const int	FIRMWARE	= 0x0800 >> 2;	// Firmware ID code "CN" + version numbers
const int	ADC		= 0x0808 >> 2;	// ADC Control Register
const int	TIMING_IO	= 0x0810 >> 2;	// Timing I/O Register
const int	TESTCTRL	= 0x0818 >> 2;	// Test Data Control Register
const int	TESTLEN		= 0x0820 >> 2;	// Test Data Pattern Length

// Per Pick-Up registers
const int	PU0_REG_BASE	= 0x0880 >> 2;	// PickUp 0 Register Base
const int	PU1_REG_BASE	= 0x0900 >> 2;	// PickUp 1 Register Base
const int	PU2_REG_BASE	= 0x0980 >> 2;	// PickUp 2 Register Base

const int	CONTROL		= 0x0000 >> 2;	// PU General Control and Status register
const int	CYCLE		= 0x0008 >> 2;	// Cycle number
const int	TIME		= 0x0010 >> 2;	// Time in ms from start of cycle
const int	TIME_TBLADDR	= 0x0018 >> 2;	// Last write address in timing table
const int	PLL_FREQUENCY	= 0x0020 >> 2;	// PLL Reference orbit frequency
const int	PLL_FREQDELAY	= 0x0028 >> 2;	// PLL frequency load delay
const int	PLL_PHASEDELAY	= 0x0030 >> 2;	// PLL phase delay
const int	PLL_GAIN	= 0x0038 >> 2;	// PLL gain
const int	DDS_FREQ_MIN	= 0x0040 >> 2;	// PLL DDS minimum frequency
const int	DDS_FREQ_MAX	= 0x0048 >> 2;	// PLL DDS maximum frequency
const int	DIAG_CTRL	= 0x0050 >> 2;	// Diagnostics Control/Status
const int	DIAG_TRIGGER	= 0x0058 >> 2;	// Diagnostics Trigger
const int	TEST		= 0x0060 >> 2;	// Timing Test

// Memory access window
const int	FPGA_DATA_ADDRESS	= 0x200000;	// The FPGA memory
const int	FPGA_MEM		= 0x200000 >> 2;	// The FPGA memory

// SDRAM Data Banks
const int	SDRAM_DATA_PU0		= 0x08;	// PickUp 0 data
const int	SDRAM_DATA_PU1		= 0x09;	// PickUp 0 data
const int	SDRAM_DATA_PU2		= 0x0A;	// PickUp 0 data
const int	SDRAM_DATA_TEST		= 0x0B;	// Test input data
const int	SDRAM_BLOCKRAM		= 0x00;	// Test input data

const int	PU0_BRAM_BASE		= 0;	// PickUp 0 Block Ram ID base
const int	PU1_BRAM_BASE		= 8;	// PickUp 0 Block Ram ID base
const int	PU2_BRAM_BASE		= 16;	// PickUp 0 Block Ram ID base

const int	BRAM_CYCLE_TIMING	= 0;	// Cycle Timing Table
const int	BRAM_CYCLE_INFO		= 1;	// Cycle Information Table
const int	BRAM_PHASE_TABLE	= 2;	// Timing Phase Table
const int	BRAM_SWITCH_TABLE	= 3;	// Timing Switch Table
const int	BRAM_DIAG_TABLE		= 4;	// Diagnostics table
const int	BRAM_BUNCH_MEAN0	= 5;	// Bunch Mean Table #0
const int	BRAM_BUNCH_MEAN1	= 6;	// Bunch Mean Table #1

// Control register bits
const int	CONTROL_INIT				= 0x01;	// Reset
const int	CONTROL_LOOP_CONTROL			= 0x02;	// 
const int	CONTROL_DDS_FREQUENCY_LIMIT_ENABLE	= 0x04;	// 

// Interrupt bits
const int	INT_PU0_CYCLE_START	(1<<0);		// 0 PU #0 CYCLE_START
const int	INT_PU0_CYCLE_STOP	(1<<1);		// 1 PU #0 CYCLE_STOP
const int	INT_PU0_ERROR		(1<<2);		// 2 PU #0 ERROR
const int	INT_PU0_DIAG		(1<<3);		// 3 PU #0 DIAGNOSTIC INFO CAPTURED
const int	INT_PU0_WRITE_FIFO	(1<<4);		// 4 PU #0 SDRAM Write FIFO Half Full
const int	INT_PU0_USER0		(1<<5);		// 5 PU #0 User Interrupts (Switch Table bits 5)
const int	INT_PU0_USER1		(1<<6);		// 6 PU #0 User Interrupts (Switch Table bits 6)
const int	INT_PU0_USER2		(1<<7);		// 7 PU #0 User Interrupts (Switch Table bits 7)

const int	INT_PU1_CYCLE_START	(1<<8);		// 8 PU #1 CYCLE_START
const int	INT_PU1_CYCLE_STOP	(1<<9);		// 9 PU #1 CYCLE_STOP
const int	INT_PU1_ERROR		(1<<10);	// 10 PU #1 ERROR
const int	INT_PU1_DIAG		(1<<11);	// 11 PU #1 DIAGNOSTIC INFO CAPTURED
const int	INT_PU1_WRITE_FIFO	(1<<12);	// 12 PU #1 SDRAM Write FIFO Half Full
const int	INT_PU1_USER0		(1<<13);	// 13 PU #1 User Interrupts (Switch Table bits 5)
const int	INT_PU1_USER1		(1<<14);	// 14 PU #1 User Interrupts (Switch Table bits 6)
const int	INT_PU1_USER2		(1<<15);	// 15 PU #1 User Interrupts (Switch Table bits 7)

const int	INT_PU2_CYCLE_START	(1<<16);	// 16 PU #2 CYCLE_START
const int	INT_PU2_CYCLE_STOP	(1<<17);	// 17 PU #2 CYCLE_STOP
const int	INT_PU2_ERROR		(1<<18);	// 18 PU #2 ERROR
const int	INT_PU2_DIAG		(1<<19);	// 19 PU #2 DIAGNOSTIC INFO CAPTURED
const int	INT_PU2_WRITE_FIFO	(1<<20);	// 20 PU #2 SDRAM Write FIFO Half Full
const int	INT_PU2_USER0		(1<<21);	// 21 PU #2 User Interrupts (Switch Table bits 5)
const int	INT_PU2_USER1		(1<<22);	// 22 PU #2 User Interrupts (Switch Table bits 6)
const int	INT_PU2_USER2		(1<<23);	// 23 PU #2 User Interrupts (Switch Table bits 7)


enum TimingSig		{
			TimingSigClock=0x01, TimingSigCycleStart=0x02, TimingSigCycleStop=0x04,
			TimingSigCalStart=0x08, TimingSigCalStop=0x10, TimingSigInjection=0x20,
			TimingSigHChange=0x40, TimingSigFRef=0x80
			};

#define	DEBUG	0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

int		BufferSize	= (1024*1024*2);
int		MemoryArea	= SDRAM_DATA_TEST;

char		driver[] = "/dev/admxrc0";
char		fpgafile[] = "tms-fpga.bit";

ADMXRC2_HANDLE		card;
ADMXRC2_STATUS		status;
ADMXRC2_DMADESC		dmaDesc;
ADMXRC2_CARD_INFO	ocardInfo;
ADMXRC2_SPACE_INFO	spInfo;
volatile uint*		fpga;

char*			recvbuf;

void hd8(void* data, int n){
	unsigned char* d = (unsigned char*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%2.2x ", *d++);
		if((i & 0xF) == 0xF)
			printf("\n");
	}
	printf("\n");
}

void hd32(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7)
			printf("\n");
	}
	printf("\n");
}

void hd64(void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i = 0;
	
	printf("%8.8x: ", i/2);
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7){
			printf("\n");
			printf("%8.8x: ", (i+1)/2);
		}
	}
	printf("\n");
}

int readBlock(){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA\n");	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return BufferSize;
}

int writeBlock(){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[MEM_REG] = MemoryArea;
	fpga[ADDR_REG] = 0;
	
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA: To %x\n", fpgaAddress);	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, BufferSize, fpgaAddress, ADMXRC2_PCITOLOCAL, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return BufferSize;
}

void writeTestPattern(){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		p[n] = n;
	}
	
	writeBlock();
}

void readTestPattern(){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	readBlock();
	
	for(n = 0; n < BufferSize / sizeof(uint32_t); n++){
		if(p[n] != n){
			printf("Error at: %x = %x\n", n, p[n]);
			hd32(recvbuf, 64);
			break;
		}
	}
}

void readMem(){
	readBlock();
	hd32(recvbuf, 32);
}

void testm1(){
	writeTestPattern();
	readTestPattern();

//	readMem();
	printf("\n");
}

int fpgaWriteData64(void* fromAddress, uint32_t fpgaOffset, int nbytes){
	ADMXRC2_STATUS	status;
	uint32_t	dmaMode;
	
	dmaMode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType,
				  ADMXRC2_IOWIDTH_64, 0,
				  ADMXRC2_DMAMODE_USEREADY |
				  ADMXRC2_DMAMODE_USEBTERM |
				  ADMXRC2_DMAMODE_BURSTENABLE);
				  
	memcpy(recvbuf, fromAddress, nbytes);

	status = ADMXRC2_DoDMA(card, dmaDesc, 0, nbytes,
			FPGA_DATA_ADDRESS + fpgaOffset, ADMXRC2_PCITOLOCAL, 0, dmaMode, 0, 0, 0);

	return status;
}

int fpgaReadData64(uint32_t fpgaOffset, void* toAddress, int nbytes){
	ADMXRC2_STATUS	status;
	uint32_t	dmaMode;
	
	dmaMode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType,
				  ADMXRC2_IOWIDTH_64, 0,
				  ADMXRC2_DMAMODE_USEREADY |
				  ADMXRC2_DMAMODE_USEBTERM |
				  ADMXRC2_DMAMODE_BURSTENABLE);
				  
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, nbytes,
			FPGA_DATA_ADDRESS + fpgaOffset, ADMXRC2_LOCALTOPCI, 0, dmaMode, 0, 0, 0);

	memcpy(toAddress, recvbuf, nbytes);

	return status;
}

void pupeLoadTestData(){
	FILE*		file;
	uint32_t	data[1024];
	int		nb;

	printf("Load test data\n");	
	fpga[TESTCTRL] = 0;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;
	fpga[MEM_REG] = SDRAM_DATA_TEST;
	fpga[ADDR_REG] = 0;

	// Delay needed so that TEST DATA engine has released access to SDRAM
	usleep(1000);

	if(!(file = fopen("beam3-437000-8.psd", "r"))){
		fprintf(stderr, "Cannot open file beam3-437000-8.psd\n");
		exit(1);
	}
	
	if((nb = fread(data, 1, sizeof(data), file)) <= 0){
		fprintf(stderr, "Error: Reading data items from file\n");
		exit(1);
	}
	fpgaWriteData64(data, 0, nb);
	
	fclose(file);
	
	fpga[TESTLEN] = nb / 8;
	fpga[TESTCTRL] = (1 << 25);
	fpga[ADC] = 0x00;
	fpga[PU0_REG_BASE + TEST] = fpga[PU0_REG_BASE + TEST] | 0x8000;
}

void pupeLoadStatePhaseTables(){
	int		n;
	uint32_t	states[] = { 0xe31ef15, 0xeee2f1d, 0xe3eef15, 0x4eeef1d, 0xeeeef15, 0x00 };
	uint8_t		phase0[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase1[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xcb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase2[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa };
	uint8_t		phase3[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xcb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x4,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4b,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb,0xb };
	uint8_t		phase4[] = { 0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xcb,0xcb,0xcb,0xca,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0xa,0xa,0xa,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xc,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0xd,0x4b,0x4b,0x4b,0x4a,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3,0x3 };
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	
	printf("Load state/phase tables\n");	
	fpga[regBase + PLL_PHASEDELAY] = 128;
	fpga[regBase + PLL_FREQUENCY]  = 30030411;
	fpga[regBase + PLL_FREQDELAY]  = 0;
	fpga[regBase + PLL_GAIN]       = (9 << 16) | 4096;
	fpga[regBase + DDS_FREQ_MIN]   = 0;
	fpga[regBase + DDS_FREQ_MAX]   = 0;
	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_SWITCH_TABLE;
	fpgaWriteData64(states, 0, sizeof(states));

	fpga[IMEM_REG] = brBase + BRAM_PHASE_TABLE;
	fpgaWriteData64(phase0, 0 * 512, 512);
	fpgaWriteData64(phase1, 1 * 512, 512);
	fpgaWriteData64(phase2, 2 * 512, 512);
	fpgaWriteData64(phase3, 3 * 512, 512);
	fpgaWriteData64(phase4, 4 * 512, 512);

}

void pupeSetDiagnostics(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint32_t	v;
	
	printf("Start Diagnostics capture\n");	
	v = (0 << 2);		// Source
	v |= (0 << 4);		// Trigger Store
	v |= (0 << 8);		// Clock Select
	v |= (0 << 16);		// Post trigger delay
	v |= 0x1;		// Enable

	fpga[regBase + DIAG_TRIGGER]	= TimingSigInjection;
	fpga[regBase + DIAG_CTRL]	= v;
	
}

int	getBitValue(uint64_t value, int startBit, int nBits, int s){
	int64_t		v;
	uint64_t	bm = (1ULL << nBits) - 1;
	uint64_t	sm = (1ULL << (nBits - 1));
	
	v = (value >> startBit) & bm;
	
	if(s){
		v = -(v & sm) | v;
	}
	
	return v;
}

void pupeDumpDiagnostics(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint64_t	data[1024];
	FILE*		file;
	int		i;
	
	printf("Dump Diagnostics\n");	
	if(! (file = fopen("diag0.txt", "w"))){
		fprintf(stderr, "Unable to open diag0.txt\n");
		exit(1);
	}
	
	// Wait for data
	while((fpga[regBase + DIAG_CTRL] & 0x01)){
		usleep(100000);
	}
	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_DIAG_TABLE;
	fpgaReadData64(0, data, sizeof(data));
	
	// Now dump data
//	fprintf(file, "FrefIn PllMsb PhaseTableMsb Lo1 Sigma Lo2 Gate Blr Mean1 Mean2 RfSelect1 RfSelect2 SelFilter SwitchState DdsFreq\n");
	for(i = 0; i < 1024; i++){
		fprintf(file, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n",
			getBitValue(data[i], 63, 1, 0),		// FREF
			!getBitValue(data[i], 62, 1, 0),	// PllMsb (FRef Reference)
			!getBitValue(data[i], 61, 1, 0),	// PhaseTableMsb (Local FREF)
			getBitValue(data[i], 60, 1, 0),		// LO1
//			getBitValue(data[i], 57, 3, 1),		// Sigma (3 bits)
			getBitValue(data[i], 32, 12, 1),	// Sigma (12 bits)
			getBitValue(data[i], 56, 1, 0),		// LO2

			getBitValue(data[i], 49, 1, 0),		// Gate
			getBitValue(data[i], 48, 1, 0),		// Blr
			getBitValue(data[i], 50, 1, 0),		// Mean1
			getBitValue(data[i], 51, 1, 0),		// Mean2

			getBitValue(data[i], 52, 1, 0),		// RfSelect1
			getBitValue(data[i], 53, 1, 0),		// RfSelect2
			getBitValue(data[i], 54, 1, 0),		// SelFilter

			getBitValue(data[i], 44, 4, 0),		// SwitchState

			getBitValue(data[i], 0, 32, 1)		// DDS_FREQ
			);
	}
	fclose(file);
}

void pupeCycle(){
	printf("Perform Pupe Cycle\n");	
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStart;	// Cycle Start
	usleep(1000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStart;	// Cycle Start

	usleep(100000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigInjection;		// Injection
	usleep(1000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigInjection;	// Cycle Start

	usleep(200000);
	fpga[TIMING_IO] = fpga[TIMING_IO] | TimingSigCycleStop;		// Cycle Stop
	usleep(1000);
	fpga[TIMING_IO] = fpga[TIMING_IO] & ~TimingSigCycleStop;	// Cycle Start

	usleep(100000);
}

const int	FpgaCycleInfoTableSize		= 512;
const int	FpgaCycleTimingTableSize	= 4096;
const int	FpgaCycleTimingTableMask	= (4096-1);

void pupeDataView(){
	uint32_t	regBase = PU0_REG_BASE;
	uint32_t	brBase = PU0_BRAM_BASE;
	uint32_t	ocycleInfoTable[FpgaCycleInfoTableSize];
	uint64_t	ocycleTimingTable[FpgaCycleTimingTableSize];
	uint64_t	odata[1024];
	uint32_t	cycleTimingAddress = 0;
	uint64_t	cycleTimingEntry;
	uint32_t	cycleDataAddress;
	uint32_t	cycle = 0;
	int		i;

	printf("View Data\n");	
	fpga[MEM_REG] = SDRAM_BLOCKRAM;
	fpga[IMEM_REG] = brBase + BRAM_CYCLE_INFO;
	fpgaReadData64(0, ocycleInfoTable, sizeof(ocycleInfoTable));

	fpga[IMEM_REG] = brBase + BRAM_CYCLE_TIMING;
	fpgaReadData64(0, ocycleTimingTable, sizeof(ocycleTimingTable));

	cycleTimingAddress = ocycleInfoTable[(((cycle & 0x03)) << 4) + 1] & 0x00FFFFFF;	// Injection entry

	// Get Cycle Data Table Address
	cycleTimingAddress &= FpgaCycleTimingTableMask;
	cycleTimingEntry = ocycleTimingTable[cycleTimingAddress];
	cycleDataAddress = cycleTimingEntry >> 32;

	printf("CycleInfoTable:\n");
	hd32(ocycleInfoTable, 4 * 16);

	printf("CycleTimingAddress: %u(%x) CycleTime: %u CycleDataAddress: %u(%x)\n", cycleTimingAddress, cycleTimingAddress, uint32_t(cycleTimingEntry & 0xFFFFFFFF), cycleDataAddress, cycleDataAddress);
	printf("CycleTimingTable:\n");
	hd64(&ocycleTimingTable[cycleTimingAddress], 16);

	// Fetch Data (Only page 0)
	fpga[MEM_REG] = SDRAM_DATA_PU0;
	fpga[ADDR_REG] = 0x00;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;
	fpgaReadData64(cycleDataAddress * 8, odata, sizeof(odata));
	
	printf("CycleDataTable:\n");
	hd64(odata, 64);
	
	for(i = 0; i < 16; i++){
		printf("Data: %d,%d,%d\n", getBitValue(odata[i], 0, 16, 1), getBitValue(odata[i], 16, 16, 1), getBitValue(odata[i], 32, 16, 1));
	}
}

int main(int argc, char *argv[])
{
//	float		clock_freq0 = 40.0e6f;
	float		clock_freq0 = 50.0e6f;
	float		clock_freq1 = 125.0e6f;
	ADMXRC2_STATUS	status;
	int		data;
	
	recvbuf = (char*)ADMXRC2_Malloc(BufferSize);
	memset(recvbuf, 0x55, BufferSize); 

	status = ADMXRC2_OpenCard(0, &card);
	if(status != ADMXRC2_SUCCESS) {
		printf("Failed to open driver (%s) Status (%x)\n", driver, status);
		return 1;
	}

	if((status = ADMXRC2_GetCardInfo(card, &ocardInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to get info: %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}

	if((status = ADMXRC2_GetSpaceInfo(card, 0, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}
	fpga = (volatile uint*)spInfo.VirtualBase;

	// Set up Clocks
	status = ADMXRC2_SetClockRate(card, 0, clock_freq0, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set LCLK to %.1fMHz\n", clock_freq0 / 1.0e6f);
		return 1;
	}

	status = ADMXRC2_SetClockRate(card, 1, clock_freq1, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set MCLK to %.1fMHz\n", clock_freq1 / 1.0e6f);
		return 1;
	}

	printf("Set Clocks to: Lclk: %.2fMHz MClk: %.2fMHz\n", clock_freq0 / 1.0e6f, clock_freq1 / 1.0e6f);
        printf("Programming FPGA: %s\n", fpgafile);
        if((status = ADMXRC2_ConfigureFromFile(card, fpgafile)) != ADMXRC2_SUCCESS) {
                printf("Unable to load fpga: status(%x)\n", status);
                return 1;
        }

	status = ADMXRC2_SetupDMA(card, recvbuf, BufferSize, 0, &dmaDesc);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	// Wait a bit
	sleep(1);

	printf("FPGA Firmware: %x\n", fpga[FIRMWARE]);

	// Initialise
	fpga[PU0_REG_BASE + CONTROL] = CONTROL_INIT;
	usleep(1000);
	fpga[PU0_REG_BASE + CONTROL] = 0x00;
	usleep(1000);
	fpga[ADC] = 0x00;
	fpga[TIMING_IO] = 0xFF00;
	
	pupeLoadTestData();
	pupeLoadStatePhaseTables();
	pupeSetDiagnostics();
	
	pupeCycle();
	
	pupeDumpDiagnostics();
	pupeDataView();
	
	// Close the device
	ADMXRC2_CloseCard(card);

	return 0;
}
