/*******************************************************************************
 *	BoapMc.h	BoapMc RPC protocol
 *	T.Barnaby,	BEAM Ltd,	2012-11-14
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#ifndef	BoapMc_HH
#define	BoapMc_HH

#include <BTypes.h>
#include <BMutex.h>
#include <BSemaphore.h>
#include <BQueue.h>
#include <BFifo.h>
#include <BComms.h>

// Main BOAP Types
enum BoapMcType		{ BoapMcTypeRequest = 0x00, BoapMcTypeReply = 0x80 };

// BoapMc packet header
struct BoapMcPacketHead {
	BUInt8		length;
	BUInt8		addressTo;
	BUInt8		addressFrom;
	BUInt8		cmd;
	BUInt16		error;
	BUInt16		checksum;
} __attribute__((aligned(8),packed));

// BoapMc packet
class BoapMcPacket {
public:
	BoapMcPacketHead	head;
	char			data[256 - sizeof(BoapMcPacketHead)];
};

/*******************************************************************************
 *	Base for all client interface objects
 *******************************************************************************
 */
class BoapMcClientObject {
public:
			BoapMcClientObject(BComms& comms);
	virtual		~BoapMcClientObject();

	void		setAddress(BUInt8 addressTo, BUInt8 addressFrom);
	BUInt32		getApiVersion();			///< Returns the API version

protected:
	BError		performCall();				///< Performs a RPC call to the named service
	BError		performSend();				///< Performs a send to the named service
	BError		performRecv();				///< Performs a receive

	BUInt32		oapiVersion;
	BComms&		ocomms;
	BUInt8		oaddressTo;
	BUInt8		oaddressFrom;
	BoapMcPacket	opacket;
};

class BoapMcSignalObject {
public:
			BoapMcSignalObject(BComms& comms);

protected:
	BError		performSend(BoapMcPacket& tx);			// Performs a send to the named service
	BComms&		ocomms;
};


/*******************************************************************************
 *	Base for server communications objects
 *******************************************************************************
 */
class BoapMcServiceObject {
public:
			BoapMcServiceObject();
	virtual		~BoapMcServiceObject();

	virtual BError	process(BoapMcPacket& rx, BoapMcPacket& tx);
	virtual BError	processEvent(BoapMcPacket& rx);
protected:
	BError		sendEvent(BoapMcPacket& tx);
	BUInt32		oapiVersion;
};

/*******************************************************************************
 *	Base for bidirectional communications objects
 *******************************************************************************
 */
class BoapMcComms {
public:
			BoapMcComms(Bool threaded = 0, BUInt rxQueueSize = 4);
	virtual		~BoapMcComms();

	void		setCommsMode(Bool slave, BUInt txQueueSize);		///< Sets slave mode
	void		setComms(BComms& comms);				///< Sets the communications interface to use
	void		setComms(BComms* comms);				///< Sets the communications interface to use
	void		setAddress(BUInt8 addressTo, BUInt8 addressFrom);	///< Sets the to and from addresses
	BUInt32		getApiVersion();					///< Returns the API version
	BUInt32		setTimeout(BUInt32 timeoutUs);				///< Sets the call timeout returning the current value

	virtual BError	processRx(BTimeout timeoutUs = BTimeoutForever);	///< Process any RX packets queuing them as needed
	virtual BError	processRequests(BTimeout timeoutUs = BTimeoutForever);	///< Check and process all requests
	virtual BError	processRequest(BTimeout timeoutUs = BTimeoutForever);	///< Check and process any request
	virtual BError	processPacket(BoapMcPacket& rx, BoapMcPacket& tx);	///< Process a recieved packet

protected:
	// Communications functions
	BError		performCall();						///< Performs a RPC call to the remote side
	BError		performSend();						///< Performs a RPC send to the remote side

	BError		packetSend(BoapMcPacket& packet);			///< Receives a packet
	BError		packetRecv(BoapMcPacket& packet);			///< Receives a packet

	Bool			othreaded;
	BMutex			olockCall;				///< Lock for RPC calls. Only one at a time
	BMutex			olockTx;				///< Lock for TX.
	BComms*			ocomms;
	BUInt32			oapiVersion;
	Bool			oslave;					///< Set slave mode
	BUInt32			otimeout;				///< The timeout in us for calls
	BUInt8			oaddressTo;
	BUInt8			oaddressFrom;
	BoapMcPacket		opacket;				///< Packet RX buffer
	BoapMcPacket		opacketTx;				///< Packet TX buffer for calls
	BoapMcPacket		opacketRx;				///< Packet RX buffer for calls
	BSemaphore		opacketRxSema;				///< Wait RX semaphore
	BoapMcPacket		opacketReqTx;				///< Packet TX buffer for requests
	BoapMcPacket		opacketReqRx;				///< Packet RX buffer for requests
	BQueue<BoapMcPacket>	opacketReqQueue;			///< Packet RX buffer queue for requests

	BFifo<BoapMcPacket>	opacketTxQueue;				///< Packet TX Queue
	BSemaphoreCount		opacketTxQueueWriteNum;			///< Packet TX Queue number
	BSemaphore		opacketTxSema;				///< Wait for TX semaphore
};


#endif
