/*
** File: model_admxrc5t1.c
** Project: ADMXRC2 module driver
** Purpose: ADM-XRC-5T1 specific code, including the function to bootstrap
**          the device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include <core/adb1.h>
#include <core/admxrc5t1.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdmxrc5t1Rev0* pVpd = (VpdAdmxrc5t1Rev0*)pVpdBuffer;
  unsigned int order;

  if (bankIndex >= 3) {
    return ADMXRC2_INVALID_INDEX;
  }

  if (bankIndex < 2) {
    order = pVpd->sdram[bankIndex].order;
    pIoctl->out.width = 64U;
    pIoctl->out.typeMask = ADMXRC2_RAM_SDRAM_DDR2;
  } else {
    bankIndex -= 2;
    order = pVpd->ssram[bankIndex].order;
    pIoctl->out.width = 36U;
    pIoctl->out.typeMask = ADMXRC2_RAM_SRAM_DDR2;
  }
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.bFitted = order ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdmxrc5t1Rev0* pVpd = (VpdAdmxrc5t1Rev0*)pVpdBuffer;
  unsigned int i;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = pVpd->serialNumber;
  pIoctl->out.numMemoryBank = 3;
  pIoctl->out.memoryBankFitted = 0;
  for (i = 0; i < 2; i++) {
    if (pVpd->sdram[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << i);
    }
  }
  for (; i < 3; i++) {
    if (pVpd->ssram[i - 2].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << i);
    }
  }
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdmxrc5t1Rev0* pVpd = (VpdAdmxrc5t1Rev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1136");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  switch (pVpd->tempGrade) {
  default:
    pIoctl->out.tempGrade = (uint32_t)-1;
    break;
  }
  pIoctl->out.scd = pVpd->scd;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdmxrc5t1(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdmxrc5t1: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 4U;
  pDevCtx->configureDmaAddress = 0x800020U; /* TODO - should make this dynamic, i.e. based on current value of LS1B register */
  pDevCtx->configureDmaMode =
    ADB1_DMA_CFG_32BIT | ADB1_DMA_CFG_USE_READY | ADB1_DMA_CFG_USE_BTERM |
    ADB1_DMA_CFG_ENABLE_BURST | ADB1_DMA_CFG_FIXED_LOCAL;
}
