/*
** File: info.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for returning information about a
**          card. None of these IOCTL handlers modifies the state of a card,
**          so they can be considered "read-only".
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "info.h"

DfIoStatus
ioctlGetBankInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl = (IOCTLS_ADMXRC2_GETBANKINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  ADMXRC2_STATUS methodStatus;
  const void* pVpd;
  unsigned int bankIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  bankIndex = pIoctl->in.index;

  pVpd = pDevCtx->coreInterface.pGetVpdPointer(pDevCtx->pCoreContext);
  if (NULL == pVpd) {
    /* No VPD, return card info structure that is zeroes except for model field */
    dfDebugPrint(0, ("+++ ioctlGetBankInfo: no VPD\n"));
    status = DfIoStatusError(ADMXRC2_FAILED);
    return status;
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  methodStatus = pDevCtx->methods.pGetBankInfo(pDevCtx, bankIndex, pVpd, pIoctl);
  if (ADMXRC2_SUCCESS != methodStatus) {
    status = DfIoStatusError(methodStatus);
  }
  return status;
}

DfIoStatus
ioctlGetCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl = (IOCTLS_ADMXRC2_GETCARDINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  const void* pVpd;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  /* For reason why 1U is subtracted from model in next line, see definition of CoreModelType */
  pIoctl->out.boardType = pDevCtx->info.model - 1U;
  pIoctl->out.numClock = pDevCtx->coreInterface.pGetNumClockGen(pDevCtx->pCoreContext);
  pIoctl->out.numDmaChannel = pDevCtx->coreInterface.pGetNumDmaChannel(pDevCtx->pCoreContext);
  pIoctl->out.numSpace = pDevCtx->coreInterface.pGetNumWindow(pDevCtx->pCoreContext);
  pIoctl->out.numSensor = pDevCtx->coreInterface.pGetNumSensor(pDevCtx->pCoreContext);
  pIoctl->out.numTargetFpga = pDevCtx->coreInterface.pGetNumTargetFpga(pDevCtx->pCoreContext);

  pVpd = pDevCtx->coreInterface.pGetVpdPointer(pDevCtx->pCoreContext);
  if (NULL == pVpd) {
    /* No VPD, return card info structure that is zeroes except for model field */
    dfDebugPrint(3, ("+++ ioctlGetCardInfo: no VPD\n"));
  } else {
    pDevCtx->methods.pGetCardInfo(pDevCtx, pVpd, pIoctl);
  }

  return status;
}

DfIoStatus
ioctlGetFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl = (IOCTLS_ADMXRC2_GETFPGAINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  ADMXRC2_STATUS methodStatus;
  const void* pVpd;
  unsigned int targetIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  targetIndex = pIoctl->in.index;

  pVpd = pDevCtx->coreInterface.pGetVpdPointer(pDevCtx->pCoreContext);
  if (NULL == pVpd) {
    dfDebugPrint(0, ("+++ ioctlGetFpgaInfo: no VPD\n"));
    status = DfIoStatusError(ADMXRC2_FAILED);
    return status;
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  methodStatus = pDevCtx->methods.pGetFpgaInfo(pDevCtx, targetIndex, pVpd, pIoctl);
  if (ADMXRC2_SUCCESS != methodStatus) {
    status = DfIoStatusError(methodStatus);
  }

  return status;
}

DfIoStatus
ioctlGetDriverVersion(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETDRIVERVERSION* pIoctl = (IOCTLS_ADMXRC2_GETDRIVERVERSION*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  pIoctl->out.major = ADMXRC_VERSION_0;
  pIoctl->out.minor = ADMXRC_VERSION_1;
  pIoctl->out.bugfix = ADMXRC_VERSION_2;
  pIoctl->out.build = ADMXRC_VERSION_3;

  return status;
}
