/*
** File: dma.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for DMA transfers.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "dma.h"
#include "lock.h"
#include <ioctl_admxrc2.h>
#include <admxrc2/types.h>

#define DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_CANCEL (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_NORMAL (6) /* Level of debug messages relating to normal operation */

typedef enum _AcquireState {
  AcquireStateInvalid    = 0, /* Not a legal state (sanity checking) */
  AcquireStateAcquiring  = 1, /* Acquiring ticket */
  AcquireStateCancelling = 2, /* Cancelling acquisition of ticket */
  AcquireStateAcquired   = 3, /* Acquired ticket */
  AcquireStateFailed     = 4  /* Failed to acquire ticket */
} AcquireState;

typedef enum _DmaState {
  DmaStateNotStarted     = 0, /* DMA transfer is not yet started */
  DmaStateTransfer       = 1, /* DMA transfer is ongoing */
  DmaStateCancelling     = 2, /* DMA transfer is being cancelled */
  DmaStateCompleted      = 3  /* DMA transfer is completed */
} DmaState;

typedef enum _TimerState {
  TimerStateNotScheduled = 0, /* Timer is not scheduled */
  TimerStateScheduled    = 1  /* Timer is scheduled */
} TimerState;

typedef enum _CallbackState {
  CallbackStateNoCallback  = 0, /* Client request does not have a cancel function */
  CallbackStateHasCallback = 1  /* Client request has a cancel function */
} CallbackState;

/* Forward declarations */
static void dmaCallback(CoreTicket* pTicket, void* pCallbackContext, CoreDmaStatus dmaStatus);
static void dmaCancelCallback(DfClientRequest* pRequest, DfIoStatus reason);
static void dmaTicketAcquiredCallback(void* pDevice, CoreTicket* pTicket, CoreAcquireStatus result);
static void dmaTimerCallback(DfTimer* pTimer, void* pTimerContext, void* pContextIgnored);

/*
** -----------------------------------------------------------------------------
** Utility code
** -----------------------------------------------------------------------------
*/

static DfIoStatus
mapDescriptionResult(
  DfDescriptionResult result)
{
  switch (result) {
  case DfDescriptionSuccess:
    return DfIoStatusSuccess;

  case DfDescriptionNullPointer:
    return DfIoStatusError(ADMXRC2_NULL_POINTER);

  case DfDescriptionInvalidBuffer:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  case DfDescriptionNoMemory:
    return DfIoStatusError(ADMXRC2_NO_MEMORY);

  case DfDescriptionUnexpectedError:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapDmaStatus(
  CoreDmaStatus dmaStatus)
{
  switch (dmaStatus) {
  case CoreDmaSuccess:
    return DfIoStatusSuccess;

  case CoreDmaHardwareError:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreDmaInvalidChannel:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreDmaInvalidRegion:
    return DfIoStatusError(ADMXRC2_INVALID_REGION);

  case CoreDmaDirectionNotSupported:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreDmaCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreDmaUnexpectedError:
  case CoreDmaInvalidTicket:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapAcquireStatus(
  CoreAcquireStatus acqStatus)
{
  switch (acqStatus) {
  case CoreAcquireSuccess:
    return DfIoStatusSuccess;

  case CoreAcquireMustWait:
    return DfIoStatusError(ADMXRC2_DEVICE_BUSY);

  case CoreAcquireCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreAcquireRemoved:
  case CoreAcquireFailed:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreAcquireInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreAcquireInvalid:
  case CoreAcquireUnexpectedError:
  case CoreAcquireWaiting:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static void
completeRequest(
  DmaTicket* pTicket)
{
  DfClientRequest* pRequest = pTicket->pRequest;
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  boolean_t bDmaLocked = pTicket->bDmaLocked;
  DfBufferDescription* pDescription;
  DfTime now, delta;
  uint64_t microseconds;

  if (pTicket->acquireState == AcquireStateAcquired) {
    pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &pTicket->ticket);
  }
  if (bDmaLocked) {
    dereferenceBufferDescription(pTicket->pDevCtx, pTicket->pClCtx, pTicket->variant.pBuffer);
  } else {
    pDescription = pTicket->dmaParameters.pBufferDescription;
    dfBufferDescriptionUnlock(pDescription);
    dfBufferDescriptionDelete(pDescription);
  }
  dfRequestComplete(pRequest, pTicket->status);
  if (pTicket->flags & ADMXRC2_DMAFLAG_TIMEOUT) {
    dfTimerUninit(&pTicket->timer);
    now = dfTimeGet();
    if (now >= pTicket->timeoutTicks) {
      delta = 0;
    } else {
      delta = pTicket->timeoutTicks - now;
    }
    microseconds = dfTimeToMicroseconds(delta);
    if (bDmaLocked) {
      IOCTLS_ADMXRC2_DODMA* pIoctl = pTicket->ioctl.pLocked;
      pIoctl->out.timeoutUs = (microseconds < pIoctl->in.timeoutUs) ? microseconds : pIoctl->in.timeoutUs;
    } else {
      IOCTLS_ADMXRC2_DODMAIMMEDIATE* pIoctl = pTicket->ioctl.pImmediate;
      pIoctl->out.timeoutUs = (microseconds < pIoctl->in.timeoutUs) ? microseconds : pIoctl->in.timeoutUs;
    }
  }
  dfPoolFree(&pDevCtx->dma.ticketPool, pTicket);
}

DF_DECLARE_INLINE_FUNC(void, dereferenceTicket)(
  DmaTicket* pTicket)
{
  int usage;

  usage = dfAtomicDec(&pTicket->usage);
  dfAssert(usage >= 0);
  if (usage == 0) {
    completeRequest(pTicket);
  }
}

DF_DECLARE_INLINE_FUNC(void, referenceTicket)(
  DmaTicket* pTicket)
{
  (void)dfAtomicInc(&pTicket->usage);
}

DF_DECLARE_INLINE_FUNC(void, setStatus)(
  DmaTicket* pTicket,
  DfIoStatus reason)
{
  if (pTicket->status == DfIoStatusSuccess) {
    pTicket->status = reason;
  }
}

static void
acquireCancel(
  DmaTicket* pTicket)
{
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  DfSpinLockFlags f;
  unsigned int acquireState;

  f = dfSpinLockGet(&pTicket->lock);
  acquireState = pTicket->acquireState;
  switch (acquireState) {
  case AcquireStateAcquiring:
    pTicket->acquireState = AcquireStateCancelling;
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("acquireCancel: Cancelling acquisition, pTicket=%p\n", pTicket));
    /* Attempt to cancel acquisition */
    pDevCtx->coreInterface.pAcquireCancel(pDevCtx->pCoreContext, &pTicket->ticket);
    break;

  case AcquireStateCancelling:
  case AcquireStateAcquired:
  case AcquireStateFailed:
    dfSpinLockPut(&pTicket->lock, f);
    /* Nothing to do. */
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("acquireCancel: Nothing to do, pTicket=%p\n", pTicket));
    break;

  default:
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** acquireCancel: illegal state %u(0x%x), pTicket=%p\n", acquireState, acquireState, pTicket));
    break;
  }
}

static void
dmaCancel(
  DmaTicket* pTicket)
{
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  DfSpinLockFlags f;
  unsigned int dmaState;

  f = dfSpinLockGet(&pTicket->lock);
  dmaState = pTicket->dmaState;
  switch (dmaState) {
  case DmaStateNotStarted:
  case DmaStateCancelling:
  case DmaStateCompleted:
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaCancel: Nothing to do, pTicket=%p\n", pTicket));
    /* Nothing to do */
    break;

  case DmaStateTransfer:
    pTicket->dmaState = DmaStateCancelling;
    dfSpinLockPut(&pTicket->lock, f);
    /* Cancel the DMA on the hardware */
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaCancel: Cancelling on hardware, pTicket=%p\n", pTicket));
    if (!(pTicket->flags & ADMXRC2_DMAFLAG_QUEUETEST)) {
      pDevCtx->coreInterface.pDmaCancel(pDevCtx->pCoreContext, &pTicket->ticket, pTicket->dmaChannel);
    }
    break;
    
  default:
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** dmaCancel: illegal state %u(0x%x), pTicket=%p\n", dmaState, dmaState, pTicket));
    break;
  }
}

static void
timerCancel(
  DmaTicket* pTicket)
{
  DfSpinLockFlags f;
  unsigned int timerState;

  f = dfSpinLockGet(&pTicket->lock);
  timerState = pTicket->timerState;
  switch (timerState) {
  case TimerStateNotScheduled:
    dfSpinLockPut(&pTicket->lock, f);
    break;

  case TimerStateScheduled:
    if (dfTimerCancel(&pTicket->timer)) {
      pTicket->timerState = TimerStateNotScheduled;
      dfSpinLockPut(&pTicket->lock, f);
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("timerCancel: Cancelled timer, pTicket=%p\n", pTicket));
      dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
    } else {
      dfSpinLockPut(&pTicket->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("timerCancel: Failed to cancel timer, pTicket=%p\n", pTicket));
    }
    break;

  default: /* Illegal state */
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** timerCancel: illegal state %u(0x%x), pTicket=%p\n", timerState, timerState, pTicket));
    break;
  }
}

static void
callbackRemove(
  DmaTicket* pTicket)
{
  DfClientRequest* pRequest = pTicket->pRequest;
  DfSpinLockFlags f;
  unsigned int callbackState;

  f = dfSpinLockGet(&pTicket->lock);
  callbackState = pTicket->callbackState;
  switch (callbackState) {
  case CallbackStateNoCallback:
    dfSpinLockPut(&pTicket->lock, f);
    break;

  case CallbackStateHasCallback:
    if (dfRequestClearCallback(pRequest)) {
      pTicket->callbackState = CallbackStateNoCallback;
      dfSpinLockPut(&pTicket->lock, f);
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("callbackRemove: Removed cancel function, pTicket=%p pRequest=%p\n", pTicket, pRequest));
      dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
    } else {
      dfSpinLockPut(&pTicket->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("callbackRemove: Failed to remove cancel function, pTicket=%p pRequest=%p\n", pTicket, pRequest));
    }
    break;

  default: /* Illegal state */
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("callbackRemove: Illegal state %u(0x%x), pTicket=%p pRequest=%p\n", callbackState, callbackState, pTicket, pRequest));
    break;
  }
}

/*
** -----------------------------------------------------------------------------
** Callbacks
** -----------------------------------------------------------------------------
*/

static void
dmaCancelCallback(
  DfClientRequest* pRequest,
  DfIoStatus reason)
{
  DmaTicket* pTicket = (DmaTicket*)dfRequestGetContext(pRequest);
  DfSpinLockFlags f;

  dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaCancelCallback: entered, pRequest=%p pTicket=%p\n", pRequest, pTicket));

  f = dfSpinLockGet(&pTicket->lock);
  dfAssert(pTicket->callbackState == CallbackStateHasCallback);
  pTicket->callbackState = CallbackStateNoCallback;
  pTicket->bCancel = TRUE;
  setStatus(pTicket, DfIoStatusError(ADMXRC2_CANCELLED));
  dfSpinLockPut(&pTicket->lock, f);

  acquireCancel(pTicket);
  dmaCancel(pTicket);
  timerCancel(pTicket);
  dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
}

static void
dmaTicketAcquiredCallback(
  void* pDevice,
  CoreTicket* pCoreTicket,
  CoreAcquireStatus acqStatus)
{
  DmaTicket* pTicket = DF_CONTAINER_OF(pCoreTicket, DmaTicket, ticket);
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  unsigned int dmaChannel;
  CoreDmaStatus dmaStatus;
  DfSpinLockFlags f;

  /* Get the actual resource index from the ticket, in case the DMA channel requested was "ANY" */
  pTicket->dmaChannel = dmaChannel = pTicket->ticket.resources[0].actualIndex;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaTicketAcquiredCallback: entered, pTicket=%p dmaChannel=%u\n", pTicket, dmaChannel));

  f = dfSpinLockGet(&pTicket->lock);
  dfAssert(pTicket->acquireState == AcquireStateAcquiring || pTicket->acquireState == AcquireStateCancelling);
  switch (acqStatus) {
  case CoreAcquireSuccess:
    pTicket->acquireState = AcquireStateAcquired;
    if (pTicket->bCancel) {
      dfSpinLockPut(&pTicket->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaTicketAcquiredCallback: acqStatus=AcquireSuccess, cancel flag set, pTicket=%p\n", pTicket));
      /* Don't need to call dmaCancel(), as DMA transfer cannot yet be started */
      callbackRemove(pTicket);
      timerCancel(pTicket);
    } else {
      dmaStatus = pDevCtx->coreInterface.pDmaInitialize(pDevCtx->pCoreContext, &pTicket->ticket, dmaChannel, &pTicket->dmaParameters, dmaCallback, NULL);
      if (CoreDmaSuccess != dmaStatus) {
        pTicket->dmaState = DmaStateCompleted;
        setStatus(pTicket, mapDmaStatus(dmaStatus));
        dfSpinLockPut(&pTicket->lock, f);
        /* Don't need to call dmaCancel(), as DMA transfer cannot yet be started */
        callbackRemove(pTicket);
        timerCancel(pTicket);
      } else {
        pTicket->dmaState = DmaStateTransfer;
        dfSpinLockPut(&pTicket->lock, f);
        referenceTicket(pTicket);
        dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaTicketAcquiredCallback: acqStatus=AcquireSuccess, starting DMA, pTicket=%p\n", pTicket));
        if (pTicket->flags & ADMXRC2_DMAFLAG_QUEUETEST) {
          dmaCallback(&pTicket->ticket, NULL, CoreDmaSuccess);
        } else {
          pDevCtx->coreInterface.pDmaStart(pDevCtx->pCoreContext, &pTicket->ticket, dmaChannel);
        }
      }
    }
    dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
    break;

  case CoreAcquireCancelled:
    pTicket->acquireState = AcquireStateFailed;
    dfSpinLockPut(&pTicket->lock, f);
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaTicketAcquiredCallback: acqStatus=AcquireCancelled, pTicket=%p\n", pTicket));
    /* Don't need to call dmaCancel(), as DMA transfer cannot yet be started */
    timerCancel(pTicket);
    callbackRemove(pTicket);
    dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
    break;

  default: /* Unexpected acquisition status */
    dfSpinLockPut(&pTicket->lock, f);
    dfAssert(FALSE);
    break;
  }
}

static void
dmaCallback(
  CoreTicket* pCoreTicket,
  void* pCallbackContext, /* ignored */
  CoreDmaStatus dmaStatus)
{
  DmaTicket* pTicket = DF_CONTAINER_OF(pCoreTicket, DmaTicket, ticket);
  DfSpinLockFlags f;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCallback: entered, pTicket=%p dmaChannel=%u\n", pTicket, pTicket->dmaChannel));

  f = dfSpinLockGet(&pTicket->lock);
  dfAssert(pTicket->dmaState == DmaStateTransfer || pTicket->dmaState == DmaStateCancelling);
  pTicket->dmaState = DmaStateCompleted;
  setStatus(pTicket, mapDmaStatus(dmaStatus));
  dfSpinLockPut(&pTicket->lock, f);

  /* Don't need to call acquireCancel(), as getting here means acquisition must have succeeded */
  timerCancel(pTicket);
  callbackRemove(pTicket);
  dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
}

static void
dmaTimerCallback(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pContextIgnored)
{
  DmaTicket* pTicket = (DmaTicket*)pTimerContext;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pTicket->lock);
  dfAssert(pTicket->timerState == TimerStateScheduled);
  pTicket->timerState = TimerStateNotScheduled;
  pTicket->bCancel = TRUE;
  setStatus(pTicket, DfIoStatusError(ADMXRC2_TIMEOUT));
  dfSpinLockPut(&pTicket->lock, f);

  dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaTimerCallback: pTicket=%p\n", pTicket));
  acquireCancel(pTicket);
  dmaCancel(pTicket);
  callbackRemove(pTicket);
  dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */
}

/*
** -----------------------------------------------------------------------------
** IOCTL handlers
** -----------------------------------------------------------------------------
*/

#if defined(ADMXRC2_CHECK_FLAGS)
static const uint32_t legalFlagsMask = ADMXRC2_DMAFLAG_DONOTQUEUE | ADMXRC2_DMAFLAG_TIMEOUT | ADMXRC2_DMAFLAG_QUEUETEST;
#endif

DfIoStatus
ioctlDoDma(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_DODMA* pIoctl = (IOCTLS_ADMXRC2_DODMA*)pBuffer;
  DmaTicket* pTicket = NULL;
  CoreAcquireStatus acqStatus;
  unsigned int dmaChannel;
  BufferDescription* pUserBuffer;
  DfBufferDescription* pDescription;
  size_t offset, length;
  uint32_t acquireFlags, flags, direction, mode, hDmaDesc;
  uint32_t localAddress;
  DfSpinLockFlags f;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  dmaChannel = pIoctl->in.channel;
  if (dmaChannel != ADMXRC2_DMACHAN_ANY && dmaChannel >= pDevCtx->coreInterface.pGetNumDmaChannel(pDevCtx->pCoreContext)) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  hDmaDesc = pIoctl->in.hDmaDesc;
  offset = pIoctl->in.offset;
  length = pIoctl->in.length;
  if (0 == length) {
    dfRequestPending(pRequest);
    dfRequestComplete(pRequest, DfIoStatusSuccess);
    return DfIoStatusDeferred;
  }
  if (offset + length < offset) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  mode = pIoctl->in.mode;
  flags = pIoctl->in.dirFlags & 0xFFFFFFU;
  direction = (pIoctl->in.dirFlags >> 24) & 0x1U;
  localAddress = pIoctl->in.local;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif
  if (!(mode & ADMXRC2_DMAMODE_FIXEDLOCAL)) {
    if (localAddress + length < localAddress || localAddress + length > pDevCtx->localAddressLimit) {
      return DfIoStatusError(ADMXRC2_INVALID_REGION);
    }
  } else {
    /* TO DO - validate localAddress based on width of DMA specified in 'mode' */
    uint8_t tmp = (uint8_t)(pDevCtx->localWidth - (localAddress & (pDevCtx->localWidth - 1U)));

    if (localAddress + length < localAddress || localAddress + tmp > pDevCtx->localAddressLimit) {
      return DfIoStatusError(ADMXRC2_INVALID_REGION);
    }
  }
  acquireFlags = (flags & ADMXRC2_DMAFLAG_DONOTQUEUE) ? CORE_ACQUIRE_NOWAIT : 0;
  pUserBuffer = referenceBufferHandle(pDevCtx, pClCtx, hDmaDesc - 1U);
  if (NULL == pUserBuffer) {
    return DfIoStatusError(ADMXRC2_INVALID_DMADESC);
  }
  pDescription = pUserBuffer->pDfBufferDesc;
  if (offset + length > pDescription->length) {
    dereferenceBufferDescription(pDevCtx, pClCtx, pUserBuffer);
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }

  pTicket = dfPoolAlloc(&pDevCtx->dma.ticketPool, DmaTicket);
  if (NULL == pTicket) {
    dereferenceBufferDescription(pDevCtx, pClCtx, pUserBuffer);
    return DfIoStatusError(ADMXRC2_NO_MEMORY);
  }

  dfSpinLockInit(&pTicket->lock);
  dfAtomicSet(&pTicket->usage, 1); /* This thread is currently manipulating the ticket => initial usage = 1 */
  pTicket->pRequest = pRequest;
  pTicket->pDevCtx = pDevCtx;
  pTicket->pClCtx = pClCtx;
  pTicket->dmaChannel = dmaChannel; /* This is updated with "actualIndex" from ticket, in dmaTicketAcquiredCallback */
  pTicket->flags = flags;
  pTicket->bDmaLocked = TRUE;
  pTicket->variant.pBuffer = pUserBuffer;
  pTicket->dmaParameters.pBufferDescription = pDescription;
  pTicket->dmaParameters.bWriteToLocal = (direction == ADMXRC2_PCITOLOCAL) ? TRUE : FALSE;
  pTicket->dmaParameters.bFixedLocalAddress = (mode & ADMXRC2_DMAMODE_FIXEDLOCAL) ? TRUE : FALSE;
  pTicket->dmaParameters.region.offset = offset;
  pTicket->dmaParameters.region.length = length;
  pTicket->dmaParameters.region.localAddress = localAddress;
  pTicket->dmaParameters.modeFlags = mode;
  pTicket->bCancel = FALSE;
  pTicket->status = DfIoStatusSuccess;
  pTicket->callbackState = CallbackStateNoCallback;
  pTicket->dmaState = DmaStateNotStarted;
  pTicket->acquireState = AcquireStateInvalid;
  pTicket->timerState = TimerStateNotScheduled;
  pTicket->ticket.count = 1;
  pTicket->ticket.resources[0].code = CORE_RESOURCE_DMAENGINE;
  pTicket->ticket.resources[0].index = (uint8_t)((dmaChannel == ADMXRC2_DMACHAN_ANY) ? CORE_RESOURCE_INDEX_ANY : (uint8_t)dmaChannel);
  if (flags & ADMXRC2_DMAFLAG_TIMEOUT) {
    /* TO DO - ensure that arithmetic for conversion from microseconds to ticks cannot overflow */
    pTicket->timeoutTicks = dfTimeGet() + dfMicrosecondsToTime(pIoctl->in.timeoutUs);
    pTicket->ioctl.pLocked = pIoctl;
    dfTimerInit(&pTicket->timer, pTicket);
  } else {
    pTicket->ioctl.pLocked = NULL;
  }
  dfRequestSetContext(pRequest, pTicket);

  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &pTicket->ticket);
  if (acqStatus != CoreAcquireSuccess) {
    dfPoolFree(&pDevCtx->dma.ticketPool, pTicket);
    dereferenceBufferDescription(pDevCtx, pClCtx, pUserBuffer);
    dfTimerUninit(&pTicket->timer);
    return mapAcquireStatus(acqStatus);
  }

  /* From here on, this function is permitted to return only DfIoStatusDeferred */
  dfRequestPending(pRequest);

  f = dfSpinLockGet(&pTicket->lock);

  if (dfRequestSetCallback(pRequest, dmaCancelCallback)) {
    pTicket->callbackState = CallbackStateHasCallback;
    referenceTicket(pTicket);

    dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlDoDmaImmediate: acquiring, pRequest=%p pTicket=%p dmaChannel=%u\n", pRequest, pTicket, dmaChannel));

    pTicket->acquireState = AcquireStateAcquiring;
    acqStatus = pDevCtx->coreInterface.pAcquireAsyncPoll(pDevCtx->pCoreContext, &pTicket->ticket, acquireFlags, dmaTicketAcquiredCallback);
    switch (acqStatus) {
    case CoreAcquireWaiting:
    case CoreAcquireSuccess:
      if (flags & ADMXRC2_DMAFLAG_TIMEOUT) {
        pTicket->timerState = TimerStateScheduled;
        dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlDoDmaImmediate: Scheduling timer, pTicket=%p\n", pTicket));
        referenceTicket(pTicket);
        dfTimerSchedule(&pTicket->timer, pTicket->timeoutTicks, dmaTimerCallback, NULL);
      } else {
        pTicket->timerState = TimerStateNotScheduled;
      }
      dfSpinLockPut(&pTicket->lock, f);
      if (CoreAcquireSuccess == acqStatus) {
        /* Acquire was successful, so proceed to start DMA transfer. */
        referenceTicket(pTicket);
        dmaTicketAcquiredCallback(NULL, &pTicket->ticket, acqStatus); 
      }
      break;

    case CoreAcquireMustWait:
    default:
      pTicket->acquireState = AcquireStateFailed;
      setStatus(pTicket, DfIoStatusError(acqStatus == CoreAcquireMustWait ? ADMXRC2_DEVICE_BUSY : ADMXRC2_UNKNOWN_ERROR));
      dfSpinLockPut(&pTicket->lock, f);
      callbackRemove(pTicket);
      break;
    }

  } else {
    setStatus(pTicket, DfIoStatusError(ADMXRC2_CANCELLED));
    dfSpinLockPut(&pTicket->lock, f);
  }

  dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */

  return DfIoStatusDeferred;

}

#if DF_NEED_THUNK
DfIoStatus
ioctlDoDmaThunk(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_DODMA* pIoctl32 = (IOCTLS32_ADMXRC2_DODMA*)pBuffer;
  IOCTLS_ADMXRC2_DODMA ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.timeoutUs = pIoctl32->in.timeoutUs;
  ioctl.in.offset = (size_t)pIoctl32->in.offset;
  ioctl.in.length = (size_t)pIoctl32->in.length;
  ioctl.in.local = pIoctl32->in.local;
  ioctl.in.hDmaDesc = pIoctl32->in.hDmaDesc;
  ioctl.in.channel = pIoctl32->in.channel;
  ioctl.in.mode = pIoctl32->in.mode;
  ioctl.in.dirFlags = pIoctl32->in.dirFlags;
  return ioctlDoDma(pDevCtx, pClCtx, pRequest, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
}
#endif

DfIoStatus
ioctlDoDmaImmediate(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_DODMAIMMEDIATE* pIoctl = (IOCTLS_ADMXRC2_DODMAIMMEDIATE*)pBuffer;
  DmaTicket* pTicket = NULL;
  CoreAcquireStatus acqStatus;
  unsigned int dmaChannel;
  DfBufferDescription* pDescription;
  DfDescriptionResult descResult;
  size_t length;
  uint32_t acquireFlags, flags, direction, mode;
  uint32_t localAddress;
  DfSpinLockFlags f;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  dmaChannel = pIoctl->in.channel;
  if (dmaChannel != ADMXRC2_DMACHAN_ANY && dmaChannel >= pDevCtx->coreInterface.pGetNumDmaChannel(pDevCtx->pCoreContext)) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  length = pIoctl->in.length;
  if (0 == length) {
    dfRequestPending(pRequest);
    dfRequestComplete(pRequest, DfIoStatusSuccess);
    return DfIoStatusDeferred;
  }
  mode = pIoctl->in.mode;
  flags = pIoctl->in.dirFlags & 0xFFFFFFU;
  direction = (pIoctl->in.dirFlags >> 24) & 0x1U;
  localAddress = pIoctl->in.local;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif
  if (!(mode & ADMXRC2_DMAMODE_FIXEDLOCAL)) {
    if (localAddress + length < localAddress || localAddress + length > pDevCtx->localAddressLimit) {
      return DfIoStatusError(ADMXRC2_INVALID_REGION);
    }
  } else {
    /* TO DO - validate localAddress based on width of DMA specified in 'mode' */
    uint8_t tmp = (uint8_t)(pDevCtx->localWidth - (localAddress & (pDevCtx->localWidth - 1U)));

    if (localAddress + length < localAddress || localAddress + tmp > pDevCtx->localAddressLimit) {
      return DfIoStatusError(ADMXRC2_INVALID_REGION);
    }
  }
  acquireFlags = (flags & ADMXRC2_DMAFLAG_DONOTQUEUE) ? CORE_ACQUIRE_NOWAIT : 0;
  descResult = dfBufferDescriptionCreateUser(pIoctl->in.pBuffer, length, &pDescription);
  if (DfDescriptionSuccess != descResult) {
    return mapDescriptionResult(descResult);
  }
  if (!dfBufferDescriptionLock(pDescription)) {
    dfBufferDescriptionDelete(pDescription);
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }

  pTicket = dfPoolAlloc(&pDevCtx->dma.ticketPool, DmaTicket);
  if (NULL == pTicket) {
    dfBufferDescriptionUnlock(pDescription);
    dfBufferDescriptionDelete(pDescription);
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */
  }

  dfSpinLockInit(&pTicket->lock);
  dfAtomicSet(&pTicket->usage, 1); /* This thread is currently manipulating the ticket => initial usage = 1 */
  pTicket->pRequest = pRequest;
  pTicket->pDevCtx = pDevCtx;
  pTicket->pClCtx = pClCtx;
  pTicket->dmaChannel = dmaChannel; /* This is updated with "actualIndex" from ticket, in dmaTicketAcquiredCallback */
  pTicket->flags = flags;
  pTicket->bDmaLocked = FALSE;
  pTicket->variant.pDescription = pDescription;
  pTicket->dmaParameters.pBufferDescription = pDescription;
  pTicket->dmaParameters.bWriteToLocal = (direction == ADMXRC2_PCITOLOCAL) ? TRUE : FALSE;
  pTicket->dmaParameters.bFixedLocalAddress = (mode & ADMXRC2_DMAMODE_FIXEDLOCAL) ? TRUE : FALSE;
  pTicket->dmaParameters.region.offset = 0;
  pTicket->dmaParameters.region.length = length;
  pTicket->dmaParameters.region.localAddress = localAddress;
  pTicket->dmaParameters.modeFlags = mode;
  pTicket->bCancel = FALSE;
  pTicket->status = DfIoStatusSuccess;
  pTicket->callbackState = CallbackStateNoCallback;
  pTicket->dmaState = DmaStateNotStarted;
  pTicket->acquireState = AcquireStateInvalid;
  pTicket->timerState = TimerStateNotScheduled;
  pTicket->ticket.count = 1;
  pTicket->ticket.resources[0].code = CORE_RESOURCE_DMAENGINE;
  pTicket->ticket.resources[0].index = (uint8_t)((dmaChannel == ADMXRC2_DMACHAN_ANY) ? CORE_RESOURCE_INDEX_ANY : (uint8_t)dmaChannel);
  if (flags & ADMXRC2_DMAFLAG_TIMEOUT) {
    /* TO DO - ensure that arithmetic for conversion from microseconds to ticks cannot overflow */
    pTicket->timeoutTicks = dfTimeGet() + dfMicrosecondsToTime(pIoctl->in.timeoutUs);
    pTicket->ioctl.pImmediate = pIoctl;
    dfTimerInit(&pTicket->timer, pTicket);
  } else {
    pTicket->ioctl.pImmediate = NULL;
  }
  dfRequestSetContext(pRequest, pTicket);

  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &pTicket->ticket);
  if (acqStatus != CoreAcquireSuccess) {
    dfPoolFree(&pDevCtx->dma.ticketPool, pTicket);
    dfBufferDescriptionUnlock(pDescription);
    dfBufferDescriptionDelete(pDescription);
    dfTimerUninit(&pTicket->timer);
    return mapAcquireStatus(acqStatus);
  }

  /* From here on, this function is permitted to return only DfIoStatusDeferred */
  dfRequestPending(pRequest);

  f = dfSpinLockGet(&pTicket->lock);

  if (dfRequestSetCallback(pRequest, dmaCancelCallback)) {
    pTicket->callbackState = CallbackStateHasCallback;
    referenceTicket(pTicket);

    dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlDoDmaImmediate: acquiring, pRequest=%p pTicket=%p dmaChannel=%u\n", pRequest, pTicket, dmaChannel));

    pTicket->acquireState = AcquireStateAcquiring;
    acqStatus = pDevCtx->coreInterface.pAcquireAsyncPoll(pDevCtx->pCoreContext, &pTicket->ticket, acquireFlags, dmaTicketAcquiredCallback);
    switch (acqStatus) {
    case CoreAcquireWaiting:
    case CoreAcquireSuccess:
      if (flags & ADMXRC2_DMAFLAG_TIMEOUT) {
        pTicket->timerState = TimerStateScheduled;
        dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlDoDmaImmediate: Scheduling timer, pTicket=%p\n", pTicket));
        referenceTicket(pTicket);
        dfTimerSchedule(&pTicket->timer, pTicket->timeoutTicks, dmaTimerCallback, NULL);
      } else {
        pTicket->timerState = TimerStateNotScheduled;
      }
      dfSpinLockPut(&pTicket->lock, f);
      if (CoreAcquireSuccess == acqStatus) {
        /* Acquire was successful, so proceed to start DMA transfer. */
        referenceTicket(pTicket);
        dmaTicketAcquiredCallback(NULL, &pTicket->ticket, acqStatus); 
      }
      break;

    case CoreAcquireMustWait:
    default:
      pTicket->acquireState = AcquireStateFailed;
      setStatus(pTicket, DfIoStatusError(acqStatus == CoreAcquireMustWait ? ADMXRC2_DEVICE_BUSY : ADMXRC2_UNKNOWN_ERROR));
      dfSpinLockPut(&pTicket->lock, f);
      callbackRemove(pTicket);
      break;
    }

  } else {
    setStatus(pTicket, DfIoStatusError(ADMXRC2_CANCELLED));
    dfSpinLockPut(&pTicket->lock, f);
  }

  dereferenceTicket(pTicket); /* MUST NOT touch pTicket after here */

  return DfIoStatusDeferred;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlDoDmaImmediateThunk(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADMXRC2_DODMAIMMEDIATE* pIoctl32 = (IOCTLS32_ADMXRC2_DODMAIMMEDIATE*)pBuffer;
  IOCTLS_ADMXRC2_DODMAIMMEDIATE ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.timeoutUs = pIoctl32->in.timeoutUs;
  ioctl.in.pBuffer = dfThunkPtr(pIoctl32->in.pBuffer);
  ioctl.in.length = (size_t)pIoctl32->in.length;
  ioctl.in.local = pIoctl32->in.local;
  ioctl.in.channel = pIoctl32->in.channel;
  ioctl.in.mode = pIoctl32->in.mode;
  ioctl.in.dirFlags = pIoctl32->in.dirFlags;
  return ioctlDoDmaImmediate(pDevCtx, pClCtx, pRequest, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
}
#endif

/*
** -----------------------------------------------------------------------------
** Cleanup code; called when a device handle is being closed
** -----------------------------------------------------------------------------
*/

void
cleanupDma(
  Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx)
{
  dfDebugPrint(DEBUGLEVEL_CANCEL, ("cleanupDma: entered, pDevCtx=%p pClCtx=%p\n", pDevCtx, pClCtx)); 
}
