/*
** File: debug.c  
** Project: Linux Driver Framework
** Purpose: Debugging functions
**
** (C) Copyright Alpha Data 2010
*/

#include "df.h"
#include "dflinux.h"

#if DF_DBG_BUILD
int g_dfDebugLevel = 0;
#endif

#if DF_DBG_BUILD

# define BUFSIZE (256)

void
_dfDebugPrint(
  const char* fmt, ...)
{
  va_list	ap;
  char buffer[BUFSIZE];
  char* p;
  size_t n, m = BUFSIZE;
 
  p = buffer;
  n = strlen(g_dfDriverName);
  if (n >= m) {
    n = m;
  }
  memcpy(p, g_dfDriverName, n);
  p += n;
  m -= n;
  if (m) {
    *p++ = ':';
    m--;
  }
  if (m) {
    *p++ = ' ';
    m--;
  }
  if (m) {
    va_start(ap, fmt);
    vsnprintf(p, m, fmt, ap);
    va_end(ap);
  }
  buffer[BUFSIZE - 1] = '\0';

  printk(KERN_ALERT "%s", buffer);
}

static __inline size_t
remaining(
  const char* pPos,
  const char* line,
  size_t max)
{
  size_t n = pPos - line;
  if (n > max) {
    return 0;
  } else {
    return max - n;
  }
}

void
dfDebugDumpMemory(
  int level,
  unsigned int indent,
  unsigned int nColumn,
  unsigned int radix,
  boolean_t bAlign,
  const void* pBuffer,
  size_t n)
{
  uint64_t address, limit, addressCol, m;
  unsigned int addressLo, columnMask;
  uint8_t* p;
  uint8_t* p8;
  char line[257];
  uint8_t columnBuf[256];
  char* pPos;
  uint8_t val8;
  uint16_t val16;
  uint32_t val32;
  uint64_t val64;
  unsigned int i, j, k, chunk, rowBytes;

  if (sizeof(void*) != 4 && sizeof(void*) != 8) {
    dfDebugPrint(0, ("*** dfDebugDumpMemory: strange value '%lu' for 'sizeof(void*)'\n", (unsigned long)sizeof(void*)));
    return;
  }
  if (indent > 128) {
    dfDebugPrint(0, ("*** dfDebugDumpMemory: invalid value '%lu' for indent\n", (unsigned long)indent));
    return;
  }
  switch (nColumn) {
  case 1:
  case 2:
  case 4:
  case 8:
  case 16:
  case 32:
    break;

  default:
    dfDebugPrint(0, ("*** dfDebugDumpMemory: invalid value '%lu' for nColumn\n", (unsigned long)nColumn));
    return;
  }
  if (radix != 1 && radix != 2 && radix != 4 && radix != 8) {
    dfDebugPrint(0, ("*** dfDebugDumpMemory: invalid value '%lu' for radix\n", (unsigned long)radix));
    return;
  }
  
  rowBytes = nColumn * radix;

  /* Make sure everything is aligned */
  columnMask = rowBytes - 1U;
  address = (uint64_t)(uintptr_t)pBuffer;
  limit = address + n;
  if (bAlign) {
    addressCol = address & ~(uint64_t)columnMask;
    addressLo = ((unsigned int)address) & columnMask;
  } else {
    addressCol = address;
    addressLo = 0;
  }
  
  m = limit - address;
  p = (uint8_t*)pBuffer;
  
  pPos = line;
  memset(pPos, ' ', indent);
  pPos += indent;
  snprintf(pPos, remaining(pPos, line, sizeof(line)), "Dump of memory at %p + %lu(0x%lx) bytes:\n",
	   pBuffer, (unsigned long)n, (unsigned long)n);
  dfDebugPrint(level, ("%s", line));
  
  pPos = line;
  if (sizeof(void*) != 8) {
    memset(pPos, ' ', indent + 12);
    pPos += indent + 12;
  } else {
    memset(pPos, ' ', indent + 21);
    pPos += indent + 21;
  }
  for (i = 0; i < nColumn; i += radix) {
    switch (radix) {
    case 0x1U:
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "%02lx ", (unsigned long)i);
      pPos += 3;
      break;

    case 0x2U:
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "  %02lx ", (unsigned long)i);
      pPos += 5;
      break;

    case 0x4U:
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "      %02lx ", (unsigned long)i);
      pPos += 9;
      break;

    case 0x8U:
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "               %02lx ", (unsigned long)i);
      pPos += 18;
      break;
    }
  }
  *pPos++ = '\n';
  *pPos++ = '\0';
  dfDebugPrint(level, ("%s", line));
  
  while (m) {
    chunk = rowBytes - addressLo;
    if (chunk > m) {
      chunk = (unsigned int)m;
    }
    
    memset(columnBuf, 0, rowBytes);
    memcpy(columnBuf + addressLo, p, chunk);
    
    p8 = (uint8_t*)columnBuf;
    
    pPos = line;
    memset(pPos, ' ', indent);
    pPos += indent;
    if (sizeof(void*) != 8) {
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "0x%08lx: ", (unsigned long)(uint32_t)addressCol);
      pPos += 12;
    } else {
      snprintf(pPos, remaining(pPos, line, sizeof(line)), "0x%08lx_%08lx: ",
	       (unsigned long)(uint32_t)(addressCol >> 32U), (unsigned long)(uint32_t)addressCol);
      pPos += 21;
    }
    
    j = addressLo + chunk;
    for (i = 0; i < rowBytes; i += radix) {
      switch (radix) {
      case 0x1U:
	val8 = *p8++;
	if (i >= addressLo && i < j) {
	  snprintf(pPos, remaining(pPos, line, sizeof(line)), "%02lx", (unsigned long)val8);
	  pPos += 2;
	} else {
	  *pPos++ = '.';
	  *pPos++ = '.';
	}
	break;
	
      case 0x2U:
	val16 = dfReadUnaligned16((uint16_t*)p8);
	val16 = dfLe32ToCpu(val16);
	p8 += 2;
	for (k = 2; k != 0;) {
	  k--;
	  if (i + k >= addressLo && i + k < j) {
	    snprintf(pPos, remaining(pPos, line, sizeof(line)), "%02lx", (unsigned long)(uint8_t)(val16 >> (k << 3)));
	    pPos += 2;
	  } else {
	    *pPos++ = '.';
	    *pPos++ = '.';
	  }
	  if (k == 3) {
	    *pPos++ = '_';
	  }
	}
	break;
	
      case 0x4U:
	val32 = dfReadUnaligned32((uint32_t*)p8);
	val32 = dfLe32ToCpu(val32);
	p8 += 4;
	for (k = 4; k != 0;) {
	  k--;
	  if (i + k >= addressLo && i + k < j) {
	    snprintf(pPos, remaining(pPos, line, sizeof(line)), "%02lx", (unsigned long)(uint8_t)(val32 >> (k << 3)));
	    pPos += 2;
	  } else {
	    *pPos++ = '.';
	    *pPos++ = '.';
	  }
	}
	break;
	
      case 0x8U:
	val64 = dfReadUnaligned64((uint64_t*)p8);
	val64 = dfLe64ToCpu(val64);
	p8 += 8;
	for (k = 8; k != 0;) {
	  k--;
	  if (i + k >= addressLo && i + k < j) {
	    snprintf(pPos, remaining(pPos, line, sizeof(line)), "%02lx", (unsigned long)(uint8_t)(val64 >> (k << 3)));
	    pPos += 2;
	  } else {
	    *pPos++ = '.';
	    *pPos++ = '.';
	  }
	  if (k == 4) {
	    *pPos++ = '_';
	  }
	}
	break;
      }
      *pPos++ = ' ';
    }
    
    p8 = (uint8_t*)columnBuf;
    for (i = 0; i < rowBytes; i++) {
      char c = (char)*p8++;
      
      if (isprint(c)) {
	*pPos++ = c;
      } else {
	*pPos++ = '.';
      }
    }
    *pPos++ = '\n';
    *pPos++ = '\0';
    dfDebugPrint(level, (line));
    addressLo = 0;
    addressCol += rowBytes;
    m -= chunk;
    p += chunk;
  }
}

#endif

extern void
dfLinuxCheckDataTypes(
  void)
{
  dfAssert(sizeof(uint8_t) == 1);
  dfAssert(sizeof(uint16_t) == 2 * sizeof(uint8_t));
  dfAssert(sizeof(uint32_t) == 4 * sizeof(uint8_t));
  dfAssert(sizeof(uint32_t) >= sizeof(unsigned int));
  dfAssert(sizeof(uint64_t) == 8 * sizeof(uint8_t));
  dfAssert(sizeof(uint64_t) <= sizeof(unsigned long long));
  dfAssert(sizeof(uint64_t) >= sizeof(size_t));
  dfAssert(sizeof(int8_t) == 1);
  dfAssert(sizeof(int16_t) == 2 * sizeof(int8_t));
  dfAssert(sizeof(int32_t) == 4 * sizeof(int8_t));
  dfAssert(sizeof(int32_t) >= sizeof(int));
  dfAssert(sizeof(int64_t) == 8 * sizeof(int8_t));
  dfAssert(sizeof(size_t) == sizeof(void*));
  dfAssert(sizeof(size_t) >= sizeof(unsigned long));
  dfAssert(sizeof(uintptr_t) == sizeof(void*));
}

#if DF_DBG_BUILD && DF_DBG_ALLOCATION

typedef struct _DfKMallocRecord {
  void* p;
  size_t n;
  unsigned int flags;
  const char* pFile;
  int line;
  struct _DfKMallocRecord* pNext;
} DfKMallocRecord;

typedef struct _DfVMallocRecord {
  void* p;
  size_t n;
  const char* pFile;
  int line;
  struct _DfVMallocRecord* pNext;
} DfVMallocRecord;

typedef struct _DfPoolAllocRecord {
  void* p;
  DfPool* pPool;
  const char* pFile;
  int line;
  struct _DfPoolAllocRecord* pNext;
} DfPoolAllocRecord;

static spinlock_t g_dbgKMallocLock = SPIN_LOCK_UNLOCKED;
static spinlock_t g_dbgVMallocLock = SPIN_LOCK_UNLOCKED;
static spinlock_t g_dbgPoolAllocLock = SPIN_LOCK_UNLOCKED;
static DfKMallocRecord* g_pDbgKMallocList = NULL;
static DfVMallocRecord* g_pDbgVMallocList = NULL;
static DfPoolAllocRecord* g_pDbgPoolAllocList = NULL;

void
dfLinuxRecordKMalloc(
  void* p,
  size_t n,
  int flags,
  const char* pFile,
  int line)
{
  DfKMallocRecord* pRecord = (DfKMallocRecord*)kmalloc(sizeof(DfKMallocRecord), GFP_KERNEL);
  if (NULL == pRecord) {
    dfDebugPrint(0, ("*** dfLinuxRecordKMalloc: couldn't allocate record\n"));
    return;
  }
  pRecord->p = p;
  pRecord->n = n;
  pRecord->flags = flags;
  pRecord->pFile = pFile;
  pRecord->line = line;

  spin_lock(&g_dbgKMallocLock);
  pRecord->pNext = g_pDbgKMallocList;
  g_pDbgKMallocList = pRecord;
  spin_unlock(&g_dbgKMallocLock);
}

void
dfLinuxRecordKFree(
  void* p)
{
  DfKMallocRecord** ppRecord = &g_pDbgKMallocList;
  DfKMallocRecord* pRecord;

  pRecord = *ppRecord;
  while (NULL != pRecord) {
    if (pRecord->p == p) {
      *ppRecord = pRecord->pNext;
      kfree(pRecord);
      return;
    }
    ppRecord = &pRecord->pNext;
    pRecord = *ppRecord;
  }
  dfDebugPrint(0, ("*** dfLinuxRecordKFree: couldn't find record\n"));
}

void
dfLinuxRecordVMalloc(
  void* p,
  size_t n,
  const char* pFile,
  int line)
{
  DfVMallocRecord* pRecord = (DfVMallocRecord*)kmalloc(sizeof(DfVMallocRecord), GFP_KERNEL);
  if (NULL == pRecord) {
    dfDebugPrint(0, ("*** dfLinuxRecordVMalloc: couldn't allocate record\n"));
    return;
  }
  pRecord->p = p;
  pRecord->n = n;
  pRecord->pFile = pFile;
  pRecord->line = line;

  spin_lock(&g_dbgVMallocLock);
  pRecord->pNext = g_pDbgVMallocList;
  g_pDbgVMallocList = pRecord;
  spin_unlock(&g_dbgVMallocLock);
}

void
dfLinuxRecordVFree(
  void* p)
{
  DfVMallocRecord** ppRecord = &g_pDbgVMallocList;
  DfVMallocRecord* pRecord;

  pRecord = *ppRecord;
  while (NULL != pRecord) {
    if (pRecord->p == p) {
      *ppRecord = pRecord->pNext;
      kfree(pRecord);
      return;
    }
    ppRecord = &pRecord->pNext;
    pRecord = *ppRecord;
  }
  dfDebugPrint(0, ("*** dfLinuxRecordVFree: couldn't find record\n"));
}

void
dfLinuxRecordPoolAlloc(
  void* p,
  DfPool* pPool,
  const char* pFile,
  int line)
{
  DfPoolAllocRecord* pRecord = (DfPoolAllocRecord*)kmalloc(sizeof(DfPoolAllocRecord), GFP_KERNEL);
  if (NULL == pRecord) {
    dfDebugPrint(0, ("*** dfLinuxRecordPoolAlloc: couldn't allocate record\n"));
    return;
  }
  pRecord->p = p;
  pRecord->pPool = pPool;
  pRecord->pFile = pFile;
  pRecord->line = line;

  spin_lock(&g_dbgPoolAllocLock);
  pRecord->pNext = g_pDbgPoolAllocList;
  g_pDbgPoolAllocList = pRecord;
  spin_unlock(&g_dbgPoolAllocLock);
}

void
dfLinuxRecordPoolFree(
  void* p)
{
  DfPoolAllocRecord** ppRecord = &g_pDbgPoolAllocList;
  DfPoolAllocRecord* pRecord;

  pRecord = *ppRecord;
  while (NULL != pRecord) {
    if (pRecord->p == p) {
      *ppRecord = pRecord->pNext;
      kfree(pRecord);
      return;
    }
    ppRecord = &pRecord->pNext;
    pRecord = *ppRecord;
  }
  dfDebugPrint(0, ("*** dfLinuxRecordPoolFree: couldn't find record\n"));
}

void
dfLinuxReportLeaks(
  void)
{
  DfKMallocRecord* pKMalloc = g_pDbgKMallocList;
  DfVMallocRecord* pVMalloc = g_pDbgVMallocList;
  DfPoolAllocRecord* pPoolAlloc = g_pDbgPoolAllocList;

  dfDebugPrint(0, ("dfLinuxReportLeaks: Resource leak report:\n"));

  while (pKMalloc != NULL) {
    DfKMallocRecord* pNext;

    dfDebugPrint(0, ("dfLinuxReportLeaks:   kmalloc leak at %s:%d of %llu(0x%llx) bytes \n",
      pKMalloc->pFile, pKMalloc->line, (unsigned long long)pKMalloc->n, (unsigned long long)pKMalloc->n));
    pNext = pKMalloc->pNext;
    kfree(pKMalloc);
    pKMalloc = pNext;
  }

  while (pVMalloc != NULL) {
    DfVMallocRecord* pNext;

    dfDebugPrint(0, ("dfLinuxReportLeaks:   vmalloc leak at %s:%d of %llu(0x%llx) bytes \n",
      pVMalloc->pFile, pVMalloc->line, (unsigned long long)pVMalloc->n, (unsigned long long)pVMalloc->n));
    pNext = pVMalloc->pNext;
    kfree(pVMalloc);
    pVMalloc = pNext;
  }

  while (pPoolAlloc != NULL) {
    DfPoolAllocRecord* pNext;

    dfDebugPrint(0, ("dfLinuxReportLeaks:   pool leak at %s:%d, pPool=%p) bytes \n",
      pPoolAlloc->pFile, pPoolAlloc->line, pPoolAlloc->pPool));
    pNext = pPoolAlloc->pNext;
    kfree(pPoolAlloc);
    pPoolAlloc = pNext;
  }

  dfDebugPrint(0, ("dfLinuxReportLeaks: End of resource leak report.\n"));
}

#endif
