/*
** File: v4clksynth.c  
** Project: ADB3 core driver
** Purpose: Common functions for Virtex-4 DCM clock synthesizer
**
** (C) Copyright Alpha Data 2013
**
** NOTES
**
** 1. Clock word layout is:
**
**   d0[15:0]   M multiplier value - 1
**   d0[31:16]  D divider value - 1
**   d1         (unused)
**   d2         (unused)
*/

#include <df.h>

#include "adb1.h"
#include "v4clksynth.h"
#include "v4clksynth_common.h"

/* Min. & max. CLKFX output frequency for Virtex-4 -10 speed grade */
static const uint32_t g_minFrequency = 31999000U, g_maxFrequency = 210001000U;

V4ClkSynthClockWordStatus
v4ClkSynthClockWord(
  uint32_t flags,
	uint32_t refFrequency,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
	CoreClockWord* pClockWord)
{
	uint32_t m, bestM = 0, d, bestD = 0; /* init. to avoid compiler whine */
	uint32_t f, bestFrequency = 0; /* init. to avoid compiler whine */
	uint32_t error, bestError = (uint32_t)-1;

	dfDebugPrint(5, ("v4ClkSynthClockWord: refFrequency=%lu reqFrequency=" DF_FMT_U64 "\n",
    (unsigned long)refFrequency, (unsigned long long)frequencyReq));

  if (refFrequency == 0 || refFrequency > 268000000) {
		return V4ClkSynthClockWordStatusGeneralFailure;
	}
	if (frequencyReq < g_minFrequency || frequencyReq > g_maxFrequency) {
		return V4ClkSynthClockWordStatusOutOfRange;
	}

	/* Won't overflow assuming (refFrequency <= 268e6) */
	for (d = 1; d <= 32; d++) {
		for (m = 2; m <= 32; m++) {
			f = ((refFrequency >> 1) * m / d) << 1;
      if ((flags & CoreClockMinimumFrequency) && f < frequencyMin) {
        continue;
      }
      if ((flags & CoreClockMaximumFrequency) && f > frequencyMax) {
        continue;
      }
			if (f > (uint32_t)frequencyReq) {
				error = f - (uint32_t)frequencyReq;
			} else {
				error = (uint32_t)frequencyReq - f;
			}
			if (error < bestError) {
				bestError = error;
				bestFrequency = f;
				bestM = m;
				bestD = d;
			}
		}
	}

	if (bestError == (uint32_t)-1) {
		/* Failed to find a suitable value */
		return V4ClkSynthClockWordStatusGeneralFailure;
	}

	pClockWord->frequency = bestFrequency;
	pClockWord->d0 = ((bestD - 1) << 16) | (bestM - 1);

	dfDebugPrint(5, ("v4ClkSynthClockWord: bestFrequency=%lu clockWord=0x%08lx\n", (unsigned long)bestFrequency, (unsigned long)pClockWord->d0));

	return V4ClkSynthClockWordStatusSuccess;
}

boolean_t
v4ClkSynthInit(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  unsigned int lclkRef,
  uint32_t* pFOut )
{
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32, lclkStat3;
  uint16_t lclkD, lclkM;
  uint64_t frequency;

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  dfTimerInit(&pV4ClkSynthCtx->timer, pDevCtx);
  pV4ClkSynthCtx->save.bLastValid = FALSE;

  if (lclkRef == 0 || lclkRef > 268000000) {
    dfDebugPrint(0, ("*** v4ClkSynthInit: reference clock frequency %u Hz is out of range\n", lclkRef));
	}

  /* Figure out current LCLK frequency */
  lclkStat3 = dfPciMemRead32(hModel, &pModelRegs->clocking.others.lclkstat3);
  if (lclkStat3 & 0x4U) {
    val32 = dfPciMemRead32(hModel, &pModelRegs->clocking.others.lclkstat2);
  } else {
    val32 = dfPciMemRead32(hModel, &pModelRegs->clocking.others.lclkstat1);
  }
  lclkM = (uint16_t)(val32 + 1U);
  if (lclkM < 2 || lclkM > 32) {
    dfDebugPrint(0, ("*** v4ClkSynthInit: M multipler value %u is out of range\n", (unsigned int)lclkM));
    return FALSE;
  }
  lclkD = (uint16_t)((val32 >> 16) + 1U);
  if (lclkD < 1 || lclkD > 32) {
    dfDebugPrint(0, ("*** v4ClkSynthInit: D divider value %u is out of range\n", (unsigned int)lclkD));
    return FALSE;
  }
	/* Won't overflow assuming (lclkRef <= 268e6) */
  frequency = ((lclkRef >> 1) * lclkM / lclkD) << 1;
  if (frequency < g_minFrequency || frequency > g_maxFrequency) {
    dfDebugPrint(0, ("*** v4ClkSynthInit: initial LCLK frequency is " DF_FMT_U64 " Hz, out of range\n", (unsigned long long)frequency));
  } else {
    dfDebugPrint(1, ("v4ClkSynthInit: initial LCLK frequency is " DF_FMT_U64 " Hz\n", (unsigned long long)frequency));
  }
  *pFOut = (uint32_t)frequency; /* cast is safe due to above range checks */

  return TRUE;
}

void
v4ClkSynthUninit(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx)
{
  dfTimerUninit(&pV4ClkSynthCtx->timer);
}

static void
onCheckV4ClkSynth(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  V4ClkSynthProgramRequest* pRequest = (V4ClkSynthProgramRequest*)pCallbackContext;
  V4ClkSynthDeviceContext* pV4ClkSynthCtx = pRequest->pV4ClkSynthCtx;
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
	uint32_t val32, mask;
  V4ClkSynthProgramStatus status = V4ClkSynthProgramStatusSuccess;

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;
  val32 = dfPciMemRead32(hModel, &pModelRegs->clocking.others.lclkstat3);

  dfDebugPrint(5, ("onCheckV4ClkSynth: LCLKSTAT3=0x%lx\n", (unsigned long)val32));	
	mask = (val32 & 0x4U) ? 0x2U : 0x1U;
	if (!(val32 & mask)) {
		/* DCM not locked */
		status = V4ClkSynthProgramStatusHardwareError;
  }
  if (val32 & 0x28U) {
    /* Control logic busy or write of DCM parameters timed out */
    status = V4ClkSynthProgramStatusHardwareError;
  }
  if (val32 & 0x10U) {
    /* Error */
    status = V4ClkSynthProgramStatusHardwareError;
  }
  if (V4ClkSynthProgramStatusSuccess == status) {
    pV4ClkSynthCtx->save.bLastValid = TRUE;
    pV4ClkSynthCtx->save.last = pRequest->clockWord;
  }
  pRequest->pCallback(pDevCtx, pV4ClkSynthCtx, pRequest, status);
}

V4ClkSynthProgramStatus
v4ClkSynthProgram(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  V4ClkSynthProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  V4ClkSynthProgramCallback* pCallback)
{
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  pRequest->pV4ClkSynthCtx = pV4ClkSynthCtx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  dfPciMemWrite32(hModel, &pModelRegs->clocking.others.lclkctl, pClockWord->d0 | (0x1U << 31));
  dfPciMemRead32(hModel, &pModelRegs->clocking.others.lclkstat3);
  dfTimerSchedule(&pV4ClkSynthCtx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckV4ClkSynth, (void*)pRequest);

  return V4ClkSynthProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  V4ClkSynthProgramRequest* pRequest,
  V4ClkSynthProgramStatus status)
{
  V4ClkSynthProgramRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, V4ClkSynthProgramRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

V4ClkSynthProgramStatus
v4ClkSynthProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  const CoreClockWord* pClockWord)
{
  V4ClkSynthProgramRequestSync req;
  V4ClkSynthProgramStatus status;

  dfEventInit(&req.ev);
  status = v4ClkSynthProgram(pDevCtx, pV4ClkSynthCtx, &req.request, pClockWord, onProgramDone);
  if (V4ClkSynthProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

extern void
v4ClkSynthSaveState(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx)
{
  /* Nothing to do yet */
}

V4ClkSynthProgramStatus
v4ClkSynthRestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx)
{
  V4ClkSynthProgramStatus status = V4ClkSynthProgramStatusSuccess;

  /* If the clock synthesizer had been programmed before power-down, reprogram it with the same thing */
  if (pV4ClkSynthCtx->save.bLastValid) {
    status = v4ClkSynthProgramSync(pDevCtx, pV4ClkSynthCtx, &pV4ClkSynthCtx->save.last);
  }

  return status;
}

ClockProgramStatus
v4ClkSynthMapProgramStatus(
  V4ClkSynthProgramStatus status)
{
  switch (status) {
  case V4ClkSynthProgramStatusSuccess:
    return ClockProgramSuccess;

  case V4ClkSynthProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case V4ClkSynthProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
v4ClkSynthMapClockWordStatus(
  V4ClkSynthClockWordStatus status)
{
  switch (status) {
  case V4ClkSynthClockWordStatusSuccess:
    return ClockWordSuccess;

  case V4ClkSynthClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case V4ClkSynthClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
