/*
** File: model_adpexrc6tadv.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADPE-XRC-6T-ADV, including function to bootstrap
**          device context.
**
** (C) Copyright Alpha Data 2012
**
** NOTES
**
** 1. Organization of Flash bank 0 (U52) (all offsets in bytes):
**
**  0x0000000 - 0x07FFFFF  Alternate ADB3 PCI-E to OCP Bridge bitstream
**  0x0800000 - 0x0FFFFFF  Default ADB3 PCI-E to OCP Bridge bitstream
**  0x1000000 - 0x10FFFFF  (reserved)
**  0x1100000 - 0x11FFFFF  Customer region
**  0x1200000 - 0x28FFFFF  Target FPGA 0 bitstream
**  0x2900000 - 0x3FFFFFF  Failsafe target FPGA 0 bitstream
**
** 2. Organization of Flash bank 1 (U51) (all offsets in bytes):
**
**  0x0000000 - 0x0FFFFFF  Customer region
**  0x1000000 - 0x10FFFFF  (reserved)
**  0x1100000 - 0x11FFFFF  Customer region
**  0x1200000 - 0x28FFFFF  Target FPGA 1 bitstream
**  0x2900000 - 0x3FFFFFF  Failsafe target FPGA 1 bitstream
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "adpexrc6tadv.h"
#include "admxrc6tx_common.h"
#include "avr_common.h"
#include "avr_flash.h"
#include "avr_sensor.h"
#include "coreclock.h"
#include "coredma.h"
#include "coreif.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"

static CoreSensorInfo g_avrVoltageInfo[] = {
  { "5V supply rail",              0, { 0,  2359 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A1 */
  { "3.3V PCI-E supply rail",      0, { 0,  1311 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A2 */
  { "12V supply rail",             0, { 0,  1966 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A3 */
  { "2.5V supply rail",            0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A4 */
  { "1.8V supply rail",            0, { 0,  6554 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A5 */
  { "1.5V supply rail",            0, { 0,  4325 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A6 */
  { "1V supply rail (Target 0)",   0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A7 */
  { "1V supply rail (Target 1)",   0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* AVR ADC-B1 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrCurrentInfo[] = {
  { "3.3V PCI-E supply current",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }, /* AVR ADC-B2 */
  { "12V supply current",          0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }  /* AVR ADC-B3 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrTempInfo[] = {
  { "uC internal temperature",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Microcontroller internal temp */
  { "Board temp. at Target 0",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* TMP422 internal temp */
  { "Board temp. at Target 1",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* TMP422 internal temp */
  { "Bridge FPGA temp. diode",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Temperature sensor diode in Bridge FPGA */
  { "Target FPGA 0 temp. diode",   0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* External sensor diode in TMP422 */
  { "Target FPGA 1 temp. diode",   0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* External sensor diode in TMP422 */
};

static CoreSensorInfo g_avrMiscInfo[] = {
  { "Total powered on time",       0, { 0, 16384 }, CoreSensorDataDouble, CoreSensorUnitS,    0 }, /* Number of seconds spent powered on */
  { "Event count",                 0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* Number of elapsed events */
};

/* These correspond to 'g_avrVoltageInfo' above, to convert the raw ADC values into Volt */
static AvrVoltageScaling g_avrVoltageScaling[] = {
  { 10U, 0x0FFFU, FALSE, 0x17F5C  /* 5V supply rail,         factor = 0.00146240f */ },
  { 12U, 0x0FFFU, FALSE, 0x17F5C  /* 3.3V PCI-E supply rail, factor = 0.00146240f */ },
  { 14U, 0x0FFFU, FALSE, 0x40000  /* 12V supply rail,  factor = 0.00390625f */ },
  { 16U, 0x0FFFU, FALSE, 0x17F5C  /* 2.5V supply rail,       factor = 0.00146240f */ },
  { 18U, 0x0FFFU, FALSE, 0x17F5C  /* 1.8V supply rail,       factor = 0.00146240f */ },
  { 20U, 0x0FFFU, FALSE, 0x17F5C  /* 1.5V supply rail,       factor = 0.00146240f */ },
  { 22U, 0x0FFFU, FALSE, 0x05D17  /* 1V supply rail (target 0), factor = 0.00035511f */ },
  { 24U, 0x0FFFU, FALSE, 0x05D17  /* 1V supply rail (target 1), factor = 0.00035511f */ }
};

/* These correspond to 'g_avrCurrentInfo' above, to convert the raw ADC values into Volt */
static AvrCurrentScaling g_avrCurrentScaling[] = {
  { 26U, 0x0FFFU, FALSE, 0x13FFF  /* 3.3V PCI-E supply rail, factor = 0.0012207 */  },
  { 28U, 0x0FFFU, FALSE, 0x42AAA  /* 12V supply rail,        factor = 0.00406901 */ }
};

/* These correspond to 'g_avrTempInfo' above, to convert the raw ADC values into deg. C */
static AvrTempScaling g_avrTempScaling[] = {
  { 34U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 36U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 38U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 40U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 42U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 44U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ }
};

static AvrMiscScaling g_avrMiscScaling = {
  { 0U },
  { 7U, TRUE }
};

static void
initSensorInfoController(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0;

  dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfo, sizeof(g_avrVoltageInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrCurrentInfo, sizeof(g_avrCurrentInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrCurrentInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrTempInfo, sizeof(g_avrTempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrMiscInfo, sizeof(g_avrMiscInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
}

static boolean_t
createFakeVpd(
  Adb3CoreDeviceContext* pDevCtx,
  uint32_t serialFromReg)
{
  VpdAdpexrc6tadvRev0* pVpd = &pDevCtx->info.vpd.pData->adpexrc6tadv.rev0;
  int index, bank;

  DF_ZERO_OBJECT(pVpd);
  pVpd->version = 0;
  pVpd->length = sizeof(VpdAdpexrc6tadvRev0);
  if (!pDevCtx->model.adpexrc6tadv.bIsController) {
    pVpd->board.serial = serialFromReg & 0x7FFFFFFFU;
  } else {
    pVpd->board.serial = 0x100U;
  }
  pVpd->board.modifications = 0U;
  pVpd->board.tempGrade = ADPEXRC6TADV_VPD_TEMPGRADE_COMMERCIAL;
  pVpd->board.pcbRev = 0x10U;
  pVpd->board.si5338Ref = 25000000;
  for (index = 0; index < DF_ARRAY_LENGTH(pVpd->target); index++) {
    for (bank = 0; bank < DF_ARRAY_LENGTH(pVpd->target[bank].sdram); bank++) {
      pVpd->target[index].sdram[bank].order = 27; /* 512 MiB per bank */
      pVpd->target[index].sdram[bank].speed = 5333;
      pVpd->target[index].sdram[bank].link = 0;
    }
    pVpd->target[index].fpga.device = 171; /* 6VLX240T */
    dfAssert(sizeof(char) == sizeof(uint8_t));
    dfStrCpy((char*)pVpd->target[index].fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->target[index].fpga.speedGrade), "1");
    dfStrCpy((char*)pVpd->target[index].fpga.stepping, DF_ARRAY_LENGTH(pVpd->target[index].fpga.stepping), "");
    pVpd->target[index].advFitted = 0xFU;
  }
  pVpd->checksum = 0U;

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx,
  uint32_t serialFromReg)
{
  VpdAdpexrc6tadvRev0* pVpd = &pDevCtx->info.vpd.pData->adpexrc6tadv.rev0;
  VpdAdpexrc6tadvRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  int index, bank;

  admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  admxrc6txReadAvrVpdData(pDevCtx, 2, sizeof(vpdLength), &vpdLength);
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    if (vpdLength != sizeof(VpdAdpexrc6tadvRev0)) {
  	  dfDebugPrint(0, ("*** readVpd: invalid VPD length %lu(0x%lx), expecting %lu(0x%lx)\n",
        vpdLength, vpdLength, sizeof(VpdAdpexrc6tadvRev0), sizeof(VpdAdpexrc6tadvRev0)));
      return FALSE;
    }
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  admxrc6txVerifyAvrVpdChecksum(pDevCtx, vpdLength);

  admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->length = vpdLength;
  pVpd->board.serial = dfLe32ToCpu(vpdRaw.board.serial);
  if (!pDevCtx->model.adpexrc6tadv.bIsController) {
    pVpd->board.serial = serialFromReg & 0x7FFFFFFFU;
  }
  pVpd->board.modifications = dfLe32ToCpu(vpdRaw.board.modifications);
  pVpd->board.tempGrade = dfLe8ToCpu(vpdRaw.board.tempGrade);
  pVpd->board.pcbRev = dfLe8ToCpu(vpdRaw.board.pcbRev);
  pVpd->board.si5338Ref = dfLe32ToCpu(vpdRaw.board.si5338Ref);
  for (index = 0; index < DF_ARRAY_LENGTH(pVpd->target); index++) {
    for (bank = 0; bank < DF_ARRAY_LENGTH(pVpd->target[bank].sdram); bank++) {
      pVpd->target[index].sdram[bank].order = dfLe8ToCpu(vpdRaw.target[index].sdram[bank].order);
      pVpd->target[index].sdram[bank].speed = dfLe16ToCpu(vpdRaw.target[index].sdram[bank].speed);
      pVpd->target[index].sdram[bank].link = dfLe16ToCpu(vpdRaw.target[index].sdram[bank].link);
    }
    pVpd->target[index].fpga.device = dfLe16ToCpu(vpdRaw.target[index].fpga.device);
    dfAssert(sizeof(char) == sizeof(uint8_t));
    dfStrCpy((char*)pVpd->target[index].fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->target[index].fpga.speedGrade), (const char*)vpdRaw.target[index].fpga.speedGrade);
    dfStrCpy((char*)pVpd->target[index].fpga.stepping, DF_ARRAY_LENGTH(pVpd->target[index].fpga.stepping), (const char*)vpdRaw.target[index].fpga.stepping);
    pVpd->target[index].advFitted = dfLe8ToCpu(vpdRaw.target[index].advFitted);
  }
  pVpd->checksum = dfLe8ToCpu(vpdRaw.checksum);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
enableAvrRxNotEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.adpexrc6tadv.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.adpexrc6tadv.shadow.intCtl |= ADPEXRC6TADVCTL_V_INTCTL_ADVRXNE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

static void
enableAvrTxEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.adpexrc6tadv.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.adpexrc6tadv.shadow.intCtl |= ADPEXRC6TADVCTL_V_INTCTL_ADVTXE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.adpexrc6tadv.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  return ClockProgramInvalidIndex;
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  return ClockWordInvalidIndex;
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  dfDebugPrint(9, ("admxrc6txFlashSingleBank: bankIndex=%u bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    return;
  }

  admxrc6txFlashBank(pDevCtx, bankIndex ? &pDevCtx->hardware.model.pAdpexrc6tadv->flash1 : &pDevCtx->hardware.model.pAdpexrc6tadv->flash0, bWrite, location, pValue);
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;
  boolean_t bOk;
  uint32_t* pFpgaCtl;

  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (index >= pDevCtx->info.bootstrap.numTargetFpga) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  pFpgaCtl = index ? &pModelRegs->fpga1.ctl : &pModelRegs->fpga0.ctl;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    bOk = TRUE;
    val32 = dfPciMemRead32(hModel, pFpgaCtl);
    if (!(val32 & 0x4U)) {
      dfDebugPrint(9, ("fpgaControl: fpgaStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    bOk = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, pFpgaCtl);
    *pbValue = (val32 & 0x2U) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[index];
    if (*pbValue) {
      val32 |= 0x1U;
    } else {
      val32 &= ~0x1U;
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, pFpgaCtl, val32);
    dfPciMemRead32(hModel, pFpgaCtl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;
  uint8_t* pFpgaSelectMap;

  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (index >= pDevCtx->info.bootstrap.numTargetFpga) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  pFpgaSelectMap = index ? &pModelRegs->fpga1.selectMap[0] : &pModelRegs->fpga0.selectMap[0];

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, pFpgaSelectMap, val8);
    }

    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)pFpgaSelectMap, val32);
      length32 -= 4;
    }

    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, pFpgaSelectMap, val8);
    }

    dfPciMemRead32(hModel, (uint32_t*)pFpgaSelectMap); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, pFpgaSelectMap);
      *pBuffer8++ = val8;
    }
  }

  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;
  uint32_t serialReg = 0U;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        pDevCtx->info.model = CoreModelAdpexrc6tadvControl;
      } else {
        serialReg = dfPciMemRead32(hModel, &pModelRegs->serial);
        /* Fix for board revision 1 bug, where there are no connections between controller
        ** and target 1; this results in serial number register always returning 0 */
        if (0 == serialReg) {
          /* Fix up serial number so it is interpreted as target 1, board SN #1*/
          dfDebugPrint(0, ("+++ initHardware: Fixing up serial number of 0. This is probably a revision 1 ADPE-XRC-6T-ADV.\n"));
          serialReg = (0x2U << 24) | 1U;
        }
        if (((serialReg >> 24) & 0x3U) == 0x1U) {
          pDevCtx->info.model = CoreModelAdpexrc6tadvTarget0;
        } else {
          pDevCtx->info.model = CoreModelAdpexrc6tadvTarget1;
        }
      }

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.adpexrc6tadv.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        pDevCtx->model.adpexrc6tadv.shadow.intCtl = ADMXRC6Tx_V_INTCTL_SYSENABLE;
        dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
      } else {
        pDevCtx->model.adpexrc6tadv.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
        dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVTGT_V_INTCTL_LEVELBITS);
      }
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        /* Initialize FPGACTL register for 8-bit SelectMap port */
        pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[0] = 0;
        dfPciMemWrite32(hModel, &pModelRegs->fpga0.ctl, pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[0]);
        pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[1] = 0;
        dfPciMemWrite32(hModel, &pModelRegs->fpga1.ctl, pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[1]);
		    dfPciMemRead32(hModel, &pModelRegs->fpga1.ctl);
      }

      /* Identify Flash devices */
      for (i = 0; i < pDevCtx->info.bootstrap.numFlashBank; i++) {
        admxrc6txFlashResetBank(pDevCtx, i ? &pDevCtx->hardware.model.pAdpexrc6tadv->flash1 : &pDevCtx->hardware.model.pAdpexrc6tadv->flash0);
        if (flashCfiIdentify(pDevCtx, i, 0)) {
          pDevCtx->info.flash[i].bPresent = TRUE;
          pDevCtx->info.flash[i].targetArea.start = 0x1200000U;
          pDevCtx->info.flash[i].targetArea.length = pDevCtx->info.flash[i].totalSize - pDevCtx->info.flash[i].targetArea.start;
        } else {
          dfDebugPrint(0, ("*** initHardware: failed to identify Flash bank %u\n", i));
        }
      }

      /* Read in VPD and pass in serial number from register, in order to adjust serial number */
      if (!readVpd(pDevCtx, serialReg)) {
        createFakeVpd(pDevCtx, serialReg);
      }

      /* Initialize information needed for production test */
      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->adpexrc6tadv.rev0.board.serial;
      pDevCtx->info.prodTest.cpldRevision = 0;
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->adpexrc6tadv.rev0.board.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
        pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].localBase = 0;
        pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
        pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
        i++;
      }
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        initSensorInfoController(pDevCtx);
      }
      break;

    case 1:
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        /* Initialise the AVR uC interface */
        if (avrInit(
          &pDevCtx->model.adpexrc6tadv.avr.context,
          &pDevCtx->interrupt.object,
          &pDevCtx->hardware.model.pAdpexrc6tadv->avr.ctlStat,
          pDevCtx->hardware.hModel,
          ADPEXRC6TADVCTL_AVR_FLASH_SIZE,
          ADPEXRC6TADVCTL_AVR_FLASH_PAGE_SIZE,
          enableAvrRxNotEmptyInt,
          enableAvrTxEmptyInt))
        {
          pDevCtx->model.adpexrc6tadv.bAvrInitialized = TRUE;
        }
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Deinitialize the AVR interface */
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        if (pDevCtx->model.adpexrc6tadv.bAvrInitialized) {
          avrUninit(&pDevCtx->model.adpexrc6tadv.avr.context);
          pDevCtx->model.adpexrc6tadv.bAvrInitialized = FALSE;
        }
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isrController(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.adpexrc6tadv.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADPEXRC6TADVCTL_V_INTSTAT_LEVELBITS) {
      if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
        /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
        intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      if (intStat & ADPEXRC6TADVCTL_V_INTSTAT_ADVRXNE) {
        /* Disable RX FIFO not empty interrupt. */
        intCtl &= ~ADPEXRC6TADVCTL_V_INTSTAT_ADVRXNE;
      }
      if (intStat & ADPEXRC6TADVCTL_V_INTSTAT_ADVTXE) {
        /* Disable TX FIFO empty interrupt. */
        intCtl &= ~ADPEXRC6TADVCTL_V_INTSTAT_ADVTXE;
      }
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.adpexrc6tadv.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADPEXRC6TADVCTL_V_INTSTAT_ADVRXNE) {
      dfDebugPrint(9, ("isr: AVR RX FIFO not empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.adpexrc6tadv.avr.context.transport.rx.dpc, NULL);
    }
    if (intStat & ADPEXRC6TADVCTL_V_INTSTAT_ADVTXE) {
      dfDebugPrint(9, ("isr: AVR TX FIFO empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.adpexrc6tadv.avr.context.transport.tx.dpc, NULL);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isrController: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.adpexrc6tadv.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
isrTarget(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.adpexrc6tadv.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isrTarget: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->model.adpexrc6tadv.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc6tadv* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc6tadv;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        pDevCtx->model.adpexrc6tadv.shadow.intCtl = ADMXRC6Tx_V_INTCTL_SYSENABLE;
        dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVCTL_V_INTCTL_LEVELBITS);
      } else {
        pDevCtx->model.adpexrc6tadv.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
        dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6tadv.shadow.intCtl | ADPEXRC6TADVTGT_V_INTCTL_LEVELBITS);
      }
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      if (pDevCtx->model.adpexrc6tadv.bIsController) {
        /* Restore FPGACTL register for 8-bit SelectMap port */
        dfPciMemWrite32(hModel, &pModelRegs->fpga0.ctl, pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[0]);
        dfPciMemWrite32(hModel, &pModelRegs->fpga1.ctl, pDevCtx->model.adpexrc6tadv.shadow.fpgaCtl[1]);
		    dfPciMemRead32(hModel, &pModelRegs->fpga0.ctl);
      }

      /* Reset Flash banks */
      for (i = 0; i < pDevCtx->info.bootstrap.numFlashBank; i++) {
        admxrc6txFlashResetBank(pDevCtx, i ? &pDevCtx->hardware.model.pAdpexrc6tadv->flash1 : &pDevCtx->hardware.model.pAdpexrc6tadv->flash0);
      }
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = avrSensorVoltageRead(pDevCtx, index, g_avrVoltageScaling, DF_ARRAY_LENGTH(g_avrVoltageScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageScaling);
  }

  status = avrSensorCurrentRead(pDevCtx, index, g_avrCurrentScaling, DF_ARRAY_LENGTH(g_avrCurrentScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrCurrentScaling);
  }

  status = avrSensorTempRead(pDevCtx, index, g_avrTempScaling, DF_ARRAY_LENGTH(g_avrTempScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  }

  status = avrSensorMiscRead(pDevCtx, index, &g_avrMiscScaling, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  }

  return status;
}

static CoreVpdStatus
mapAvrFlashStatus(
  AvrFlashStatus status)
{
  switch (status) {
  case AvrFlashSuccess:
    return CoreVpdSuccess;

  case AvrFlashTimeout:
    return CoreVpdTimeout;

  case AvrFlashNotServiceMode:
    return CoreVpdNotPresent;

  case AvrFlashInvalidRegion:
    return CoreVpdInvalidRegion;

  case AvrFlashGeneralFailure:
    return CoreVpdGeneralFailure;

  default:
    if (status >= AvrFlashHardwareFailure && status < AvrFlashHardwareFailure + 0x100U) {
      return CoreVpdGeneralFailure;
    }
    return CoreVpdGeneralFailure;
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  AvrFlashStatus status;

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    /* Casts are safe because of above bounds check */
    status = avrFlashWrite(pDevCtx, &pDevCtx->model.adpexrc6tadv.avr.context, (uint32_t)address, (uint32_t)length, pData);
  } else {
    /* Casts are safe because of above bounds check */
    status = avrFlashRead(pDevCtx, &pDevCtx->model.adpexrc6tadv.avr.context, (uint32_t)address, (uint32_t)length, pData);
  }
  return mapAvrFlashStatus(status);
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdpexrc6tadv)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdpexrc6tadv)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

CoreWindowConfigStatus
windowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  boolean_t bConfigure,
  uint32_t flags,
  CoreWindowConfig* pConfig)
{
  if (pDevCtx->model.adpexrc6tadv.bIsController) {
    /* Controller windows 0 & 1 don't go anywhere in current firmware. */
    return CoreWindowConfigNotSupported;
  }

  /* Windows 0 & 1 on target FPGAs 0 and 1 behave as normal. */
  return adb3WindowConfig(pDevCtx, windowIndex, bConfigure, flags, pConfig);
}

void
adb3CoreBootstrapAdpexrc6tadv(
	Adb3CoreDeviceContext* pDevCtx)
{
  boolean_t bExposeAllClocks = FALSE;

  dfDebugPrint(1, ("adb3CoreBootstrapAdpexrc6tadv: entered\n"));

  dfAssert(DF_OFFSET_OF(VpdAdpexrc6tadvRev0, checksum) == sizeof(VpdAdpexrc6tadvRev0) - 1);

  pDevCtx->model.adpexrc6tadv.bIsController = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADPEXRC6TADV_C) ? TRUE : FALSE;

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = pDevCtx->model.adpexrc6tadv.bIsController ? isrController : isrTarget;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = windowConfig;

  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "Si5338ExposeAllClocks", &bExposeAllClocks);
  if (pDevCtx->model.adpexrc6tadv.bIsController) {
    pDevCtx->info.bootstrap.numDmaChannel = ADPEXRC6TADVCTL_NUM_DMA_CHANNEL;
    pDevCtx->info.bootstrap.numClockGen = bExposeAllClocks ? 4U : 0U;
    pDevCtx->info.bootstrap.numSensor = DF_ARRAY_LENGTH(g_avrVoltageInfo) +
                                        DF_ARRAY_LENGTH(g_avrCurrentInfo) +
                                        DF_ARRAY_LENGTH(g_avrTempInfo) +
                                        DF_ARRAY_LENGTH(g_avrMiscInfo);
    pDevCtx->info.bootstrap.numTargetFpga = ADPEXRC6TADVCTL_NUM_TARGET_FPGA;
    pDevCtx->info.bootstrap.vpdLength = ADPEXRC6TADVCTL_AVR_FLASH_SIZE;
    pDevCtx->info.bootstrap.numFlashBank = 2;
    pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
    pDevCtx->info.bootstrap.flash[0].width = 2;
    pDevCtx->info.bootstrap.flash[1].deviceWidth = 2;
    pDevCtx->info.bootstrap.flash[1].width = 2;
  } else {
    pDevCtx->info.bootstrap.numDmaChannel = ADPEXRC6TADVTGT_NUM_DMA_CHANNEL;
    pDevCtx->info.bootstrap.numClockGen = 0U;
    pDevCtx->info.bootstrap.numSensor = 0;
    pDevCtx->info.bootstrap.numTargetFpga = 1U;
    pDevCtx->info.bootstrap.vpdLength = 0U;
    pDevCtx->info.bootstrap.numFlashBank = 0;
  }
  pDevCtx->info.bootstrap.numIoModule = 0U;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdpexrc6tadvRev0);
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}
