/*
** File: model_adpexrc6t.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADPE-XRC-6T, including function to bootstrap
**          device context.
**
** (C) Copyright Alpha Data 2011, 2015
**
** NOTES
**
** 1. Organization of Flash device U37 (all offsets in bytes):
**
**  0x0000000 - 0x07FFFFF  Alternate ADB3 PCI-E to OCP Bridge bitstream
**  0x0800000 - 0x0FFFFFF  Default ADB3 PCI-E to OCP Bridge bitstream
**  0x1000000 - 0x11FFFFF  Alpha Data VPD region
**  0x1100000 - 0x11FFFFF  Customer VPD region
**  0x1200000 - 0x28FFFFF  Target FPGA bitstream
**  0x2900000 - 0x3FFFFFF  Failsafe target FPGA bitstream
**
** 2. Frequency synthesizer in PCI-E Bridge is programmable clock index 0.
**
** 3. Usage of SI5338 clock outputs:
**
**  Clock index 1 corresponds to MultiSynth 1, user-programmable
**    - routed to Target FPGA MGTREFCLK0_116 (pins M8, M7)
**
**  The other multisynths are not user-programmable unless the
**  Si5338ExposeAllClocks driver parameter is nonzero. If nonzero,
**  clock indices 2, 3 and 4 correspond to multisynths 0, 2 and 3.
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "adpexrc6t.h"
#include "admxrc6tx_common.h"
#include "avr_common.h"
#include "avr_flash.h"
#include "avr_fmc.h"
#include "avr_sensor.h"
#include "coreclock.h"
#include "coredma.h"
#include "coreif.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"
#include "si5338.h"
#include "si5338_common.h"
#include "v6clksynth.h"
#include "v6clksynth_common.h"

/* TO DO - error values */
static CoreSensorInfo g_avrVoltageInfo[] = {
  { "12V supply rail",             0, { 0,  1966 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A0 */
  { "5V supply rail",              0, { 0,  2359 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A1 */
  { "3.3V slot supply rail",       0, { 0,  1311 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A2 */
  { "3.3V internal supply rail",   0, { 0,  1311 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A3 */
  { "2.5V supply rail",            0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A4 */
  { "FMC VADJ supply rail",        0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A5 */
  { "FMC VIOB supply rail",        0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A6 */
  { "1.8V supply rail",            0, { 0,  6554 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-A7 */
  { "1.5V supply rail",            0, { 0,  4325 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-B0 */
  { "1V supply rail",              0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-B1 */
  { "1V GTX supply rail",          0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-B2 */
  { "1.2V GTX supply rail",        0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC-B3 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrTempInfo[] = {
  { "uC internal temperature",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Microcontroller internal temp */
  { "Board temperature",           0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* TMP422 internal temp */
  { "Target FPGA temp. diode",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* External sensor diode in TMP422 */
  { "Bridge FPGA temp. diode",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Temperature sensor diode in Bridge FPGA */
};

/* TO DO - error values */
static CoreSensorInfo g_avrMiscInfo[] = {
  { "Total powered on time",       0, { 0, 16384 }, CoreSensorDataDouble, CoreSensorUnitS,    0 }, /* Number of seconds spent powered on */
  { "Event count",                 0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* Number of elapsed events */
};

/* These correspond to 'g_avrVoltageInfo' above, to convert the raw ADC values into Volt */
static AvrVoltageScaling g_avrVoltageScaling[] = {
  {  8U, 0x0FFFU, FALSE, 0x40000  /* factor = 0.00390625f */ },
  { 10U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 12U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 14U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 16U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 18U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 20U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 22U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 24U, 0x0FFFU, FALSE, 0x17F5C  /* factor = 0.00146240f */ },
  { 26U, 0x0FFFU, FALSE, 0x05D17  /* factor = 0.00035511f */ },
  { 28U, 0x0FFFU, FALSE, 0x05D17  /* factor = 0.00035511f */ },
  { 30U, 0x0FFFU, FALSE, 0x05D17  /* factor = 0.00035511f */ },
};

/* These correspond to 'g_avrTempInfo' above, to convert the raw ADC values into deg. C */
static AvrTempScaling g_avrTempScaling[] = {
  { 34U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 36U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 38U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 40U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ }
};

static AvrMiscScaling g_avrMiscScaling = {
  { 0U },
  { 5U, TRUE }
};

static void
initSensorInfo(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0;

  dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfo, sizeof(g_avrVoltageInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrTempInfo, sizeof(g_avrTempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  i += admxrc6txInitSensorInfoXsmon(pDevCtx, i, TRUE);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrMiscInfo, sizeof(g_avrMiscInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
}

static boolean_t
verifyChecksum(
	Adb3CoreDeviceContext* pDevCtx,
  uint16_t vpdLength)
{
  CoreVpdStatus status;
  uint8_t buff[16];
  uint8_t sum = 0, checksum;
  uint16_t offset = 0;
  uint16_t remaining = (uint16_t)(vpdLength - 1U);
  uint16_t i, chunk;

  if (vpdLength < 5) {
  	dfDebugPrint(0, ("*** verifyChecksum: invalid VPD length %lu(0x%lx)\n",
      (unsigned long)vpdLength, (unsigned long)vpdLength));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Get checksum from VPD memory; always last byte of structure */
  status = admxrc6txReadAvrVpdData(pDevCtx, remaining, sizeof(checksum), &checksum);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** verifyChecksum: failed to read checksum from VPD area\n"));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Compute checksum over all but last byte of structure */
  while (remaining) {
    chunk = (uint16_t)((remaining > sizeof(buff)) ? sizeof(buff) : remaining);
    status = admxrc6txReadAvrVpdData(pDevCtx, offset, chunk, buff);
    if (CoreVpdSuccess != status) {
  	  dfDebugPrint(0, ("*** verifyChecksum: failed to data chunk from VPD area for checksumming\n"));
      pDevCtx->info.vpd.bChecksumError = TRUE;
      return FALSE;
    }

    for (i = 0; i < chunk; i++) {
      sum = (uint8_t)(sum + buff[i]);
    }
    offset = (uint16_t)(offset + chunk);
    remaining = (uint16_t)(remaining - chunk);
  }
  sum = (uint8_t)-sum;

  if (sum == checksum) {
    return TRUE;
  } else {
  	dfDebugPrint(0, ("*** verifyChecksum: checksum invalid, computed 0x%02lx found 0x%02lx\n",
      (unsigned long)sum, (unsigned long)checksum));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdpexrc6tRev0* pVpd = &pDevCtx->info.vpd.pData->adpexrc6t.rev0;
  VpdAdpexrc6tRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  int i;

  admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  admxrc6txReadAvrVpdData(pDevCtx, 2, sizeof(vpdLength), &vpdLength);
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  verifyChecksum(pDevCtx, vpdLength);

  admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->length = vpdLength;
  pVpd->serial = dfLe32ToCpu(vpdRaw.serial);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->tempGrade = dfLe8ToCpu(vpdRaw.tempGrade);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->si5338Ref = dfLe32ToCpu(vpdRaw.si5338Ref);
  if (0 == pVpd->si5338Ref) {
    /* If invalid value, assume 25 MHz */
    pVpd->si5338Ref = 25000000U;
  }
  pVpd->idelayRef = dfLe32ToCpu(vpdRaw.idelayRef);
  pVpd->gtpRef = dfLe32ToCpu(vpdRaw.gtpRef);
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
    pVpd->sdram[i].speed = dfLe16ToCpu(vpdRaw.sdram[i].speed);
    pVpd->sdram[i].link = dfLe16ToCpu(vpdRaw.sdram[i].link);
  }
  pVpd->fpga.device = dfLe16ToCpu(vpdRaw.fpga.device);
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->fpga.speedGrade), (const char*)vpdRaw.fpga.speedGrade);
  dfStrCpy((char*)pVpd->fpga.stepping, DF_ARRAY_LENGTH(pVpd->fpga.stepping), (const char*)vpdRaw.fpga.stepping);
  pVpd->checksum = dfLe8ToCpu(vpdRaw.checksum);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
enableAvrRxNotEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.adpexrc6t.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.adpexrc6t.shadow.intCtl |= ADPEXRC6T_V_INTCTL_ADVRXNE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6t.shadow.intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

static void
enableAvrTxEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.adpexrc6t.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.adpexrc6t.shadow.intCtl |= ADPEXRC6T_V_INTCTL_ADVTXE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6t.shadow.intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.adpexrc6t.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6t.shadow.intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static void
onV6ClkSynthProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  V6ClkSynthProgramRequest* pRequest,
  V6ClkSynthProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, v6ClkSynthMapProgramStatus(status));
}

static void
onSi5338ProgramDone(
  Si5338DeviceContext* pV6ClkSynthCtx,
  Si5338ProgramRequest* pRequest,
  unsigned int msIndex,
  Si5338ProgramStatus status,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
  unsigned int clockIndex = 0;

  switch (msIndex) {
  case 1:
    /* Multisynth index 1 is clock index 1 */
    clockIndex = 1;
    break;

  case 0:
    /* Multisynth index 0 is clock index 2 */
    clockIndex = 2;
    break;

  case 2:
    /* Multisynth index 2 is clock index 3 */
    clockIndex = 3;
    break;

  case 3:
    /* Multisynth index 3 is clock index 4 */
    clockIndex = 4;
    break;

  default:
    dfAssert(FALSE);
    break;
  }
  onClockProgramDone(pDevCtx, clockIndex, si5338MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  if (clockIndex == 0) {
    return v6ClkSynthMapProgramStatus(v6ClkSynthProgram(pDevCtx, &pDevCtx->model.adpexrc6t.v6ClkSynth.context, &pDevCtx->model.adpexrc6t.v6ClkSynth.programming, pClockWord, onV6ClkSynthProgramDone));
  } else {
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
    unsigned int msIndex;
#endif

    switch (clockIndex) {
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
    case 1:
      /* Clock index 1 is multisynth index 1 */
      msIndex = 1;
      break;

    case 2:
      /* Clock index 2 is multisynth index 0 */
      msIndex = 0;
      break;

    case 3:
      /* Clock index 3 is multisynth index 2 */
      msIndex = 2;
      break;

    case 4:
      /* Clock index 4 is multisynth index 3 */
      msIndex = 3;
      break;
#endif

    default:
      return ClockProgramInvalidIndex;
    }
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
    return si5338MapProgramStatus(si5338ClockProgram(&pDevCtx->model.adpexrc6t.si5338.context, &pDevCtx->model.adpexrc6t.si5338.programming[msIndex], msIndex, pClockWord, onSi5338ProgramDone, pDevCtx));
#endif
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return v6ClkSynthMapClockWordStatus(v6ClkSynthClockWord(flags, pDevCtx->model.adpexrc6t.v6ClkSynth.refClkFrequency, frequencyReq, frequencyMin, frequencyMax, pClockWord));

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
  case 1:
  case 2:
  case 3:
  case 4:
    return si5338MapClockWordStatus(si5338ClockWord(pDevCtx->model.adpexrc6t.si5338.context.vco.frequency, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord));
#endif

  default:
    return ClockWordInvalidIndex;
  }
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;
  boolean_t bOk;

  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    bOk = TRUE;
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    if (!(val32 & 0x4U)) {
      dfDebugPrint(9, ("fpgaControl: fpgaStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    bOk = TRUE;
    /* Check that MPTL is up */
    val32 = dfPciMemRead32(hModel, &pModelRegs->mptlStat);
    if ((val32 & 0x7U) != 0x7U) {
      dfDebugPrint(9, ("fpgaControl: mptlStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    *pbValue = (val32 & 0x2U) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.adpexrc6t.shadow.fpgaCtl;
    if (*pbValue) {
      val32 |= 0x1U;
    } else {
      val32 &= ~0x1U;
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, val32);
    dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
    }

    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->selectMap[0], val32);
      length32 -= 4;
    }

    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
    }

    dfPciMemRead32(hModel, &pModelRegs->fpgaCtl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->selectMap[0]);
      *pBuffer8++ = val8;
    }
  }

  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;
  uint32_t fOut;
  uint8_t fmcStatus;
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
  static const uint32_t freqStep[4] = { 0U, 0U, 0U, 0U };
#endif
  uint32_t forcePcieGen = 0U;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADPEXRC6TL) {
        pDevCtx->info.model = CoreModelAdpexrc6tl;
        dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "Adpexrc6tlForcePcieGen", &forcePcieGen);
      } else if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADPEXRC6TL_BERT) {
        pDevCtx->info.model = CoreModelAdpexrc6tlBert;
      } else if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADPEXRC6T_BERT) {
        pDevCtx->info.model = CoreModelAdpexrc6tBert;
      } else {
        pDevCtx->info.model = CoreModelAdpexrc6t;
        dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "Adpexrc6tForcePcieGen", &forcePcieGen);
      }

      /* This is a workaround for an issue where a particular BIOS doesn't automatically select PCIe Gen 2 link speed */
      if (forcePcieGen) {
        uint8_t rev = pDevCtx->pDevObj->hardwareId.variant.pci.revision;
        if (rev >= 0x0DU) {
          pDevCtx->model.adpexrc6t.forcePcieGen = (unsigned int)forcePcieGen;
          adb3ForcePcieGen(pDevCtx, (unsigned int)forcePcieGen);
        } else {
	        dfDebugPrint(0, ("+++ Force PCIe Gen %u requested but cannot comply because feature not present in Bridge PCIe revision 0x%02X\n",
            (unsigned int)forcePcieGen, (unsigned int)rev));
        }
      }

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.adpexrc6t.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6t.shadow.intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.adpexrc6t.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Reinitialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Initialize FPGACTL register for 8-bit SelectMap port */
      pDevCtx->model.adpexrc6t.shadow.fpgaCtl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.adpexrc6t.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Identify Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0x1200000U;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Read in VPD */
      readVpd(pDevCtx);

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      /* Initialize I2C stuff */
      i2cInit(&pDevCtx->model.adpexrc6t.i2c.context, NULL, admxrc6txI2cStart, admxrc6txI2cPoll, pDevCtx);
#endif

      /* Identify any XRM module fitted */
      fmcStatus = dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + ADPEXRC6T_SENSOR_OFFSET_FMCSTATUS);
      avrFmcIdentify(pDevCtx, TRUE, fmcStatus);

      /* Initialize V6 clock synthesizer stuff */
      pDevCtx->model.adpexrc6t.v6ClkSynth.refClkFrequency = pDevCtx->info.vpd.pData->adpexrc6t.rev0.idelayRef;
      if (!v6ClkSynthInit(pDevCtx, &pDevCtx->model.adpexrc6t.v6ClkSynth.context, pDevCtx->model.adpexrc6t.v6ClkSynth.refClkFrequency, &fOut)) {
        dfDebugPrint(0, ("*** initHardware: failed to initialize V6 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      /* Initialize SI5338 */
      si5338Init(&pDevCtx->model.adpexrc6t.i2c.context, &pDevCtx->model.adpexrc6t.si5338.context, 0, ADPEXRC6T_I2C_SLOT_SI5338);
      /* Find the current SI5338 output frequencies */
      if (!si5338Interrogate(&pDevCtx->model.adpexrc6t.si5338.context, pDevCtx->info.vpd.pData->adpexrc6t.rev0.si5338Ref, freqStep)) {
        dfDebugPrint(0, ("*** initHardware: failed to interrogate SI5338\n"));
      } else {
        pDevCtx->clockGenerator[1].currentFrequency = pDevCtx->model.adpexrc6t.si5338.context.multisynth[1].frequency;
        pDevCtx->clockGenerator[2].currentFrequency = pDevCtx->model.adpexrc6t.si5338.context.multisynth[0].frequency;
        pDevCtx->clockGenerator[3].currentFrequency = pDevCtx->model.adpexrc6t.si5338.context.multisynth[2].frequency;
        pDevCtx->clockGenerator[4].currentFrequency = pDevCtx->model.adpexrc6t.si5338.context.multisynth[3].frequency;
      }
      if (dfDebugLevel(2)) {
        dfDebugPrint(0, ("initHardware: SI5338 register dump:\n"));
        si5338DumpRegisters(&pDevCtx->model.adpexrc6t.si5338.context);
      }
#endif

      /* Initialize information needed for production test */
      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->adpexrc6t.rev0.serial;
      pDevCtx->info.prodTest.cpldRevision = 0; /* TO DO - get microcontroller revision */
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->adpexrc6t.rev0.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
        pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].localBase = 0;
        pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
        pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
        i++;
      }
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      initSensorInfo(pDevCtx);
      break;

    case 1:
      /* Initialise the AVR uC interface */
      if (avrInit(
        &pDevCtx->model.adpexrc6t.avr.context,
        &pDevCtx->interrupt.object,
        &pDevCtx->hardware.model.pAdmxrc6tx->model2.avr.curr.ctlStat,
        pDevCtx->hardware.hModel,
        ADPEXRC6T_AVR_FLASH_SIZE,
        ADPEXRC6T_AVR_FLASH_PAGE_SIZE,
        enableAvrRxNotEmptyInt,
        enableAvrTxEmptyInt))
      {
        pDevCtx->model.adpexrc6t.bAvrInitialized = TRUE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize V6 clock synthesizer stuff */
      v6ClkSynthUninit(pDevCtx, &pDevCtx->model.adpexrc6t.v6ClkSynth.context);

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      /* Uninitialize I2C stuff */
      i2cUninit(&pDevCtx->model.adpexrc6t.i2c.context);
#endif

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Deinitialize the AVR interface */
      if (pDevCtx->model.adpexrc6t.bAvrInitialized) {
        avrUninit(&pDevCtx->model.adpexrc6t.avr.context);
        pDevCtx->model.adpexrc6t.bAvrInitialized = FALSE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.adpexrc6t.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADPEXRC6T_V_INTSTAT_LEVELBITS) {
      if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
        /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
        intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      if (intStat & ADPEXRC6T_V_INTSTAT_ADVRXNE) {
        /* Disable RX FIFO not empty interrupt. */
        intCtl &= ~ADPEXRC6T_V_INTSTAT_ADVRXNE;
      }
      if (intStat & ADPEXRC6T_V_INTSTAT_ADVTXE) {
        /* Disable TX FIFO empty interrupt. */
        intCtl &= ~ADPEXRC6T_V_INTSTAT_ADVTXE;
      }
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.adpexrc6t.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADPEXRC6T_V_INTSTAT_ADVRXNE) {
      dfDebugPrint(9, ("isr: AVR RX FIFO not empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.adpexrc6t.avr.context.transport.rx.dpc, NULL);
    }
    if (intStat & ADPEXRC6T_V_INTSTAT_ADVTXE) {
      dfDebugPrint(9, ("isr: AVR TX FIFO empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.adpexrc6t.avr.context.transport.tx.dpc, NULL);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.adpexrc6t.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  V6ClkSynthProgramStatus v6ClkSynthStatus;
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
  Si5338ProgramStatus si5338Status;
#endif

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save V6 clock synthesizer state */
      v6ClkSynthSaveState(pDevCtx, &pDevCtx->model.adpexrc6t.v6ClkSynth.context);

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      /* Save SI5338 clock generator state */
      si5338SaveState(&pDevCtx->model.adpexrc6t.si5338.context);
#endif
    } else { /* Powering up (from D3) */
      /* This is a workaround for an issue where a particular BIOS doesn't automatically select PCIe Gen 2 link speed */
      unsigned int forcePcieGen = pDevCtx->model.adpexrc6t.forcePcieGen;
      if (forcePcieGen) {
        adb3ForcePcieGen(pDevCtx, forcePcieGen);
      }

      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.adpexrc6t.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.adpexrc6t.shadow.intCtl | ADPEXRC6T_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Reinitialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Restore FPGACTL register for 8-bit SelectMap port */
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.adpexrc6t.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Reset Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);

#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      /* Restore SI5338 clock generator state */
      si5338Status = si5338RestoreState(&pDevCtx->model.adpexrc6t.si5338.context);
      if (Si5338ProgramStatusSuccess != si5338Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore SI5338 state, status=%u\n", si5338Status));
      }
#endif

      /* Restore V6 clock synthesizer state */
      v6ClkSynthStatus = v6ClkSynthRestoreState(pDevCtx, &pDevCtx->model.adpexrc6t.v6ClkSynth.context);
      if (V6ClkSynthProgramStatusSuccess != v6ClkSynthStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore V6 clock synthesizer state, status=%u\n", v6ClkSynthStatus));
      }
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = avrSensorVoltageRead(pDevCtx, index, g_avrVoltageScaling, DF_ARRAY_LENGTH(g_avrVoltageScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageScaling);
  }

  status = avrSensorTempRead(pDevCtx, index, g_avrTempScaling, DF_ARRAY_LENGTH(g_avrTempScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  }

  status = admxrc6txReadSensorXsmon(pDevCtx, index, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= ADMXRC6TX_NUM_XSMON_SENSOR;
  }

  status = avrSensorMiscRead(pDevCtx, index, &g_avrMiscScaling, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  }

  return status;
}

static CoreVpdStatus
mapAvrFlashStatus(
  AvrFlashStatus status)
{
  switch (status) {
  case AvrFlashSuccess:
    return CoreVpdSuccess;

  case AvrFlashTimeout:
    return CoreVpdTimeout;

  case AvrFlashNotServiceMode:
    return CoreVpdNotPresent;

  case AvrFlashInvalidRegion:
    return CoreVpdInvalidRegion;

  case AvrFlashGeneralFailure:
    return CoreVpdGeneralFailure;

  default:
    if (status >= AvrFlashHardwareFailure && status < AvrFlashHardwareFailure + 0x100U) {
      return CoreVpdGeneralFailure;
    }
    return CoreVpdGeneralFailure;
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  AvrFlashStatus status;

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    /* Casts are safe because of above bounds check */
    status = avrFlashWrite(pDevCtx, &pDevCtx->model.adpexrc6t.avr.context, (uint32_t)address, (uint32_t)length, pData);
  } else {
    /* Casts are safe because of above bounds check */
    status = avrFlashRead(pDevCtx, &pDevCtx->model.adpexrc6t.avr.context, (uint32_t)address, (uint32_t)length, pData);
  }
  return mapAvrFlashStatus(status);
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

void
adb3CoreBootstrapAdpexrc6t(
	Adb3CoreDeviceContext* pDevCtx)
{
  boolean_t bExposeAllClocks = FALSE;

  dfDebugPrint(1, ("adb3CoreBootstrapAdpexrc6t: entered\n"));

  dfAssert(DF_OFFSET_OF(VpdAdpexrc6tRev0, checksum) == sizeof(VpdAdpexrc6tRev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = admxrc6txFlashSingleBank;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb3WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 1U;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "Si5338ExposeAllClocks", &bExposeAllClocks);
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
  pDevCtx->info.bootstrap.numClockGen += bExposeAllClocks ? 4U : 1U;
#endif
  if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADPEXRC6TL) {
    pDevCtx->info.bootstrap.numDmaChannel = ADPEXRC6TL_NUM_DMA_CHANNEL;
  } else {
    pDevCtx->info.bootstrap.numDmaChannel = ADPEXRC6T_NUM_DMA_CHANNEL;
  }
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = ADMXRC6TX_NUM_XSMON_SENSOR + DF_ARRAY_LENGTH(g_avrVoltageInfo) + DF_ARRAY_LENGTH(g_avrTempInfo) + DF_ARRAY_LENGTH(g_avrMiscInfo);
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = ADPEXRC6T_AVR_FLASH_SIZE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdpexrc6tRev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}
