/*
** File: model_admpcie6s1.c  
** Project: ADB3 core driver
** Purpose: ADM-PCIE-6S1-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include "admpcie6s1.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"
#include "model_boot.h"
#include "pci9xxx.h"
#include "pci9xxx_common.h"
#include "pci9xxx_93x6.h"
#include "si5338_common.h"

/*
** Bit-banging I2C interface on ADM-PCIE-6S1.
**
** NOTE: implement I2C arbitration not implemented because we are guaranteed to be only master.
*/

/* If 1, always waits for SCL to be sampled high after releasing SCL */
#define CONSERVATIVE_SCL_WAIT (1)

#define I2C_DELAY() dfDelayMicroseconds(1)

static boolean_t /* TRUE => timeout occurred */
admpcie6s1I2cWaitSclHigh(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs)
{
  static const unsigned int numConsecutive = 3U;
  unsigned int timeoutCount = 10U;
  uint8_t tmp8;
  unsigned int n = 0;

  /* Wait for SCL to go high, in case slave is delaying by asserting SCL */
  /* Looks for a few consecutive samples of SCL high before exiting loop */
  while (timeoutCount) {
    tmp8 = (uint8_t)(dfPciMemRead8(hModelRegs, &pModelRegs->ccon) & ADMPCIE6S1_CCON_SCL);
    if (tmp8) {
      n++;
      if (numConsecutive == n) {
        break;
      }
    } else {
      n = 0;
    }
    I2C_DELAY();
    timeoutCount--;
  }
  if (0 == timeoutCount) {
    return TRUE; /* Timed out waiting for SCL high */
  }

  return FALSE; /* OK */
}

static boolean_t
admpcie6s1I2cSendStart(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon)
{
  /* Ensure both SDA and SCL are undriven */
  *pCcon = 0x0U;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := undriven */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();
#if CONSERVATIVE_SCL_WAIT
  if (admpcie6s1I2cWaitSclHigh(hModelRegs, pModelRegs)) {
    return TRUE; /* Timeout */
  }
#endif

  /* Take SDA low */
  *pCcon |= ADMPCIE6S1_CCON_SDA;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := 0 */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  return FALSE; /* OK */
}

static boolean_t
admpcie6s1I2cSendStop(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon)
{
  /* NOTE: on entry, assumes SCL is undriven */

  /* Take SCL low */
  *pCcon |= ADMPCIE6S1_CCON_SCL;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := 0, SDA := ? */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  /* Take SDA low */
  *pCcon |= ADMPCIE6S1_CCON_SDA;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := 0, SDA := 0 */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  /* Take SCL high */
  *pCcon = (uint8_t)(*pCcon & ~ADMPCIE6S1_CCON_SCL);
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := 0 */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();
#if CONSERVATIVE_SCL_WAIT
  if (admpcie6s1I2cWaitSclHigh(hModelRegs, pModelRegs)) {
    return TRUE; /* Timeout */
  }
#endif

  /* Take SDA high */
  *pCcon = (uint8_t)(*pCcon & ~ADMPCIE6S1_CCON_SDA);
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := undriven */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  return FALSE; /* OK */
}

static boolean_t /* TRUE => timeout occurred */
admpcie6s1I2cSendBit(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon,
  uint8_t bit)
{
  /* NOTE: on entry, assumes SCL is undriven */

  /* Take SCL low ready for transmitting 1st bit */
  *pCcon |= ADMPCIE6S1_CCON_SCL;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := 0, SDA := ? */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  /* Put bit on SDA */
  *pCcon = (uint8_t)((*pCcon & ~ADMPCIE6S1_CCON_SDA) | (bit ? 0x0U : ADMPCIE6S1_CCON_SDA));
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := 0, SDA := bit */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  /* Release SCL so it can go high */
  *pCcon = (uint8_t)(*pCcon & ~ADMPCIE6S1_CCON_SCL);
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := bit */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();
#if CONSERVATIVE_SCL_WAIT
  if (admpcie6s1I2cWaitSclHigh(hModelRegs, pModelRegs)) {
    return TRUE; /* Timeout occurred */
  }
#endif

  return FALSE; /* OK */
}

static boolean_t /* TRUE => timeout occurred */
admpcie6s1I2cSendNack(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon)
{
  return admpcie6s1I2cSendBit(hModelRegs, pModelRegs, pCcon, 1);
}

static boolean_t /* TRUE => timeout occurred */
admpcie6s1I2cReceiveBit(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon,
  uint8_t* pBit)
{
  uint8_t cstat;

  /* NOTE: on entry, assumes SCL is undriven */

  /* Take SCL low ready for receiving 1st bit */
  *pCcon |= ADMPCIE6S1_CCON_SCL;
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := 0, SDA := 0 */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();

  /* Release SCL so it can go high */
  *pCcon = (uint8_t)(*pCcon & ~ADMPCIE6S1_CCON_SCL);
  dfPciMemWrite8(hModelRegs, &pModelRegs->ccon, *pCcon); /* SCL := undriven, SDA := ? */
  (void)dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  I2C_DELAY();
#if CONSERVATIVE_SCL_WAIT
  if (admpcie6s1I2cWaitSclHigh(hModelRegs, pModelRegs)) {
    return TRUE; /* Timeout occurred */
  }
#endif

  /* I2C slave drives SDA low if bit is 0; leaves undriven if bit is 1 */
  cstat = dfPciMemRead8(hModelRegs, &pModelRegs->ccon);
  *pBit = (uint8_t)((cstat & ADMPCIE6S1_CCON_SDA) ? 0x1U : 0x0U);

  return FALSE; /* OK */
}

static boolean_t /* TRUE => timeout occurred */
admpcie6s1I2cReceiveAckNack(
  DfMemoryHandle hModelRegs,
  ModelRegsAdmpcie6s1* pModelRegs,
  uint8_t* pCcon,
  uint8_t* pNack)
{
  return admpcie6s1I2cReceiveBit(hModelRegs, pModelRegs, pCcon, pNack);
}

static I2cStatus
admpcie6s1I2cRead(
  Adb3CoreDeviceContext* pDevCtx,
  uint8_t deviceAddress,
  uint8_t regAddress,
  uint8_t* pVal8)
{
  DfMemoryHandle hModelRegs = pDevCtx->hardware.hModel;
  ModelRegsAdmpcie6s1* pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  unsigned int i;
  uint8_t ccon = 0x0U, nAck, bit, val8;
  boolean_t bTimeout;

  dfDebugPrint(7, ("admpcie6s1I2cRead: device=0x%02x address=0x%02x\n", (unsigned int)deviceAddress, (unsigned int)regAddress));

  dfAssertPassiveContext();

  /* Send start sequence */
  bTimeout = admpcie6s1I2cSendStart(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending start sequence\n"));
    return I2cStatusTimeout;
  }

  /* Send device address (7 bits and then a 0) */
  val8 = (uint8_t)((deviceAddress & 0x7FU) << 1U);
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((val8 >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending device address (1)\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout receiving device address ACK (1)\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: received NACK for device address (1)\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Send register address (8 bits) */
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((regAddress >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending register address\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout receiving register address ACK\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: received NACK for register address\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Send stop sequence */
  bTimeout = admpcie6s1I2cSendStop(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending stop sequence (1)\n"));
    return I2cStatusTimeout;
  }

  /* Send start sequence */
  admpcie6s1I2cSendStart(hModelRegs, pModelRegs, &ccon);

  /* Send device address (7 bits and then a 1) */
  val8 = (uint8_t)((deviceAddress & 0x7FU) << 1U) | 0x1U;
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((val8 >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending device address (2)\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout receiving device address ACK (2)\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: received NACK for device address (2)\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Get read data (8 bits) */
  val8 = 0U;
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cReceiveBit(hModelRegs, pModelRegs, &ccon, &bit);
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout receiving data byte\n"));
      return I2cStatusTimeout;
    }
    val8 |= (uint8_t)(bit << i);
  }

  /* Send NACK */
  bTimeout = admpcie6s1I2cSendNack(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending NACK\n"));
    return I2cStatusTimeout;
  }

  /* Send stop sequence */
  bTimeout = admpcie6s1I2cSendStop(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cRead: timeout sending stop sequence (2)\n"));
    return I2cStatusTimeout;
  }

  *pVal8 = val8;
  dfDebugPrint(8, ("admpcie6s1I2cRead: success, data=0x%02x\n", (unsigned int)val8));

  return I2cStatusSuccess;
}

static I2cStatus
admpcie6s1I2cWrite(
  Adb3CoreDeviceContext* pDevCtx,
  uint8_t deviceAddress,
  uint8_t regAddress,
  uint8_t val8)
{
  DfMemoryHandle hModelRegs = pDevCtx->hardware.hModel;
  ModelRegsAdmpcie6s1* pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  unsigned int i;
  uint8_t ccon = 0x0U, nAck, tmp8;
  boolean_t bTimeout;

  dfDebugPrint(7, ("admpcie6s1I2cWrite: device=0x%02x address=0x%02x val8=0x%02x\n",
    (unsigned int)deviceAddress, (unsigned int)regAddress, (unsigned int)val8));

  dfAssertPassiveContext();

  /* Send start sequence */
  bTimeout = admpcie6s1I2cSendStart(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout sending start sequence\n"));
    return I2cStatusTimeout;
  }

  /* Send device address (7 bits and then a 0) */
  tmp8 = (uint8_t)((deviceAddress & 0x7FU) << 1U);
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((tmp8 >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout sending device address\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout receiving device address ACK\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: received NACK for device address\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Send register address (8 bits) */
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((regAddress >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout sending register address\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout receiving device address ACK\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: received NACK for device address\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Send data (8 bits) */
  for (i = 8; i > 0;) {
    i--;
    bTimeout = admpcie6s1I2cSendBit(hModelRegs, pModelRegs, &ccon, (uint8_t)((val8 >> i) & 0x1U));
    if (bTimeout) {
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout sending data\n"));
      return I2cStatusTimeout;
    }
  }

  /* Get ACK/NACK */
  bTimeout = admpcie6s1I2cReceiveAckNack(hModelRegs, pModelRegs, &ccon, &nAck);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout receiving data ACK\n"));
    return I2cStatusTimeout;
  } else {
    if (nAck) {
      /* I2C slave responded with NACK */
      dfDebugPrint(0, ("*** admpcie6s1I2cWrite: received NACK for data\n"));
      return I2cStatusHardwareError;
    }
  }

  /* Send stop sequence */
  bTimeout = admpcie6s1I2cSendStop(hModelRegs, pModelRegs, &ccon);
  if (bTimeout) {
    dfDebugPrint(0, ("*** admpcie6s1I2cWrite: timeout sending stop sequence\n"));
    return I2cStatusTimeout;
  }

  dfDebugPrint(8, ("admpcie6s1I2cWrite: success\n"));

  return I2cStatusSuccess;
}

static void
admpcie6s1I2cStart(
	I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t slot,
  uint8_t address,
  boolean_t bWrite,
  uint8_t data,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;

  if (bus != 0U) {
    pDevCtx->model.admpcie6s1.i2c.status = I2cStatusInvalidBus;
    return;
  }

  if (bWrite) {
    pDevCtx->model.admpcie6s1.i2c.status = admpcie6s1I2cWrite(pDevCtx, slot, address, data);
  } else {
    pDevCtx->model.admpcie6s1.i2c.status = admpcie6s1I2cRead(pDevCtx, slot, address, &pDevCtx->model.admpcie6s1.i2c.readValue);
  }
}

static boolean_t
admpcie6s1I2cPoll(
	I2cContext* pI2cCtx,
  boolean_t bWrite,
  uint8_t* pData,
  boolean_t* pbSuccess,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;

  *pData = pDevCtx->model.admpcie6s1.i2c.readValue;
  *pbSuccess = (I2cStatusSuccess == pDevCtx->model.admpcie6s1.i2c.status) ? TRUE : FALSE;
  return TRUE; /* Always indicate that I2C read/write has finished */
}

static void
setupDirectMaster(
  Adb3CoreDeviceContext* pDevCtx)
{
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  uint32_t dmpbam;

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;

  /*
  ** Set up Direct Master to translate local bus accesses from 0xFF000000-0xFFFFFFFF
  ** to PCI Express memory cycles.
  */
  dmpbam = dfPciMemRead32(hBridge, &pRegs->DMPBAM) & ~0x3U; /* First, ensure that the Direct Master window is disabled. */
  dfPciMemWrite32(hBridge, &pRegs->DMPBAM, dmpbam);
  dfPciMemWrite32(hBridge, &pRegs->DMRR, 0xFF000000U); /* Uses LA[31:24] in decode for Direct Master. */
  dfPciMemWrite32(hBridge, &pRegs->DMLBAM, 0xFF000000U); /* LA[31:24] == 8'hFF => hits Direct Master window. */
  dfPciMemWrite32(hBridge, &pRegs->DMPBAM, dmpbam | 0x1U); /* Reenable the Direct Master window to PCI memory. */
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData);

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  Si5338ProgramStatus si5338Status;
  ClockProgramStatus status;

  switch (clockIndex) {
  case 0U:
    si5338Status = si5338ClockProgramSteppingSync(&pDevCtx->model.admpcie6s1.si5338.context, 3U, pClockWord);
    break;

  case 1U:
    if (pDevCtx->model.admpcie6s1.si5338.bMClkGlitchFree) {
      si5338Status = si5338ClockProgramSteppingSync(&pDevCtx->model.admpcie6s1.si5338.context, 1U, pClockWord);
    } else {
      si5338Status = si5338ClockProgramSync(&pDevCtx->model.admpcie6s1.si5338.context, 1U, pClockWord);
    }
    break;

 case 2U:
    if (pDevCtx->model.admpcie6s1.si5338.bRefClk200GlitchFree) {
      si5338Status = si5338ClockProgramSteppingSync(&pDevCtx->model.admpcie6s1.si5338.context, 0U, pClockWord);
    } else {
      si5338Status = si5338ClockProgramSync(&pDevCtx->model.admpcie6s1.si5338.context, 0U, pClockWord);
    }
    break;

  default:
	  return ClockProgramInvalidIndex;
  }

  status = si5338MapProgramStatus(si5338Status);
  if (ClockProgramSuccess == status) {
    onClockProgramDone(pDevCtx, clockIndex, status);
  }
  return status;
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  Si5338ClockWordStatus status;

  switch (clockIndex) {
  case 0U:
    status = si5338ClockWordStepping(&pDevCtx->model.admpcie6s1.si5338.context, flags, 3U, frequencyReq, frequencyMin, frequencyMax, pClockWord);
    break;

  case 1U:
    if (pDevCtx->model.admpcie6s1.si5338.bMClkGlitchFree) {
      status = si5338ClockWordStepping(&pDevCtx->model.admpcie6s1.si5338.context, flags, 1U, frequencyReq, frequencyMin, frequencyMax, pClockWord);
    } else {
      status = si5338ClockWord(pDevCtx->model.admpcie6s1.si5338.context.vco.frequency, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord);
    }
    break;

 case 2U:
    if (pDevCtx->model.admpcie6s1.si5338.bRefClk200GlitchFree) {
      status = si5338ClockWordStepping(&pDevCtx->model.admpcie6s1.si5338.context, flags, 0U, frequencyReq, frequencyMin, frequencyMax, pClockWord);
    } else {
      status = si5338ClockWord(pDevCtx->model.admpcie6s1.si5338.context.vco.frequency, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord);
    }
    break;

  default:
	  return ClockWordInvalidIndex;
  }

  return si5338MapClockWordStatus(status);
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t page = (uint8_t)(location >> 21);
  uint8_t* pFlash;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x10000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;
  pFlash = ((uint8_t*)pDevCtx->hardware.model.pAdmpcie6s1) + 0x200000U;

  /* Set Flash page register */
  dfPciMemWrite8(hModel, &pModelRegs->flpage, (uint8_t)(page & 0x7fU));
  (void)dfPciMemRead8(hModel, &pModelRegs->flpage);

  location &= 0x1fffffU;
  if (bWrite) {
    dfPciMemWrite8(hModel, pFlash + location, (uint8_t)*pValue);
    (void)dfPciMemRead8(hModel, &pModelRegs->flpage);
  } else {
    *pValue = (uint32_t)dfPciMemRead8(hModel, pFlash + location);
  }
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t val8;

  dfDebugPrint(9, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val8 = dfPciMemRead8(hModel, &pModelRegs->fcon);
    *pbValue = (val8 & ADMPCIE6S1_FCON_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val8 = dfPciMemRead8(hModel, &pModelRegs->fcon);
    *pbValue = (val8 & ADMPCIE6S1_FCON_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val8 = pDevCtx->model.admpcie6s1.shadow.fcon;
    if (*pbValue) {
      val8 = (uint8_t)(val8 | ADMPCIE6S1_FCON_PROG);
    } else {
      val8 = (uint8_t)(val8 & ~ADMPCIE6S1_FCON_PROG);
    }
    dfDebugPrint(9, ("fpgaControlCommonPlxBased: opCode=SetProg val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite8(hModel, &pModelRegs->fcon, val8);
    (void)dfPciMemRead8(hModel, &pModelRegs->fcon);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->smap[0], val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->smap[0], val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->smap[0], val8);
    }
    
    (void)dfPciMemRead8(hModel, &pModelRegs->fcon); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->smap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  CoreVpdStatus status;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  VpdAdmpcie6s1Rev0* pVpd, vpdRaw;
  uint32_t freqStep[4];

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmpcie6s1;

      /* Initialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, FALSE);

      /* Initialize Direct Master in a way specific to ADM-PCIE-6S1 */
      setupDirectMaster(pDevCtx);

      /* Ensure that all interrupts are disabled in PCI9656 */
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaster = PLX_INTCSR_PCI_IE | PLX_INTCSR_LOC_IE | PLX_INTCSR_DMA0_IE | PLX_INTCSR_DMA1_IE;
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaskMaster = PLX_INTCSR_LOC_INT | PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT;
      dfPciMemWrite32(hBridge, &pRegs->INTCSR, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->INTCSR);
      dfPciMemWrite16(hBridge, &pRegs->DMACSR, 0x0808U);
      (void)dfPciMemRead16(hBridge, &pRegs->DMACSR);

      /* Enable target FPGA interrupt in CPLD */
      pDevCtx->model.admpcie6s1.shadow.ienab = 0x1U;
      dfPciMemWrite8(hModel, &pModelRegs->imset, 0xffU);
      dfPciMemWrite8(hModel, &pModelRegs->imclr, pDevCtx->model.admpcie6s1.shadow.ienab);
      dfPciMemWrite8(hModel, &pModelRegs->icon, 0xffU);
      (void)dfPciMemRead8(hModel, &pModelRegs->imset);

      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize I2C stuff */
      i2cInit(&pDevCtx->model.admpcie6s1.i2c.context, NULL, admpcie6s1I2cStart, admpcie6s1I2cPoll, pDevCtx);

      /* Initialize SI5338 */
      si5338Init(&pDevCtx->model.admpcie6s1.i2c.context, &pDevCtx->model.admpcie6s1.si5338.context, 0, ADMPCIE6S1_I2C_SLOT_SI5338);

      status = readWriteVpd(pDevCtx, FALSE, ADMPCIE6S1_EEPROM_VPD_OFFSET, sizeof(vpdRaw), &vpdRaw);
      if (CoreVpdSuccess == status) {
        pDevCtx->info.vpd.bValid = TRUE;
        pVpd = &pDevCtx->info.vpd.pData->admpcie6s1.rev0;
        pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
        pVpd->cardId = dfLe16ToCpu(vpdRaw.cardId);
        pVpd->modifications = dfLe16ToCpu(vpdRaw.modifications);
        pVpd->si5338RefClk = dfLe32ToCpu(vpdRaw.si5338RefClk);
        dfCopyKK(pVpd->fpgaSCD, vpdRaw.fpgaSCD, sizeof(pVpd->fpgaSCD));
        pVpd->fpgaSpeedGrade = dfLe8ToCpu(vpdRaw.fpgaSpeedGrade);
        pVpd->fpgaTempGrade = dfLe8ToCpu(vpdRaw.fpgaTempGrade);
        pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
        pVpd->cpldRev = dfLe8ToCpu(vpdRaw.cpldRev);
        pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
        pVpd->_reserved1 = 0;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to read VPD from EEPROM, status=0x%lx\n", (unsigned long)status));
      }

      /* Find the current SI5338 output frequencies */
      pDevCtx->model.admpcie6s1.si5338.bRefClk200GlitchFree = FALSE;
      dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "ADMPCIE6S1GlitchFreeRefClk200", &pDevCtx->model.admpcie6s1.si5338.bRefClk200GlitchFree);
      pDevCtx->model.admpcie6s1.si5338.bMClkGlitchFree = FALSE;
      dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "ADMPCIE6S1GlitchFreeMClk", &pDevCtx->model.admpcie6s1.si5338.bMClkGlitchFree);
      freqStep[3] = freqStep[2] = freqStep[1] = freqStep[0] = 0U;
      if (pDevCtx->model.admpcie6s1.si5338.bRefClk200GlitchFree) {
        dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "ADMPCIE6S1FreqStepRefClk200", &freqStep[0]);
      }
      if (pDevCtx->model.admpcie6s1.si5338.bMClkGlitchFree) {
        dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "ADMPCIE6S1FreqStepMClk", &freqStep[1]);
      }
      dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "ADMPCIE6S1FreqStepLClk", &freqStep[2]);
      freqStep[3] = freqStep[2];
      if (!si5338Interrogate(&pDevCtx->model.admpcie6s1.si5338.context, pDevCtx->info.vpd.pData->admpcie6s1.rev0.si5338RefClk, freqStep)) {
        dfDebugPrint(0, ("*** initHardware: failed to interrogate SI5338\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = pDevCtx->model.admpcie6s1.si5338.context.multisynth[1].frequency;
        pDevCtx->clockGenerator[1].currentFrequency = pDevCtx->model.admpcie6s1.si5338.context.multisynth[2].frequency;
        pDevCtx->clockGenerator[2].currentFrequency = pDevCtx->model.admpcie6s1.si5338.context.multisynth[0].frequency;
        pDevCtx->clockGenerator[3].currentFrequency = pDevCtx->model.admpcie6s1.si5338.context.multisynth[3].frequency;
      }
      if (dfDebugLevel(2)) {
        dfDebugPrint(0, ("initHardware: SI5338 register dump:\n"));
        si5338DumpRegisters(&pDevCtx->model.admpcie6s1.si5338.context);
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->LAS0BA);
      ls0r = dfPciMemRead32(hBridge, &pRegs->LAS0RR);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->LAS1BA);
      ls1r = dfPciMemRead32(hBridge, &pRegs->LAS1RR);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdmpcie6s1;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Uninitialize I2C stuff */
      i2cUninit(&pDevCtx->model.admpcie6s1.i2c.context);

      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intcsr, clearMask;
  uint8_t istat;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts */
  intcsr = dfPciMemRead32(hBridge, &pRegs->INTCSR);
  
  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  intcsr &= pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMask;
  /* Clear the interrupts we saw */
  dfPciMemWrite32(hBridge, &pRegs->INTCSR, intcsr | pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr);

  if (intcsr & PLX_INTCSR_LOC_INT) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    istat = (uint8_t)(dfPciMemRead8(hModel, &pModelRegs->icon) & pDevCtx->model.admpcie6s1.shadow.ienab);
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite8(hModel, &pModelRegs->icon, istat);

    /* Take action for the model-specific interrupts that we saw. */
    if (istat & ADMPCIE6S1_ICON_FPGA) {
      dfDebugPrint(9, ("isr: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  if (intcsr & (PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT)) {
    clearMask = 0x0U;
    if (intcsr & PLX_INTCSR_DMA0_INT) {
      dfDebugPrint(9, ("isr: DMA0 interrupt\n"));
      clearMask |= 0x0008U;
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
      dfDebugPrint(9, ("isr: DMA1 interrupt\n"));
      clearMask |= 0x0800U;
    }
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, (uint16_t)(clearMask | pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr));
    if (intcsr & PLX_INTCSR_DMA0_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[0].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 0);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[0].dpc, (void*)(uintptr_t)0);
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[1].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 1);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[1].dpc, (void*)(uintptr_t)1);
    }
  }

  return intcsr ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap PCI9656\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap CPLD\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("mapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, sizeof(Pci9656Registers), &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map PCI9656\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped PCI9656 @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map CPLD\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped CPLD @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  ModelRegsAdmpcie6s1* pModelRegs;
  DfMemoryHandle hModel;
  Si5338ProgramStatus si5338Status;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdmpcie6s1;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save SI5338 clock generator state */
      si5338SaveState(&pDevCtx->model.admpcie6s1.si5338.context);
    } else { /* Powering up (from D3) */
      /* Reinitialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, FALSE);

      /* Initialize Direct Master in a way specific to ADM-PCIE-6S1 */
      setupDirectMaster(pDevCtx);

      /* Reenable target FPGA interrupt in CPLD */
      pDevCtx->model.admpcie6s1.shadow.ienab = 0x1U;
      dfPciMemWrite8(hModel, &pModelRegs->imset, 0xffU);
      dfPciMemWrite8(hModel, &pModelRegs->imclr, pDevCtx->model.admpcie6s1.shadow.ienab);
      dfPciMemWrite8(hModel, &pModelRegs->icon, 0xffU);
      (void)dfPciMemRead8(hModel, &pModelRegs->imset);

      /* Restore SI5338 clock generator state */
      si5338Status = si5338RestoreState(&pDevCtx->model.admpcie6s1.si5338.context);
      if (Si5338ProgramStatusSuccess != si5338Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore SI5338 state, status=%u\n", si5338Status));
      }
    }
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    return write93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  } else {
    return read93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Pci9656Registers)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x%lx bytes\n", (unsigned long)sizeof(Pci9656Registers)));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("validateBusResources: OK\n"));
  
  return TRUE;
}

void
adb3CoreBootstrapAdmpcie6s1(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmpcie6s1: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = pci9xxxDmaList;
  pDevCtx->methods.pDmaTransfer = pci9xxxDmaTransfer;
  pDevCtx->methods.pEnableInterrupts = pci9xxxEnableInterrupts;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = pci9xxxWindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 3;
  pDevCtx->info.bootstrap.numDmaChannel = PCI9656_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = eepromSize9356;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmpcie6s1Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 1;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 16;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
