/*
** File: i2c.c  
** Project: ADB3 core driver
** Purpose: Functions for I2C bus access.
**
** (C) Copyright Alpha Data 2011
*/

#include "device.h"
#include "i2c.h"
#include "i2c_common.h"

typedef struct _I2cRequestSync {
  I2cRequest request;
  DfEvent* pEvent;
  I2cStatus status;
} I2cRequestSync;

static const unsigned int g_i2cTimeoutUs = 500000U;   /* 500 ms (!!) timeout for I2C transfers */
static const unsigned int g_i2cPollIntervalUs = 50U; /* Aim for 50us polling interval */

void
i2cInit(
  I2cContext* pI2cCtx,
  DfInterruptObject* pInterruptObject,
  I2cStartMethod* pI2cStart,
  I2cPollMethod* pI2cPoll,
  void* pContext)
{
  DfTime tmp = dfMicrosecondsToTime(g_i2cTimeoutUs);

  DF_ZERO_OBJECT(pI2cCtx);

  /* I2C timeout is 2 system clock ticks, or g_i2cTimeoutUs, whichever is the larger number of ticks */
  tmp = dfMicrosecondsToTime(g_i2cTimeoutUs);
  if (tmp < 2) {
    pI2cCtx->timeout.interval = 2;
  } else {
    pI2cCtx->timeout.interval = tmp;
  }
  /* I2C poll interval is12 system clock tick, or g_i2cPollIntervalUs, whichever is the larger number of ticks */
  tmp = dfMicrosecondsToTime(g_i2cPollIntervalUs);
  if (tmp < 1) {
    pI2cCtx->polling.interval = 1;
  } else {
    pI2cCtx->polling.interval = tmp;
  }
  serializerQueueInit(&pI2cCtx->serializer);
  dfTimerInit(&pI2cCtx->timer, pI2cCtx);
  pI2cCtx->method.pContext = pContext;
  pI2cCtx->method.pI2cStart = pI2cStart;
  pI2cCtx->method.pI2cPoll = pI2cPoll;
  if (NULL != pInterruptObject) {
    dfDpcInit(&pI2cCtx->dpc, pInterruptObject, i2cCompletedRoutine, (void*)pI2cCtx);
    pI2cCtx->bDpcInitialized = TRUE;
    pI2cCtx->bInterruptDriven = TRUE;
  }
}

void
i2cUninit(
  I2cContext* pI2cCtx)
{
  dfTimerUninit(&pI2cCtx->timer);
  if (pI2cCtx->bDpcInitialized) {
    dfDpcUninit(&pI2cCtx->dpc);
    pI2cCtx->bDpcInitialized = FALSE;
  }
}

static void
i2cPollRoutine(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  I2cContext* pI2cCtx = (I2cContext*)pTimerContext;
  SerializerRequest* pSerRequest = serializerGetHead(&pI2cCtx->serializer);
  I2cRequest* pI2cRequest;
  DfTime now;
  boolean_t bSuccess;

  if (NULL == pSerRequest) {
    /* Should not happen, but catch anyway */
    dfBugCheck(("*** i2cPollRoutine: no requests in I2C serializer queue\n"));
    return;
  }
  pI2cRequest = DF_CONTAINER_OF(pSerRequest, I2cRequest, serializerRequest);
  /* Poll for completion of I2C operation and get data byte if it was a read */
  if (pI2cCtx->method.pI2cPoll(pI2cCtx, pI2cRequest->bWrite, &pI2cRequest->data, &bSuccess, pI2cCtx->method.pContext)) {
    /* The I2C operation has finished */
    serializerPut(&pI2cCtx->serializer);
    pI2cRequest->pCallback(pI2cCtx, pI2cRequest, pI2cRequest->data, bSuccess ? I2cStatusSuccess : I2cStatusHardwareError, pI2cRequest->pContext);
  } else {
    now = dfTimeGet();
    if (dfTimeGreaterEqual(now, pI2cCtx->timeout.expires)) {
      /* Operation has timed out */
      /* Let the next I2C operation begin */
      serializerPut(&pI2cCtx->serializer);
      dfDebugPrint(1, ("i2cPollRoutine: timeout waiting for I2C completion\n"));
      pI2cRequest->pCallback(pI2cCtx, pI2cRequest, 0, I2cStatusTimeout, pI2cRequest->pContext);
    } else {
      /* Reschedule polling timer */
      pI2cCtx->polling.expires += pI2cCtx->polling.interval;
      dfTimerSchedule(&pI2cCtx->timer, pI2cCtx->polling.expires, i2cPollRoutine, NULL);
    }
  }
}

void
i2cCompletedRoutine(
  DfDpc* pDpc,
  void* pContext,
  void* pArg)
{
  I2cContext* pI2cCtx = (I2cContext*)pContext;
  SerializerRequest* pSerRequest;
  I2cRequest* pI2cRequest;
  boolean_t bSuccess;

  pSerRequest = serializerGetHead(&pI2cCtx->serializer);
  if (NULL == pSerRequest) {
    /* Should not happen, but catch anyway */
    return;
  }
  pI2cRequest = DF_CONTAINER_OF(pSerRequest, I2cRequest, serializerRequest);
  /* Get data byte if I2C operation was a read */
  pI2cCtx->method.pI2cPoll(pI2cCtx, pI2cRequest->bWrite, &pI2cRequest->data, &bSuccess, pI2cCtx->method.pContext);
  /* Let the next I2C operation begin */
  serializerPut(&pI2cCtx->serializer);
  pI2cRequest->pCallback(pI2cCtx, pI2cRequest, pI2cRequest->data, bSuccess ? I2cStatusSuccess : I2cStatusHardwareError, pI2cRequest->pContext);
}

static void
i2cSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContext)
{
  I2cRequest* pI2cRequest = DF_CONTAINER_OF(pRequest, I2cRequest, serializerRequest);
  I2cContext* pI2cCtx = (I2cContext*)pContext;
  DfTime now;

  /* Start the I2C operation on the hardware */
  pI2cCtx->method.pI2cStart(pI2cCtx, pI2cRequest->bus, pI2cRequest->device, pI2cRequest->address, pI2cRequest->bWrite, pI2cRequest->data, pI2cCtx->method.pContext);
  if (pI2cCtx->bInterruptDriven) {
    /* Wait for I2C interrupt, so do nothing more here */
  } else {
    /* Start I2C polling timer for non-interrupt driven interface */
    now = dfTimeGet();
    pI2cCtx->polling.expires = now + pI2cCtx->polling.interval;
    pI2cCtx->timeout.expires = now + pI2cCtx->timeout.interval;
    dfTimerSchedule(&pI2cCtx->timer, pI2cCtx->polling.expires, i2cPollRoutine, NULL);
  }
}

static void
i2cCallbackSync(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t data,
  I2cStatus status,
  void* pContextIgnored)
{
  I2cRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, I2cRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(pRequestSync->pEvent);
}

void
i2cRead(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t bus,
  uint8_t device,
  uint8_t address,
  I2cCallback* pCallback,
  void* pContext)
{
  dfDebugPrint(7, ("i2cRead: pRequest=%p bus=0x%02x device=0x%02x address=0x%02x pCallback=%p\n",
    pRequest, bus, device, address, pCallback));

  pRequest->bus = bus;
  pRequest->device = device;
  pRequest->address = address;
  pRequest->bWrite = FALSE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  serializerGet(&pI2cCtx->serializer, &pRequest->serializerRequest, i2cSerializerAcquired, pI2cCtx);
}

void
i2cWrite(
  I2cContext* pI2cCtx,
  I2cRequest* pRequest,
  uint8_t bus,
  uint8_t device,
  uint8_t address,
  uint8_t data,
  I2cCallback* pCallback,
  void* pContext)
{
  dfDebugPrint(7, ("i2cWrite: pRequest=%p bus=0x%02x device=0x%02x address=0x%02x data=0x%02x pCallback=%p\n",
    pRequest, bus, device, address, data, pCallback));

  pRequest->bus = bus;
  pRequest->device = device;
  pRequest->address = address;
  pRequest->data = data;
  pRequest->bWrite = TRUE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  serializerGet(&pI2cCtx->serializer, &pRequest->serializerRequest, i2cSerializerAcquired, pI2cCtx);
}

I2cStatus
i2cReadSync(
  I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t device,
  uint8_t address,
  uint8_t* pData)
{
  I2cRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  i2cRead(pI2cCtx, &request.request, bus, device, address, i2cCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  *pData = request.request.data;
  return request.status;
}

I2cStatus
i2cWriteSync(
  I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t device,
  uint8_t address,
  uint8_t data)
{
  I2cRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  i2cWrite(pI2cCtx, &request.request, bus, device, address, data, i2cCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  return request.status;
}

void
i2cDebugScanBus(
  I2cContext* pI2cCtx,
  uint8_t bus)
{
  I2cStatus i2cStatus;
  uint8_t slot;
  uint8_t val8;

  dfDebugPrint(0, ("i2cDebugScanBus: Scanning bus %u...\n", (unsigned int)bus));
  for (slot = 0; slot < 0x80U; slot++) {
    i2cStatus = i2cReadSync(pI2cCtx, bus, slot, 0U, &val8);
    switch (i2cStatus) {
    case I2cStatusSuccess:
      dfDebugPrint(0, ("i2cDebugScanBus: read succeeded at bus %u slot 0x%02x\n", (unsigned int)bus, (unsigned int)slot));
      i2cDebugDumpDevice(pI2cCtx, bus, (uint8_t)slot);
      break;

    case I2cStatusTimeout:
      dfDebugPrint(0, ("*** i2cDebugScanBus:  read failed at bus %u slot 0x%02x: interface timeout\n", (unsigned int)bus, (unsigned int)slot));
      break;

    case I2cStatusHardwareError:
      /* No response - don't treat as error. */
      break;

    case I2cStatusInvalidBus:
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus %u slot 0x%02x: invalid bus\n", (unsigned int)bus, (unsigned int)slot));
      return;

    case I2cStatusInvalidDevice:
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus %u slot 0x%02x: invalid slot\n", (unsigned int)bus, (unsigned int)slot));
      break;

    case I2cStatusInvalidAddress:
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus %u slot 0x%02x: invalid address\n", (unsigned int)bus, (unsigned int)slot));
      break;

    case I2cStatusGeneralFailure:
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus %u slot 0x%02x: general failure\n", (unsigned int)bus, (unsigned int)slot));
      break;

    default:
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus %u slot 0x%02x: unknown error code %u\n", (unsigned int)bus, (unsigned int)slot, i2cStatus));
      break;
    }
  }
}

void
i2cDebugDumpDevice(
  I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t slot)
{
  I2cStatus i2cStatus;
  unsigned int addr;
  uint8_t buf[0x100U];

  dfDebugPrint(0, ("i2cDebugDumpDevice: Dump of bus %u slot 0x%02x:\n", (unsigned int)bus, (unsigned int)slot));
  dfZeroMemory(buf, DF_ARRAY_LENGTH(buf));
  for (addr = 0; addr < DF_ARRAY_LENGTH(buf); addr++) {
    i2cStatus = i2cReadSync(pI2cCtx, bus, slot, (uint8_t)addr, &buf[addr]);
    if (i2cStatus != I2cStatusSuccess) {
      dfDebugPrint(0, ("*** i2cDebugScanBus: read failed at bus=%u slot=0x%02x addr=0x%02x: status=%u\n",
        (unsigned int)bus, (unsigned int)slot, (unsigned int)addr, i2cStatus));
      break;
    }
  }
  dfDebugDumpMemory(0, 0, 16, 1, FALSE, buf, addr);
}
