/*
** File: hwmon.c  
** Project: ADB3 core driver
** Purpose: Implements functions for hardare monitoring infrastructure in
**          ADB3 core driver.
**
** (C) Copyright Alpha Data 2010
**
** The hardware monitoring infrastructure maintains a single timer, which
** always has a timeout of 'g_pollIntervalUs'.
**
** This timer can be enabled any number of times, but the timer is only
** started when the enable count transitions from 0 to 1.
**
** When the timer callback runs, it checks the enable count. If it is
** nonzero, the timer reschedules itself.
**
** The timer callback performs the following tasks:
** - Calls a model-specific function to pump a state machine that handles
**   the system monitor alert interrupts.
** - Calls a model-specific function to read all sensors that are enabled
**   for history, and update minimum / maximum etc.
*/

#include <df.h>

#include "hwmon.h"
#include "corenotify.h"

const unsigned int g_pollIntervalUs = 1000000U;

static void
hwMonTimerCallback(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  boolean_t bReschedule, bActive[MAX_NUM_TARGET_FPGA];
  DfSpinLockFlags f;
  unsigned int i;

  dfDebugPrint(7, ("hwMonTimerCallback: entered"));

  f = dfSpinLockGet(&pDevCtx->hwMon.lock);
  if (pDevCtx->hwMon.polling.bShutdown) {
    dfDebugPrint(8, ("hwMonTimerCallback: shutdown detected"));
    pDevCtx->hwMon.polling.bTimerRunning = FALSE;
    dfSpinLockPut(&pDevCtx->hwMon.lock, f);
    dfEventSignal(&pDevCtx->hwMon.polling.shutdownEvent);
    return;
  }
  dfSpinLockPut(&pDevCtx->hwMon.lock, f);

  /* Poll alert interrupts of each target FPGA. */
  for (i = 0; i < pDevCtx->info.bootstrap.numTargetFpga; i++) {
    if (pDevCtx->hwMon.polling.bPolling[i]) {
      if (pDevCtx->methods.pAlertPoll(pDevCtx, i)) {
        bActive[i] = TRUE;
      } else {
        bActive[i] = FALSE;
      }
    }
  }

  f = dfSpinLockGet(&pDevCtx->hwMon.lock);
  for (i = 0; i < pDevCtx->info.bootstrap.numTargetFpga; i++) {
    if (pDevCtx->hwMon.polling.bPolling[i]) {
      if (!bActive[i]) {
        pDevCtx->hwMon.polling.enable--;
        pDevCtx->hwMon.polling.bPolling[i] = FALSE;
      }
    }
  }
  /* Reschedule the timer if at least one target FPGA is still alerting */
  bReschedule = (pDevCtx->hwMon.polling.enable > 0) ? TRUE : FALSE;
  if (!bReschedule) {
    pDevCtx->hwMon.polling.bTimerRunning = FALSE;
  }
  dfSpinLockPut(&pDevCtx->hwMon.lock, f);

  /* Notify listeners */
  coreNotifyListeners(pDevCtx, &pDevCtx->notification.sensorPoll);

  if (bReschedule) {
    dfDebugPrint(8, ("hwMonTimerCallback: rescheduling"));
    dfTimerSchedule(&pDevCtx->hwMon.polling.timer, dfTimeGet() + dfMicrosecondsToTime(g_pollIntervalUs), hwMonTimerCallback, pDevCtx);
  }
}

void
hwMonAlertDpc(
  DfDpc* pDpc,
  void* pContext,
  void* pArg)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
  unsigned int targetIndex = (unsigned int)(uintptr_t)pArg;

  dfDebugPrint(7, ("hwMonAlertDpc: entered, targetIndex=%u", targetIndex));

  pDevCtx->hwMon.polling.bPolling[targetIndex] = TRUE;

  /* Ensure that the polling timer is running */
  hwMonEnable(pDevCtx);

  /* Notify anyone waiting for FPGA alert notifications */
  coreNotifyListeners(pDevCtx, &pDevCtx->notification.fpgaAlert[targetIndex]);
}

void
hwMonEnable(
  Adb3CoreDeviceContext* pDevCtx)
{
  boolean_t bStart;
  DfSpinLockFlags f;

  dfAssert(!pDevCtx->hwMon.polling.bShutdown);

  f = dfSpinLockGet(&pDevCtx->hwMon.lock);
  bStart = (pDevCtx->hwMon.polling.enable == 0) ? TRUE : FALSE;
  if (bStart) {
    pDevCtx->hwMon.polling.bTimerRunning = TRUE;
  }
  pDevCtx->hwMon.polling.enable++;
  dfSpinLockPut(&pDevCtx->hwMon.lock, f);

  /*
  ** This is safe to do outside spinlock because we don't allow a particular
  ** subsystem to call hwWonDisable before hwMonEnable returns.
  */
  if (bStart) {
    dfTimerSchedule(&pDevCtx->hwMon.polling.timer, dfTimeGet() + dfMicrosecondsToTime(g_pollIntervalUs), hwMonTimerCallback, pDevCtx);
  }
}
