/*
** File: cfi.c  
** Project: ADB3 core driver
** Purpose: Implements functions for dealing with CFI Flash devices.
**
** (C) Copyright Alpha Data 2009-2014
**
** Notes:
**
** 1. CFI vendor-specific data structure for Intel performance command set
** (CFI_ALG_INTEL_PERF, 0x2000U) is the same as that of Intel / Sharp
** extended command set (CFI_ALG_INTELSHARP_EXT, 0x0001U). There are
** differences in the erase & programming algorithms, though.
*/

#include "cfi.h"
#include "flash_cfi.h"
#include "device.h"

static uint32_t
bcdToBinary(
  uint32_t bcd)
{
  return ((bcd >> 0) & 0xfU) + 10 * ((bcd >> 4) & 0xfU) +
         100 * ((bcd >> 8) & 0xfU) + 1000 * ((bcd >> 12) & 0xfU) +
         10000 * ((bcd >> 16) & 0xfU) + 100000 * ((bcd >> 20) & 0xfU) +
         1000000 * ((bcd >> 24) & 0xfU) + 10000000 * ((bcd >> 28) & 0xfU);
}

static void
cfiReadChunk(
  Adb3CoreDeviceContext*  pDevCtx,
  unsigned int bankIndex,
  uint64_t baseAddr,
  uint32_t offset,
  uint64_t length,
  void* pBuffer)
{
  uint8_t* p;
  uint32_t val32;

  dfDebugPrint(6, ("cfiReadChunk: entered, bankIndex=%lu baseAddr=0x%08lx_%08lx offset=0x%lx length=0x%08lx_%08lx pBuffer=%p\n",
    (unsigned long)bankIndex, dfSplitUint64(baseAddr), offset, dfSplitUint64(length), pBuffer));

  p = (uint8_t*)pBuffer;
  switch (pDevCtx->info.bootstrap.flash[bankIndex].deviceWidth) {
  case 1U:
    /* Flash device has 8-bit data bus */
    while (length) {
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + offset, &val32);
      *p++ = (uint8_t)val32;
      length--;
      offset++;
    }
    break;

  case 2U:
    /* Flash device has 16-bit data bus, but only low 8 bits used for CFI data */
    while (length) {
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + (offset << 1), &val32);
      *p++ = (uint8_t)val32;
      length--;
      offset++;
    }
    break;

  case 4U:
    /* Flash device has 32-bit data bus, but only low 8 bits used for CFI data */
    while (length) {
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + (offset << 2), &val32);
      *p++ = (uint8_t)val32;
      length--;
      offset++;
    }
    break;

  default:
    return;
  }
}

/* Enter CFI-query mode */
static void
cfiQueryEnter(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t address)
{
  uint32_t val32;

  /* Issue CFI-query-mode command */
  val32 = 0x98U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, address + 0x55U, &val32);
}

/* Enter read-array mode */
static void
readArrayEnter(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t address)
{
  uint32_t val32;

  /* Issue read-array command */
  val32 = 0xffU;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, address, &val32);
}

/* Enter read-identifier mode */
static void
readIdentifierEnter(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t address)
{
  uint32_t val32;

  /* Issue read-identifier command */
  val32 = 0x90U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, address, &val32);
}

/* Clear status register */
static void
cfiClearStatus(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t address)
{
  uint32_t val32;
  unsigned int i;

  /* Issue a few clear-status-register commands, to make sure multi-write commands are cleared out */
  val32 = 0x50U;
  for (i = 0; i < 4; i++) {
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, address, &val32);
  }
}

static CoreFlashStatus
flashCfiWaitReady0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t address,
  uint32_t timeoutUs,
  uint32_t* pStatus)
{
  const uint32_t busyWaitThreshold = 10000U;
  uint32_t val32;
  DfTime start, expire;

  if (timeoutUs <= busyWaitThreshold) {
    /* Use busy wait */
    while (1) {
      /* Read status */
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, address, &val32);
      *pStatus = val32;
      if ((val32 & 0x80U) == 0x80U) {
        break;
      }
      if (timeoutUs == 0) {
        return CoreFlashTimeout;
      }
      dfDelayMicroseconds(10);
      if (timeoutUs < 10) {
        timeoutUs = 0;
      } else {
        timeoutUs -= 10;
      }
    }
  } else {
    /* Use sleeping wait */
    start = dfTimeGet();
    expire = start + dfMicrosecondsToTime(timeoutUs);
    while (1) {
      /* Read status */
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, address, &val32);
      *pStatus = val32;
      if ((val32 & 0x80U) == 0x80U) {
        break;
      }
      if (dfTimeGet() >= expire) {
        return CoreFlashTimeout;
      }
      if (dfDelayThreadFor(dfMicrosecondsToTime(busyWaitThreshold))) {
        return CoreFlashInterrupted;
      }
    }
  }
  return CoreFlashSuccess;
}

static void
initInformation(
  CfiInformation* pCfiInfo)
{
  pCfiInfo->header.primary.pExtended = NULL;
  pCfiInfo->header.alt.pExtended = NULL;
  pCfiInfo->geometry.pEraseRegion = NULL;
  pCfiInfo->validation.pLink = NULL;
}

static CoreFlashStatus
flashCfiSetBlockLock0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  boolean_t bLock)
{
  const uint32_t timeoutUs = 5000000;
  CoreFlashStatus status = CoreFlashSuccess;
  uint32_t val32;

  /* Issue lock-block / unlock-block command */
  val32 = 0x60U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  val32 = bLock ? 0x01U : 0xd0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  /* Read status */
  status = flashCfiWaitReady0001(pDevCtx, bankIndex, blockAddress, timeoutUs, &val32);
  if (status == CoreFlashSuccess) {
    if ((val32 & 0x7eU) != 0x0U) {
      dfDebugPrint(0, ("*** flashCfiSetBlockLock0001: failed, bankIndex=%lu blockAddress=0x%08lx_%08lx bLock=%s SR=0x%02lx\n",
        (unsigned long)bankIndex, dfSplitUint64(blockAddress), bLock ? "TRUE" : "FALSE", (unsigned long)val32));
      /* An error occurred */
      status = CoreFlashLockFailed;
    }
  } else {
    dfDebugPrint(0, ("*** flashCfiSetBlockLock0001: wait failed, bankIndex=%lu blockAddress=0x%08lx_%08lx bLock=%s SR=0x%02lx status=%lu\n",
      (unsigned long)bankIndex, dfSplitUint64(blockAddress), bLock ? "TRUE" : "FALSE", (unsigned long)val32, (unsigned long)status));
  }
  /* Return to read-array mode */
  readArrayEnter(pDevCtx, bankIndex, blockAddress);

  return status;
}

static CoreFlashStatus
flashCfiSetBlockLock0200(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  boolean_t bLock)
{
  CoreFlashStatus status = CoreFlashSuccess;
  uint32_t val32;

  /* Issue lock-block / unlock-block command */
  val32 = 0x60U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  val32 = bLock ? 0x01U : 0xd0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);

  /* No need to wait, according to G18 data sheet */

  /* Return to read-array mode */
  readArrayEnter(pDevCtx, bankIndex, blockAddress);

  return status;
}

/* Dump CFI translated information to kernel debugger for Intel/Sharp extended command set */
static void
cfiDumpExtInformation0001(
  CfiIntelSharpFixed* pExtInfo)
{
  if (NULL == pExtInfo) {
    dfDebugPrint(0, ("  +++ cfiDumpExtInformation0001: NULL pointer!\n"));
    return;
  }
  dfDebugPrint(0, ("  Vendor ID, Device ID: 0x%04lX, 0x%04lX\n", (unsigned long)pExtInfo->vendorId, (unsigned long)pExtInfo->deviceId));
  dfDebugPrint(0, ("  Magic number: %c%c%c (0x%lx 0x%lx 0x%lx)\n",
    (char)pExtInfo->header.magic[0], (char)pExtInfo->header.magic[1], (char)pExtInfo->header.magic[2],
    (unsigned long)pExtInfo->header.magic[0], (unsigned long)pExtInfo->header.magic[1], (unsigned long)pExtInfo->header.magic[2]));
  dfDebugPrint(0, ("  Data structure version: %c.%c (0x%lx.0x%lx)\n",
    (char)pExtInfo->header.major, (char)pExtInfo->header.minor, (unsigned long)pExtInfo->header.major, (unsigned long)pExtInfo->header.minor));
  dfDebugPrint(0, ("  Capabilities: 0x%lx\n", (unsigned long)pExtInfo->caps));
  dfDebugPrint(0, ("    Chip Erase: %s\n", (pExtInfo->caps & CFI_ALG0001_CAPS_CHIP_ERASE) ? "yes" : "no"));
  dfDebugPrint(0, ("    Legacy block lock mechanism: %s\n", (pExtInfo->caps & CFI_ALG0001_CAPS_LEGACY_BLOCK_LOCK) ? "yes" : "no"));
  dfDebugPrint(0, ("    Instant block lock mechanism: %s\n", (pExtInfo->caps & CFI_ALG0001_CAPS_INSTANT_BLOCK_LOCK) ? "yes" : "no"));
  dfDebugPrint(0, ("    Link to next CFI structure: %s\n", (pExtInfo->caps & CFI_ALG0001_CAPS_LINK_NEXT) ? "yes" : "no"));
  dfDebugPrint(0, ("  Suspend functions: 0x%lx\n", (unsigned long)pExtInfo->caps2));
  dfDebugPrint(0, ("  Block lock status mask: 0x%lx\n", (unsigned long)pExtInfo->blkProtStat));
  dfDebugPrint(0, ("  Optimum VCC during programming: %lu.%lu V\n", pExtInfo->vddProgOpt / 10, pExtInfo->vddProgOpt % 10));
  if (pExtInfo->vppProgOpt != 0) {
    dfDebugPrint(0, ("  Optimum VPP during programming: %lu.%lu V\n", pExtInfo->vppProgOpt / 10, pExtInfo->vppProgOpt % 10));
  } else {
    dfDebugPrint(0, ("  Optimum VPP during programming: no VPP pin\n"));
  }
  if (pExtInfo->caps & CFI_ALG0001_CAPS_LINK_NEXT) {
    dfDebugPrint(0, ("  Next CFI structure address: 0x%08lx_%08lx\n", dfSplitUint64(pExtInfo->linkAddress)));
    dfDebugPrint(0, ("  Next CFI structure offset: 0x%08lx\n", (unsigned long)pExtInfo->linkOffset));
  }
}

/* Dump CFI translated information to kernel debugger */
static void
cfiDumpInformation(
  CfiInformation* pCfiInfo)
{
  uint16_t algorithm, address;
  char* pAlgorithm;
  char* pInterfaceType;
  unsigned int i;

  dfDebugPrint(0, ("CFI basic query header\n"));
  dfDebugPrint(0, ("  Magic number: %c%c%c (0x%lx 0x%lx 0x%lx)\n",
    (char)pCfiInfo->header.magic[0], (char)pCfiInfo->header.magic[1], (char)pCfiInfo->header.magic[2],
    (unsigned long)pCfiInfo->header.magic[0], (unsigned long)pCfiInfo->header.magic[1], (unsigned long)pCfiInfo->header.magic[2]));
  for (i = 0; i < 2; i++) {
    algorithm = (uint16_t)(i ? pCfiInfo->header.alt.algorithm : pCfiInfo->header.primary.algorithm);
    address = (uint16_t)(i ? pCfiInfo->header.alt.address : pCfiInfo->header.primary.address);
    switch (algorithm) {
    case CFI_ALG_NONE:
      pAlgorithm = "None";
      break;

    case CFI_ALG_INTELSHARP_EXT:
      pAlgorithm = "Intel / Sharp Extended";
      break;

    case CFI_ALG_AMD_FUJITSU_STD:
      pAlgorithm = "AMD / Fujitsu Standard";
      break;

    case CFI_ALG_INTEL_STD:
      pAlgorithm = "Intel Standard";
      break;

    case CFI_ALG_AMD_FUJITSU_EXT:
      pAlgorithm = "AMD / Fujitsu Extended";
      break;

    case CFI_ALG_WINBOND_STD:
      pAlgorithm = "Winbond Standard";
      break;

    case CFI_ALG_MITSUBISHI_STD:
      pAlgorithm = "Mitsubishi Standard";
      break;

    case CFI_ALG_MITSUBISHI_EXT:
      pAlgorithm = "Mitsubishi Extended";
      break;

    case CFI_ALG_SST_PAGE_WRITE:
      pAlgorithm = "SST Page Write";
      break;

    case CFI_ALG_INTEL_PERF:
      pAlgorithm = "Intel Performance";
      break;

    case CFI_ALG_INTEL_DATA:
      pAlgorithm = "Intel Data";
      break;

    default:
      pAlgorithm = "Unknown";
      break;
    }
    dfDebugPrint(0, ("  %s algorithm command set: %s (%lu)\n",
      i ? "Alternate" : "Primary", pAlgorithm, (unsigned long)algorithm));
    dfDebugPrint(0, ("  %s algorithm extended query address: 0x%lx\n",
      i ? "Alternate" : "Primary", (unsigned long)address));
  }
  if (pCfiInfo->bGeometryValid) {
    dfDebugPrint(0, ("CFI system interface table\n"));
    dfDebugPrint(0, ("  VCC during programming: %lu.%lu - %lu.%lu V\n",
      pCfiInfo->interface.vccProgMin / 10, pCfiInfo->interface.vccProgMin % 10,
      pCfiInfo->interface.vccProgMax / 10, pCfiInfo->interface.vccProgMax % 10));
    if (pCfiInfo->interface.vppProgMax != 0) {
      dfDebugPrint(0, ("  VPP during programming: %lu.%lu - %lu.%lu V\n",
        pCfiInfo->interface.vppProgMin / 10, pCfiInfo->interface.vppProgMin % 10,
        pCfiInfo->interface.vppProgMax / 10, pCfiInfo->interface.vppProgMax % 10));
    } else {
      dfDebugPrint(0, ("  VPP during programming: no VPP pin\n"));
    }
    if (pCfiInfo->interface.toSingleTyp) {
      dfDebugPrint(0, ("  Typical timeout for single-word programming: %lu us\n", (unsigned long)pCfiInfo->interface.toSingleTyp));
    } else {
      dfDebugPrint(0, ("  Typical timeout for single-word programming: mode not supported\n"));
    }
    if (pCfiInfo->interface.toFullTyp) {
      dfDebugPrint(0, ("  Typical timeout for full multi-word programming: %lu us\n", (unsigned long)pCfiInfo->interface.toFullTyp));
    } else {
      dfDebugPrint(0, ("  Typical timeout for full multi-word programming: mode not supported\n"));
    }
    if (pCfiInfo->interface.toEraseTyp) {
      dfDebugPrint(0, ("  Typical timeout for block erase: %lu ms\n", (unsigned long)pCfiInfo->interface.toEraseTyp));
    } else {
      dfDebugPrint(0, ("  Typical timeout for block erase: mode not supported\n"));
    }
    if (pCfiInfo->interface.toChipTyp) {
      dfDebugPrint(0, ("  Typical timeout for full full-chip erase: %lu ms\n", (unsigned long)pCfiInfo->interface.toChipTyp));
    } else {
      dfDebugPrint(0, ("  Typical timeout for full full-chip erase: mode not supported\n"));
    }
    if (pCfiInfo->interface.toSingleMax) {
      dfDebugPrint(0, ("  Maximum timeout for single-word programming: %lu us\n", (unsigned long)pCfiInfo->interface.toSingleMax));
    } else {
      dfDebugPrint(0, ("  Maximum timeout for single-word programming: mode not supported\n"));
    }
    if (pCfiInfo->interface.toFullMax) {
      dfDebugPrint(0, ("  Maximum timeout for full multi-word programming: %lu us\n", (unsigned long)pCfiInfo->interface.toFullMax));
    } else {
      dfDebugPrint(0, ("  Maximum timeout for full multi-word programming: mode not supported\n"));
    }
    if (pCfiInfo->interface.toEraseMax) {
      dfDebugPrint(0, ("  Maximum timeout for block erase: %lu ms\n", (unsigned long)pCfiInfo->interface.toEraseMax));
    } else {
      dfDebugPrint(0, ("  Maximum timeout for block erase: mode not supported\n"));
    }
    if (pCfiInfo->interface.toChipMax) {
      dfDebugPrint(0, ("  Maximum timeout for full full-chip erase: %lu ms\n", (unsigned long)pCfiInfo->interface.toChipMax));
    } else {
      dfDebugPrint(0, ("  Maximum timeout for full full-chip erase: mode not supported\n"));
    }
  } else {
    dfDebugPrint(0, ("CFI system interface table NOT FOUND\n"));
  }
  if (pCfiInfo->bGeometryValid) {
    dfDebugPrint(0, ("CFI geometry table\n"));
    dfDebugPrint(0, ("  Die size: 0x%08lx_%08lx bytes\n", dfSplitUint64(pCfiInfo->geometry.fixed.size)));
    switch (pCfiInfo->geometry.fixed.interfaceCode) {
      case CFI_DIC_X8_ASYNC:
        pInterfaceType = "x8, asynchronous";
        break;

      case CFI_DIC_X16_ASYNC:
        pInterfaceType = "x16, asynchronous";
        break;

      case CFI_DIC_X8X16_ASYNC:
        pInterfaceType = "x8 or x16, asynchronous";
        break;

      case CFI_DIC_X32_ASYNC:
        pInterfaceType = "x32, asynchronous";
        break;

      case CFI_DIC_X16X32_ASYNC:
        pInterfaceType = "x16 or x32, asynchronous";
        break;

      default:
        pInterfaceType = "Unknown";
        break;
    }
    dfDebugPrint(0, ("  Device interface: %s (%lu)\n", pInterfaceType, (unsigned long)pCfiInfo->geometry.fixed.interfaceCode));
    dfDebugPrint(0, ("  Multi-byte programming maximum length: %lu(0x%lx)\n",
      (unsigned long)pCfiInfo->geometry.fixed.multiByteMax, (unsigned long)pCfiInfo->geometry.fixed.multiByteMax));
    dfDebugPrint(0, ("  Number of distinct erase regions: %lu\n", (unsigned long)pCfiInfo->geometry.fixed.numEraseRegion));
    for (i = 0; i < pCfiInfo->geometry.fixed.numEraseRegion; i++) {
      dfDebugPrint(0, ("  Erase region %lu: address=0x%08lx_%08lx blockSize=0x%08lx_%08lx numBlock=0x%08lx_%08lx\n",
        (unsigned long)i, dfSplitUint64(pCfiInfo->geometry.pEraseRegion[i].startAddress),
        dfSplitUint64(pCfiInfo->geometry.pEraseRegion[i].blockSize), dfSplitUint64(pCfiInfo->geometry.pEraseRegion[i].numBlock)));
    }
  } else {
    dfDebugPrint(0, ("CFI geometry table NOT FOUND\n"));
  }
  switch (pCfiInfo->header.primary.algorithm) {
  case CFI_ALG_NONE:
    break;

  case CFI_ALG_INTELSHARP_EXT:
  case CFI_ALG_INTEL_PERF:
    dfDebugPrint(2, ("Primary algorithm (%lu) extended header:\n", (unsigned long)pCfiInfo->header.primary.algorithm));
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 2) {
      cfiDumpExtInformation0001((CfiIntelSharpFixed*)pCfiInfo->header.primary.pExtended);
    }
#endif
    break;

  default:
    dfDebugPrint(0, ("+++ Primary algorithm extended header: algorithm %lu not recognised\n", (unsigned long)pCfiInfo->header.primary.algorithm));
  }

  switch (pCfiInfo->header.alt.algorithm) {
  case CFI_ALG_NONE:
    break;

  case CFI_ALG_INTELSHARP_EXT:
  case CFI_ALG_INTEL_PERF:
    dfDebugPrint(2, ("Alternate algorithm (%lu) extended header:\n", (unsigned long)pCfiInfo->header.alt.algorithm));
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= 2) {
      cfiDumpExtInformation0001((CfiIntelSharpFixed*)pCfiInfo->header.alt.pExtended);
    }
#endif
    break;

  default:
    dfDebugPrint(0, ("+++ Alternate algorithm extended header: algorithm %lu not recognised\n", (unsigned long)pCfiInfo->header.alt.algorithm));
  }
}

/* Checks CFI information for inconsistencies / apparent errors. Returns TRUE if OK. */
static boolean_t
cfiValidateInformation(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  CfiInformation* pCfiInfo)
{
  uint16_t algorithm;
  CfiExtendedQueryHeader* pExtended;
  unsigned int i;
  uint16_t interfaceCode;
  uint32_t j;
  uint64_t total;

  DF_ZERO_OBJECT(&pCfiInfo->validation);
  pCfiInfo->validation.preferredAlgorithm = CFI_ALG_NONE;
  pCfiInfo->validation.algorithm.generic.pExtended = NULL;
  pCfiInfo->validation.bSingleWordSupported = FALSE;
  pCfiInfo->validation.bMultiWordSupported = FALSE;
  pCfiInfo->validation.bBlockEraseSupported = FALSE;
  pCfiInfo->validation.bChipEraseSupported = FALSE;
  pCfiInfo->validation.largestBlock = 0U;
  pCfiInfo->validation.vendorId = 0xffffU;
  pCfiInfo->validation.deviceId = 0xffffU;
  pCfiInfo->validation.bAnotherStruct = FALSE; /* Assume FALSE, may change later */
  pCfiInfo->validation.linkAddress = 0;
  pCfiInfo->validation.linkOffset = 0;
  pCfiInfo->validation.pLink = NULL;

  if (pCfiInfo->header.magic[0] != 'Q' || pCfiInfo->header.magic[1] != 'R' || pCfiInfo->header.magic[2] != 'Y') {
    dfDebugPrint(0, ("*** cfiValidate: magic number in header is not 'QRY'.\n"));
    return FALSE;
  }
  for (i = 0; i < 2; i++) {
    algorithm = (uint16_t)(i ? pCfiInfo->header.alt.algorithm : pCfiInfo->header.primary.algorithm);
    pExtended = i ? pCfiInfo->header.alt.pExtended : pCfiInfo->header.primary.pExtended;
    if (NULL != pExtended) {
      /* Check the magic number in the extended command set header */
      if (i) {
        if (pExtended->magic[0] != 'A' || pExtended->magic[1] != 'L' || pExtended->magic[2] != 'T') {
          dfDebugPrint(0, ("*** cfiValidate: magic number in alternate programming algorithm table header is not 'ALT'.\n"));
          return FALSE;
        }
      } else{
        if (pExtended->magic[0] != 'P' || pExtended->magic[1] != 'R' || pExtended->magic[2] != 'I') {
          dfDebugPrint(0, ("*** cfiValidate: magic number in primary programming algorithm table header is not 'PRI'.\n"));
          return FALSE;
        }
      }
    }
    switch (algorithm) {
    case CFI_ALG_INTELSHARP_EXT:
    case CFI_ALG_INTEL_PERF:
      pCfiInfo->validation.preferredAlgorithm = algorithm;
      pCfiInfo->validation.algorithm.generic.pExtended = (void*)pExtended;
      break;

    default:
      break;
    }
    if (pCfiInfo->validation.preferredAlgorithm != CFI_ALG_NONE) {
      break;
    }
  }
  if (pCfiInfo->validation.preferredAlgorithm == CFI_ALG_NONE) {
    dfDebugPrint(0, ("*** cfiValidate: chip supports no command sets that we understand.\n"));
    return FALSE;
  }

  /* Check system interface table */
  if (pCfiInfo->bInterfaceValid) {
    /* We don't really care about VCC / VPP programming voltages, not checked in this driver */
    
    /* Check timeout values */
    pCfiInfo->validation.bSingleWordSupported = pCfiInfo->interface.toSingleTyp ? TRUE : FALSE;
    pCfiInfo->validation.bMultiWordSupported = pCfiInfo->interface.toFullTyp ? TRUE : FALSE;
    pCfiInfo->validation.bBlockEraseSupported = pCfiInfo->interface.toEraseTyp ? TRUE : FALSE;
    pCfiInfo->validation.bChipEraseSupported = pCfiInfo->interface.toChipTyp ? TRUE : FALSE;
    if (!pCfiInfo->validation.bSingleWordSupported && !pCfiInfo->validation.bMultiWordSupported) {
      dfDebugPrint(0, ("*** cfiValidate: neither single-word nor multi-word programming supported.\n"));
      return FALSE;
    }
    if (!pCfiInfo->validation.bChipEraseSupported && !pCfiInfo->validation.bBlockEraseSupported) {
      dfDebugPrint(0, ("*** cfiValidate: neither block-erase nor chip-erase supported.\n"));
      return FALSE;
    }
    if (pCfiInfo->validation.bSingleWordSupported) {
      if (!pCfiInfo->interface.toSingleMax) { 
        dfDebugPrint(0, ("*** cfiValidate: single-word programming maximum timeout invalid (%lu us).\n", (unsigned long)pCfiInfo->interface.toSingleMax));
        return FALSE;
      }
    } else {
      if (pCfiInfo->interface.toSingleMax) { 
        dfDebugPrint(0, ("*** cfiValidate: single-word programming typical timeout conflicts with maximum timeout.\n"));
        return FALSE;
      }
    }
    if (pCfiInfo->validation.bMultiWordSupported) {
      if (!pCfiInfo->interface.toFullMax) { 
        dfDebugPrint(0, ("*** cfiValidate: multi-word programming maximum timeout invalid (%lu us).\n", (unsigned long)pCfiInfo->interface.toFullMax));
        return FALSE;
      }
    } else {
      if (pCfiInfo->interface.toFullMax) { 
        dfDebugPrint(0, ("*** cfiValidate: multi-word programming typical timeout conflicts with maximum timeout.\n"));
        return FALSE;
      }
    }
    if (pCfiInfo->validation.bBlockEraseSupported) {
      if (!pCfiInfo->interface.toEraseMax) {
        dfDebugPrint(0, ("*** cfiValidate: block erase maximum timeout invalid (%lu us).\n", (unsigned long)pCfiInfo->interface.toEraseMax));
        return FALSE;
      }
    } else {
      if (pCfiInfo->interface.toEraseMax) {
        dfDebugPrint(0, ("*** cfiValidate: block erase typical timeout conflicts with maximum timeout.\n"));
        return FALSE;
      }
    }
    if (pCfiInfo->validation.bChipEraseSupported) {
      if (!pCfiInfo->interface.toChipMax) {
        dfDebugPrint(0, ("*** cfiValidate: chip erase maximum timeout invalid (%lu us).\n", (unsigned long)pCfiInfo->interface.toChipMax));
        return FALSE;
      }
    } else {
      if (pCfiInfo->interface.toChipMax) {
        dfDebugPrint(0, ("*** cfiValidate: chip erase typical timeout conflicts with maximum timeout.\n"));
        return FALSE;
      }
    }
  }

  /* Check geometry table */
  if (pCfiInfo->bGeometryValid) {
    /* We never fit a device less than 128 kiB in size */
    if (pCfiInfo->geometry.fixed.size < 0x20000U) {
      dfDebugPrint(0, ("*** cfiValidate: device size too small (0x%08lx_%08lx bytes).\n", dfSplitUint64(pCfiInfo->geometry.fixed.size)));
      return FALSE;
    }
    interfaceCode = pCfiInfo->geometry.fixed.interfaceCode;
    switch (interfaceCode) {
    case CFI_DIC_X8_ASYNC:
    case CFI_DIC_X16_ASYNC:
    case CFI_DIC_X8X16_ASYNC:
    case CFI_DIC_X32_ASYNC:
    case CFI_DIC_X16X32_ASYNC:
      break;

    default:
      dfDebugPrint(0, ("*** cfiValidate: interface code is invalid (%lu).\n", (unsigned long)pCfiInfo->geometry.fixed.interfaceCode));
      return FALSE;
    }
    switch (pDevCtx->info.bootstrap.flash[bankIndex].width) {
    case 1U:
      if (interfaceCode != CFI_DIC_X8_ASYNC && interfaceCode != CFI_DIC_X8X16_ASYNC) {
        dfDebugPrint(0, ("*** cfiValidate: interface code (%lu) conflicts with flash width from bootstrap data (%lu)\n",
          (unsigned long)pCfiInfo->geometry.fixed.interfaceCode, (unsigned long)pDevCtx->info.bootstrap.flash[bankIndex].width));
      }
      break;

    case 2U:
      if (interfaceCode != CFI_DIC_X16_ASYNC && interfaceCode != CFI_DIC_X8X16_ASYNC && interfaceCode != CFI_DIC_X16_ASYNC) {
        dfDebugPrint(0, ("*** cfiValidate: interface code (%lu) conflicts with flash width from bootstrap data (%lu)\n",
          (unsigned long)pCfiInfo->geometry.fixed.interfaceCode, (unsigned long)pDevCtx->info.bootstrap.flash[bankIndex].width));
      }
      break;

    case 4U:
      if (interfaceCode != CFI_DIC_X32_ASYNC && interfaceCode != CFI_DIC_X16X32_ASYNC) {
        dfDebugPrint(0, ("*** cfiValidate: interface code (%lu) conflicts with flash width from bootstrap data (%lu)\n",
          (unsigned long)pCfiInfo->geometry.fixed.interfaceCode, (unsigned long)pDevCtx->info.bootstrap.flash[bankIndex].width));
      }
      break;

    default:
      dfDebugPrint(0, ("*** cfiValidate: cannot validate interface code due to invalid flash width from bootstrap data (%lu)\n",
        (unsigned long)pDevCtx->info.bootstrap.flash[bankIndex].width));
      break;
    }
    if (pCfiInfo->validation.bMultiWordSupported && (pCfiInfo->geometry.fixed.multiByteMax > pCfiInfo->geometry.fixed.size || pCfiInfo->geometry.fixed.multiByteMax < 4U)) {
      dfDebugPrint(0, ("*** cfiValidate: maximum multibyte programming length is invalid (%lu bytes).\n", (unsigned long)pCfiInfo->geometry.fixed.multiByteMax));
      return FALSE;
    }
    if (pCfiInfo->geometry.fixed.numEraseRegion == 0) {
      if (!pCfiInfo->validation.bBlockEraseSupported) {
        dfDebugPrint(0, ("*** cfiValidate: block-erase supported => number of erase-regions must be nonzero.\n"));
        return FALSE;
      }
    } else {
      if (pCfiInfo->geometry.pEraseRegion == NULL) {
        dfDebugPrint(0, ("*** cfiValidate: number of erase regions is nonzero, but erase region array is NULL.\n"));
        return FALSE;
      } else {
        total = 0;
        for (j = 0; j < pCfiInfo->geometry.fixed.numEraseRegion; j++) {
          if (pCfiInfo->geometry.pEraseRegion[j].numBlock == 0 || pCfiInfo->geometry.pEraseRegion[j].blockSize == 0) {
            dfDebugPrint(0, ("*** cfiValidate: erase region %lu invalid, numBlock=" DF_FMT_U64 "(0x%08lx_%08lx) blockSize=" DF_FMT_U64 "(0x%08lx_%08lx).\n",
              (unsigned long)j, (unsigned long long)pCfiInfo->geometry.pEraseRegion[j].numBlock, dfSplitUint64(pCfiInfo->geometry.pEraseRegion[j].numBlock),
              (unsigned long long)pCfiInfo->geometry.pEraseRegion[j].blockSize, dfSplitUint64(pCfiInfo->geometry.pEraseRegion[j].blockSize)));
            return FALSE;
          }
          total += pCfiInfo->geometry.pEraseRegion[j].numBlock * pCfiInfo->geometry.pEraseRegion[j].blockSize;
          if (pCfiInfo->geometry.pEraseRegion[j].blockSize > pCfiInfo->validation.largestBlock) {
            pCfiInfo->validation.largestBlock = pCfiInfo->geometry.pEraseRegion[j].blockSize;
          }
        }
        if (total != pCfiInfo->geometry.fixed.size) {
          dfDebugPrint(0, ("*** cfiValidate: total size from all erase regions (0x%08lx_%08lx) conflicts with die size (0x%08lx_%08lx).\n",
            dfSplitUint64(total), dfSplitUint64(pCfiInfo->geometry.fixed.size)));
          return FALSE;
        }
      }
    }
  }

  /* Checks specific to particular algorithms */
  switch (pCfiInfo->validation.preferredAlgorithm) {
  case CFI_ALG_INTELSHARP_EXT:
  case CFI_ALG_INTEL_PERF:
    /* For Intel/Sharp Extended command set, we must have a valid geometry table */
    if (!pCfiInfo->bInterfaceValid) {
      dfDebugPrint(0, ("*** cfiValidate: system interface table not found, required for INTEL/SHARP Extended command set.\n"));
      return FALSE;
    }
    if (!pCfiInfo->bGeometryValid) {
      dfDebugPrint(0, ("*** cfiValidate: geometry table not found, required for INTEL/SHARP Extended command set.\n"));
      return FALSE;
    }
    if (pCfiInfo->validation.algorithm.x0001.pExtended == NULL) {
      dfDebugPrint(0, ("*** cfiValidate: Sharp / Intel Extended table is NULL.\n"));
      return FALSE;
    } else {
      CfiIntelSharpFixed* pExtInfo = (CfiIntelSharpFixed*)pCfiInfo->validation.algorithm.x0001.pExtended;
      boolean_t bChipEraseSupported;

      pCfiInfo->validation.vendorId = pExtInfo->vendorId;
      pCfiInfo->validation.deviceId = pExtInfo->deviceId;
      bChipEraseSupported = (pExtInfo->caps & CFI_ALG0001_CAPS_CHIP_ERASE) ? TRUE : FALSE;
      pCfiInfo->validation.algorithm.x0001.bLegacyBlockLockSupported = (pExtInfo->caps & CFI_ALG0001_CAPS_LEGACY_BLOCK_LOCK) ? TRUE : FALSE;
      pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported = (pExtInfo->caps & CFI_ALG0001_CAPS_INSTANT_BLOCK_LOCK) ? TRUE : FALSE;
      if (bChipEraseSupported != pCfiInfo->validation.bChipEraseSupported) {
        dfDebugPrint(0, ("*** cfiValidate: extended query table disagrees with basic query table about Chip Erase support"));
      }
      if (pExtInfo->caps & CFI_ALG0001_CAPS_LINK_NEXT) {
        pCfiInfo->validation.bAnotherStruct = TRUE;
        pCfiInfo->validation.linkAddress = pExtInfo->linkAddress;
        pCfiInfo->validation.linkOffset = pExtInfo->linkOffset;
      }
    }
    break;
  }

  if (!pCfiInfo->validation.bBlockEraseSupported) {
    /* Block-erase not supported, so treat entire chip as one big block */
    pCfiInfo->validation.largestBlock = pCfiInfo->geometry.fixed.size;
  }

  return TRUE;
}

static boolean_t
cfiIdentify0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t baseAddr,
  uint32_t offset,
  boolean_t bPrimary,
  CfiInformation* pCfiInfo)
{
  CfiIntelSharpFixedRaw1 rawFixed1;
  CfiIntelSharpFixedRaw2 rawFixed2;
  CfiIntelSharpFixedRaw3 rawFixed3;
  CfiIntelSharpFixedRaw4 rawFixed4;
  CfiIntelSharpFixedRaw5 rawFixed5;
  uint8_t extraCaps[4];
  CfiIntelSharpPartitionInfoRaw rawPartInfo;
  CfiIntelSharpLinkRaw rawLink;
  CfiIntelSharpFixed* pFixed;
  uint32_t val32;
  unsigned int i;
  uint64_t linkAddress = 0U;
  uint32_t linkOffset = 0U;
  uint16_t vendorId = 0xffffU, deviceId = 0xffffU;

  /* Get vendor & device ID of chip using in read identifier mode  */
  readIdentifierEnter(pDevCtx, bankIndex, baseAddr);
  switch (pDevCtx->info.bootstrap.flash[bankIndex].deviceWidth) {
  case 1U:
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr, &val32);
    vendorId = (uint8_t)val32;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + 1, &val32);
    deviceId = (uint8_t)val32;
    break;

  case 2U:
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr, &val32);
    vendorId = (uint16_t)val32;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + 2, &val32);
    deviceId = (uint16_t)val32;
    break;

  case 4U:
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr, &val32);
    vendorId = (uint16_t)val32;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + 4, &val32);
    deviceId = (uint16_t)val32;
    break;
  }
  /* Return to CFI query mode */
  cfiQueryEnter(pDevCtx, bankIndex, baseAddr);
  dfDebugPrint(2, ("cfiIdentify0001: vendor=0x%04lx device=0x%04lx\n", (unsigned long)vendorId, (unsigned long)deviceId));

  /* Get CFI extended header fixed part 1 */
  cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawFixed1), &rawFixed1);
  dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI extended header information part 1 @ 0x%08lx_%08lx:\n",
    dfSplitUint64(baseAddr + offset)));
  dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawFixed1, sizeof(rawFixed1));
  offset += (uint32_t)sizeof(rawFixed1);
  if (bPrimary) {
    if (rawFixed1.header.magic[0] != 'P' || rawFixed1.header.magic[1] != 'R' || rawFixed1.header.magic[2] != 'I') {
      dfDebugPrint(0, ("+++ cfiIdentify0001: CFI primary extended signature not found - abandoning\n"));
      return FALSE;
    }
  } else {
    if (rawFixed1.header.magic[0] != 'A' || rawFixed1.header.magic[1] != 'L' || rawFixed1.header.magic[2] != 'T') {
      dfDebugPrint(0, ("+++ cfiIdentify0001: CFI alternate extended signature not found - abandoning\n"));
      return FALSE;
    }
  }
  if (rawFixed1.capsByte3 & 0x80U) {
    /* There is another 4-byte capabilities field, but we skip it */
    do {
      cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(extraCaps), &extraCaps);
      dfDebugPrint(2, ("cfiIdentify0001: extra caps found - 0x%02lx 0x%02lx 0x%02lx 0x%02lx\n",
        (unsigned long)extraCaps[0], (unsigned long)extraCaps[1], (unsigned long)extraCaps[2], (unsigned long)extraCaps[3]));
      offset += 4;
    } while (!(extraCaps[3] & 0x80U));
  }

  /* Get CFI extended header fixed part 2 */
  cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawFixed2), &rawFixed2);
  dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI extended header information part 2 @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
  dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawFixed2, sizeof(rawFixed2));
  offset += (uint32_t)sizeof(rawFixed2);

  if (rawFixed1.header.major > 1 || (rawFixed1.header.major == 1 && rawFixed1.header.minor >= 1)) {
    /* Protection register info is valid */
    cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawFixed3), &rawFixed3);
    dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI extended header information part 3 @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
    dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawFixed3, sizeof(rawFixed3));
    offset += (uint32_t)sizeof(rawFixed3);

    if (rawFixed3.numProtReg < 1 || rawFixed3.numProtReg > 2) {
      /* CFI documentation is very sketchy as to how the number of protection registers affects
      ** layout of the CFI data structures. All of the Flash devices we've seen so far have 2
      ** for this value, and it is unclear how to skip over these fields if the value is other
      ** than 2. So if the value is not 2, we will give up, to be on the safe side. */
      dfDebugPrint(0, ("*** cfiIdentify0001: number of protection registers (%u) is not 1 or 2. Giving up in order to be on the safe side.\n", rawFixed3.numProtReg));
      return FALSE;
    }

    /* Skip over protection register info */
    dfDebugPrint(2, ("cfiIdentify0001: Skipping %ld protection registers of 0x%lx bytes each\n",
      (unsigned long)(rawFixed3.numProtReg - 1), (unsigned long)sizeof(CfiIntelSharpProtRegRaw)));
    offset += (uint32_t)((rawFixed3.numProtReg - 1) * sizeof(CfiIntelSharpProtRegRaw));

    /* Get burst read capability info */
    cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawFixed4), &rawFixed4);
    dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI extended header information part 4 @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
    dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawFixed4, sizeof(rawFixed4));
    offset += (uint32_t)sizeof(rawFixed4);

    /* Skip over remaining burst read capability info */
    offset += rawFixed4.numSyncMode;

    /* Get number of hardware partition regions */
    cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawFixed5), &rawFixed5);
    dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI extended header information part 5 @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
    dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawFixed5, sizeof(rawFixed5));
    offset += (uint32_t)sizeof(rawFixed5);

    for (i = 0; i < rawFixed5.numPartitionRegion; i++) {
      if (rawFixed1.header.major > 1 || (rawFixed1.header.major == 1 && rawFixed1.header.minor >= 4)) {
        /* Skip over size indicator */
        offset += 2;
      }

      /* Get partition region info */
      cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawPartInfo.fixed), &rawPartInfo.fixed);
      dfDebugPrint(2, ("cfiIdentify0001: Dump of raw partition %ld info fixed part @ 0x%08lx_%08lx:\n",
        (unsigned long)i, dfSplitUint64(baseAddr + offset)));
      dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawPartInfo.fixed, sizeof(rawPartInfo.fixed));
      offset += (uint32_t)sizeof(rawPartInfo.fixed);

      offset += (uint32_t)(rawPartInfo.fixed.numEraseBlockRegions * sizeof(rawPartInfo.region[0]));
    }
  }

  if (rawFixed1.capsByte3 & 0x40U) {
    uint32_t link;

    /* There is CFI link field */
    cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawLink), &rawLink);
    dfDebugPrint(2, ("cfiIdentify0001: Dump of raw CFI link field @ 0x%08lx_%08lx:\n",
      (unsigned long)(uint32_t)((baseAddr + offset) >> 32), (unsigned long)(uint32_t)(baseAddr + offset)));
    dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawLink, sizeof(rawLink));
    offset += (uint32_t)sizeof(rawLink);

    link = ((uint32_t)rawLink.linkByte0 << 0) | ((uint32_t)rawLink.linkByte1 << 8) |
      ((uint32_t)rawLink.linkByte2 << 16) | ((uint32_t)rawLink.linkByte3 << 24);
    linkAddress = (uint64_t)((link >> 10) & 0x3ffffU) << 22;
    linkOffset = (link & 0x3ffU);
  }

  pFixed = (CfiIntelSharpFixed*)dfMalloc(sizeof(*pFixed));
  if (NULL == pFixed) {
    dfDebugPrint(0, ("*** cfiIdentify0001: extended header allocation failed, size=%p bytes\n", (void*)sizeof(*pFixed)));
    return FALSE;
  }
  pFixed->header.magic[0] = rawFixed1.header.magic[0];
  pFixed->header.magic[1] = rawFixed1.header.magic[1];
  pFixed->header.magic[2] = rawFixed1.header.magic[2];
  pFixed->header.major = rawFixed1.header.major;
  pFixed->header.minor = rawFixed1.header.minor;
  pFixed->caps =
    ((uint32_t)rawFixed1.capsByte0 << 0) | ((uint32_t)rawFixed1.capsByte1 << 8) |
    ((uint32_t)rawFixed1.capsByte2 << 16) | ((uint32_t)rawFixed1.capsByte3 << 24);
  pFixed->caps2 = rawFixed2.caps2;
  pFixed->blkProtStat = (uint16_t)(((uint16_t)rawFixed2.blkProtStatLo << 0) | ((uint16_t)rawFixed2.blkProtStatHi << 8));
  pFixed->vddProgOpt = (uint8_t)bcdToBinary(rawFixed2.vddProgOpt);
  pFixed->vppProgOpt = (uint8_t)bcdToBinary(rawFixed2.vppProgOpt);
  if (bPrimary) {
    pCfiInfo->header.primary.pExtended = (CfiExtendedQueryHeader*)pFixed;
  } else {
    pCfiInfo->header.alt.pExtended = (CfiExtendedQueryHeader*)pFixed;
  }
  pFixed->linkAddress = linkAddress;
  pFixed->linkOffset = linkOffset;
  pFixed->vendorId = vendorId;
  pFixed->deviceId = deviceId;

  return TRUE;
}

static CoreFlashStatus
cfiBlockLock0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t chipAddress,
  uint64_t blockAddress,
  boolean_t bLock)
{
  CfiInformation* pCfiInfo = pDevCtx->info.flash[bankIndex].detail.pCfi;
  CoreFlashStatus status = CoreFlashSuccess;

  /*
  ** Instant block-lock mechanism supports locking and unlock of blocks individually.
  ** Legacy block-lock mechanism supports locking of blocks individually but can only unlock all blocks.
  **
  ** The legacy ADMXRC2 driver doesn't use the legacy block locking mechanism if it finds a J3 Flash
  ** device, which is OK because J3 Flash devices come from the factory with all blocks unlocked. In
  ** order not to render legacy cards like the ADM-XRC-II unusable, if this driver ever controls
  ** them, we will only ever unlock using the legacy block locking mechanism.
  */

  if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported || pCfiInfo->validation.algorithm.x0001.bLegacyBlockLockSupported) {
    if (bLock) {
      if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported) {
        /* Lock the block using instant block lock mechanism */
        status = flashCfiSetBlockLock0001(pDevCtx, bankIndex, blockAddress, TRUE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      }
    } else {
      if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported) {
        /* Unlock the block using instant block lock mechanism */
        status = flashCfiSetBlockLock0001(pDevCtx, bankIndex, blockAddress, FALSE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      } else {
        /* Unlock all blocks using legacy block lock mechanism */
        status = flashCfiSetBlockLock0001(pDevCtx, bankIndex, chipAddress, FALSE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      }
    }
  } else {
    /* There is no block locking mechanism on this chip */
  }
  /* Go to read-array mode */
  readArrayEnter(pDevCtx, bankIndex, blockAddress);

  return status;
}

static CoreFlashStatus
cfiBlockLock0200(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t chipAddress,
  uint64_t blockAddress,
  boolean_t bLock)
{
  CfiInformation* pCfiInfo = pDevCtx->info.flash[bankIndex].detail.pCfi;
  CoreFlashStatus status = CoreFlashSuccess;

  /*
  ** Instant block-lock mechanism supports locking and unlock of blocks individually.
  ** Legacy block-lock mechanism supports locking of blocks individually but can only unlock all blocks.
  **
  ** The legacy ADMXRC2 driver doesn't use the legacy block locking mechanism if it finds a J3 Flash
  ** device, which is OK because J3 Flash devices come from the factory with all blocks unlocked. In
  ** order not to render legacy cards like the ADM-XRC-II unusable, if this driver ever controls
  ** them, we will only ever unlock using the legacy block locking mechanism.
  */

  if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported || pCfiInfo->validation.algorithm.x0001.bLegacyBlockLockSupported) {
    if (bLock) {
      if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported) {
        /* Lock the block using instant block lock mechanism */
        status = flashCfiSetBlockLock0001(pDevCtx, bankIndex, blockAddress, TRUE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      }
    } else {
      if (pCfiInfo->validation.algorithm.x0001.bInstantBlockLockSupported) {
        /* Unlock the block using instant block lock mechanism */
        status = flashCfiSetBlockLock0200(pDevCtx, bankIndex, blockAddress, FALSE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      } else {
        /* Unlock all blocks using legacy block lock mechanism */
        status = flashCfiSetBlockLock0200(pDevCtx, bankIndex, chipAddress, FALSE);
        if (CoreFlashSuccess != status) {
          return status;
        }
      }
    }
  } else {
    /* There is no block locking mechanism on this chip */
  }
  /* Go to read-array mode */
  readArrayEnter(pDevCtx, bankIndex, blockAddress);

  return status;
}

static boolean_t
cfiIdentify(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t baseAddr,
  uint32_t offset,
  CfiInformation* pTranslated)
{
  CfiQueryStringRaw rawHeader;
  CfiSystemInterfaceRaw rawInterface;
  CfiGeometryFixedRaw rawGeometry;
  CfiEraseRegionRaw rawEraseRegion;
  size_t size, rawSize;
  uint64_t startAddr;
  uint32_t extAddress;
  unsigned int i;
  boolean_t bSuccess = FALSE;

  dfDebugPrint(1, ("cfiIdentify: entered, pDevCtx=%p bankIndex=%lu baseAddr=0x%08lx_%08lx offset=0x%lx\n",
    (void*)pDevCtx, (unsigned long)bankIndex, dfSplitUint64(baseAddr), (unsigned long)offset));

  /* Enter CFI query mode */
  cfiQueryEnter(pDevCtx, bankIndex, baseAddr);

  /* Get CFI header */
  DF_ZERO_OBJECT(pTranslated);
  cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawHeader), &rawHeader);
  dfDebugPrint(2, ("cfiIdentify: Dump of raw CFI header information @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
  dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawHeader, sizeof(rawHeader));
  offset += (uint32_t)sizeof(CfiQueryStringRaw);
  if (rawHeader.magic[0] != 'Q' || rawHeader.magic[1] != 'R' || rawHeader.magic[2] != 'Y') {
    dfDebugPrint(0, ("+++ cfiIdentify: CFI signature not found - abandoning\n"));
    goto done;
  }
  pTranslated->header.magic[0] = dfLe8ToCpu(rawHeader.magic[0]);
  pTranslated->header.magic[1] = dfLe8ToCpu(rawHeader.magic[1]);
  pTranslated->header.magic[2] = dfLe8ToCpu(rawHeader.magic[2]);
  pTranslated->header.primary.algorithm = (uint16_t)((uint16_t)rawHeader.priAlgorithmLo | ((uint16_t)rawHeader.priAlgorithmHi << 8));
  pTranslated->header.primary.address = (uint16_t)((uint16_t)rawHeader.priAddressLo | ((uint16_t)rawHeader.priAddressHi << 8));
  pTranslated->header.primary.pExtended = NULL;
  pTranslated->header.alt.algorithm = (uint16_t)((uint16_t)rawHeader.altAlgorithmLo | ((uint16_t)rawHeader.altAlgorithmHi << 8));
  pTranslated->header.alt.address = (uint16_t)((uint16_t)rawHeader.altAddressLo | ((uint16_t)rawHeader.altAddressHi << 8));
  pTranslated->header.alt.pExtended = NULL;
  extAddress = (uint32_t)-1;
  extAddress = (pTranslated->header.primary.algorithm != 0 && pTranslated->header.primary.address < extAddress) ? pTranslated->header.primary.address : extAddress;
  extAddress = (pTranslated->header.alt.algorithm != 0 && pTranslated->header.alt.address < extAddress) ? pTranslated->header.alt.address : extAddress;
  if (extAddress < sizeof(rawHeader)) {
    dfDebugPrint(0, ("*** cfiIdentify: CFI header information obscured by Extended Query info - abandoning\n"));
    goto done;
  }
  /* Get CFI system interface information */
  if (extAddress >= offset + sizeof(rawInterface)) {
    cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawInterface), &rawInterface);
    dfDebugPrint(2, ("cfiIdentify: Dump of raw CFI system interface information @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
    dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawInterface, sizeof(rawInterface));
    pTranslated->interface.vccProgMin = (uint8_t)bcdToBinary(rawInterface.vccProgMin);
    pTranslated->interface.vccProgMax = (uint8_t)bcdToBinary(rawInterface.vccProgMax);
    pTranslated->interface.vppProgMin = (uint8_t)bcdToBinary(rawInterface.vppProgMin);
    pTranslated->interface.vppProgMax = (uint8_t)bcdToBinary(rawInterface.vppProgMax);
    pTranslated->interface.toSingleTyp = rawInterface.toSingleTyp ? 1U << rawInterface.toSingleTyp : 0;
    pTranslated->interface.toFullTyp = rawInterface.toFullTyp ? 1U << rawInterface.toFullTyp : 0;
    pTranslated->interface.toEraseTyp = rawInterface.toEraseTyp ? 1U << rawInterface.toEraseTyp : 0;
    pTranslated->interface.toChipTyp = rawInterface.toChipTyp ? 1U << rawInterface.toChipTyp : 0;
    pTranslated->interface.toSingleMax = rawInterface.toSingleMax ? pTranslated->interface.toSingleTyp * (1U << rawInterface.toSingleMax) : 0;
    pTranslated->interface.toFullMax = rawInterface.toFullMax ? pTranslated->interface.toFullTyp * (1U << rawInterface.toFullMax) : 0;
    pTranslated->interface.toEraseMax = rawInterface.toEraseMax ? pTranslated->interface.toEraseTyp * (1U << rawInterface.toEraseMax) : 0;
    pTranslated->interface.toChipMax = rawInterface.toChipMax ? pTranslated->interface.toChipTyp * (1U << rawInterface.toChipMax) : 0;
    offset += (uint32_t)sizeof(CfiSystemInterfaceRaw);
    pTranslated->bInterfaceValid = TRUE;
   /* Get CFI geometry information */
    if (extAddress >= offset + sizeof(rawGeometry)) {
      cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawGeometry), &rawGeometry);
      dfDebugPrint(2, ("cfiIdentify: Dump of raw CFI geometry information (fixed part) @ 0x%08lx_%08lx:\n", dfSplitUint64(baseAddr + offset)));
      dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawGeometry, sizeof(rawGeometry));
      offset += (uint32_t)sizeof(rawGeometry);
      pTranslated->geometry.fixed.size = 1ULL << rawGeometry.order;
      pTranslated->geometry.fixed.interfaceCode = (uint16_t)((uint16_t)rawGeometry.interfaceCodeLo | ((uint16_t)rawGeometry.interfaceCodeHi << 8));
      pTranslated->geometry.fixed.multiByteMax = 1U << ((uint16_t)rawGeometry.multiByteMaxLo | ((uint16_t)rawGeometry.multiByteMaxHi << 8));
      pTranslated->geometry.fixed.numEraseRegion = rawGeometry.numEraseRegion;
      if (0 != pTranslated->geometry.fixed.numEraseRegion) {
        /* Get CFI erase region information */
        rawSize = sizeof(CfiEraseRegionRaw) * pTranslated->geometry.fixed.numEraseRegion;
        if (extAddress >= offset + rawSize) {
          size = sizeof(CfiEraseRegion) * pTranslated->geometry.fixed.numEraseRegion;
          pTranslated->geometry.pEraseRegion = (CfiEraseRegion*)dfMalloc(size);
          if (NULL == pTranslated->geometry.pEraseRegion) {
            dfDebugPrint(0, ("*** cfiIdentify: CfiEraseRegion array allocation failed, size=%p bytes\n", (void*)size));
            goto done;
          }
          startAddr = baseAddr;
          for (i = 0; i < pTranslated->geometry.fixed.numEraseRegion; i++) {
            cfiReadChunk(pDevCtx, bankIndex, baseAddr, offset, sizeof(rawEraseRegion), &rawEraseRegion);
            dfDebugPrint(2, ("cfiIdentify: Dump of raw CFI erase region %lu information @ 0x%08lx_%08lx:\n",
              (unsigned long)i, dfSplitUint64(baseAddr + offset)));
            dfDebugDumpMemory(2, 2, 16, 1, FALSE, &rawEraseRegion, sizeof(rawEraseRegion));
            pTranslated->geometry.pEraseRegion[i].numBlock = dfLe16ToCpu(rawEraseRegion.numBlock) + 1;
            pTranslated->geometry.pEraseRegion[i].blockSize = dfLe16ToCpu(rawEraseRegion.blockSize) * 256;
            pTranslated->geometry.pEraseRegion[i].startAddress = startAddr;
            offset += (uint32_t)sizeof(rawEraseRegion);
            startAddr += pTranslated->geometry.pEraseRegion[i].numBlock * pTranslated->geometry.pEraseRegion[i].blockSize;
          }
        } else {
          dfDebugPrint(0, ("*** cfiIdentify: basic query information specifies at least one erase region, but erase region info obscured by Extended Query info - abandoning\n"));
          goto done;
        }
      }
      pTranslated->bGeometryValid = TRUE;
    } else {
      dfDebugPrint(0, ("+++ cfiIdentify: CFI geometry information (fixed part) obscured by Extended Query info\n"));
    }
  } else {
    dfDebugPrint(0, ("+++ cfiIdentify: CFI system interface information obscured by Extended Query info\n"));
  }
  pTranslated->chipAddress = baseAddr;

  switch (pTranslated->header.primary.algorithm) {
  case CFI_ALG_INTELSHARP_EXT:
  case CFI_ALG_INTEL_PERF:
    /* Get additional information for Intel/Sharp extended command set */
    if (!cfiIdentify0001(pDevCtx, bankIndex, baseAddr, pTranslated->header.primary.address, TRUE, pTranslated)) {
      goto done;
    }
    break;
  }

  switch (pTranslated->header.alt.algorithm) {
  case CFI_ALG_INTELSHARP_EXT:
  case CFI_ALG_INTEL_PERF:
    /* Get additional information for Intel/Sharp extended command set */
    if (!cfiIdentify0001(pDevCtx, bankIndex, baseAddr, pTranslated->header.alt.address, FALSE, pTranslated)) {
      goto done;
    }
    break;
  }

#if DF_DBG_BUILD
  if (g_dfDebugLevel >= 2) {
    cfiDumpInformation(pTranslated);
  }
#endif

  if (!cfiValidateInformation(pDevCtx, bankIndex, pTranslated)) {
    /* Detected inconsistency or downright impossible value in CFI information */
    dfDebugPrint(0, ("+++ cfiIdentify: CFI information contains apparent errors - abandoning\n"));
    goto done;
  }

  bSuccess = TRUE;

done:
  /* Return to read-array mode */
  readArrayEnter(pDevCtx, bankIndex, baseAddr);

  return bSuccess;
}

/*
** -----------------------------------------------------------------
** Exported routines
** -----------------------------------------------------------------
*/

void
flashCfiCleanup(
  CfiInformation* pCfiInfo)
{
  CfiInformation* pLink;

  while (NULL != pCfiInfo) {
    if (NULL != pCfiInfo->header.primary.pExtended) {
      dfFree(pCfiInfo->header.primary.pExtended);
      pCfiInfo->header.primary.pExtended = NULL;
    }
    if (NULL != pCfiInfo->header.alt.pExtended) {
      dfFree(pCfiInfo->header.alt.pExtended);
      pCfiInfo->header.alt.pExtended = NULL;
    }
    if (NULL != pCfiInfo->geometry.pEraseRegion) {
      dfFree(pCfiInfo->geometry.pEraseRegion);
      pCfiInfo->geometry.pEraseRegion = NULL;
    }
    pLink = pCfiInfo->validation.pLink;
    dfFree(pCfiInfo);
    pCfiInfo = pLink;
  }
}

CoreFlashStatus
flashCfiEraseBlock0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress)
{
  CoreFlashStatus status;
  CfiInformation* pCfiInfo = pDevCtx->info.flash[bankIndex].detail.pCfi;
  uint32_t val32;
  uint32_t timeoutUs = pCfiInfo->interface.toEraseMax * 1000;

  /* Unlock the block (or chip) */
  status = cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, FALSE);
  if (CoreFlashSuccess != status) {
    return status;
  }

  /* Issue erase-block command */
  val32 = 0x20U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  /* Issue erase-confirm command */
  val32 = 0xd0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  /* Wait for completion */
  status = flashCfiWaitReady0001(pDevCtx, bankIndex, blockAddress, timeoutUs, &val32);
  if (CoreFlashSuccess != status) {
    goto failed;
  }
  if ((val32 & 0x7eU) != 0U) {
    status = CoreFlashEraseFailed;
    goto failed;
  }

  /* Re-lock the block against accidental modifications */
  status = cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  if (CoreFlashSuccess != status) {
    /* Clear status register */
    cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  }
  return status;

failed:
  /* Clear status register */
  cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  /* Something went wrong, so attempt to re-lock the block */
  cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  return status;
}

CoreFlashStatus
flashCfiEraseBlock0200(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress)
{
  CoreFlashStatus status;
  CfiInformation* pCfiInfo = pDevCtx->info.flash[bankIndex].detail.pCfi;
  uint32_t val32;
  uint32_t timeoutUs = pCfiInfo->interface.toEraseMax * 1000;

  /* Unlock the block (or chip) */
  status = cfiBlockLock0200(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, FALSE);
  if (CoreFlashSuccess != status) {
    return status;
  }

  /* Issue erase-block command */
  val32 = 0x20U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  /* Issue erase-confirm command */
  val32 = 0xd0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  /* Wait for completion */
  status = flashCfiWaitReady0001(pDevCtx, bankIndex, blockAddress, timeoutUs, &val32);
  if (CoreFlashSuccess != status) {
    goto failed;
  }
  if ((val32 & 0x7eU) != 0U) {
    status = CoreFlashEraseFailed;
    goto failed;
  }

  /* Re-lock the block against accidental modifications */
  status = cfiBlockLock0200(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  if (CoreFlashSuccess != status) {
    /* Clear status register */
    cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  }
  return status;

failed:
  /* Clear status register */
  cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  /* Something went wrong, so attempt to re-lock the block */
  cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  return status;
}

boolean_t
flashCfiIdentify(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t baseAddr)
{
  const unsigned int maxDie = 4; /* Maximum number of dies to check */
  Adb3CoreFlashInfo* pFlashInfo;
  CfiInformation* pTranslated = NULL;
  uint64_t totalSize = 0;
  uint32_t offset;
  boolean_t bSuccess = TRUE;
  unsigned int i = 0;

  dfDebugPrint(1, ("flashCfiIdentify: entered, pDevCtx=%p bankIndex=%lu baseAddr=0x%08lx_%08lx\n",
    (void*)pDevCtx, (unsigned long)bankIndex, dfSplitUint64(baseAddr)));

  if (bankIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    dfDebugPrint(0, ("*** flashCfiIdentify: invalid bank index %u\n", bankIndex));
    return FALSE;
  }
  pFlashInfo = &pDevCtx->info.flash[bankIndex];
  if (pDevCtx->info.bootstrap.flash[bankIndex].width == 0) {
    dfDebugPrint(0, ("+++ flashCfiIdentify: bootstrap Flash width is 0, not identifying\n"));
    return FALSE;
  }
  if (pDevCtx->methods.pFlash == NULL) {
    dfDebugPrint(0, ("+++ flashCfiIdentify: no Flash method, not identifying\n"));
    return FALSE;
  }

  pTranslated = (CfiInformation*)dfMalloc(sizeof(*pTranslated));
  if (NULL == pTranslated) {
    dfDebugPrint(0, ("*** flashCfiIdentify: failed to allocate root CFI structure\n"));
    return FALSE;
  }
  initInformation(pTranslated);
  pFlashInfo->detail.pCfi = pTranslated;

  /* Attempt to read CFI information */

  offset = 0x10U; /* Offset of start of CFI standard header */

  while (NULL != pTranslated) {
    CfiInformation* pNextTranslated;

    bSuccess = cfiIdentify(pDevCtx, bankIndex, baseAddr, offset, pTranslated);
    if (!bSuccess) {
      break;
    }
    totalSize += pTranslated->geometry.fixed.size;
    if (i < (maxDie - 1) && pTranslated->validation.bAnotherStruct) {
      pNextTranslated = dfMalloc(sizeof(*pNextTranslated));
      if (NULL == pNextTranslated) {
        dfDebugPrint(0, ("*** flashCfiIdentify: failed to allocate next CFI structure\n"));
        bSuccess = FALSE;
        break;
      }
      initInformation(pNextTranslated);  
      pTranslated->validation.pLink = pNextTranslated;
      baseAddr = pTranslated->validation.linkAddress;
      offset = pTranslated->validation.linkOffset;
      pTranslated = pNextTranslated;
      i++;
    } else {
      break;
    }
  }

  if (!bSuccess) {
    /* Clean up on error */
    if (NULL != pFlashInfo->detail.pCfi) {
      flashCfiCleanup(pFlashInfo->detail.pCfi);
      pFlashInfo->detail.pCfi = NULL;
    }
  } else {
    pFlashInfo->totalSize = totalSize;
  }

  return bSuccess;
}

CoreFlashStatus
flashCfiWriteBlock0001(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  uint64_t length,
  const void* pData)
{
  CoreFlashStatus status;
  Adb3CoreFlashInfo* pFlashInfo = &pDevCtx->info.flash[bankIndex];
  CfiInformation* pCfiInfo = pFlashInfo->detail.pCfi;
  unsigned int flashWidth = pDevCtx->info.bootstrap.flash[bankIndex].width;
  uint64_t writeAddress;
  uint32_t multiByteMax;
  uint32_t val32, i, count, timeoutUs;
  uint8_t* p;

  /* Unlock the block (or chip) */
  status = cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, FALSE);
  if (CoreFlashSuccess != status) {
    return status;
  }

  p = (uint8_t*)pData;
  writeAddress = blockAddress;
  dfAssert((blockAddress & (flashWidth - 1U)) == 0U);
  if (pCfiInfo->validation.bMultiWordSupported) {
    /* Using multi-word programming */
    timeoutUs = pCfiInfo->interface.toFullMax;
    multiByteMax = pCfiInfo->geometry.fixed.multiByteMax;
    dfAssert((blockAddress & (multiByteMax - 1U)) == 0);
    dfAssert((length & (multiByteMax - 1U)) == 0);
    count = multiByteMax / flashWidth;
    while (length) {
      /* Issue buffer-write command */
      val32 = 0xe8U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Write word count */
      val32 = count - 1U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Write words to buffer */
      for (i = 0; i < count; i++) {
        switch (flashWidth) {
        case 1U:
          val32 = *p++;
          break;

        case 2U:
          val32 = (uint32_t)*p++ << 0U;
          val32 |= (uint32_t)*p++ << 8U;
          break;

        case 4U:
          val32 = (uint32_t)*p++ << 0U;
          val32 |= (uint32_t)*p++ << 8U;
          val32 |= (uint32_t)*p++ << 16U;
          val32 |= (uint32_t)*p++ << 24U;
          break;
        }
        pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
        writeAddress += flashWidth;
      }
      /* Issue write-confirm command */
      val32 = 0xd0U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Wait for completion */
      status = flashCfiWaitReady0001(pDevCtx, bankIndex, blockAddress, timeoutUs, &val32);
      if (CoreFlashSuccess != status) {
        goto failed;
      }
      if ((val32 & 0x7eU) != 0U) {
        status = CoreFlashWriteFailed;
        goto failed;
      }
      length -= multiByteMax;
    }
  } else {
    /* Using single-word programming */
    timeoutUs = pCfiInfo->interface.toSingleMax;
    while (length) {
      /* Issue program command */
      val32 = 0x40U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
      switch (flashWidth) {
      case 1U:
        val32 = *p++;
        break;

      case 2U:
        val32 = (uint32_t)*p++ << 0U;
        val32 |= (uint32_t)*p++ << 8U;
        break;

      case 4U:
        val32 = (uint32_t)*p++ << 0U;
        val32 |= (uint32_t)*p++ << 8U;
        val32 |= (uint32_t)*p++ << 16U;
        val32 |= (uint32_t)*p++ << 24U;
        break;
      }
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
      /* Wait for completion */
      status = flashCfiWaitReady0001(pDevCtx, bankIndex, writeAddress, timeoutUs, &val32);
      if (CoreFlashSuccess != status) {
        goto failed;
      }
      if ((val32 & 0x7eU) != 0U) {
        status = CoreFlashWriteFailed;
        goto failed;
      }
      writeAddress += flashWidth;
      length -= flashWidth;
    }
  }

  /* Re-lock the block against accidental modifications */
  status = cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  if (CoreFlashSuccess != status) {
    return status;
  }
  return CoreFlashSuccess;

failed:
  /* Clear status register */
  cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  /* Something went wrong, so attempt to re-lock the block */
  cfiBlockLock0001(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  return status;
}

CoreFlashStatus
flashCfiWriteBlock0200(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  uint64_t length,
  const void* pData)
{
  CoreFlashStatus status;
  Adb3CoreFlashInfo* pFlashInfo = &pDevCtx->info.flash[bankIndex];
  CfiInformation* pCfiInfo = pFlashInfo->detail.pCfi;
  unsigned int flashWidth = pDevCtx->info.bootstrap.flash[bankIndex].width;
  uint64_t writeAddress;
  uint32_t multiByteMax;
  uint32_t val32, i, count, timeoutUs;
  uint8_t* p;

  /* Unlock the block (or chip) */
  status = cfiBlockLock0200(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, FALSE);
  if (CoreFlashSuccess != status) {
    return status;
  }

  p = (uint8_t*)pData;
  writeAddress = blockAddress;
  dfAssert((blockAddress & (flashWidth - 1U)) == 0U);
  if (pCfiInfo->validation.bMultiWordSupported) {
    /* Using multi-word programming */
    timeoutUs = pCfiInfo->interface.toFullMax;
    multiByteMax = pCfiInfo->geometry.fixed.multiByteMax;
    dfAssert((blockAddress & (multiByteMax - 1U)) == 0);
    dfAssert((length & (multiByteMax - 1U)) == 0);
    count = multiByteMax / flashWidth;
    while (length) {
      /* Issue buffer-write command */
      val32 = 0xE9U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Write word count */
      val32 = count - 1U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Write words to buffer */
      for (i = 0; i < count; i++) {
        switch (flashWidth) {
        case 1U:
          val32 = *p++;
          break;

        case 2U:
          val32 = (uint32_t)*p++ << 0U;
          val32 |= (uint32_t)*p++ << 8U;
          break;

        case 4U:
          val32 = (uint32_t)*p++ << 0U;
          val32 |= (uint32_t)*p++ << 8U;
          val32 |= (uint32_t)*p++ << 16U;
          val32 |= (uint32_t)*p++ << 24U;
          break;
        }
        pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
        writeAddress += flashWidth;
      }
      /* Issue write-confirm command */
      val32 = 0xD0U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
      /* Wait for completion */
      status = flashCfiWaitReady0001(pDevCtx, bankIndex, blockAddress, timeoutUs, &val32);
      if (CoreFlashSuccess != status) {
        goto failed;
      }
      if ((val32 & 0x7eU) != 0U) {
        status = CoreFlashWriteFailed;
        goto failed;
      }
      length -= multiByteMax;
    }
  } else {
    /* Using single-word programming */
    timeoutUs = pCfiInfo->interface.toSingleMax;
    while (length) {
      /* Issue program command */
      val32 = 0x41U;
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
      switch (flashWidth) {
      case 1U:
        val32 = *p++;
        break;

      case 2U:
        val32 = (uint32_t)*p++ << 0U;
        val32 |= (uint32_t)*p++ << 8U;
        break;

      case 4U:
        val32 = (uint32_t)*p++ << 0U;
        val32 |= (uint32_t)*p++ << 8U;
        val32 |= (uint32_t)*p++ << 16U;
        val32 |= (uint32_t)*p++ << 24U;
        break;
      }
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, writeAddress, &val32);
      /* Wait for completion */
      status = flashCfiWaitReady0001(pDevCtx, bankIndex, writeAddress, timeoutUs, &val32);
      if (CoreFlashSuccess != status) {
        goto failed;
      }
      if ((val32 & 0x7eU) != 0U) {
        status = CoreFlashWriteFailed;
        goto failed;
      }
      writeAddress += flashWidth;
      length -= flashWidth;
    }
  }

  /* Re-lock the block against accidental modifications */
  status = cfiBlockLock0200(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  if (CoreFlashSuccess != status) {
    return status;
  }
  return CoreFlashSuccess;

failed:
  /* Clear status register */
  cfiClearStatus(pDevCtx, bankIndex, blockAddress);
  /* Something went wrong, so attempt to re-lock the block */
  cfiBlockLock0200(pDevCtx, bankIndex, pCfiInfo->chipAddress, blockAddress, TRUE);
  return status;
}
