/*
** File: ddr3_spd_common.c  
** Project: ADB3 core driver
** Purpose: Functions for reading DDR3 DIMM SPD information via an I2C interface.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include "ddr3_spd_common.h"

boolean_t
admxrc6txGetDdr3DimmSpd(
  I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t slot,
  Ddr3DimmSummary* pSummary,
  uint8_t* pRaw)
{
  uint8_t val8, raw[DDR3_SPD_ROM_SIZE];
  unsigned int addr;
  I2cStatus i2cStatus;

  for (addr = 0; addr < DDR3_SPD_ROM_SIZE; addr++) {
    i2cStatus = i2cReadSync(pI2cCtx, bus, slot, (uint8_t)addr, &val8);
    switch (i2cStatus) {
    case I2cStatusSuccess:
      raw[addr] = val8;
      break;

    case I2cStatusTimeout:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: I2C interface timeout, bus=%u slot=0x%02x addr=0x%02x\n",
        (unsigned int)bus, (unsigned int) slot, (unsigned int)addr));
      return FALSE;

    case I2cStatusHardwareError:
      /* DIMM not present, not an error. */
      return FALSE;

    case I2cStatusInvalidBus:
    case I2cStatusInvalidDevice:
    case I2cStatusInvalidAddress:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: I2C interface invalid bus/slot/adddress error, bus=%u slot=0x%02x addr=0x%02x\n",
        (unsigned int)bus, (unsigned int) slot, (unsigned int)addr));
      return FALSE;

    case I2cStatusGeneralFailure:
    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: I2C interface read failure, i2cStatus=%u bus=%u slot=0x%02x addr=0x%02x\n",
        i2cStatus, (unsigned int)bus, (unsigned int) slot, (unsigned int)addr));
      return FALSE;
    }
  }

  dfDebugPrint(2, ("admxrc6txGetDdr3DimmSpd: Dump of DIMM SPD raw data @ I2C bus %u slot %u:\n", bus, slot));
  dfDebugDumpMemory(2, 2, 16, 1, FALSE, raw, DDR3_SPD_ROM_SIZE);

  if (NULL != pSummary) {
    Ddr3SdramSpd* pStruct = (Ddr3SdramSpd*)&raw[0];
    uint8_t tmp8;
    unsigned int spdBytesUsedValue = 0U;
    unsigned int spdBytesTotalValue = 0U;
    uint64_t deviceCapacityBits = 0U;
    uint8_t deviceWidthValue = 0U;
    uint8_t numRanksValue = 0U;
    uint16_t primaryBusWidthValue = 0U;
    uint16_t busWidthExtensionValue = 0U;
    uint64_t termA;
    uint32_t termB;
    int32_t termA1, termA2, termB1, termB2;
    uint8_t Fn, Fd, Mn, Md, tCKM;
    int8_t tCKF;
    uint64_t fCKmax = 0U;
    boolean_t bCalcFreq = TRUE;

    DF_ZERO_OBJECT(pSummary);

    /* Sanity check for SPD Bytes Used field */
    tmp8 = (uint8_t)(pStruct->size & 0xFU);
    switch (tmp8) {
    case 0U:
      break;

    case 1U:
      spdBytesUsedValue = 128U;
      break;

    case 2U:
      spdBytesUsedValue = 176U;
      break;

    case 3U:
      spdBytesUsedValue = 256U;
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid SPD Bytes Used field 0x%x\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    /* Sanity check for SPD Bytes Total field */
    tmp8 = (uint8_t)((pStruct->size >> 4) & 0x7U);
    switch (tmp8) {
    case 0U:
      break;

    case 1U:
      spdBytesTotalValue = 256U;
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid SPD Bytes Total field 0x%x\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    /* Sanity check for DRAM type = DDR3 SDRAM */
    tmp8 = pStruct->dramType;
    switch (tmp8) {
    case 0x0BU:
      /* DDR3 SDRAM */
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, not DDR3 SDRAM (dramType=0x%02x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    tmp8 = (uint8_t)(pStruct->densityBanks & 0xFU);
    deviceCapacityBits = ((uint64_t)0x10000000U) << tmp8;
    if (tmp8 > 6) {
      /* Field value of 0x6 */
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Total SDRAM capacity field (0x%x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
    }

    tmp8 = (uint8_t)(pStruct->organization & 0x7U);
    switch (tmp8) {
    case 0x0U:
    case 0x1U:
    case 0x2U:
    case 0x3U:
      deviceWidthValue = (uint8_t)(4U << tmp8);
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid SDRAM Device Width field (0x%x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    tmp8 = (uint8_t)((pStruct->organization >> 3) & 0x7U);
    switch (tmp8) {
    case 0x0U:
    case 0x1U:
    case 0x2U:
    case 0x3U:
      numRanksValue = (uint8_t)(tmp8 + 1U);
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Number of Ranks field (0x%x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    tmp8 = (uint8_t)(pStruct->busWidth & 0x7U);
    switch (tmp8) {
    case 0x0U:
    case 0x1U:
    case 0x2U:
    case 0x3U:
      primaryBusWidthValue = (uint8_t)(8U << tmp8);
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Primary bus width field (0x%x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    tmp8 = (uint8_t)((pStruct->busWidth >> 3) & 0x7U);
    switch (tmp8) {
    case 0x0U:
      break;

    case 0x1U:
      busWidthExtensionValue = 8U;
      break;

    default:
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Bus width extension field (0x%x)\n", bus, slot, (unsigned int)tmp8));
      pSummary->bRomCorrupt = TRUE;
      break;
    }

    /* Store summary ECC and data widths */
    pSummary->dataWidth = primaryBusWidthValue;
    pSummary->eccWidth = busWidthExtensionValue;

    /* Calculate summary total DIMM size in words of 'primaryBusWidthValue' bits. */
    dfDebugPrint(2, ("admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, geometry: deviceCapacityBits=0x%08lx_%08lx deviceWidth=%u numRanks=%u\n",
      bus, slot, dfSplitUint64(deviceCapacityBits), deviceWidthValue, numRanksValue));
    if (0 != deviceWidthValue) {
      pSummary->physicalSize = dfDivide64By32(deviceCapacityBits, deviceWidthValue) * numRanksValue;
    }

    dfDebugPrint(1, ("admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, geometry: dataWidth=%u eccWidth=%u physicalSize=0x%08lx_%08lx\n",
      bus, slot, (unsigned int)pSummary->dataWidth, (unsigned int)pSummary->eccWidth, dfSplitUint64(pSummary->physicalSize)));

    /*
    ** Calculate maximum operating frequency (for any supported CAS latency). Start from
    ** expression in JEDEC Standard No. 21-C Annex K, which can be written as
    **
    **    tCK = tCKM * (Mn/Md) + tCKF * 0.001*(Fn/Fd)                                   (ns)
    **
    ** => tCK = 10e-9 * (tCKM * (Mn/Md) + tCKF * 0.001*(Fn/Fd))                         (s)
    **
    ** => fCK = 10e9 / (tCKM * (Mn/Md) + tCKF * 0.001*(Fn/Fd))                          (Hz)
    **
    ** Multiply top & buttom of RHS by Md * 1000 * Fd:
    **
    **    fCK = 10e9 * Md * 1000 * Fd / (Mn * tCK(M) * 1000 * Fd + tCK(F) * Fn * Md)    (Hz)
    **
    ** where:
    **    tCKM = tCK (in medium timebase units, (1..255)
    **    tCKF = tCK fine correction (in fine timebase units), -128..127
    **    Fn = fine timebase value numerator, 1..15
    **    Fd = fine timebase value denominator, 1..15
    **    (Fn/Fd) = fine timebase value (ps)
    **    Mn = medium timebase value numerator, 1..255
    **    Md = medium timebase value denominator, 1..255
    **    (Mn/Md) = fine timebase value (ns)
    **
    ** Define terms as follows:
    **
    **    a1 = 10e9
    **    a2 = Md * 1000 * Fd;
    **    a = a1 * a2                   requires ~55 bits => uint64_t
    **    b1 = Mn * tCKM * 1000 * Fd    requires ~30 bits => int32_t
    **    b2 = tCKF * Fn * Md           requires ~20 bits, signed => int32_t
    **    b = b1 + b2                   requires ~30 bits => uint32_t
    **
    ** NOTE: although b1 and b2 are signed, it is an error if b1 <= b2. So OK to use uint32_t for b.
    */
    tmp8 = pStruct->ftb;
    Fn = (uint8_t)((tmp8 >> 4) & 0xFU);
    if (0U == Fn) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Fine Timebase Dividend field (%u)\n", bus, slot, (unsigned int)Fn));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    Fd = (uint8_t)((tmp8 >> 0) & 0xFU);
    if (0U == Fd) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Fine Timebase Divisor field (%u)\n", bus, slot, (unsigned int)Fd));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    Mn = pStruct->mtbDividend;
    if (0U == Mn) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Medium Timebase Dividend field (%u)\n", bus, slot, (unsigned int)Mn));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    Md = pStruct->mtbDivisor;
    if (0U == Md) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid Medium Timebase Divisor field (%u)\n", bus, slot, (unsigned int)Md));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    dfDebugPrint(2, ("admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x timebase values: Fn=%u Fd=%u Mn=%u Md=%u\n",
      bus, slot, (unsigned int)Fn, (unsigned int)Fd, (unsigned int)Mn, (unsigned int)Md));
    tCKM = pStruct->tCKmin;
    if (0U == tCKM) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, invalid tCKmin value (%u)\n", bus, slot, (unsigned int)tCKM));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    tCKF = pStruct->tCKminOffset;
    dfDebugPrint(2, ("admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x tCKmin values: tCKM=%u tCKF=%d\n", bus, slot, (unsigned int)tCKM, (int)tCKF));
    termA1 = 1000000000U;
    termA2 = (uint32_t)Md * (uint32_t)1000U * (uint32_t)Fd;
    termA = (uint64_t)termA1 * termA2;
    termB1 = (int32_t)Mn * (int32_t)tCKM * (int32_t)1000 * (int32_t)Fd;
    termB2 = (int32_t)tCKF * (int32_t)Fn * (int32_t)Md;
    if (termB1 <= termB2) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, tCK medium (%u) & fine (%u) values imply period is zero or negative (%u)\n",
        bus, slot, (unsigned int)tCKM, (unsigned int)tCKF));
      pSummary->bRomCorrupt = TRUE;
      bCalcFreq = FALSE;
    }
    termB = (uint32_t)(termB1 + termB2);
    if (bCalcFreq) {
      fCKmax = dfDivide64By32(termA, termB);
    }
    if (fCKmax > 0xFFFFFFFFU) {
      dfDebugPrint(0, ("*** admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x, calculated fCKmax value (" DF_FMT_U64 "( overflows 32 bits!\n", bus, slot, fCKmax));
      pSummary->freqMax = 0U;
      pSummary->bRomCorrupt = TRUE;
    } else {
      dfDebugPrint(1, ("admxrc6txGetDdr3DimmSpd: bus=%u slot=0x%02x fCKmax=" DF_FMT_U64 " Hz\n", bus, slot, fCKmax));
      pSummary->freqMax = (uint32_t)fCKmax;
    }
  }

  if (NULL != pRaw) {
    dfCopyKK(pRaw, raw, DDR3_SPD_ROM_SIZE);
  }

  return TRUE;
}
