/*
** File: coredebugts.c  
** Project: ADB3 core driver
** Purpose: Core Interface I/O debug timestamping functions.
**
** (C) Copyright Alpha Data 2016
*/

#include <df.h>
#include "device.h"
#include "coredebugts.h"

#if defined(BUILD_DEBUGTS)
#include <admxrc2/debugtst.h>

/* Must be called while holding the lock appropriate to the facility */
static CoreDebugTSResult
fillStatusStruct(
  DfTimestampBuffer* pTSBuf,
  CoreDebugTSStatus* pStatus)
{
  uint32_t head, tail, count, size;
  boolean_t bOverflow;
  DfSpinLockFlags f;

  if (!dfTimestampBufferEnabled(pTSBuf)) {
    return CoreDebugTSNotEnabled;
  }

  size = pTSBuf->buffer.size;

  f = dfSpinLockGet(&pTSBuf->lock);
  head = pTSBuf->state.head;
  tail = pTSBuf->state.tail;
  count = (tail >= head) ? tail - head : size + tail - head;
  bOverflow = pTSBuf->state.bOverflow;
  dfSpinLockPut(&pTSBuf->lock, f);

  pStatus->count = count;
  pStatus->status = bOverflow ? ADMXRC2_DEBUGTS_STATUS_OVERFLOW : 0;

  return CoreDebugTSSuccess;
}

CoreDebugTSResult
coreGetDebugTSInfo(
  void* pInterfaceContext,
  unsigned int facility,
  CoreDebugTSInfo* pDebugTSInfo)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfTimestampBuffer* pTSBuf;

  if (facility < pDevCtx->info.bootstrap.numDmaChannel) {
    DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[facility];
    pTSBuf = &pDmaContext->timestampBuffer;
  } else {
    return CoreDebugTSInvalidFacility;
  }

  if (!dfTimestampBufferEnabled(pTSBuf)) {
    return CoreDebugTSNotEnabled;
  }

  /* Constant fields that don't need to be accessed while holding a lock */
  pDebugTSInfo->bufferSize = pTSBuf->buffer.size;
  pDebugTSInfo->frequency = pTSBuf->attributes.frequency;
  pDebugTSInfo->initTimestamp = pTSBuf->attributes.initTimestamp;
  pDebugTSInfo->initTime = pTSBuf->attributes.initTime;

  return CoreDebugTSSuccess;
}

CoreDebugTSResult
coreGetDebugTSStatus(
  void* pInterfaceContext,
  unsigned int facility,
  CoreDebugTSStatus* pDebugTSStatus)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreDebugTSResult result;

  if (facility < pDevCtx->info.bootstrap.numDmaChannel) {
    DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[facility];
    DfTimestampBuffer* pTSBuf = &pDmaContext->timestampBuffer;

    result = fillStatusStruct(pTSBuf, pDebugTSStatus);
  } else {
    return CoreDebugTSInvalidFacility;
  }

  return result;
}

CoreReadDebugTSResult
coreReadDebugTS(
  void* pInterfaceContext,
  unsigned int facility,
  size_t offset,
  size_t count,
  uint32_t flags,
  void* pBuffer)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  if (facility < pDevCtx->info.bootstrap.numDmaChannel) {
    DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[facility];
    DfTimestampBuffer* pTSBuf = &pDmaContext->timestampBuffer;

    if (!dfTimestampBufferEnabled(pTSBuf)) {
      return CoreReadDebugTSNotEnabled;
    }

    if (offset > 0xFFFFFFFFU || (offset + count) > 0xFFFFFFFFU) {
      return CoreReadDebugTSInvalidRegion;
    }

    if (!dfTimestampBufferRead(pTSBuf, (uint32_t)offset, (uint32_t)count, pBuffer, (flags & ADMXRC2_DEBUGTS_FLAG_CONSUME) ? TRUE : FALSE)) {
      return CoreReadDebugTSInvalidRegion;
    }

    return CoreReadDebugTSSuccess;
  } else {
    return CoreReadDebugTSInvalidFacility;
  }
}

CoreDebugTSResult
coreResetDebugTS(
  void* pInterfaceContext,
  unsigned int facility)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  if (facility < pDevCtx->info.bootstrap.numDmaChannel) {
    DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[facility];
    DfTimestampBuffer* pTSBuf = &pDmaContext->timestampBuffer;

    if (!dfTimestampBufferReset(pTSBuf)) {
      return CoreDebugTSNotEnabled;
    }

    return CoreDebugTSSuccess;
  } else {
    return CoreDebugTSInvalidFacility;
  }
}
#endif
