/*
** File: cfi.h  
** Project: ADB3 core driver
** Purpose: Definitions for Common Flash Interface (CFI).
**
** (C) Copyright Alpha Data 2009-2010
*/

#ifndef _ADATA_CORE_CFI_H
#define _ADATA_CORE_CFI_H

#include <df.h>

/*
** ---------------------------------------------------------------------------
** Basic query information
** ---------------------------------------------------------------------------
*/

/* Values for CfiQueryString::{primary,alt}::algorithm */
#define CFI_ALG_NONE            (0x0U)    /* No command set */
#define CFI_ALG_INTELSHARP_EXT  (0x1U)    /* Intel / Sharp extended command set */
#define CFI_ALG_AMD_FUJITSU_STD (0x2U)    /* AMD / Fujitsu standard command set */
#define CFI_ALG_INTEL_STD       (0x3U)    /* Intel standard command set */
#define CFI_ALG_AMD_FUJITSU_EXT (0x4U)    /* AMD / Fujitsu extended command set */
#define CFI_ALG_WINBOND_STD     (0x6U)    /* Winbond standard command set */
#define CFI_ALG_MITSUBISHI_STD  (0x100U)  /* Mitsubishi standard command set */
#define CFI_ALG_MITSUBISHI_EXT  (0x101U)  /* Mitsubishi extended command set */
#define CFI_ALG_SST_PAGE_WRITE  (0x102U)  /* SST page write command set */
#define CFI_ALG_INTEL_PERF      (0x200U)  /* Intel performance command set */
#define CFI_ALG_INTEL_DATA      (0x210U)  /* Intel data command set */

struct _CfiEraseRegion;
typedef struct _CfiEraseRegion CfiEraseRegion;
struct _CfiEraseRegionRaw;
typedef struct _CfiEraseRegionRaw CfiEraseRegionRaw;
struct _CfiExtendedQueryHeader;
typedef struct _CfiExtendedQueryHeader CfiExtendedQueryHeader;
struct _CfiIntelSharpFixed;
typedef struct _CfiIntelSharpFixed CfiIntelSharpFixed;
struct _CfiIntelSharpFixedRaw1;
typedef struct _CfiIntelSharpFixedRaw1 CfiIntelSharpFixedRaw1;
struct _CfiIntelSharpFixedRaw2;
typedef struct _CfiIntelSharpFixedRaw2 CfiIntelSharpFixedRaw2;
struct _CfiIntelSharpFixedRaw3;
typedef struct _CfiIntelSharpFixedRaw3 CfiIntelSharpFixedRaw3;
struct _CfiIntelSharpProtRegRaw;
typedef struct _CfiIntelSharpProtRegRaw CfiIntelSharpProtRegRaw;
struct _CfiIntelSharpFixedRaw4;
typedef struct _CfiIntelSharpFixedRaw4 CfiIntelSharpFixedRaw4;
struct _CfiIntelSharpFixedRaw5;
typedef struct _CfiIntelSharpFixedRaw5 CfiIntelSharpFixedRaw5;
struct _CfiIntelSharpPartitionInfoRaw;
typedef struct _CfiIntelSharpPartitionInfoRaw CfiIntelSharpPartitionInfoRaw;
struct _CfiIntelSharpLinkRaw;
typedef struct _CfiIntelSharpLinkRaw CfiIntelSharpLinkRaw;
struct _CfiGeometry;
typedef struct _CfiGeometry CfiGeometry;
struct _CfiGeometryFixed;
typedef struct _CfiGeometryFixed CfiGeometryFixed;
struct _CfiGeometryFixedRaw;
typedef struct _CfiGeometryFixedRaw CfiGeometryFixedRaw;
struct _CfiInformation;
typedef struct _CfiInformation CfiInformation;
struct _CfiQueryString;
typedef struct _CfiQueryString CfiQueryString;
struct _CfiQueryStringRaw;
typedef struct _CfiQueryStringRaw CfiQueryStringRaw;
struct _CfiSystemInterface;
typedef struct _CfiSystemInterface CfiSystemInterface;
struct _CfiSystemInterfaceRaw;
typedef struct _CfiSystemInterfaceRaw CfiSystemInterfaceRaw;

#pragma pack(1)
struct _CfiQueryStringRaw {
  uint8_t                   magic[3];  /* ASCII 'Q', 'R', 'Y' */
  uint8_t                   priAlgorithmLo;
  uint8_t                   priAlgorithmHi;
  uint8_t                   priAddressLo;
  uint8_t                   priAddressHi;
  uint8_t                   altAlgorithmLo;
  uint8_t                   altAlgorithmHi;
  uint8_t                   altAddressLo;
  uint8_t                   altAddressHi;
};
#pragma pack()

struct _CfiQueryString {
  uint8_t                   magic[3];  /* ASCII 'Q', 'R', 'Y' */
  struct {
    uint16_t                algorithm; /* See above definitions */
    uint16_t                address;
    CfiExtendedQueryHeader* pExtended;
  } primary, alt;
};

#pragma pack(1)
struct _CfiSystemInterfaceRaw {
  /* Following 4 members are all BCD-encoded in units of 0.1V */
  uint8_t            vccProgMin;       /* Minimum VCC during programming */
  uint8_t            vccProgMax;       /* Maximum VCC during programming */
  uint8_t            vppProgMin;       /* Minimum VPP during programming, or 0 if no VPP pin */
  uint8_t            vppProgMax;       /* Maximum VPP during programming, or 0 if no VPP pin */
  /* Following 4 members are all log base 2 of number of units */
  uint8_t            toSingleTyp;      /* Typical timeout (us) for single word programming, or 0 if not supported */
  uint8_t            toFullTyp;        /* Typical timeout (us) for full multi-word programming, or 0 if not supported */
  uint8_t            toEraseTyp;       /* Typical timeout (ms) for block erase, or 0 if not supported */
  uint8_t            toChipTyp;        /* Typical timeout (ms) for full-chip erase, or 0 if not supported */
  /* Following 4 members are all log base 2 of multiple of corresponding above 4 members */
  uint8_t            toSingleMax;      /* Maximum timeout for single word programming, or 0 if not supported */
  uint8_t            toFullMax;        /* Maximum timeout for full multi-word programming, or 0 if not supported */
  uint8_t            toEraseMax;       /* Maximum timeout for block erase, or 0 if not supported */
  uint8_t            toChipMax;        /* Maximum timeout for full-chip erase, or 0 if not supported */
};
#pragma pack()

struct _CfiSystemInterface {
  /* Following 4 members are all in units of 0.1V */
  uint8_t            vccProgMin;       /* Minimum VCC during programming */
  uint8_t            vccProgMax;       /* Maximum VCC during programming */
  uint8_t            vppProgMin;       /* Minimum VPP during programming, or 0 if no VPP pin */
  uint8_t            vppProgMax;       /* Maximum VPP during programming, or 0 if no VPP pin */
  uint32_t           toSingleTyp;      /* Typical timeout (us) for single word programming, or 0 if not supported */
  uint32_t           toFullTyp;        /* Typical timeout (us) for full multi-word programming, or 0 if not supported */
  uint32_t           toEraseTyp;       /* Typical timeout (ms) for block erase, or 0 if not supported */
  uint32_t           toChipTyp;        /* Typical timeout (ms) for full-chip erase, or 0 if not supported */
  /* Following 4 members are all in microseconds */
  uint32_t           toSingleMax;     /* Maximum timeout (us) for single word programming, or 0 if not supported */
  uint32_t           toFullMax;       /* Maximum timeout (us) for full multi-word programming, or 0 if not supported */
  uint32_t           toEraseMax;      /* Maximum timeout (ms) for block erase, or 0 if not supported */
  uint32_t           toChipMax;       /* Maximum timeout (ms) for full-chip erase, or 0 if not supported */
};

#pragma pack(1)
struct _CfiEraseRegionRaw {
  uint16_t           numBlock;        /* Number of blocks, less one, in this region (eg. 0xff => 256) */
  /* The following member is in units of 256 bytes */
  uint16_t           blockSize;       /* Size of each block in this region */
};
#pragma pack()

struct _CfiEraseRegion {
  uint64_t           startAddress;     /* Start address of this region */
  uint64_t           blockSize;        /* Size of each block in this region */
  uint64_t           numBlock;         /* Number of blocks in this region */
};

/* Codes for CfiGeometry::interfaceCode */
#define CFI_DIC_X8_ASYNC     (0)       /* x8 only, asynchronous interface                     */
#define CFI_DIC_X16_ASYNC    (1)       /* x16 only, asynchronous interface                    */
#define CFI_DIC_X8X16_ASYNC  (2)       /* x8 or x16 via BYTE# signal, asynchronous interface  */
#define CFI_DIC_X32_ASYNC    (3)       /* x32 only, asynchronous interface                    */
#define CFI_DIC_X16X32_ASYNC (4)       /* x16 or x32 via WORD# signal, asynchronous interface */

#pragma pack(1)
struct _CfiGeometryFixedRaw {
  uint8_t          order;              /* Log base 2 of device size in bytes */
  uint8_t          interfaceCodeLo;    /* Device interface code */
  uint8_t          interfaceCodeHi;    /* Device interface code */
  /* The following member is log base 2 of number of bytes */
  uint8_t          multiByteMaxLo;     /* Maximum number of bytes per multi-word programming sequence */
  uint8_t          multiByteMaxHi;     /* Maximum number of bytes per multi-word programming sequence */
  uint8_t          numEraseRegion;     /* Number of distinct erase-block regions, or 0 if entire device erases at once */
};
#pragma pack()

struct _CfiGeometryFixed {
  uint64_t         size;               /* Die size, in bytes */
  uint16_t         interfaceCode;      /* Die interface code */
  uint32_t         multiByteMax;       /* Maximum number of bytes per multi-word programming sequence */
  uint8_t          numEraseRegion;     /* Number of distinct erase-block regions, or 0 if entire device erases at once */
};

struct _CfiGeometry {
  CfiGeometryFixed   fixed;
  CfiEraseRegion*    pEraseRegion;     /* Erase-block region information, variable length array */
};

/*
** This structure is used to represent a Flash device, OR a single die within
** a stacked-die Flash device. In the case of a stacked-die device, the validation.pLink
** field is non-NULL for all dies except the last.
*/
struct _CfiInformation {
  /* Following fields are initialized by cfiIdentify */
  CfiQueryString header;
  boolean_t bInterfaceValid;
  CfiSystemInterface interface;
  boolean_t bGeometryValid;
  CfiGeometry geometry;
  uint64_t chipAddress;

  /* Following substruct is filled in by cfiValidateInformation */
  struct {
    uint64_t largestBlock;
    uint16_t preferredAlgorithm;
    uint16_t vendorId;
    uint16_t deviceId;
    boolean_t bSingleWordSupported;
    boolean_t bMultiWordSupported;
    boolean_t bBlockEraseSupported;
    boolean_t bChipEraseSupported;
    union {
      struct {
        void* pExtended;
      } generic;
      struct {
        /*
        ** Member pExtended below is set to this::header::primary::pExtended or this::header::alt::pExtended
        ** depending on whether primary or alternate algorithm is preferred.
        */
        CfiIntelSharpFixed* pExtended;
        boolean_t bLegacyBlockLockSupported;
        boolean_t bInstantBlockLockSupported;
      } x0001;
    } algorithm;
    /* bAnotherStruct is true and pLink is non-NULL for each die on a stack-die device that is not the last */
    boolean_t bAnotherStruct;
    uint64_t linkAddress;
    uint32_t linkOffset;
    CfiInformation* pLink;
  } validation;
};

/* Table for primary extended algorithm or alternative extended algorithm */
#pragma pack(1)
struct _CfiExtendedQueryHeader {
  uint8_t            magic[3];         /* ASCII 'P', 'R', 'I' or 'A', 'L', 'T' */
  uint8_t            major;            /* Major version, ASCII character */
  uint8_t            minor;            /* Minor version, ASCII character */
};
#pragma pack()

/*
** ---------------------------------------------------------------------------
** Intel / Sharp extended command set information
**
** Used by Xilinx XCFxxxX, Numonyx, Intel among others.
**
** The data structure is of variable length and looks like:
**
** {
**   CfiIntelSharpFixedRaw1
**   Zero or more 32-bit fields defining additional capabilities
**   CfiIntelSharpFixedRaw2
**   CfiIntelSharpFixedRaw3
**   Zero or more CfiIntelSharpProtRegRaw fields
**   CfiIntelSharpFixedRaw4
**   Zero or more burst read configuration fields (1 byte each)
**   CfiIntelSharpFixedRaw5
**   Zero or CfiIntelSharpPartitionInfoRaw fields (if CFI version is 1.4+,
**     each field will be preceeded by a 2-byte length value)
**   CfiIntelSharpLinkRaw
** }
**
** ---------------------------------------------------------------------------
*/

#pragma pack(1)
struct _CfiIntelSharpFixedRaw1 {
  CfiExtendedQueryHeader header;
  /* 32-bit field that defines capabilities and features of the device */
  uint8_t                capsByte0;
  uint8_t                capsByte1;
  uint8_t                capsByte2;
  uint8_t                capsByte3;
};
#pragma pack()

#pragma pack(1)
struct _CfiIntelSharpFixedRaw2 {
  /* Capabilities related to what functions are permitted after suspend */
  uint8_t                caps2;
  /* 16-bit field that can be read to get block lockdown status */
  uint8_t                blkProtStatLo;
  uint8_t                blkProtStatHi;
  /* Following 2 members are BCD-encoded in units of 0.1V */
  uint8_t                vddProgOpt;   /* Optimum VDD during programming */
  uint8_t                vppProgOpt;   /* Optimum VPP during programming */
};
#pragma pack()

/* This structure in V1.1+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpFixedRaw3 {
  uint8_t                numProtReg;   /* Number of protection register regions */
  struct {
    uint8_t              physLo;       /* Lock byte JEDEC address low */
    uint8_t              physHi;       /* Lock byte JEDEC address high */
    uint8_t              factory;      /* Log base 2 of number of bytes */
    uint8_t              user;         /* Log base 2 of number of bytes */
  } reg0;
};
#pragma pack()

/* This structure in V1.1+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpProtRegRaw {
  uint8_t              phys0;               /* Lock word JEDEC address bits 7:0 */
  uint8_t              phys1;               /* Lock word JEDEC address bits 15:8 */
  uint8_t              phys2;               /* Lock word JEDEC address bits 23:16 */
  uint8_t              phys3;               /* Lock word JEDEC address bits 31:24 */
  uint8_t              numFactoryGroupLo;   /* Low byte of no. of factory groups */
  uint8_t              numFactoryGroupHi;   /* High byte of no. of factory groups */
  uint8_t              factoryGroupSize;    /* Log base 2 of number of bytes in factory group */
  uint8_t              numUserGroupLo;      /* Low byte of no. of user groups */
  uint8_t              numUserGroupHi;      /* High byte of no. of user groups */
  uint8_t              userGroupSize;       /* Log base 2 of number of bytes in user group */
};
#pragma pack()

/* This structure in V1.1+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpFixedRaw4 {
  uint8_t              pageRead;            /* Log base 2 of max. page read length */
  uint8_t              numSyncMode;         /* Number of synchronous read modes */
};
#pragma pack()

/* This structure in V1.1+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpLinkRaw {
  /*
  ** The following 4 bytes make up a 32-bit field as follows:
  ** 9:0    offset in referenced 32 MBit segment, in bytes
  ** 27:10  index of referenced 32 MBit segment
  ** 30:28  memory type
  ** 31     1 => another link field follows immediately
  */
  uint8_t              linkByte0;           /* Link lowest byte */
  uint8_t              linkByte1;
  uint8_t              linkByte2;
  uint8_t              linkByte3;           /* Link highest byte */
  /*
  ** Quantity and flags:
  ** 3:0    quantity, minus 1
  ** 4      1 => table and die relative location
  ** 5      1 => link field and table relative location
  ** 7:6    (reserved)
  */
  uint8_t              quantityFlags;
};
#pragma pack()

/* This structure in V1.3+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpFixedRaw5 {
  uint8_t              numPartitionRegion;  /* Number of hardware partition regions in the device */
};
#pragma pack()

/* This structure in V1.3+ of data structure */
#pragma pack(1)
struct _CfiIntelSharpPartitionInfoRaw {
  struct {
    uint8_t            numIdenticalLo;      /* Lo byte of number of identical partitions within the region */
    uint8_t            numIdenticalHi;      /* Hi byte of number of identical partitions within the region */
    /*
    ** Number of program or erase operations allowed in a partition
    ** 3:0    no. of program operations
    ** 7:4    no. of erase operations
    */
    uint8_t            numProgErase;
    /*
    ** Simultaneous program or erase operations allowed in this partition
    ** 3:0    no. of simultaneous program operations
    ** 7:4    no. of simultaneous erase operations
    */
    uint8_t            numProgEraseSimul;
    /*
    ** Simultaneous program or erase operations allowed in other partitions while a partition in this region is in program mode
    ** 3:0    no. of simultaneous program operations
    ** 7:4    no. of simultaneous erase operations
    */
    uint8_t            numProgEraseSimulProg;
    /* Number of types of erase block regions in this partition region; 0 => partition erases in bulk */
    uint8_t            numEraseBlockRegions;
  } fixed;
  struct {
    uint8_t            numEraseBlockLo;     /* Lo byte of no. of identically-sized erase blocks in partition */
    uint8_t            numEraseBlockHi;     /* Hi byte of no. of identically-sized erase blocks in partition */
    uint8_t            eraseBlockSizeLo;    /* Lo byte of erase block size in partition, divided by 256, in bytes */
    uint8_t            eraseBlockSizeHi;    /* Hi byte of erase block size in partition, divided by 256, in bytes */
    uint8_t            eraseCyclesLo;       /* Lo byte of no. of guaranteed erase cycles for a block, divided by 1000 */
    uint8_t            eraseCyclesHi;       /* Lo byte of no. of guaranteed erase cycles for a block, divided by 1000 */
    /*
    ** 3:0    bits per cell
    ** 4      1 => internal EDAC used
    ** 7:5    (reserved)
    */
    uint8_t            flags;
    /* Capabilities:
    ** 0      1 => page mode host reads permitted
    ** 1      1 => synchronous host reads permitted
    ** 2      1 => synchronous host writes permitted
    ** 7:3    (reserved)
    */
    uint8_t            caps;
    uint8_t            progAlignedSize;     /* Log base 2 of size of a programming block, in bytes */
    uint8_t            progLegacy;          /* Bit 7 = 1 => legacy operation, ignore 'progAlignedSize' */
    uint8_t            controlAlignedSize;  /* Control mode valid size, in bytes */
    uint8_t            controlLegacy;       /* Bit 7 = 1 => legacy operation, ignore 'controlAlignedSize' */
    uint8_t            invalidAlignedSize;  /* Control mode invalid size, in bytes */
    uint8_t            invalidLegacy;       /* Bit 7 = 1 => legacy operation, ignore 'invalidAlignedSize' */
  } region[1];
};
#pragma pack()

#define CFI_ALG0001_CAPS_CHIP_ERASE         (0x1U << 0)
#define CFI_ALG0001_CAPS_LEGACY_BLOCK_LOCK  (0x1U << 3)
#define CFI_ALG0001_CAPS_INSTANT_BLOCK_LOCK (0x1U << 5)
#define CFI_ALG0001_CAPS_LINK_NEXT          (0x1U << 30)

struct _CfiIntelSharpFixed {
  CfiExtendedQueryHeader header;
  uint32_t               caps;
  uint8_t                caps2;
  uint16_t               blkProtStat;
  /* Following 2 members are in units of 0.1V */
  uint8_t                vddProgOpt;   /* Optimum VDD during programming */
  uint8_t                vppProgOpt;   /* Optimum VPP during programming */
  /* Following 2 members are not part of the CFI data structure, but are obtained via query device ID command */
  uint16_t               vendorId;
  uint16_t               deviceId;
  /* If caps[30] = 1, following two fields give address of next CFI structure in a stacked-die device */
  uint64_t               linkAddress;
  uint32_t               linkOffset;
};

#endif
