/*
** File: avr2_clock.c  
** Project: ADB3 core driver
** Purpose: Routines for programming clock generators via AVR2 interface.
**
** (C) Copyright Alpha Data 2015
**
** NOTES
**
** 1. Clock word layout is:
**
**   d0[31:0]   (unused)
**   d1[31:0]   (unused)
**   d2[31:0]   (unused)
**   frequency  Frequency, in Hz
*/

#include "device.h"
#include "avr2_common.h"
#include "avr2_clock.h"
#include "avr2_clock_common.h"

#define AVR2_FACILITY_USER_EXT (0x05)
#define AVR2_COMMAND_SET_FREQ  (0x8E)
#define AVR2_COMMAND_GET_FREQ  (0x8F)

static Avr2ClockStatus
mapAvr2TransStatus(
  Avr2TransStatus transStatus)
{
  switch (transStatus) {
  case Avr2TransStatusSuccess:
    return Avr2ClockStatusSuccess;

  case Avr2TransStatusRXTimeout:
  case Avr2TransStatusTXTimeout:
  case Avr2TransStatusQueueTimeout:
    return Avr2ClockStatusTimeout;

  default:
    return Avr2ClockStatusGeneralFailure;
  }
}

boolean_t
avr2ClockInit(
  Avr2ClockContext* pAvr2ClockCtx,
  Avr2DeviceContext* pAvr2DevCtx,
  unsigned int numClock,
  uint32_t* pFInit)
{
  Avr2ClockStatus status;
  boolean_t bOk = TRUE; /* Assume successful */
  uint32_t f;
  unsigned int i;

  if (numClock > AVR2_CLOCK_MAX_CLOCKS) {
    dfDebugPrint(0, ("*** avr2ClockInit: Too many clocks, numClock=%u\n", numClock));
    return FALSE;
  }
  pAvr2ClockCtx->pAvr2DevCtx = pAvr2DevCtx;
  pAvr2ClockCtx->numClock = numClock;
  for (i = 0; i < numClock; i++) {
    pAvr2ClockCtx->clocks[i].index = i;
    status = avr2ClockGetFrequencySync(pAvr2ClockCtx, i, &f);
    if (Avr2ClockStatusSuccess != status) {
      dfDebugPrint(0, ("*** avr2ClockInit: Failed to get initial frequency for clock index %u, status=%u\n", i, status));
      pFInit[i] = 0;
      bOk = FALSE;
    } else {
      dfDebugPrint(2, ("avr2ClockInit: Initial frequency of clock index %u is %lu Hz\n", i, (unsigned long)f));
      pFInit[i] = f;
    }
    pAvr2ClockCtx->clocks[i].save.bLastValid = FALSE;
  }

  return bOk;
}

void
avr2ClockUninit(
  Avr2ClockContext* pAvr2ClockCtx)
{
  /* Nothing to do yet */
}

static void
onAvr2GetFrequencyDone(
  Avr2TransRequest* pTransactionIgnored,
  Avr2TransStatus avr2Status,
  size_t actualResponseLength,
  void* pContext)
{
  Avr2ClockRequest* pRequest = (Avr2ClockRequest*)pContext;
  unsigned int index = pRequest->index;
  Avr2ClockContext* pAvr2ClockCtx = pRequest->pAvr2ClockCtx;
  Avr2ClockGenContext* pClockGen = &pAvr2ClockCtx->clocks[index];
  Avr2ClockStatus avr2ClkStatus = Avr2ClockStatusSuccess;
  uint32_t frequency = 0;
  uint8_t reply;

  if (Avr2TransStatusSuccess == avr2Status) {
    reply = pClockGen->transaction.getFrequency.reply[2];
    if (0 == reply) {
      /* Get frequency response from AVR indicates success */
      frequency = ((uint32_t)pClockGen->transaction.getFrequency.reply[3] << 0)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[4] << 8)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[5] << 16)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[6] << 24);
    } else {
      /* Get frequency response from AVR indicated error */
      switch (reply) {
      case 0x1U:
        avr2ClkStatus = Avr2ClockStatusInvalidIndex;
        break;

      default:
        avr2ClkStatus = Avr2ClockStatusGeneralFailure;
        break;
      }
    }
  } else {
    /* AVR2 transaction failed */
    avr2ClkStatus = mapAvr2TransStatus(avr2Status);
  }
  pRequest->callback.pGetFrequency(pAvr2ClockCtx, pRequest, avr2ClkStatus, index, frequency, pRequest->pContext);
}

Avr2ClockStatus
avr2ClockGetFrequency(
  Avr2ClockContext* pAvr2ClockCtx,
  Avr2ClockRequest* pRequest,
  unsigned int index,
  Avr2ClockGetFrequencyCallback* pCallback,
  void* pContext)
{
  Avr2ClockGenContext* pClockGen;

  if (index >= pAvr2ClockCtx->numClock) {
    return Avr2ClockStatusInvalidIndex;
  }

  pClockGen = &pAvr2ClockCtx->clocks[index];
  pRequest->pAvr2ClockCtx = pAvr2ClockCtx;
  pRequest->callback.pGetFrequency = pCallback;
  pRequest->pContext = pContext;
  pRequest->index = index;
  pClockGen->transaction.getFrequency.command[0] = AVR2_FACILITY_USER_EXT;
  pClockGen->transaction.getFrequency.command[1] = AVR2_COMMAND_GET_FREQ;
  pClockGen->transaction.getFrequency.command[2] = (uint8_t)index;
  avr2Transaction(
    pAvr2ClockCtx->pAvr2DevCtx,
    &pClockGen->transaction.getFrequency.request,
    pClockGen->transaction.getFrequency.command,
    DF_ARRAY_LENGTH(pClockGen->transaction.getFrequency.command),
    pClockGen->transaction.getFrequency.reply,
    DF_ARRAY_LENGTH(pClockGen->transaction.getFrequency.reply),
    1000000U,
    onAvr2GetFrequencyDone,
    pRequest);

  return Avr2ClockStatusSuccess;
}

static void
onAvr2SetFrequencyDone(
  Avr2TransRequest* pTransactionIgnored,
  Avr2TransStatus avr2Status,
  size_t actualResponseLength,
  void* pContext)
{
  Avr2ClockRequest* pRequest = (Avr2ClockRequest*)pContext;
  unsigned int index = pRequest->index;
  Avr2ClockContext* pAvr2ClockCtx = pRequest->pAvr2ClockCtx;
  Avr2ClockGenContext* pClockGen = &pAvr2ClockCtx->clocks[index];
  Avr2ClockStatus avr2ClkStatus = Avr2ClockStatusSuccess;
  uint8_t reply;

  if (Avr2TransStatusSuccess == avr2Status) {
    reply = pClockGen->transaction.setFrequency.reply[2];
    if (0 == reply) {
      /* Set frequency command to AVR was successful */
    } else {
      /* Set frequency response from AVR indicated error */
      switch (reply) {
      case 0x1U:
        /* Invalid clock index */
        avr2ClkStatus = Avr2ClockStatusInvalidIndex;
        break;

      case 0x11U:
        /* Invalid requested frequency */
        avr2ClkStatus = Avr2ClockStatusOutOfRange;
        break;

      default:
        avr2ClkStatus = Avr2ClockStatusGeneralFailure;
        break;
      }
    }
  } else {
    /* AVR2 transaction failed */
    avr2ClkStatus = mapAvr2TransStatus(avr2Status);
  }
  pRequest->callback.pSetFrequency(pAvr2ClockCtx, pRequest, avr2ClkStatus, index, pRequest->pContext);
}

Avr2ClockStatus
avr2ClockSetFrequency(
  Avr2ClockContext* pAvr2ClockCtx,
  Avr2ClockRequest* pRequest,
  unsigned int index,
  const CoreClockWord* pClockWord,
  Avr2ClockSetFrequencyCallback* pCallback,
  void* pContext)
{
  Avr2ClockGenContext* pClockGen;

  if (index >= pAvr2ClockCtx->numClock) {
    return Avr2ClockStatusInvalidIndex;
  }
  if (pClockWord->frequency >= 0xFFFFFFFFU) {
    return Avr2ClockStatusOutOfRange;
  }

  pClockGen = &pAvr2ClockCtx->clocks[index];
  pRequest->pAvr2ClockCtx = pAvr2ClockCtx;
  pRequest->callback.pSetFrequency = pCallback;
  pRequest->pContext = pContext;
  pRequest->index = index;
  pClockGen->transaction.setFrequency.command[0] = AVR2_FACILITY_USER_EXT;
  pClockGen->transaction.setFrequency.command[1] = AVR2_COMMAND_SET_FREQ;
  pClockGen->transaction.setFrequency.command[2] = (uint8_t)index;
  pClockGen->transaction.setFrequency.command[3] = (uint8_t)(pClockWord->frequency >> 0);
  pClockGen->transaction.setFrequency.command[4] = (uint8_t)(pClockWord->frequency >> 8);
  pClockGen->transaction.setFrequency.command[5] = (uint8_t)(pClockWord->frequency >> 16);
  pClockGen->transaction.setFrequency.command[6] = (uint8_t)(pClockWord->frequency >> 24);
  avr2Transaction(
    pAvr2ClockCtx->pAvr2DevCtx,
    &pClockGen->transaction.setFrequency.request,
    pClockGen->transaction.setFrequency.command,
    DF_ARRAY_LENGTH(pClockGen->transaction.setFrequency.command),
    pClockGen->transaction.setFrequency.reply,
    DF_ARRAY_LENGTH(pClockGen->transaction.setFrequency.reply),
    1000000U,
    onAvr2SetFrequencyDone,
    pRequest);
    
  return Avr2ClockStatusSuccess;
}

static void
getFrequencySyncCallback(
  Avr2ClockContext* pAvr2ClockCtx,
  Avr2ClockRequest* pRequest,
  Avr2ClockStatus status,
  unsigned int index,
  uint32_t frequency,
  void* pContextIgnored)
{
  Avr2ClockRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, Avr2ClockRequestSync, request);

  pRequestSync->status = status;
  pRequestSync->frequency = frequency;
  dfEventSignal(&pRequestSync->ev);
}

Avr2ClockStatus
avr2ClockGetFrequencySync(
  Avr2ClockContext* pAvr2ClockCtx,
  unsigned int index,
  uint32_t* pFrequency)
{
  Avr2ClockStatus status;
  Avr2ClockRequestSync req;

  dfEventInit(&req.ev);
  status = avr2ClockGetFrequency(pAvr2ClockCtx, &req.request, index, getFrequencySyncCallback, NULL);
  if (Avr2ClockStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  *pFrequency = req.frequency;
  dfEventUninit(&req.ev);
  return status;
}

static void
setFrequencySyncCallback(
  Avr2ClockContext* pAvr2ClockCtx,
  Avr2ClockRequest* pRequest,
  Avr2ClockStatus status,
  unsigned int index,
  void* pContextIgnored)
{
  Avr2ClockRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, Avr2ClockRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

Avr2ClockStatus
avr2ClockSetFrequencySync(
  Avr2ClockContext* pAvr2ClockCtx,
  unsigned int index,
  const CoreClockWord* pClockWord)
{
  Avr2ClockStatus status;
  Avr2ClockRequestSync req;

  dfEventInit(&req.ev);
  status = avr2ClockSetFrequency(pAvr2ClockCtx, &req.request, index, pClockWord, setFrequencySyncCallback, NULL);
  if (Avr2ClockStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
avr2ClockSaveState(
  Avr2ClockContext* pAvr2ClockCtx)
{
  /* Nothing to do yet */
}

Avr2ClockStatus
avr2ClockRestoreState(
  Avr2ClockContext* pAvr2ClockCtx)
{
  Avr2ClockStatus status = Avr2ClockStatusSuccess;
  unsigned int i;
  CoreClockWord clockWord;

  for (i = 0; i < pAvr2ClockCtx->numClock; i++) {
    /* If the clock synthesizer had been programmed before power-down, reprogram it with the same thing */
    if (pAvr2ClockCtx->clocks[i].save.bLastValid) {
      clockWord.frequency = pAvr2ClockCtx->clocks[i].save.lastFrequency;
      status = avr2ClockSetFrequencySync(pAvr2ClockCtx, i, &clockWord);
      if (Avr2ClockStatusSuccess != status) {
        dfDebugPrint(0, ("*** avr2ClockInit: Failed to restore frequency (%lu Hz) for clock index %u, status=%u\n",
          (unsigned long)clockWord.frequency, i, status));
        break;
      }
    }
  }

  return status;
}

ClockProgramStatus
avr2ClockMapStatus(
  Avr2ClockStatus status)
{
  switch (status) {
  case Avr2ClockStatusSuccess:
    return ClockProgramSuccess;

  case Avr2ClockStatusTimeout:
  case Avr2ClockStatusServiceMode:
    return ClockProgramHardwareError;

  case Avr2ClockStatusOutOfRange:
    return ClockProgramOutOfRange;

  case Avr2ClockStatusInvalidIndex:
    return ClockProgramInvalidIndex;

  case Avr2ClockStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}
