/*
** File: admxrc2.c  
** Project: ADMXRC2 API module library
** Purpose: OS-independent code for ADMXRC2 API module library
**
** (C) Copyright Alpha Data 2009-2013
*/

#if defined(__linux)
# include <malloc.h> /* For memalign() */
# include <unistd.h> /* For sysconf() */
#endif

#if defined(__VXWORKS__) || defined(__vxworks)
# include <cacheLib.h> /* For cacheDmaMalloc, cacheDmaFree */
#endif

#include <admxrc2.h>
#include <admxrc2/combuff.h>
#include <admxrc2/flash.h>
#include <admxrc2/sensor.h>
#include <admxrc2_platform.h>
#include <portable_file.h>
#include <portable_string.h>
#if PORTABLE_HAS_WCHAR_SUPPORT
# include <portable_wstring.h>
#endif
#include "ioctl_admxrc2.h"

#define ARRAY_LENGTH(x) (sizeof(x) / sizeof((x)[0]))

static const double g_64k = 65536.0;
static const double g_4G = 4294967296.0;

#define TO_DOUBLE32(i, f) ((double)(i) + (double)(f) / g_64k);
#define TO_DOUBLE64(i, f) ((double)(i) + (double)(f) / g_4G);

static const uint32_t g_spaceDefaultFlags =
  ADMXRC2_SPACE_SET_WIDTH | ADMXRC2_SPACE_WIDTH_DEFAULT |
  ADMXRC2_SPACE_SET_PREFETCH | ADMXRC2_SPACE_PREFETCH_DEFAULT |
  ADMXRC2_SPACE_SET_BURST | ADMXRC2_SPACE_BURST_DEFAULT;

/*
** Reverse the bits in each byte, in order to match Virtex / Virtex-II / 
** Virtex-II Pro SelectMap interface bit-ordering for certain models.
*/
static void
reverseBytes(
  ADMXRC2_IMAGE image,
  size_t n)
{
  int j;
  uint8_t d, dd;
  uint8_t* p = (uint8_t*)image;

  while (n--) {
    d = *p;
    dd = 0;
    for (j = 0; j < 8; j++) {
      dd = (uint8_t)(dd << 1);
      if (d & 1) {
        dd |= 1;
      }
      d >>= 1;
    }
    *p++ = dd;
  }
}

static ADMXRC2_STATUS
selectMapOffset(
  portable_file_t hFile,
  char* identifier,
  size_t identifierSize,
  uint64_t* pSize,
  uint64_t* pOffset)
{
  typedef enum {
    StatePreamble = 0,
    StateRecords  = 1,
    StateQuit     = 2
  } State;
  const int blocksize = 1024;
  uint8_t tmpBuf[1024];
  State state;
  ADMXRC2_STATUS status = ADMXRC2_SUCCESS;
  uint64_t fileSize, remaining, pos, numRead;
  bool_t bFound, bMore, bResult;
  size_t chunk, offset, level, skip;
  uint16_t recLen;
  uint8_t recType;

  if (!portableFileGetSize(hFile, &fileSize)) {
    status = ADMXRC2_INVALID_FILE;
    goto done;
  }
  
  /*
  ** Find the beginning of the SelectMap data in the file
  */
  remaining = fileSize;
  state = StatePreamble;
  bFound = FALSE;
  pos = 0;
  offset = 0;
  level = 0;
  bMore = TRUE;
  skip = 0;
  while (!bFound) {
    if (bMore) {
      if (!remaining) {
        status = ADMXRC2_INVALID_FILE;
        goto done;
      }
      memmove(tmpBuf + 0, tmpBuf + offset, level);
      offset = level;
      chunk = blocksize - level;
      if (remaining < chunk) {
        chunk = (size_t)remaining;
      }
      bResult = portableFileRead(hFile, tmpBuf, chunk, &numRead);
      if (!bResult || numRead != chunk) {
        status = ADMXRC2_INVALID_FILE;
        goto done;
      }
      level += (size_t)numRead; /* safe because numRead guaranteed small */
      bMore = FALSE;
      continue;
    } else {
      if (0 != skip) {
        chunk = level;
        if (chunk > skip) {
          chunk = skip;
        }
        level -= chunk;
        offset += chunk;
        skip -= chunk;
        pos += chunk;
        if (0 != skip) {
          bMore = TRUE;
        }
      } else {
        switch (state) {
        case StatePreamble:
          if (level < 2) {
            bMore = TRUE;
          } else {
            /* Skip over the preamble */
            recLen = (uint16_t)(((uint16_t)tmpBuf[offset] << 8) + tmpBuf[offset + 1]);
            skip = recLen + 2;
            offset += 2;
            level -= 2;
            pos += 2;
            state = StateRecords;
          }
          break;
        case StateRecords:
          if (level < 1) {
            bMore = TRUE;
          } else {
            recType = tmpBuf[offset];
            switch (recType) {
            case 0x62:
              /*
              ** Part type record
              */
              if (level < 3) {
                bMore = TRUE;
              } else {
                recLen = (uint16_t)(((uint16_t)tmpBuf[offset + 1] << 8) + tmpBuf[offset + 2]);
                if (recLen >= identifierSize * sizeof(char)) {
                  status = ADMXRC2_INVALID_FILE;
                  goto done;
                }
                if (level < (recLen + 3U)) {
                  bMore = TRUE;
                } else {
                  memcpy(identifier, tmpBuf + offset + 3, recLen);
                  identifier[recLen] = '\0';
                  portableStringLower(identifier, identifierSize);
                  offset += recLen + 3U;
                  level -= recLen + 3U;
                  pos += recLen + 3U;
                }
              }
              break;
            case 0x65:
              /* End of header record */
              skip = 5;
              state = StateQuit;
              break;
            case 0x61:
            case 0x63:
            case 0x64:
              /* Unknown record types */
              if (level < 3) {
                bMore = TRUE;
              } else {
                recLen = (uint16_t)(((uint16_t)tmpBuf[offset + 1] << 8) + tmpBuf[offset + 2]);
                if (level < (recLen + 3U)) {
                  bMore = TRUE;
                } else {
                  offset += recLen + 3U;
                  level -= recLen + 3U;
                  pos += recLen + 3U;
                }
              }
              break;
            default:
              /* Unexpected record type - ignore it */
              recLen = (uint16_t)(((uint16_t)tmpBuf[offset + 1] << 8) + tmpBuf[offset + 2]);
              skip = recLen + 3U;
              break;
            }
          }
          break;
        case StateQuit:
          bFound = TRUE;
          break;
        }
      }
    }
  }

  *pSize = fileSize - pos;
  *pOffset = pos;

done:
  return status;
}

static ADMXRC2_STATUS
loadBitstreamA(
  ADMXRC2_BOARD_TYPE boardType,
  portable_file_t hFile,
  ADMXRC2_IMAGE* pImage,
  size_t* pSize)
{
  ADMXRC2_STATUS status = ADMXRC2_SUCCESS;
  ADMXRC2_IMAGE image = NULL;
  char identifier[32];
  uint64_t size, offset, numRead;
  bool_t bResult;

  status = selectMapOffset(hFile, identifier, ARRAY_LENGTH(identifier), &size, &offset);
  if (ADMXRC2_SUCCESS != status) {
    goto done;
  }

  if (size >= 0x100000000ULL) {
    status = ADMXRC2_INVALID_FILE;
    goto done;
  }
  image = (ADMXRC2_IMAGE)malloc((size_t)size); /* Cast is safe because we guarantee size < 4G from above checks */
  if (NULL == image) {
    status = ADMXRC2_NO_MEMORY;
    goto done;
  }

  portableFileSetPointer(hFile, offset);
  bResult = portableFileRead(hFile, image, size, &numRead);
  if (!bResult || numRead != size) {
    status = ADMXRC2_INVALID_FILE;
    goto done;
  }

  switch (boardType) {
  case ADMXRC2_BOARD_ADMXRC:
  case ADMXRC2_BOARD_ADMXRC_P:
  case ADMXRC2_BOARD_ADMXRC2_LITE:
  case ADMXRC2_BOARD_ADMXRC2:
  case ADMXRC2_BOARD_ADMXRC2_PRO_LITE:
  case ADMXRC2_BOARD_ADMXRC2_PRO:
  case ADMXRC2_BOARD_ADPWRC2:
  case ADMXRC2_BOARD_ADPDRC2:
  case ADMXRC2_BOARD_ADPXPI:
  case ADMXRC2_BOARD_ADMPCIE6S1:
    reverseBytes(image, (size_t)size); /* Cast is safe because we guarantee size < 4G from above checks */
    break;

  default:
    /* Bit-reversal not required for other models */
    break;
  }

  *pImage = image;
  *pSize = (size_t)size; /* Cast is safe because we guarantee size < 4G from above checks */

done:
  if (status != ADMXRC2_SUCCESS) {
    if (image != NULL) {
      free(image);
    }
  }
  portableFileClose(hFile);
  return status;
}

ADMXRC2_EXPORT _ADMXRC2_DWORD
ADMXRC2_CALLING_CONVENTION
ADMXRC2_BuildDMAModeWord(
  ADMXRC2_BOARD_TYPE boardType,
  ADMXRC2_IOWIDTH busWidth,
  unsigned int waitStates,
  _ADMXRC2_DWORD miscFlags)
{
  uint32_t mode = 0;

  switch (boardType) {
  case ADMXRC2_BOARD_ADMXRC:
  case ADMXRC2_BOARD_ADMXRC_P:
  case ADMXRC2_BOARD_ADMXRC2_LITE:
  case ADMXRC2_BOARD_ADMXRC2:
  case ADMXRC2_BOARD_ADPWRC2:
  case ADMXRC2_BOARD_ADPDRC2:
  case ADMXRC2_BOARD_ADMXRC4LS:
  case ADMXRC2_BOARD_ADMXRC4LX:
  case ADMXRC2_BOARD_ADMXRC4SX:
  case ADMXRC2_BOARD_ADMPCIE6S1:
    switch (busWidth) {
    case ADMXRC2_IOWIDTH_8:
      mode |= 0 << 0;
      break;

    case ADMXRC2_IOWIDTH_16:
      mode |= 1 << 0;
      break;

    case ADMXRC2_IOWIDTH_32:
      mode |= 2 << 0;
      break;

    default:
      return 0xffffffffU;
    }
    if (waitStates > 0xf) {
      return 0xffffffffU;
    }
    mode |= waitStates << 2;
    if (miscFlags & 0xffff263fU) {
      return 0xffffffffU;
    }
    mode |= miscFlags;
    break;
    
  case ADMXRC2_BOARD_ADMXRC2_PRO_LITE:
  case ADMXRC2_BOARD_ADMXRC2_PRO:
  case ADMXRC2_BOARD_ADPXPI:
  case ADMXRC2_BOARD_ADPEXRC4FX:
  case ADMXRC2_BOARD_ADMXRC4FX:
  case ADMXRC2_BOARD_ADMXRC5LX:
  case ADMXRC2_BOARD_ADMXRC5T1:
  case ADMXRC2_BOARD_ADMXRC5T2:
  case ADMXRC2_BOARD_ADCPXRC4LX:
  case ADMXRC2_BOARD_ADMAMC5A2:
  case ADMXRC2_BOARD_ADMXRC5TZ:
  case ADMXRC2_BOARD_ADCBBP:
  case ADMXRC2_BOARD_ADMXRC5T2ADV:
  case ADMXRC2_BOARD_ADMXRC5T2ADV6:
  case ADMXRC2_BOARD_ADMXRC5TDA1:
  case ADMXRC2_BOARD_ADMXRC5T2ADVCC1:
    switch (busWidth) {
    case ADMXRC2_IOWIDTH_32:
      mode |= 2 << 0;
      break;

    case ADMXRC2_IOWIDTH_64:
      mode |= 3 << 0;
      break;

    default:
      return 0xffffffffU;
    }
    if (waitStates) {
      return 0xffffffffU;
    }
    if (miscFlags & 0xffff263fU) {
      return 0xffffffffU;
    }
    mode |= miscFlags;
    break;
    
  default:
    return 0xffffffffU;
  }
  
  return (_ADMXRC2_UINT32)mode;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_CancelWaitForInterrupt(
  ADMXRC2_HANDLE hCard)
{
  IOCTLS_ADMXRC2_CANCELWAITFORINTERRUPT ioctl;

  ioctl.in.targetIndex = 0;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_CANCELWAITFORINTERRUPT, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_CloseCard(
  ADMXRC2_HANDLE hCard)
{
  /* This has to be done in an OS-specific way */
  return admxrc2CloseDevice(hCard);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromBuffer(
  ADMXRC2_HANDLE hCard,
  const void* pBuffer,
  unsigned long length)
{
  return ADMXRC2_ConfigureFromBufferEx(hCard, 0, 0, pBuffer, length);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromBufferDMA(
  ADMXRC2_HANDLE             hCard,
  const void*                pBuffer,
  unsigned long              length,
  unsigned int               dmaChannel,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE                     hEvent)
#else
  void*                      pIgnored)
#endif
{
#if defined(_WIN32) || defined(_MSC_VER)
  return ADMXRC2_ConfigureFromBufferDMAEx(hCard, 0, 0, pBuffer, length, dmaChannel, hEvent);
#else
  return ADMXRC2_ConfigureFromBufferDMAEx(hCard, 0, 0, pBuffer, length, dmaChannel, pIgnored);
#endif
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromBufferDMAEx(
  ADMXRC2_HANDLE             hCard,
  unsigned int               targetIndex,
  _ADMXRC2_DWORD             flags,
  const void*                pBuffer,
  unsigned long              length,
  unsigned int               dmaChannel,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE                     hEvent)
#else
  void*                      pIgnored)
#endif
{
  IOCTLS_ADMXRC2_CONFIGUREDMA ioctl;

  if (length != 0 && pBuffer == NULL) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.targetIndex = (uint32_t)targetIndex;
  ioctl.in.flags = flags;
  ioctl.in.pBuffer = pBuffer;
  ioctl.in.length = length;
  ioctl.in.dmaChannel = dmaChannel;
#if defined(_WIN32) || defined(_MSC_VER)
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_CONFIGUREDMA, &ioctl, hEvent, sizeof(ioctl.in), 0);
#else
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_CONFIGUREDMA, &ioctl, pIgnored, sizeof(ioctl.in), 0);
#endif
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromBufferEx(
  ADMXRC2_HANDLE  hCard,
  unsigned int targetIndex,
  _ADMXRC2_DWORD flags,
  const void* pBuffer,
  size_t length)
{
  IOCTLS_ADMXRC2_CONFIGURE ioctl;

  if (length != 0 && pBuffer == NULL) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.targetIndex = (uint32_t)targetIndex;
  ioctl.in.flags = flags;
  ioctl.in.pBuffer = pBuffer;
  ioctl.in.length = length;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_CONFIGURE, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromFile(
  ADMXRC2_HANDLE hCard,
  const char* pFilename)
{
  return ADMXRC2_ConfigureFromFileEx(hCard, 0, 0, pFilename);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromFileDMA(
  ADMXRC2_HANDLE             hCard,
  const char*                pFilename,
  unsigned int               dmaChannel,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE                     hEvent)
#else
  void*                      pIgnored)
#endif
{
#if defined(_WIN32) || defined(_MSC_VER)
  return ADMXRC2_ConfigureFromFileDMAEx(hCard, 0, 0, pFilename, dmaChannel, hEvent);
#else
  return ADMXRC2_ConfigureFromFileDMAEx(hCard, 0, 0, pFilename, dmaChannel, pIgnored);
#endif
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromFileDMAEx(
  ADMXRC2_HANDLE             hCard,
  unsigned int               targetIndex,
  _ADMXRC2_UINT32            flags,
  const char*                pFilename,
  unsigned int               dmaChannel,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE                     hEvent)
#else
  void*                      pIgnored)
#endif
{
  unsigned long imageSize;
  ADMXRC2_IMAGE image = NULL;
  ADMXRC2_STATUS status;

  status = ADMXRC2_LoadBitstream(hCard, pFilename, &image, &imageSize);
  if (ADMXRC2_SUCCESS != status) {
    return status;
  }
#if defined(_WIN32) || defined(_MSC_VER)
  status = ADMXRC2_ConfigureFromBufferDMAEx(hCard, targetIndex, flags, image, imageSize, dmaChannel, hEvent);
#else
  status = ADMXRC2_ConfigureFromBufferDMAEx(hCard, targetIndex, flags, image, imageSize, dmaChannel, pIgnored);
#endif
  ADMXRC2_UnloadBitstream(image);
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ConfigureFromFileEx(
  ADMXRC2_HANDLE hCard,
  unsigned int targetIndex,
  uint32_t flags,
  const char* pFilename)
{
  unsigned long imageSize;
  ADMXRC2_IMAGE image = NULL;
  ADMXRC2_STATUS status;

  status = ADMXRC2_LoadBitstream(hCard, pFilename, &image, &imageSize);
  if (ADMXRC2_SUCCESS != status) {
    return status;
  }
  status = ADMXRC2_ConfigureFromBufferEx(hCard, targetIndex, flags, image, imageSize);
  ADMXRC2_UnloadBitstream(image);
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_DoDMA(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_DMADESC hDmaDesc,
  unsigned long offset,
  unsigned long length,
  _ADMXRC2_DWORD local,
  ADMXRC2_DMADIR direction,
  unsigned int channel,
  _ADMXRC2_DWORD mode,
  _ADMXRC2_DWORD flags,
  unsigned long* pTimeout,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE hEvent)
#else
  void* pIgnored)
#endif
{
  IOCTLS_ADMXRC2_DODMA ioctl;
  ADMXRC2_STATUS status;

  if (NULL != pTimeout) {
    ioctl.in.timeoutUs = *pTimeout;
    flags |= ADMXRC2_DMAFLAG_TIMEOUT;
  } else {
    ioctl.in.timeoutUs = 0;
  }
  ioctl.in.offset = offset;
  ioctl.in.length = length;
  ioctl.in.local = local;
  if (hDmaDesc > 0xFFFFFFFFU) {
    return ADMXRC2_INVALID_DMADESC;
  }
  ioctl.in.hDmaDesc = (uint32_t)hDmaDesc; /* Cast safe due to above check */
  ioctl.in.channel = channel;
  ioctl.in.mode = mode;
  ioctl.in.dirFlags = ((direction & 0x1U) << 24) | (flags & 0xFFFFFFU);
#if defined(_WIN32) || defined(_MSC_VER)
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_DODMA, &ioctl, hEvent, sizeof(ioctl.in), sizeof(ioctl.out));
#else
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_DODMA, &ioctl, pIgnored, sizeof(ioctl.in), sizeof(ioctl.out));
#endif
  if (ADMXRC2_SUCCESS == status && NULL != pTimeout) {
    /* Cast is safe since it must be no greater than *pTimeout input. */
    *pTimeout = (unsigned long)ioctl.out.timeoutUs;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_DoDMAImmediate(
  ADMXRC2_HANDLE hCard,
  void* pBuffer,
  unsigned long length,
  _ADMXRC2_DWORD local,
  ADMXRC2_DMADIR direction,
  unsigned int channel,
  _ADMXRC2_DWORD mode,
  _ADMXRC2_DWORD flags,
  unsigned long* pTimeout,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE hEvent)
#else
  void* pIgnored)
#endif
{
  IOCTLS_ADMXRC2_DODMAIMMEDIATE ioctl;
  ADMXRC2_STATUS status;

  if (NULL != pTimeout) {
    ioctl.in.timeoutUs = *pTimeout;
    flags |= ADMXRC2_DMAFLAG_TIMEOUT;
  } else {
    ioctl.in.timeoutUs = 0;
  }
  ioctl.in.pBuffer = pBuffer;
  ioctl.in.length = length;
  ioctl.in.local = local;
  ioctl.in.channel = channel;
  ioctl.in.mode = mode;
  ioctl.in.dirFlags = ((direction & 0x1U) << 24) | (flags & 0xFFFFFFU);
#if defined(_WIN32) || defined(_MSC_VER)
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_DODMAIMMEDIATE, &ioctl, hEvent, sizeof(ioctl.in), sizeof(ioctl.out));
#else
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_DODMAIMMEDIATE, &ioctl, pIgnored, sizeof(ioctl.in), sizeof(ioctl.out));
#endif
  if (ADMXRC2_SUCCESS == status && NULL != pTimeout) {
    /* Cast is safe since it must be no greater than *pTimeout input. */
    *pTimeout = (unsigned long)ioctl.out.timeoutUs;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_EnumerateCards(
  ADMXRC2_ENUM_FUNCTION enumFunc)
{
  static const unsigned int maxIndex = 256U;
  ADMXRC2_HANDLE hCard;
  ADMXRC2_STATUS status;
  unsigned int index = 0;
  bool_t bFoundOne = FALSE;

  for (index = 0; index < maxIndex; index++) {
    status = ADMXRC2_OpenCardByIndexEx(index, 0, FALSE, 0, &hCard);
    if (ADMXRC2_SUCCESS == status) { 
      enumFunc(index, hCard);
      bFoundOne = TRUE;
    } else {
      if (status == ADMXRC2_CARD_NOT_FOUND) {
        break;
      }
    }
  }
  
  return bFoundOne ? ADMXRC2_SUCCESS : ADMXRC2_CARD_NOT_FOUND;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_EraseFlash(
  ADMXRC2_HANDLE hCard,
  unsigned long start,
  unsigned long length)
{
  static const uint32_t maxUInt32 = 0xFFFFFFFFU;
  IOCTLS_ADMXRC2_ERASEFLASH ioctl;
  uint32_t limit;

  /* Catch bad values before casting to uint32_t */
  if (start > maxUInt32) {
    return ADMXRC2_INVALID_REGION;
  }
  if (length) {
    limit = (uint32_t)(start + length - 1);
    if (limit < start) {
      return ADMXRC2_INVALID_REGION;
    }
  }

  ioctl.in.flashIndex = 0;
  ioctl.in.flags = ADMXRC2_FLASHFLAG_SYNC;
  ioctl.in.offset = (uint32_t)start;
  ioctl.in.length = (uint32_t)length;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_ERASEFLASH, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT void
ADMXRC2_CALLING_CONVENTION
ADMXRC2_Free(
  void* p)
{
#if defined(_WIN32) || defined(_MSC_VER)
  free(p);
#elif defined(__linux)
  free(p);
#elif defined(__VXWORKS__) || defined(__vxworks)
  cacheDmaFree(p);
#else
# error Cannot detect target operating system.
#endif
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetBankInfo( 
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  ADMXRC2_BANK_INFO* pBankInfo)
{
  IOCTLS_ADMXRC2_GETBANKINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pBankInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = index;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETBANKINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pBankInfo, 0, sizeof(*pBankInfo));
    pBankInfo->Type = ioctl.out.typeMask;
    pBankInfo->Width = ioctl.out.width;
    pBankInfo->Size = ioctl.out.sizeWords;
    pBankInfo->Fitted = ioctl.out.bFitted ? TRUE : FALSE;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetCardInfo(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_CARD_INFO* pCardInfo)
{
  IOCTLS_ADMXRC2_GETCARDINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pCardInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETCARDINFO, &ioctl, NULL, 0, sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pCardInfo, 0, sizeof(*pCardInfo));
    pCardInfo->CardID = (ADMXRC2_CARDID)ioctl.out.cardId;
    pCardInfo->SerialNum = ioctl.out.serialNumber;
    pCardInfo->BoardType = (ADMXRC2_BOARD_TYPE)ioctl.out.boardType;
    pCardInfo->FPGAType = (ADMXRC2_FPGA_TYPE)ioctl.out.fpgaType;
    pCardInfo->NumClock = (unsigned long)ioctl.out.numClock;
    pCardInfo->NumDMAChan = (unsigned long)ioctl.out.numDmaChannel;
    pCardInfo->NumRAMBank = (unsigned long)ioctl.out.numMemoryBank;
    pCardInfo->NumSpace = (unsigned long)ioctl.out.numSpace;
    pCardInfo->RAMBanksFitted = ioctl.out.memoryBankFitted;
    pCardInfo->BoardRevision = ioctl.out.boardRevision;
    pCardInfo->LogicRevision = ioctl.out.logicRevision;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetCardInfoEx(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_CARD_INFOEX* pCardInfo)
{
  IOCTLS_ADMXRC2_GETCARDINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pCardInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETCARDINFO, &ioctl, NULL, 0, sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pCardInfo, 0, sizeof(*pCardInfo));
    pCardInfo->CardID = (ADMXRC2_CARDID)ioctl.out.cardId;
    pCardInfo->SerialNum = ioctl.out.serialNumber;
    pCardInfo->BoardType = (ADMXRC2_BOARD_TYPE)ioctl.out.boardType;
    pCardInfo->FPGAType = (ADMXRC2_FPGA_TYPE)ioctl.out.fpgaType;
    pCardInfo->NumClock = (unsigned long)ioctl.out.numClock;
    pCardInfo->NumDMAChan = (unsigned long)ioctl.out.numDmaChannel;
    pCardInfo->NumTargetFPGA = (unsigned long)ioctl.out.numTargetFpga;
    pCardInfo->NumRAMBank = (unsigned long)ioctl.out.numMemoryBank;
    pCardInfo->NumSensor = (unsigned long)ioctl.out.numSensor;
    pCardInfo->NumSpace = (unsigned long)ioctl.out.numSpace;
    pCardInfo->RAMBanksFitted = ioctl.out.memoryBankFitted;
    pCardInfo->BoardRevision = ioctl.out.boardRevision;
    pCardInfo->LogicRevision = ioctl.out.logicRevision;
  }

  return status;
}

static ADMXRC2_STATUS
getMappedCommonBuffer(
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  void** ppVirtualBase,
  uint64_t* pVirtualSize)
{
  ADMXRC2_STATUS status;
  void* pBase;
  uint64_t offset, length;

  status = admxrc2FindRegion(
    hCard,
    ADMXRC2_REGION_TAG_COMMONBUFFER | (index & ADMXRC2_REGION_TAG_MASKINV),
    0,
    &pBase,
    &offset,
    &length);
  if (ADMXRC2_SUCCESS == status && 0 == offset) {
    *ppVirtualBase = pBase;
    *pVirtualSize = length;
  } else {
    *ppVirtualBase = NULL;
    *pVirtualSize = 0;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetCommonBufferInfo(
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  ADMXRC2_COMMONBUFFER_INFO* pCommonBufferInfo)
{
  const unsigned long max_ulong = (unsigned long)-1;
  IOCTLS_ADMXRC2_GETCOMMONBUFFERINFO ioctl;
  ADMXRC2_STATUS status;
  void* pBase;
  uint64_t length;

  if (NULL == pCommonBufferInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = index;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETCOMMONBUFFERINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pCommonBufferInfo, 0, sizeof(*pCommonBufferInfo));
    (void)getMappedCommonBuffer(hCard, index, &pBase, &length);
    pCommonBufferInfo->VirtualBase = pBase;
#if defined(_WIN32)
    pCommonBufferInfo->PhysicalBase.QuadPart = ioctl.out.busAddress;
#else
    pCommonBufferInfo->PhysicalBase = ioctl.out.busAddress;
#endif
    if (ioctl.out.busSize > max_ulong) {
      pCommonBufferInfo->Size = max_ulong;
    } else  {
      pCommonBufferInfo->Size = (unsigned long)ioctl.out.busSize;
    }
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetFlashBlockInfo(
  ADMXRC2_HANDLE hCard,
  unsigned long location,
  ADMXRC2_FLASHBLOCK_INFO* pFlashBlockInfo)
{
  static const uint32_t maxUInt32 = 0xFFFFFFFFU;
  IOCTLS_ADMXRC2_GETFLASHBLOCKINFO ioctl;
  ADMXRC2_STATUS status;

  /* Catch bad values before casting to uint32_t */
  if (location > maxUInt32) {
    return ADMXRC2_INVALID_REGION;
  }

  ioctl.in.flashIndex = 0;
  ioctl.in.location = (uint32_t)location;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETFLASHBLOCKINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    pFlashBlockInfo->Flags = ioctl.out.flags;
    pFlashBlockInfo->Start = ioctl.out.start;
    pFlashBlockInfo->Length = ioctl.out.length;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetFlashInfo(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_FLASH_INFO* pFlashInfo)
{
  IOCTLS_ADMXRC2_GETFLASHINFO ioctl;
  ADMXRC2_STATUS status;
  const char* pIdentifier = NULL;

  if (NULL == pFlashInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.flashIndex = 0;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETFLASHINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    switch (ioctl.out.vendorId) {
    case 0x89U:
      /* Intel or Micron */
      switch (ioctl.out.deviceId) {
      case 0x99U:
        pIdentifier = "Micron 28F008B";
        break;

      case 0x16U:
        pIdentifier = "Intel 28F320J3A";
        break;

      case 0x17U:
        pIdentifier = "Intel 28F640J3A";
        break;

      case 0x18U:
        pIdentifier = "Intel 28F128J3A";
        break;

      case 0x8803:
        pIdentifier = "Intel 28F256K3";
        break;

      case 0x8919: /* Intel 28F256P30 or P30 stacked die flash, top boot */
        pIdentifier = "Intel 28F256P30";
        break;

      case 0x891C: /* Intel 28F256P30 or P30 stacked die flash, bottom boot */
        pIdentifier = "Intel 28F256P30";
        break;

      case 0x8999: /* Numonyx Axcell P30 512GBit flash, symmetric blocks */
      case 0x899A: /* Numonyx Axcell P30 1GBit flash, symmetric blocks */
        pIdentifier = "Numonyx Axcell P30 (Symm bl)";
        break;

      case 0x8960: /* Numonyx Axcell P30 512GBit flash, top boot */
      case 0x8962: /* Numonyx Axcell P30 1GBit flash, Top boot */
        pIdentifier = "Numonyx Axcell P30 (Top boot)";
        break;

      case 0x8961: /* Numonyx Axcell P30 512GBit flash, bottom boot */
      case 0x8963: /* Numonyx Axcell P30 1GBit flash, bottom boot */

        pIdentifier = "Numonyx Axcell P30 (Bot boot)";
        break;
      }
      break;

    case 0xbfU:
      /* SST */
      switch (ioctl.out.deviceId) {
      case 0xd9U:
        pIdentifier = "SST 39VF016Q";
        break;
      }
      break;
    }
    if (NULL == pIdentifier) {
      portableFormatStringA2(pFlashInfo->DeviceName, ARRAY_LENGTH(pFlashInfo->DeviceName), "Vendor 0x%04lX Device 0x%04lX",
        (unsigned long)ioctl.out.vendorId, (unsigned long)ioctl.out.deviceId);
    } else {
      portableStringCopy(pFlashInfo->DeviceName, ARRAY_LENGTH(pFlashInfo->DeviceName), pIdentifier);
    }
    pFlashInfo->DeviceSize = ioctl.out.deviceSize;
    pFlashInfo->DeviceBlockSize = ioctl.out.deviceBlockSize;
    pFlashInfo->UseableStart = ioctl.out.useableStart;
    pFlashInfo->UseableLength = ioctl.out.useableLength;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetFPGAInfo( 
  ADMXRC2_HANDLE hCard,
  unsigned int fpgaIndex,
  ADMXRC2_FPGA_INFO* pFpgaInfo)
{
  IOCTLS_ADMXRC2_GETFPGAINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pFpgaInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = fpgaIndex;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETFPGAINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pFpgaInfo, 0, sizeof(*pFpgaInfo));
    pFpgaInfo->Type = ioctl.out.type;
    portableStringCopy(pFpgaInfo->Package, ARRAY_LENGTH(pFpgaInfo->Package), ioctl.out.package);
    pFpgaInfo->Speed = (int)ioctl.out.speed;
    pFpgaInfo->Stepping = (int)ioctl.out.stepping;
    pFpgaInfo->Industrial = (int)ioctl.out.tempGrade;
    pFpgaInfo->SCD = (int)ioctl.out.scd;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetSensorInfo(
  ADMXRC2_HANDLE hCard,
  unsigned int sensorIndex,
  ADMXRC2_SENSOR_INFO* pSensorInfo)
{
  IOCTLS_ADMXRC2_GETSENSORINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pSensorInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.sensorIndex = sensorIndex;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETSENSORINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pSensorInfo, 0, sizeof(*pSensorInfo));
    portableStringCopy(pSensorInfo->Description, ARRAY_LENGTH(pSensorInfo->Description), ioctl.out.description);
    pSensorInfo->Capabilities = ioctl.out.capabilities;
    pSensorInfo->Error = TO_DOUBLE32(ioctl.out.error.intPart, ioctl.out.error.fracPart);
    pSensorInfo->DataType = (ADMXRC2_DATA_TYPE)ioctl.out.datatype;
    pSensorInfo->Unit = (ADMXRC2_UNIT_TYPE)ioctl.out.unit;
    pSensorInfo->Exponent = ioctl.out.exponent;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetSpaceConfig(
  ADMXRC2_HANDLE  hCard,
  unsigned int spaceIndex,
  _ADMXRC2_DWORD* pConfigFlags)
{
  IOCTLS_ADMXRC2_GETSPACECONFIG ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pConfigFlags) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = spaceIndex;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETSPACECONFIG, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    *pConfigFlags = ioctl.out.configFlags;
  }

  return status;
}

static ADMXRC2_STATUS
getStandardMappedRegion(
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  void** ppVirtualBase,
  uint64_t* pVirtualSize)
{
  ADMXRC2_STATUS status;
  void* pBase;
  uint64_t offset, length;

  status = admxrc2FindRegion(
    hCard,
    ADMXRC2_REGION_TAG_WINDOW | (index & ADMXRC2_REGION_TAG_MASKINV),
    0,
    &pBase,
    &offset,
    &length);
  if (ADMXRC2_SUCCESS == status && 0 == offset) {
    *ppVirtualBase = pBase;
    *pVirtualSize = length;
  } else {
    *ppVirtualBase = NULL;
    *pVirtualSize = 0;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetSpaceInfo(
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  ADMXRC2_SPACE_INFO* pSpaceInfo)
{
  IOCTLS_ADMXRC2_GETSPACEINFO ioctl;
  ADMXRC2_STATUS status;
  void* pBase;
  uint64_t length;

  if (NULL == pSpaceInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = index;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETSPACEINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pSpaceInfo, 0, sizeof(*pSpaceInfo));
    (void)getStandardMappedRegion(hCard, index, &pBase, &length);
    pSpaceInfo->VirtualBase = pBase;
    pSpaceInfo->VirtualSize = (unsigned long)length;
#if defined(_WIN32) || defined(_MSC_VER)
    pSpaceInfo->PhysicalBase = (DWORD)ioctl.out.busBase;
    pSpaceInfo->LocalBase = (DWORD)ioctl.out.localBase;
#else
    pSpaceInfo->PhysicalBase = (unsigned long)ioctl.out.busBase;
    pSpaceInfo->LocalBase = (unsigned long)ioctl.out.localBase;
#endif
    pSpaceInfo->LocalSize = (unsigned long)ioctl.out.localSize;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetSpaceInfoEx( 
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  ADMXRC2_SPACE_INFOEX* pSpaceInfoEx)
{
  IOCTLS_ADMXRC2_GETSPACEINFO ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pSpaceInfoEx) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.index = index;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETSPACEINFO, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pSpaceInfoEx, 0, sizeof(*pSpaceInfoEx));
    pSpaceInfoEx->VirtualSize = ioctl.out.virtualSize;
    pSpaceInfoEx->BusBase = ioctl.out.busBase;
    pSpaceInfoEx->BusSize = ioctl.out.busSize;
    pSpaceInfoEx->LocalBase = ioctl.out.localBase;
    pSpaceInfoEx->LocalSize = ioctl.out.localSize;
  }

  return status;
}

ADMXRC2_EXPORT const char*
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetStatusString(
  ADMXRC2_STATUS status)
{
  return ADMXRC2_GetStatusStringEx(status, FALSE);
}

ADMXRC2_EXPORT const char*
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetStatusStringEx(
  ADMXRC2_STATUS status,
  _ADMXRC2_BOOL bShort)
{
  if (bShort) {
    switch (status) {
    case ADMXRC2_SUCCESS:
      return "ADMXRC2_SUCCESS";

    case ADMXRC2_INTERNAL_ERROR:
      return "ADMXRC2_INTERNAL_ERROR";

    case ADMXRC2_NO_MEMORY:
      return "ADMXRC2_NO_MEMORY";

    case ADMXRC2_CARD_NOT_FOUND:
      return "ADMXRC2_CARD_NOT_FOUND";

    case ADMXRC2_FILE_NOT_FOUND:
      return "ADMXRC2_FILE_NOT_FOUND";

    case ADMXRC2_INVALID_FILE:
      return "ADMXRC2_INVALID_FILE";

    case ADMXRC2_FPGA_MISMATCH:
      return "ADMXRC2_FPGA_MISMATCH";

    case ADMXRC2_INVALID_HANDLE:
      return "ADMXRC2_INVALID_HANDLE";

    case ADMXRC2_TIMEOUT:
      return "ADMXRC2_TIMEOUT";

    case ADMXRC2_CARD_BUSY:
      return "ADMXRC2_CARD_BUSY";

    case ADMXRC2_INVALID_PARAMETER:
      return "ADMXRC2_INVALID_PARAMETER";

    case ADMXRC2_CLOSED:
      return "ADMXRC2_CLOSED";

    case ADMXRC2_CARD_ERROR:
      return "ADMXRC2_CARD_ERROR";

    case ADMXRC2_NOT_SUPPORTED:
      return "ADMXRC2_NOT_SUPPORTED";

    case ADMXRC2_DEVICE_BUSY:
      return "ADMXRC2_DEVICE_BUSY";

    case ADMXRC2_INVALID_DMADESC:
      return "ADMXRC2_INVALID_DMADESC";

    case ADMXRC2_NO_DMADESC:
      return "ADMXRC2_NO_DMADESC";

    case ADMXRC2_FAILED:
      return "ADMXRC2_FAILED";

    case ADMXRC2_PENDING:
      return "ADMXRC2_PENDING";

    case ADMXRC2_UNKNOWN_ERROR:
      return "ADMXRC2_UNKNOWN_ERROR";

    case ADMXRC2_NULL_POINTER:
      return "ADMXRC2_NULL_POINTER";

    case ADMXRC2_CANCELLED:
      return "ADMXRC2_CANCELLED";

    case ADMXRC2_BAD_DRIVER:
      return "ADMXRC2_BAD_DRIVER";

    case ADMXRC2_ACCESS_DENIED:
      return "ADMXRC2_ACCESS_DENIED";

    case ADMXRC2_INVALID_INDEX:
      return "ADMXRC2_INVALID_INDEX";

    case ADMXRC2_INVALID_REGION:
      return "ADMXRC2_INVALID_REGION";

    case ADMXRC2_REGION_TOO_LARGE:
      return "ADMXRC2_REGION_TOO_LARGE";

    case ADMXRC2_NOT_OWNER:
      return "ADMXRC2_NOT_OWNER";

    default:
      return "ADMXRC2_?";
    }
  } else {
    switch (status) {
    case ADMXRC2_SUCCESS:
      return "success";

    case ADMXRC2_INTERNAL_ERROR:
      return "an error in the API logic occurred";

    case ADMXRC2_NO_MEMORY:
      return "failed to allocate memory";

    case ADMXRC2_CARD_NOT_FOUND:
      return "failed to open specified card";

    case ADMXRC2_FILE_NOT_FOUND:
      return "failed to open file";

    case ADMXRC2_INVALID_FILE:
      return "file appears to be corrupt";

    case ADMXRC2_FPGA_MISMATCH:
      return "the bitstream target FPGA did not match the FPGA on the card";

    case ADMXRC2_INVALID_HANDLE:
      return "the card handle passed was invalid";

    case ADMXRC2_TIMEOUT:
      return "the operation was not completed within the timeout period";

    case ADMXRC2_CARD_BUSY:
      return "the card could not be opened because it was in use";

    case ADMXRC2_INVALID_PARAMETER:
      return "an invalid parameter was passed";

    case ADMXRC2_CLOSED:
      return "the card handle was closed before the operation was completed";

    case ADMXRC2_CARD_ERROR:
      return "the card could not be opened because the wrong card ID was specified";

    case ADMXRC2_NOT_SUPPORTED:
      return "the requested operation is not supported";

    case ADMXRC2_DEVICE_BUSY:
      return "the operation could not be performed because the device or resource was busy";

    case ADMXRC2_INVALID_DMADESC:
      return "an invalid handle to a DMA descriptor was passed";

    case ADMXRC2_NO_DMADESC:
      return "unable to allocate a DMA descriptor";

    case ADMXRC2_FAILED:
      return "the operation failed";

    case ADMXRC2_PENDING:
      return "the operation is still in progress";

    case ADMXRC2_UNKNOWN_ERROR:
      return "an error occurred whose origin is unknown";

    case ADMXRC2_NULL_POINTER:
      return "a NULL pointer was passed";

    case ADMXRC2_CANCELLED:
      return "the operation was cancelled because the requesting thread terminated";

    case ADMXRC2_BAD_DRIVER:
      return "the device driver revision level is too low";

    case ADMXRC2_ACCESS_DENIED:
      return "caller does not have access rights for operation / open";

    case ADMXRC2_INVALID_INDEX:
      return "the index parameter was invalid";

    case ADMXRC2_INVALID_REGION:
      return "the region specified was invalid";

    case ADMXRC2_REGION_TOO_LARGE:
      return "the region specified was too large";

    case ADMXRC2_NOT_OWNER:
      return "The caller's handle is not the current owner of the specified FPGA";

    default:
      return "unknown ADMXRC2_STATUS code";
    }
  }
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetVersionInfo(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_VERSION_INFO* pVersionInfo)
{
  IOCTLS_ADMXRC2_GETDRIVERVERSION ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pVersionInfo) {
    return ADMXRC2_NULL_POINTER;
  }

  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETDRIVERVERSION, &ioctl, NULL, 0, sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pVersionInfo, 0, sizeof(*pVersionInfo));
    pVersionInfo->DriverMajor = (_ADMXRC2_BYTE)ioctl.out.major;
    pVersionInfo->DriverMinor = (_ADMXRC2_BYTE)ioctl.out.minor;
    pVersionInfo->APIMajor = (_ADMXRC2_BYTE)ADMXRC2_VERSION_0;
    pVersionInfo->APIMinor = (_ADMXRC2_BYTE)ADMXRC2_VERSION_1;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_GetVersionInfoEx(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_VERSION_INFOEX* pVersionInfoEx)
{
  IOCTLS_ADMXRC2_GETDRIVERVERSION ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pVersionInfoEx) {
    return ADMXRC2_NULL_POINTER;
  }

  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_GETDRIVERVERSION, &ioctl, NULL, 0, sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    memset(pVersionInfoEx, 0, sizeof(*pVersionInfoEx));
    pVersionInfoEx->DriverMajor = ioctl.out.major;
    pVersionInfoEx->DriverMinor = ioctl.out.minor;
    pVersionInfoEx->DriverBugfix = ioctl.out.bugfix;
    pVersionInfoEx->DriverBuild = ioctl.out.build;
    pVersionInfoEx->APIMajor = ADMXRC2_VERSION_0;
    pVersionInfoEx->APIMinor = ADMXRC2_VERSION_1;
    pVersionInfoEx->APIBugfix = ADMXRC2_VERSION_2;
    pVersionInfoEx->APIBuild = ADMXRC2_VERSION_3;
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_InstallErrorHandler(
  ADMXRC2_ERROR_HANDLER handler)
{
  return ADMXRC2_NOT_SUPPORTED;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_LoadBitstream(
  ADMXRC2_HANDLE hCard,
  const char* pFilename,
  ADMXRC2_IMAGE* pImage,
  unsigned long* pBytesRead)
{
  portable_file_t hFile;
  ADMXRC2_CARD_INFO cardInfo;
  ADMXRC2_STATUS status;
  size_t bytesRead;

  if (NULL == pFilename) {
    return ADMXRC2_NULL_POINTER;
  }
  if (NULL == pImage) {
    return ADMXRC2_NULL_POINTER;
  }

  hFile = portableFileOpenA(pFilename, FALSE);
  if (hFile == PORTABLE_INVALID_FILE) {
    return ADMXRC2_FILE_NOT_FOUND;
  }

  status = ADMXRC2_GetCardInfo(hCard, &cardInfo);
  if (ADMXRC2_SUCCESS != status) {
    return status;
  }

  status = loadBitstreamA(cardInfo.BoardType, hFile, pImage, &bytesRead);
  if (ADMXRC2_SUCCESS == status) {
    *pBytesRead = (unsigned long)bytesRead; /* Cast is safe because loadBitstreamA rejects .bit files >= 4GiB */
  }
  return status;
}

ADMXRC2_EXPORT void*
ADMXRC2_CALLING_CONVENTION
ADMXRC2_Malloc(
  unsigned long sizeInBytes)
{
#if defined(_WIN32) || defined(_MSC_VER)
  return malloc(sizeInBytes);
#elif defined(__linux)
  unsigned long pageSize = sysconf(_SC_PAGESIZE);

  return memalign(pageSize, sizeInBytes);
#elif defined(__VXWORKS__) || defined(__vxworks)
  return cacheDmaMalloc(sizeInBytes);
#else
# error Cannot detect target operating system.
#endif
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_MapDirectMaster(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_DMADESC hDmaDesc,
  unsigned long offset,
  unsigned long length,
  ADMXRC2_BUFFERMAP* pBufferMap)
{
  return ADMXRC2_NOT_SUPPORTED;
}

/*
** Attempt to map spaces 0 and 1 in their entirety into caller's virtual address space
** Also maps any available common buffers.
*/
static void
mapStandardSpaces(
  ADMXRC2_HANDLE hCard)
{
  ADMXRC2_STATUS status;
  ADMXRC2_SPACE_INFOEX spaceInfoEx;
  ADMXRC2_CARD_INFO cardInfo;
  unsigned long i, n;
  void* pDummy;

  status = ADMXRC2_GetCardInfo(hCard, &cardInfo);
  if (ADMXRC2_SUCCESS == status) {
    n = cardInfo.NumSpace;
  } else {
    n = 2; /* Fallback value */
  }
  for (i = 0; i < n; i++) {
    status = ADMXRC2_GetSpaceInfoEx(hCard, (unsigned int)i /* n should always be a small number */, &spaceInfoEx);
    if (ADMXRC2_SUCCESS == status) {
      status = admxrc2MapRegion(
        hCard,
        ADMXRC2_REGION_TAG_WINDOW | ((unsigned int)i & ADMXRC2_REGION_TAG_MASKINV),
        0,
        spaceInfoEx.VirtualSize,
        &pDummy);
      if (ADMXRC2_SUCCESS != status) {
        /* Failed, but continue anyway */
      }
    }
  }
}

static void
mapCommonBuffers(
  ADMXRC2_HANDLE hCard)
{
  ADMXRC2_STATUS status;
  ADMXRC2_COMMONBUFFER_INFO commonBufferInfo;
  unsigned int i;
  void* pDummy;

  for (i = 0;; i++) {
    status = ADMXRC2_GetCommonBufferInfo(hCard, i, &commonBufferInfo);
    if (ADMXRC2_SUCCESS == status) {
      status = admxrc2MapRegion(
        hCard,
        ADMXRC2_REGION_TAG_COMMONBUFFER | (i & ADMXRC2_REGION_TAG_MASKINV),
        0,
        commonBufferInfo.Size,
        &pDummy);
    }
    if (ADMXRC2_SUCCESS != status) {
      break;
    }
  }
}

static void
configureSpacesDefault(
  ADMXRC2_HANDLE hCard)
{
  ADMXRC2_STATUS status;
  ADMXRC2_CARD_INFO cardInfo;
  unsigned long i, n;

  status = ADMXRC2_GetCardInfo(hCard, &cardInfo);
  if (ADMXRC2_SUCCESS == status) {
    n = cardInfo.NumSpace;
  } else {
    n = 2; /* Fallback value */
  }
  for (i = 0; i < n; i++) {
    ADMXRC2_SetSpaceConfig(hCard, 0, g_spaceDefaultFlags);
  }
}

ADMXRC2_STATUS ADMXRC2_EXPORT
ADMXRC2_CALLING_CONVENTION
ADMXRC2_OpenCard(
  ADMXRC2_CARDID cardId,
  ADMXRC2_HANDLE* phCard)
{
  static const uint32_t coopLevel = ADMXRC2_COOP_EXCLUSIVE;
  unsigned int index;
  ADMXRC2_HANDLE hCardTmp = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_HANDLE hCard = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_STATUS status;
  ADMXRC2_CARD_INFO cardInfo;

  if (NULL == phCard) {
    return ADMXRC2_NULL_POINTER;
  }

  for (index = 0;; index++) {
    /* This has to be done in an OS-specific way */
    status = admxrc2OpenDevice(index, TRUE, &hCardTmp);
    if (ADMXRC2_SUCCESS != status) {
      return ADMXRC2_CARD_NOT_FOUND;
    }

    status = ADMXRC2_GetCardInfo(hCardTmp, &cardInfo);
    if (ADMXRC2_SUCCESS == status && (0 == cardId || cardInfo.CardID == cardId)) {
      break;
    }

    admxrc2CloseDevice(hCardTmp);
  }

  /* TO DO - this should be inside the loop that iterates over indices */
  status = admxrc2OpenDevice(index, FALSE, &hCard);
  if (ADMXRC2_SUCCESS == status) {
    status = ADMXRC2_SetCoopLevel(hCard, coopLevel);
    if (ADMXRC2_SUCCESS == status) {
      configureSpacesDefault(hCard);
      mapStandardSpaces(hCard);
      mapCommonBuffers(hCard);
      *phCard = hCard;
    } else {
      admxrc2CloseDevice(hCard);
    }
  }

  admxrc2CloseDevice(hCardTmp);

  return status;
}

ADMXRC2_STATUS ADMXRC2_EXPORT
ADMXRC2_CALLING_CONVENTION
ADMXRC2_OpenCardEx(
  ADMXRC2_CARDID cardId,
  _ADMXRC2_UINT32 flags,
  _ADMXRC2_BOOL bPassive,
  _ADMXRC2_UINT32 cooperativeLevel,
  ADMXRC2_HANDLE* phCard)
{
  unsigned int cardIndex;
  ADMXRC2_HANDLE hCardTmp = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_HANDLE hCard = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_STATUS status;
  bool_t bSetCoopLevel = (flags & ADMXRC2_OPENEX_NOCOOPLEVEL) ? FALSE : TRUE;
  ADMXRC2_CARD_INFO cardInfo;

  if (NULL == phCard) {
    return ADMXRC2_NULL_POINTER;
  }
  if (bPassive && (flags & (ADMXRC2_OPENEX_AUTOMAP | ADMXRC2_OPENEX_CONFIG))) {
    return ADMXRC2_INVALID_PARAMETER;
  }

  for (cardIndex = 0;; cardIndex++) {
    /* This has to be done in an OS-specific way */
    status = admxrc2OpenDevice(cardIndex, TRUE, &hCardTmp);
    if (ADMXRC2_SUCCESS != status) {
      return ADMXRC2_CARD_NOT_FOUND;
    }

    status = ADMXRC2_GetCardInfo(hCardTmp, &cardInfo);
    if (ADMXRC2_SUCCESS == status && (0 == cardId || cardInfo.CardID == cardId)) {
      break;
    }

    admxrc2CloseDevice(hCardTmp);
  }

  status = admxrc2OpenDevice(cardIndex, bPassive, &hCard);
  if (ADMXRC2_SUCCESS != status) {
    goto out;
  }

  if (!bPassive) {
    if (bSetCoopLevel) {
      status = ADMXRC2_SetCoopLevel(hCard, cooperativeLevel);
      if (ADMXRC2_SUCCESS != status) {
        admxrc2CloseDevice(hCard);
        goto out;
      }
    }

    if (flags & ADMXRC2_OPENEX_CONFIG) {
      configureSpacesDefault(hCard);
    }

    if (flags & ADMXRC2_OPENEX_AUTOMAP) {
      mapStandardSpaces(hCard);
      mapCommonBuffers(hCard);
    }
  }


  *phCard = hCard;

out:
  admxrc2CloseDevice(hCardTmp);
  return status;
}

ADMXRC2_STATUS ADMXRC2_EXPORT
ADMXRC2_CALLING_CONVENTION
ADMXRC2_OpenCardByIndex(
  unsigned int index,
  ADMXRC2_HANDLE* phCard)
{
  static const uint32_t coopLevel = ADMXRC2_COOP_EXCLUSIVE;
  ADMXRC2_HANDLE hCard = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_STATUS status;

  if (NULL == phCard) {
    return ADMXRC2_NULL_POINTER;
  }

  status = admxrc2OpenDevice(index, FALSE, &hCard);
  if (ADMXRC2_SUCCESS == status) {
    status = ADMXRC2_SetCoopLevel(hCard, coopLevel);
    if (ADMXRC2_SUCCESS == status) {
      configureSpacesDefault(hCard);
      mapStandardSpaces(hCard);
      mapCommonBuffers(hCard);
      *phCard = hCard;
    } else {
      admxrc2CloseDevice(hCard);
    }
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_OpenCardByIndexEx(
  unsigned int cardIndex,
  _ADMXRC2_UINT32 flags,
  _ADMXRC2_BOOL bPassive,
  _ADMXRC2_UINT32 coopLevel,
  ADMXRC2_HANDLE* phCard)
{
  ADMXRC2_HANDLE hCard = ADMXRC2_HANDLE_INVALID_VALUE;
  ADMXRC2_STATUS status;
  bool_t bSetCoopLevel = (flags & ADMXRC2_OPENEX_NOCOOPLEVEL) ? FALSE : TRUE;

  if (NULL == phCard) {
    return ADMXRC2_NULL_POINTER;
  }
  if (bPassive && (flags & (ADMXRC2_OPENEX_AUTOMAP | ADMXRC2_OPENEX_CONFIG))) {
    return ADMXRC2_INVALID_PARAMETER;
  }

  status = admxrc2OpenDevice(cardIndex, bPassive, &hCard);
  if (ADMXRC2_SUCCESS != status) {
     return status;
  }

  if (!bPassive) {
    if (bSetCoopLevel) {
      status = ADMXRC2_SetCoopLevel(hCard, coopLevel);
      if (ADMXRC2_SUCCESS != status) {
        admxrc2CloseDevice(hCard);
        return status;
      }
    }

    if (flags & ADMXRC2_OPENEX_CONFIG) {
      configureSpacesDefault(hCard);
    }

    if (flags & ADMXRC2_OPENEX_AUTOMAP) {
      mapStandardSpaces(hCard);
      mapCommonBuffers(hCard);
    }
  }

  *phCard = hCard;

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ProgramBytes(
  ADMXRC2_HANDLE hCard,
  const _ADMXRC2_DWORD* pProgram,
  unsigned long programLength)
{
  return ADMXRC2_NOT_SUPPORTED;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_Read(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_IOWIDTH width,
  _ADMXRC2_DWORD flags,
  _ADMXRC2_DWORD local,
  void* pBuffer,
  unsigned long length)
{
  IOCTLS_ADMXRC2_READ ioctl;

  ioctl.in.width = width;
  ioctl.in.flags = flags;
  ioctl.in.local = local;
  ioctl.in.pBuffer = pBuffer;
  ioctl.in.length = length;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_READ, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ReadConfig(
  ADMXRC2_HANDLE hCard,
  unsigned long index,
  _ADMXRC2_DWORD* pValue)
{
  IOCTLS_ADMXRC2_READCONFIG ioctl;
  ADMXRC2_STATUS status;

  if (index > 0xFFFFFFFFU) {
    return ADMXRC2_INVALID_PARAMETER;
  }

  ioctl.in.index = (uint32_t)index; /* Cast safe due to above check */
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_READCONFIG, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    *pValue = ioctl.out.value;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ReadFlash(
  ADMXRC2_HANDLE hCard,
  unsigned long start,
  unsigned long length,
  void* pBuffer)
{
  static const uint32_t maxUInt32 = 0xFFFFFFFFU;
  IOCTLS_ADMXRC2_READFLASH ioctl;
  uint32_t limit;

  /* Catch bad values before casting to uint32_t */
  if (start > maxUInt32) {
    return ADMXRC2_INVALID_REGION;
  }
  if (length) {
    limit = (uint32_t)(start + length - 1);
    if (limit < start) {
      return ADMXRC2_INVALID_REGION;
    }
  }

  ioctl.in.flashIndex = 0;
  ioctl.in.flags = ADMXRC2_FLASHFLAG_SYNC;
  ioctl.in.offset = (uint32_t)start;
  ioctl.in.length = (uint32_t)length;
  ioctl.in.pBuffer = pBuffer;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_READFLASH, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_ReadSensor(
  ADMXRC2_HANDLE hCard,
  unsigned int sensorIndex,
  ADMXRC2_SENSOR_VALUE* pSensorValue)
{
  IOCTLS_ADMXRC2_READSENSOR ioctl;
  ADMXRC2_STATUS status;

  if (NULL == pSensorValue) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.sensorIndex = sensorIndex;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_READSENSOR, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    switch (ioctl.out.datatype) {
    case ADMXRC2_DATA_BOOL:
      pSensorValue->Bool = ioctl.out.value.booleanValue;
      break;

    case ADMXRC2_DATA_DOUBLE:
      pSensorValue->Double = TO_DOUBLE64(ioctl.out.value.doubleValue.intPart, ioctl.out.value.doubleValue.fracPart);
      break;

    case ADMXRC2_DATA_INT32:
      pSensorValue->Int32 = ioctl.out.value.int32Value;
      break;

    case ADMXRC2_DATA_UINT32:
      pSensorValue->UInt32 = ioctl.out.value.uint32Value;
      break;

    default:
      return ADMXRC2_UNKNOWN_ERROR;
    }
  }
  return status;
}

#if ADMXRC2_HAS_USER_EVENT_HANDLE
ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_RegisterInterruptEvent(
  ADMXRC2_HANDLE hCard,
  portable_user_event_handle_t hEvent)
{
  IOCTLS_ADMXRC2_REGISTEREVENT ioctl;

  ioctl.in.hEvent = hEvent;
  ioctl.in.notification = ADMXRC2_EVENT_FPGAINTERRUPT(0);
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_REGISTEREVENT, &ioctl, NULL, sizeof(ioctl.in), 0);
}
#endif

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SetClockRate(
  ADMXRC2_HANDLE hCard,
  unsigned int clockIndex,
  double requestedFrequency,
  double* pActualFrequency)
{
  return ADMXRC2_SetClockRateEx(hCard, clockIndex, 0, requestedFrequency, pActualFrequency);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SetClockRateEx(
  ADMXRC2_HANDLE hCard,
  unsigned int clockIndex,
  uint32_t flags,
  double requestedFrequency,
  double* pActualFrequency)
{
  IOCTLS_ADMXRC2_SETCLOCKFREQUENCY ioctl;
  ADMXRC2_STATUS status;

  if (requestedFrequency > 9.0e18) {
    /* Ensure that cast to int64_t (max. value 2^63-1) cannot overflow */
    return ADMXRC2_INVALID_PARAMETER;
  }

  ioctl.in.clockIndex = (uint32_t)clockIndex;
  ioctl.in.flags = flags;
  /* Casting to int64_t then uint64_t avoids missing '__fixunsdfdi' symbol when building as shared library with some GNU toolchains */
  ioctl.in.frequency = (uint64_t)(int64_t)requestedFrequency; /* Cast should be safe due to above range check */
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_SETCLOCKFREQUENCY, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    if (NULL != pActualFrequency) {
      /* Casting to int64_t then double avoids missing '__floadundidf' symbol when linking apps against shared library with some GNU toolchains */
      *pActualFrequency = (double)(int64_t)ioctl.out.actualFrequency; /* Cast should be safe due to above range check */
    }
  }

  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SetCoopLevel(
  ADMXRC2_HANDLE hCard,
  _ADMXRC2_UINT32 newLevel)
{
  IOCTLS_ADMXRC2_SETCOOPLEVEL ioctl;

  ioctl.in.coopLevel = newLevel;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_SETCOOPLEVEL, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SetSpaceConfig(
  ADMXRC2_HANDLE hCard,
  unsigned int spaceIndex,
  _ADMXRC2_DWORD flags)
{
  IOCTLS_ADMXRC2_SETSPACECONFIG ioctl;

  ioctl.in.index = spaceIndex;
  ioctl.in.configFlags = flags;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_SETSPACECONFIG, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SetupDMA(
  ADMXRC2_HANDLE hCard,
  const void* pBuffer,
  unsigned long size,
  _ADMXRC2_DWORD flags,
  ADMXRC2_DMADESC* phDmaDesc)
{
  IOCTLS_ADMXRC2_LOCK ioctl;
  ADMXRC2_STATUS status;

  if (NULL == phDmaDesc) {
    return ADMXRC2_NULL_POINTER;
  }

  ioctl.in.pBuffer = pBuffer;
  ioctl.in.length = size;
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_LOCK, &ioctl, NULL, sizeof(ioctl.in), sizeof(ioctl.out));
  if (ADMXRC2_SUCCESS == status) {
    *phDmaDesc = ioctl.out.hBuffer;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SyncCommonBuffer(
  ADMXRC2_HANDLE hCard,
  unsigned int index,
  unsigned long offset,
  unsigned long length,
  ADMXRC2_SYNCMODE syncMode)
{
  IOCTLS_ADMXRC2_SYNCCOMMONBUFFER ioctl;

  ioctl.in.index = index;
  ioctl.in.syncMode = syncMode;
  ioctl.in.offset = offset;
  ioctl.in.length = length;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_SYNCCOMMONBUFFER, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_SyncDirectMaster(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_DMADESC hDmaDesc,
  unsigned long offset,
  unsigned long length,
  ADMXRC2_SYNCMODE syncMode)
{
  return ADMXRC2_NOT_SUPPORTED;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_Unconfigure(
  ADMXRC2_HANDLE hCard,
  unsigned int targetIndex,
  uint32_t flags)
{
  IOCTLS_ADMXRC2_UNCONFIGURE ioctl;

  ioctl.in.targetIndex = targetIndex;
  ioctl.in.flags = flags;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_UNCONFIGURE, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_UnloadBitstream(
  ADMXRC2_IMAGE image)
{
  if (NULL == image) {
    return ADMXRC2_NULL_POINTER;
  }

  free(image);
  return ADMXRC2_SUCCESS;
}

#if ADMXRC2_HAS_USER_EVENT_HANDLE
ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_UnregisterInterruptEvent(
  ADMXRC2_HANDLE hCard,
  portable_user_event_handle_t hEvent)
{
  IOCTLS_ADMXRC2_UNREGISTEREVENT ioctl;

  ioctl.in.hEvent = hEvent;
  ioctl.in.notification = ADMXRC2_EVENT_FPGAINTERRUPT(0);
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_UNREGISTEREVENT, &ioctl, NULL, sizeof(ioctl.in), 0);
}
#endif

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_UnsetupDMA(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_DMADESC hDmaDesc)
{
  IOCTLS_ADMXRC2_UNLOCK ioctl;

  if (hDmaDesc > 0xFFFFFFFFU) {
    return ADMXRC2_INVALID_DMADESC;
  }
  ioctl.in.hBuffer = (uint32_t)hDmaDesc; /* Cast safe due to above check */
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_UNLOCK, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_WaitForInterrupt(
  ADMXRC2_HANDLE hCard,
  unsigned long* pTimeoutUs,
  unsigned long flags,
#if defined(_WIN32) || defined(_MSC_VER)
  HANDLE hEvent)
#else
  void* pIgnored)
#endif
{
  IOCTLS_ADMXRC2_WAITFORINTERRUPT ioctl;
  ADMXRC2_STATUS status;

  if (NULL != pTimeoutUs) {
    ioctl.in.timeoutUs = *pTimeoutUs;
    flags |= ADMXRC2_INTWAIT_TIMEOUT;
  }
  ioctl.in.targetIndex = 0;
  ioctl.in.flags = (uint32_t)flags; /* No flags currently exist in bits higher than 31, so cast them away */
#if defined(_WIN32) || defined(_MSC_VER)
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_WAITFORINTERRUPT, &ioctl, hEvent, sizeof(ioctl.in), sizeof(ioctl.out));
#else
  status = admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_WAITFORINTERRUPT, &ioctl, pIgnored, sizeof(ioctl.in), sizeof(ioctl.out));
#endif
  if (ADMXRC2_SUCCESS == status && NULL != pTimeoutUs) {
    *pTimeoutUs = (unsigned long)ioctl.out.timeoutUs;
  }
  return status;
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_Write(
  ADMXRC2_HANDLE hCard,
  ADMXRC2_IOWIDTH width,
  _ADMXRC2_DWORD flags,
  _ADMXRC2_DWORD local,
  const void* pData,
  unsigned long length)
{
  IOCTLS_ADMXRC2_WRITE ioctl;

  ioctl.in.width = width;
  ioctl.in.flags = flags;
  ioctl.in.local = local;
  ioctl.in.pData = pData;
  ioctl.in.length = length;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_WRITE, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_WriteConfig(
  ADMXRC2_HANDLE hCard,
  unsigned long index,
  _ADMXRC2_DWORD value)
{
  IOCTLS_ADMXRC2_WRITECONFIG ioctl;

  if (index > 0xFFFFFFFFU) {
    return ADMXRC2_INVALID_PARAMETER;
  }

  ioctl.in.index = (uint32_t)index; /* Cast safe due to above check */
  ioctl.in.value = value;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_WRITECONFIG, &ioctl, NULL, sizeof(ioctl.in), 0);
}

ADMXRC2_EXPORT ADMXRC2_STATUS
ADMXRC2_CALLING_CONVENTION
ADMXRC2_WriteFlash(
  ADMXRC2_HANDLE hCard,
  unsigned long start,
  unsigned long length,
  const void* pData)
{
  static const uint32_t maxUInt32 = 0xFFFFFFFFU;
  IOCTLS_ADMXRC2_WRITEFLASH ioctl;
  uint32_t limit;

  /* Catch bad values before casting to uint32_t */
  if (start > maxUInt32) {
    return ADMXRC2_INVALID_REGION;
  }
  if (length) {
    limit = (uint32_t)(start + length - 1);
    if (limit < start) {
      return ADMXRC2_INVALID_REGION;
    }
  }

  ioctl.in.flashIndex = 0;
  ioctl.in.flags = ADMXRC2_FLASHFLAG_SYNC;
  ioctl.in.offset = (uint32_t)start;
  ioctl.in.length = (uint32_t)length;
  ioctl.in.pData = pData;
  return admxrc2Ioctl(hCard, ADMXRC2_IOCTLCODE_WRITEFLASH, &ioctl, NULL, sizeof(ioctl.in), 0);
}

