/*
** File: space.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for returning information about a
**          space or getting or setting the configuration of a space.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "space.h"

DfIoStatus
ioctlGetSpaceInfo(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETSPACEINFO* pIoctl = (IOCTLS_ADMXRC2_GETSPACEINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreWindowStatus windowStatus;
  CoreWindowInfo windowInfo;
  unsigned int windowIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  windowIndex = pIoctl->in.index;
  windowStatus = pDevCtx->coreInterface.pGetWindowInfo(pDevCtx->pCoreContext, windowIndex, &windowInfo);
  if (CoreWindowSuccess != windowStatus) {
    switch (windowStatus) {
    case CoreWindowInvalidIndex:
      status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
      break;

    default:
      status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
      break;
    }
  } else {
    dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
    pIoctl->out.busBase = windowInfo.busBase;
    pIoctl->out.busSize = windowInfo.busSize;
    pIoctl->out.localSize = windowInfo.localSize;
    pIoctl->out.localBase = windowInfo.localBase;
    pIoctl->out.virtualSize = (uint64_t)windowInfo.kernelSize;
  }
  return status;
}

DfIoStatus
ioctlGetSpaceConfig(
	Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETSPACECONFIG* pIoctl = (IOCTLS_ADMXRC2_GETSPACECONFIG*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreWindowConfigStatus windowStatus;
  CoreWindowConfig config;
  unsigned int windowIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  windowIndex = pIoctl->in.index;
  windowStatus = pDevCtx->coreInterface.pGetWindowConfig(pDevCtx->pCoreContext, windowIndex, &config);
  switch (windowStatus) {
  case CoreWindowConfigSuccess:
    break;

  case CoreWindowConfigInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreWindowConfigNotSupported:
    return DfIoStatusError(ADMXRC2_NOT_SUPPORTED);

  case CoreWindowConfigInvalidValue:
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);

  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }

  pIoctl->out.configFlags = 0;
  switch (config.width & COREWINDOWCONFIG_WIDTH_MASK) {
  case COREWINDOWCONFIG_WIDTH_8BIT:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_8;
    break;

  case COREWINDOWCONFIG_WIDTH_16BIT:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_16;
    break;

  case COREWINDOWCONFIG_WIDTH_32BIT:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_32;
    break;

  case COREWINDOWCONFIG_WIDTH_64BIT:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_64;
    break;

  default:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_DEFAULT;
    break;
  }
  switch (config.prefetch & COREWINDOWCONFIG_PREFETCH_MASK) {
  case COREWINDOWCONFIG_PREFETCH_MIN:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_PREFETCH_MINIMUM;
    break;

  case COREWINDOWCONFIG_PREFETCH_NORMAL:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_PREFETCH_NORMAL;
    break;

  case COREWINDOWCONFIG_PREFETCH_MAX:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_PREFETCH_MAXIMUM;
    break;

  default:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_WIDTH_DEFAULT;
    break;
  }
  switch (config.burst & COREWINDOWCONFIG_BURST_MASK) {
  case COREWINDOWCONFIG_BURST_ENABLE:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_BURST_ENABLED;
    break;

  case COREWINDOWCONFIG_BURST_DISABLE:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_BURST_DISABLED;
    break;

  default:
    pIoctl->out.configFlags |= ADMXRC2_SPACE_BURST_DEFAULT;
    break;
  }

  return status;
}

DfIoStatus
ioctlSetSpaceConfig(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADMXRC2_SETSPACECONFIG* pIoctl = (IOCTLS_ADMXRC2_SETSPACECONFIG*)pBuffer;
  DfIoStatus status;
  CoreWindowConfigStatus windowStatus;
  CoreWindowConfig config;
  unsigned int windowIndex;
  uint32_t configFlags, windowFlags = 0;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  windowIndex = pIoctl->in.index;
  configFlags = pIoctl->in.configFlags;

  DF_ZERO_OBJECT(&config);
  if (configFlags & ADMXRC2_SPACE_SET_WIDTH) {
    windowFlags |= COREWINDOWCONFIG_SET_WIDTH;
    switch (configFlags & ADMXRC2_SPACE_MASK_WIDTH) {
    case ADMXRC2_SPACE_WIDTH_DEFAULT:
      config.width = COREWINDOWCONFIG_WIDTH_DEFAULT;
      break;

    case ADMXRC2_SPACE_WIDTH_8:
      config.width = COREWINDOWCONFIG_WIDTH_8BIT;
      break;

    case ADMXRC2_SPACE_WIDTH_16:
      config.width = COREWINDOWCONFIG_WIDTH_16BIT;
      break;

    case ADMXRC2_SPACE_WIDTH_32:
      config.width = COREWINDOWCONFIG_WIDTH_32BIT;
      break;

    case ADMXRC2_SPACE_WIDTH_64:
      config.width = COREWINDOWCONFIG_WIDTH_64BIT;
      break;

    default:
      return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
    }
  }
  if (configFlags & ADMXRC2_SPACE_SET_PREFETCH) {
    windowFlags |= COREWINDOWCONFIG_SET_PREFETCH;
    switch (configFlags & ADMXRC2_SPACE_MASK_PREFETCH) {
    case ADMXRC2_SPACE_PREFETCH_DEFAULT:
      config.prefetch = COREWINDOWCONFIG_PREFETCH_DEFAULT;
      break;

    case ADMXRC2_SPACE_PREFETCH_MINIMUM:
      config.prefetch = COREWINDOWCONFIG_PREFETCH_MIN;
      break;

    case ADMXRC2_SPACE_PREFETCH_NORMAL:
      config.prefetch = COREWINDOWCONFIG_PREFETCH_NORMAL;
      break;

    case ADMXRC2_SPACE_PREFETCH_MAXIMUM:
      config.prefetch = COREWINDOWCONFIG_PREFETCH_MAX;
      break;

    default:
      return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
    }
  }
  if (configFlags & ADMXRC2_SPACE_SET_BURST) {
    windowFlags |= COREWINDOWCONFIG_SET_BURST;
    switch (configFlags & ADMXRC2_SPACE_MASK_BURST) {
    case ADMXRC2_SPACE_BURST_DEFAULT:
      config.burst = COREWINDOWCONFIG_BURST_DEFAULT;
      break;

    case ADMXRC2_SPACE_BURST_ENABLED:
      config.burst = COREWINDOWCONFIG_BURST_ENABLE;
      break;

    case ADMXRC2_SPACE_BURST_DISABLED:
      config.burst = COREWINDOWCONFIG_BURST_DISABLE;
      break;

    default:
      return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
    }
  }

  windowStatus = pDevCtx->coreInterface.pSetWindowConfig(pDevCtx->pCoreContext, windowIndex, windowFlags, &config);
  switch (windowStatus) {
  case CoreWindowConfigSuccess:
    status = DfIoStatusSuccess;
    break;

  case CoreWindowConfigInvalidIndex:
    status = DfIoStatusError(ADMXRC2_INVALID_INDEX);
    break;

  case CoreWindowConfigNotSupported:
    status = DfIoStatusError(ADMXRC2_NOT_SUPPORTED);
    break;

  case CoreWindowConfigInvalidValue:
    status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
    break;

  default:
    status = DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
    break;
  }

  return status;
}
