/*
** File: model_adpwrc2.c
** Project: ADMXRC2 module driver
** Purpose: ADP-WRC-II specific code, including the function to bootstrap
**          the device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include <core/pci9xxx.h>
#include <core/adpdrc2.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdpwrc2Rev0* pVpd = (VpdAdpwrc2Rev0*)pVpdBuffer;
  uint8_t order;

  if (bankIndex >= DF_ARRAY_LENGTH(pVpd->sdram)) {
    return ADMXRC2_INVALID_INDEX;
  }
  order = pVpd->sdram[bankIndex].order;
  pIoctl->out.width = 512U;
  pIoctl->out.typeMask = ADMXRC2_RAM_SDRAM_DDR; /* DDR SDRAM */
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.bFitted = (order != 0) ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdpwrc2Rev0* pVpd = (VpdAdpwrc2Rev0*)pVpdBuffer;
  uint32_t serialNumber;
  unsigned int i;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  serialNumber = pVpd->serialNumLo;
  serialNumber |= (uint32_t)pVpd->serialNumHi << 16;
  pIoctl->out.serialNumber = serialNumber;
  pIoctl->out.numMemoryBank = 5;
  pIoctl->out.memoryBankFitted = 0;
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->sdram); i++) {
    if (pVpd->sdram[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << i);
    }
  }
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdpwrc2Rev0* pVpd = (VpdAdpwrc2Rev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1517");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = (uint32_t)-1;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdpwrc2(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdpwrc2: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 2U;
  pDevCtx->configureDmaAddress = 0x800008U; /* TODO - should make this dynamic, i.e. based on current value of LAS1BA */
  pDevCtx->configureDmaMode =
    PCI9656_DMAMODE_8BIT | PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM |
    PCI9656_DMAMODE_ENABLE_BURST | PCI9656_DMAMODE_FIXED_LOCAL;
}
