/*
** File: flash.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for Flash programming.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "flash.h"
#include <admxrc2/types.h>

#define READFLASH_BUFFER_SIZE  (0x100)
#define WRITEFLASH_BUFFER_SIZE (0x100)

typedef struct _Context {
  boolean_t bCancel;
  DfIoStatus reason;
  DfEvent event;
} Context;

static DfIoStatus
mapAcquireStatus(
  CoreAcquireStatus acqStatus)
{
  switch (acqStatus) {
  case CoreAcquireSuccess:
    return DfIoStatusSuccess;

  case CoreAcquireMustWait:
    return DfIoStatusError(ADMXRC2_DEVICE_BUSY);

  case CoreAcquireCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreAcquireRemoved:
  case CoreAcquireFailed:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreAcquireInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreAcquireInvalid:
  case CoreAcquireUnexpectedError:
  case CoreAcquireWaiting:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapFlashStatus(
  CoreFlashStatus flashStatus)
{
  switch (flashStatus) {
  case CoreFlashSuccess:
    return DfIoStatusSuccess;
  case CoreFlashInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  case CoreFlashNotPresent:
  case CoreFlashTimeout:
  case CoreFlashVerifyFailed:
    return DfIoStatusError(ADMXRC2_FAILED);
  case CoreFlashInvalidRegion:
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  case CoreFlashInterrupted:
    return DfIoStatusError(ADMXRC2_CANCELLED);
  case CoreFlashInvalidParameter:
  case CoreFlashGeneralFailure:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static void
cancelCallback(
  DfClientRequest* pReq,
  DfIoStatus reason)
{
  Context* pContext = (Context*)dfRequestGetContext(pReq);

  pContext->reason = reason;
  pContext->bCancel = TRUE;
  dfEventSignal(&pContext->event);
}

/* Called if a Flash bank needs to be sync'ed when a device handle is closed */
void
syncFlashOnClose(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  unsigned int bankIndex)
{
  CoreAcquireStatus acqStatus;
  CoreFlashStatus flashStatus;
  CoreTicket ticket;

  dfDebugPrint(3, ("syncFlashOnClose: bankIndex=%lu\n", (unsigned long)bankIndex));

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_FLASH;
  ticket.resources[0].index = (uint8_t)bankIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    dfDebugPrint(0, ("*** syncFlashOnClose: failed to initialize ticket for Flash bank %u\n", bankIndex));
    return;
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, CORE_ACQUIRE_SYNC_UNINTERRUPTIBLE);
  if (CoreAcquireSuccess != acqStatus) {
    dfDebugPrint(0, ("*** syncFlashOnClose: failed to acquire ticket for Flash bank %u\n", bankIndex));
    return;
  }
  flashStatus = pDevCtx->coreInterface.pFlashSync(pDevCtx->pCoreContext, &ticket, bankIndex);
  if (CoreFlashSuccess != flashStatus) {
    dfDebugPrint(0, ("*** syncFlashOnClose: failed to sync flash bank %u\n", bankIndex));
    goto done;
  }

done:
  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
}

DfIoStatus
ioctlEraseFlash(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = ADMXRC2_FLASHFLAG_SYNC;
#endif
  IOCTLS_ADMXRC2_ERASEFLASH* pIoctl = (IOCTLS_ADMXRC2_ERASEFLASH*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreFlashStatus flashStatus;
  CoreFlashInfo flashInfo;
  CoreTicket ticket;
  unsigned int bankIndex;
  uint32_t flags;
  uint64_t offset, length;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }

  bankIndex = pIoctl->in.flashIndex;
  flags = pIoctl->in.flags;
  offset = pIoctl->in.offset;
  length = pIoctl->in.length;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(3, ("ioctlEraseFlash: bankIndex=%lu flags=0x%lx offset=" DF_FMT_U64 "(0x%08lx_%08lx) length=" DF_FMT_U64 "(0x%08lx_%08lx)\n",
    (unsigned long)bankIndex, (unsigned long)flags, (unsigned long long)offset, dfSplitUint64(offset),
    (unsigned long long)length, dfSplitUint64(length)));

  if (bankIndex >= pDevCtx->info.numFlashBank) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  flashStatus = pDevCtx->coreInterface.pGetFlashInfo(pDevCtx->pCoreContext, bankIndex, &flashInfo);
  if (CoreFlashSuccess != flashStatus) {
    status = mapFlashStatus(flashStatus);
    return DfIoStatusError(status);
  }
  if (offset + length < offset) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset < flashInfo.useableStart) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset + length > flashInfo.useableStart + flashInfo.useableLength) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_FLASH;
  ticket.resources[0].index = (uint8_t)bankIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  pClCtx->bNeedFlashSync[bankIndex] = TRUE;
  flashStatus = pDevCtx->coreInterface.pFlashErase(pDevCtx->pCoreContext, &ticket, bankIndex, offset, length);
  if (CoreFlashSuccess != flashStatus) {
    status = mapFlashStatus(flashStatus);
  }

  if (flags & ADMXRC2_FLASHFLAG_SYNC) {
    flashStatus = pDevCtx->coreInterface.pFlashSync(pDevCtx->pCoreContext, &ticket, bankIndex);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
    } else {
      pClCtx->bNeedFlashSync[bankIndex] = FALSE;
    }
  }

  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
  return status;
}

DfIoStatus
ioctlGetFlashInfo(
  Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETFLASHINFO* pIoctl = (IOCTLS_ADMXRC2_GETFLASHINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreFlashStatus flashStatus;
  CoreFlashInfo flashInfo;
  unsigned int bankIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  bankIndex = pIoctl->in.flashIndex;

  dfDebugPrint(3, ("ioctlGetFlashInfo: bankIndex=%lu\n", (unsigned long)bankIndex));

  flashStatus = pDevCtx->coreInterface.pGetFlashInfo(pDevCtx->pCoreContext, bankIndex, &flashInfo);
  if (CoreFlashSuccess == flashStatus) {
    /* The casts in the following 4 assigments should be OK because no Gen1/Gen2 models have >= 4GiB of Flash in a bank */
    pIoctl->out.deviceSize = (uint32_t)flashInfo.size;
    pIoctl->out.deviceBlockSize = (uint32_t)flashInfo.largestBlock;
    pIoctl->out.useableStart = (uint32_t)flashInfo.useableStart;
    pIoctl->out.useableLength = (uint32_t)flashInfo.useableLength;
    pIoctl->out.vendorId = flashInfo.vendorId;
    pIoctl->out.deviceId = flashInfo.deviceId;
  } else {
    status = mapFlashStatus(flashStatus);
  }

  return status;
}

DfIoStatus
ioctlGetFlashBlockInfo(
  Admxrc2DeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_GETFLASHBLOCKINFO* pIoctl = (IOCTLS_ADMXRC2_GETFLASHBLOCKINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreFlashStatus flashStatus;
  CoreFlashBlockInfo blockInfo;
  unsigned int bankIndex;
  uint64_t address;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  bankIndex = pIoctl->in.flashIndex;
  address = pIoctl->in.location;

  dfDebugPrint(3, ("ioctlGetFlashBlockInfo: bankIndex=%lu address=" DF_FMT_U64 "(0x%08lx_%08lx)\n",
    (unsigned long)bankIndex, (unsigned long long)address, dfSplitUint64(address)));

  flashStatus = pDevCtx->coreInterface.pGetFlashBlockInfo(pDevCtx->pCoreContext, bankIndex, address, &blockInfo);
  if (CoreFlashSuccess == flashStatus) {
    /* The casts in the following 2 assigments should be OK because no Gen1/Gen2 models have >= 4GiB of Flash in a bank */
    pIoctl->out.start = (uint32_t)blockInfo.address;
    pIoctl->out.length = (uint32_t)blockInfo.length;
    pIoctl->out.flags = 0;
    if (blockInfo.flags & COREFLASH_BOOTBLOCK) {
      pIoctl->out.flags |= ADMXRC2_FLASHBLOCKFLAG_BOOT;
    }
  } else {
    status = mapFlashStatus(flashStatus);
  }

  return status;
}

DfIoStatus
ioctlReadFlash(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = ADMXRC2_FLASHFLAG_SYNC;
#endif
  IOCTLS_ADMXRC2_READFLASH* pIoctl = (IOCTLS_ADMXRC2_READFLASH*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreFlashStatus flashStatus;
  CoreFlashInfo flashInfo;
  CoreTicket ticket;
  uint8_t buffer[READFLASH_BUFFER_SIZE];
  unsigned int bankIndex;
  uint32_t flags;
  uint64_t offset;
  size_t length, chunk;
  uint8_t* pUserBuf;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  bankIndex = pIoctl->in.flashIndex;
  flags = pIoctl->in.flags;
  offset = pIoctl->in.offset;
  length = pIoctl->in.length;
  pUserBuf = (uint8_t*)pIoctl->in.pBuffer;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(3, ("ioctlReadFlash: bankIndex=%lu flags=0x%lx offset=" DF_FMT_U64 "(0x%08lx_%08lx) length=" DF_FMT_U64 "(0x%08lx_%08lx) pUserBuf=%p\n",
    (unsigned long)bankIndex, (unsigned long)flags, (unsigned long long)offset, dfSplitUint64(offset),
    (unsigned long long)length, dfSplitUint64((uint64_t)length), pUserBuf));

  if (bankIndex >= pDevCtx->info.numFlashBank) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  flashStatus = pDevCtx->coreInterface.pGetFlashInfo(pDevCtx->pCoreContext, bankIndex, &flashInfo);
  if (CoreFlashSuccess != flashStatus) {
    status = mapFlashStatus(flashStatus);
    return DfIoStatusError(status);
  }
  if (offset + length < offset) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset < flashInfo.useableStart) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset + length > flashInfo.useableStart + flashInfo.useableLength) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_FLASH;
  ticket.resources[0].index = (uint8_t)bankIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuf, length, TRUE, TAG1);

  while (length) {
    chunk = (length > sizeof(buffer)) ? sizeof(buffer) : length;
    flashStatus = pDevCtx->coreInterface.pFlashRead(pDevCtx->pCoreContext, &ticket, bankIndex, offset, chunk, buffer);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
      goto done;
    }
    dfCopyKU(pUserBuf, buffer, chunk, TAG1);
    offset += chunk;
    pUserBuf += chunk;
    length -= chunk;
  }

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  DF_USER_SPACE_END(TAG1)

done:
  if (flags & ADMXRC2_FLASHFLAG_SYNC) {
    flashStatus = pDevCtx->coreInterface.pFlashSync(pDevCtx->pCoreContext, &ticket, bankIndex);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
    } else {
      pClCtx->bNeedFlashSync[bankIndex] = FALSE;
    }
  }

  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlReadFlashThunk(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_READFLASH* pIoctl32 = (IOCTLS32_ADMXRC2_READFLASH*)pBuffer;
  IOCTLS_ADMXRC2_READFLASH ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.flashIndex = pIoctl32->in.flashIndex;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.offset = pIoctl32->in.offset;
  ioctl.in.length = pIoctl32->in.length;
  ioctl.in.pBuffer = dfThunkPtr(pIoctl32->in.pBuffer);
  return ioctlReadFlash(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in));
}
#endif

DfIoStatus
ioctlSyncFlash(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = 0;
#endif
  IOCTLS_ADMXRC2_SYNCFLASH* pIoctl = (IOCTLS_ADMXRC2_SYNCFLASH*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreFlashStatus flashStatus;
  CoreTicket ticket;
  unsigned int bankIndex;
  uint32_t flags;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  bankIndex = pIoctl->in.flashIndex;
  flags = pIoctl->in.flags;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(3, ("ioctlSyncFlash: bankIndex=%lu flags=0x%lx\n", (unsigned long)bankIndex, (unsigned long)flags));

  if (bankIndex >= pDevCtx->info.numFlashBank) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_FLASH;
  ticket.resources[0].index = (uint8_t)bankIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  flashStatus = pDevCtx->coreInterface.pFlashSync(pDevCtx->pCoreContext, &ticket, bankIndex);
  if (CoreFlashSuccess != flashStatus) {
    status = mapFlashStatus(flashStatus);
  } else {
    pClCtx->bNeedFlashSync[bankIndex] = FALSE;
  }

  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
  return status;
}

DfIoStatus
ioctlWriteFlash(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = ADMXRC2_FLASHFLAG_SYNC;
#endif
  IOCTLS_ADMXRC2_WRITEFLASH* pIoctl = (IOCTLS_ADMXRC2_WRITEFLASH*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreFlashStatus flashStatus;
  CoreFlashInfo flashInfo;
  CoreFlashBlockInfo blockInfo;
  CoreTicket ticket;
  Context context;
  unsigned int bankIndex;
  uint32_t flags;
  uint64_t offset, blockOffset, chunk;
  size_t length;
  const uint8_t* pUserBuf;
  void* pStagingBuffer;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  bankIndex = pIoctl->in.flashIndex;
  flags = pIoctl->in.flags;
  offset = pIoctl->in.offset;
  length = pIoctl->in.length;
  pUserBuf = (const uint8_t*)pIoctl->in.pData;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(3, ("ioctlWriteFlash: bankIndex=%lu flags=0x%lx offset=" DF_FMT_U64 "(0x%08lx_%08lx) length=" DF_FMT_U64 "(0x%08lx_%08lx), pUserBuf=%p\n",
    (unsigned long)bankIndex, (unsigned long)flags, (unsigned long long)offset, dfSplitUint64(offset),
    (unsigned long long)length, dfSplitUint64((uint64_t)length), pUserBuf));

  if (bankIndex >= pDevCtx->info.numFlashBank) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  flashStatus = pDevCtx->coreInterface.pGetFlashInfo(pDevCtx->pCoreContext, bankIndex, &flashInfo);
  if (CoreFlashSuccess != flashStatus) {
    status = mapFlashStatus(flashStatus);
    return DfIoStatusError(status);
  }
  if (offset + length < offset) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset < flashInfo.useableStart) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }
  if (offset + length > flashInfo.useableStart + flashInfo.useableLength) {
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_FLASH;
  ticket.resources[0].index = (uint8_t)bankIndex;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  dfRequestSetContext(pReq, &context);
  dfEventInit(&context.event);
  context.bCancel = FALSE;
  if (!dfRequestSetCallback(pReq, cancelCallback)) {
    status = context.reason;
    goto done2;
  }

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuf, length, FALSE, TAG1);

  pStagingBuffer = pDevCtx->flash[bankIndex].pStagingBuffer;
  while (length) {
    flashStatus = pDevCtx->coreInterface.pGetFlashBlockInfo(pDevCtx->pCoreContext, bankIndex, offset, &blockInfo);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
      goto done;
    }
    blockOffset = offset - blockInfo.address;
    chunk = blockInfo.length - blockOffset;
    if (chunk > length) {
      chunk = length;
    }
    dfCopyUK(pStagingBuffer, pUserBuf, (size_t)chunk, TAG1); /* Cast safe because (chunk <= length) and length is of type 'size_t' */
    pClCtx->bNeedFlashSync[bankIndex] = TRUE;
    /* Cast in following line is safe because (chunk <= length) and length is of type 'size_t' */
    flashStatus = pDevCtx->coreInterface.pFlashWrite(pDevCtx->pCoreContext, &ticket, bankIndex, offset, (size_t)chunk, pStagingBuffer);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
      goto done;
    }
    offset += chunk;
    pUserBuf += chunk;
    length -= (size_t)chunk;
    if (context.bCancel) {
      break;
    }
  }

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  DF_USER_SPACE_END(TAG1)

done:
  if (flags & ADMXRC2_FLASHFLAG_SYNC) {
    flashStatus = pDevCtx->coreInterface.pFlashSync(pDevCtx->pCoreContext, &ticket, bankIndex);
    if (CoreFlashSuccess != flashStatus) {
      status = mapFlashStatus(flashStatus);
    } else {
      pClCtx->bNeedFlashSync[bankIndex] = FALSE;
    }
  }

  if (!dfRequestClearCallback(pReq)) {
    dfEventWait(&context.event);
    status = context.reason;
  }

done2:
  dfEventUninit(&context.event);
  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlWriteFlashThunk(
  Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_WRITEFLASH* pIoctl32 = (IOCTLS32_ADMXRC2_WRITEFLASH*)pBuffer;
  IOCTLS_ADMXRC2_WRITEFLASH ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.flashIndex = pIoctl32->in.flashIndex;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.offset = pIoctl32->in.offset;
  ioctl.in.length = pIoctl32->in.length;
  ioctl.in.pData = dfThunkConstPtr(pIoctl32->in.pData);
  return ioctlWriteFlash(pDevCtx, pClCtx, pReq, &ioctl, sizeof(ioctl.in));
}
#endif
