/*
** File: event.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for event registration and waiting.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "event.h"
#include <admxrc2/types.h>

#define DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_CANCEL (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_NORMAL (6) /* Level of debug messages relating to normal operation */

typedef enum _WaitState {
  WaitStateInitial   = 0, /* Not a legal state for any callback */
  WaitStateInList    = 1, /* In an interrupt wait list */
  WaitStateNotInList = 2  /* Not in interrupt wait list */
} WaitState;

typedef enum _TimerState {
  TimerStateInitial      = 0, /* Not a legal state for any callback */
  TimerStateNotScheduled = 1, /* Timer is not scheduled */
  TimerStateScheduled    = 2  /* Timer is scheduled */
} TimerState;

typedef enum _CancelState {
  CancelStateInitial     = 0, /* Not a legal state for any callback */
  CancelStateHasCallback = 1, /* Client request has a cancel function */
  CancelStateNoCallback  = 2  /* Client request does not have a cancel function */
} CancelState;

#if DF_HAS_USER_EVENT_HANDLE
/* Must be called while holding the list's lock */
static void
insertNodeAtTail(
  UserEventList* pList,
  UserEventInfo* pInfo)
{
  pInfo->pNext = NULL;
  pInfo->pPrev = pList->pHead;
  if (NULL == pList->pHead) {
    pList->pHead = pInfo;
    pList->pTail = pInfo;
  } else {
    pList->pTail->pNext = pInfo;
    pList->pTail = pInfo;
  }
}
#endif

#if DF_HAS_USER_EVENT_HANDLE
/* Must be called while holding the list's lock */
static void
removeNode(
  UserEventList* pList,
  UserEventInfo* pInfo)
{
  if (NULL != pInfo->pNext) {
    pInfo->pNext->pPrev = pInfo->pPrev;
  } else {
    pList->pTail = pInfo->pNext;
  }
  if (NULL != pInfo->pPrev) {
    pInfo->pPrev->pNext = pInfo->pNext;
  } else {
    pList->pHead = pInfo->pNext;
  }
}
#endif

#if DF_HAS_USER_EVENT_HANDLE
static void
notifyAllEvents(
  Admxrc2DeviceContext* pDevCtx,
  UserEventList* pList,
  unsigned int targetIndex)
{
  UserEventInfo* pInfo;
  UserEventInfo* pNext;
  UserEventInfo* pFree = NULL;
  DfSpinLockFlags f;

  /* Signal all user events in the list */
  f = dfSpinLockGet(&pList->lock);
  pInfo = pList->pHead;
  while (NULL != pInfo) {
    pInfo->refCount++; /* Increment reference count, to prevent another thread removing this node */
    dfSpinLockPut(&pList->lock, f);
    dfDebugPrint(5, ("notifyAllEvents: userEvent=%p\n", (void*)(uintptr_t)pInfo->userEvent));
    dfUserEventSignal(pInfo->userEvent);
    if (NULL != pFree) {
      /* Free the previous node whose reference count went to 0 */
      dfUserEventPut(pFree->userEvent);
      dfFree(pFree);
      pFree = NULL;
    }
    f = dfSpinLockGet(&pList->lock);
    pNext = pInfo->pNext;
    pInfo->refCount--; /* Decrement reference count */
    if (pInfo->refCount == 0) {
      /* Reference count has gone to 0, so remove from list */
      removeNode(pList, pInfo);
      pFree = pInfo;
    }
    pInfo = pNext;
  }
  dfSpinLockPut(&pList->lock, f);
  if (NULL != pFree) {
    /* Free the previous node whose reference count went to 0 */
    dfUserEventPut(pFree->userEvent);
    dfFree(pFree);
    pFree = NULL;
  }
}
#endif

static void
completeInterruptWait(
  Admxrc2DeviceContext* pDevCtx,
  InterruptWaitTicket* pTicket,
  DfIoStatus status)
{
  DfClientRequest* pRequest;
  IOCTLS_ADMXRC2_WAITFORINTERRUPT* pIoctl;
  DfTime now, delta;
  uint64_t microseconds;

  /* Sanity check */
  dfAssert(pTicket->waitState == WaitStateNotInList);
  dfAssert(pTicket->timerState == TimerStateNotScheduled);
  dfAssert(pTicket->cancelState == CancelStateNoCallback);

  pRequest = pTicket->pRequest;
  pIoctl = pTicket->pIoctl;
  if (ADMXRC2_SUCCESS == status && NULL != pIoctl) {
    now = dfTimeGet();
    if (now >= pTicket->timeoutTicks) {
      delta = 0;
    } else {
      delta = pTicket->timeoutTicks - now;
    }
    microseconds = dfTimeToMicroseconds(delta);
    pIoctl->out.timeoutUs = (microseconds < pIoctl->in.timeoutUs) ? microseconds : pIoctl->in.timeoutUs;
    dfTimerUninit(&pTicket->timer);
  }
  dfRequestComplete(pRequest, status);
  dfPoolFree(&pDevCtx->interruptWait.ticketPool, pTicket);
}

static unsigned int
completeInterruptWaitsWork(
  Admxrc2DeviceContext* pDevCtx,
  InterruptWaitList* pWaitList,
  DfList* pCompleteList,
  unsigned int targetIndex,
  DfIoStatus reason)
{
  DfListNode* pHead;
  DfSpinLockFlags f;
  InterruptWaitTicket* pTicket;
  unsigned int count = 0;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("completeInterruptWaitsWork: pCompleteList=%p targetIndex=%u reason=%lu(0x%lx)\n",
    pCompleteList, targetIndex, (unsigned long)reason, (unsigned long)reason));

  while (1) {
    f = dfSpinLockGet(&pWaitList->lock);
    pHead = dfListGetHead(pCompleteList);
    if (pHead == NULL) {
      dfSpinLockPut(&pWaitList->lock, f);
      break;
    }
    dfListRemove(pHead);
    pTicket = DF_CONTAINER_OF(pHead, InterruptWaitTicket, node);
    dfAssert(pTicket->waitState == WaitStateInList);
    dfDebugPrint(DEBUGLEVEL_NORMAL, ("completeInterruptWaitsWork: pHead=%p pTicket=%p waitState=%u cancelState=%u timerState=%u\n",
      pHead, pTicket, pTicket->waitState, pTicket->cancelState, pTicket->timerState));
    pTicket->waitState = WaitStateNotInList;
    if (pTicket->timerState == TimerStateNotScheduled || dfTimerCancel(&pTicket->timer)) {
      /* No timer scheduled, or cancelled timer successfully; can complete request below */
      pTicket->timerState = TimerStateNotScheduled;
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("completeInterruptWaitsWork: No timer or cancelled timer, pTicket=%p\n", pTicket));
    } else {
      /* Failed to cancel timer */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("completeInterruptWaitsWork: Failed to cancel timer, pTicket=%p\n", pTicket));
      continue;
    }
    dfAssert(pTicket->cancelState == CancelStateHasCallback);
    if (dfRequestClearCallback(pTicket->pRequest)) {
      /* Successfully removed cancel callback; we can complete the request */
      pTicket->cancelState = CancelStateNoCallback;
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("completeInterruptWaitsWork: Removed cancel callback, pTicket=%p\n", pTicket));
      count++;
      completeInterruptWait(pDevCtx, pTicket, reason);
    } else {
      /* Cancel callback must be running; allow it to complete the request */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("completeInterruptWaitsWork: Failed to remove cancel callback, pTicket=%p\n", pTicket));
    }
  }

  return count;
}

static void
completeInterruptWaits(
  Admxrc2DeviceContext* pDevCtx,
  InterruptWaitList* pWaitList,
  unsigned int targetIndex,
  DfIoStatus reason)
{
  DfList completeList;
  DfSpinLockFlags f;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("completeInterruptWaits: pWaitList=%p targetIndex=%u reason=%lu(0x%lx)\n",
    pWaitList, targetIndex, (unsigned long)reason, (unsigned long)reason));

  f = dfSpinLockGet(&pWaitList->lock);
  if (dfListIsEmpty(&pWaitList->header)) {
    /* List of waiting requests is empty, so set interrupt flag and do nothing more. */
    pDevCtx->fpga[targetIndex].interrupt.bInterruptFlag = TRUE;
    dfSpinLockPut(&pWaitList->lock, f);
    return;
  }
  /* List of waiting requests is not empty, so confiscate it. */
  dfListConfiscate(&completeList, &pWaitList->header);
  dfSpinLockPut(&pWaitList->lock, f);

  /* Complete the waiting requests that we just confiscated. */
  completeInterruptWaitsWork(pDevCtx, pWaitList, &completeList, targetIndex, reason);
}

#if DF_HAS_USER_EVENT_HANDLE
/* Removes the user event list and frees all user event nodes that are associated with 'pClCtx' */
static void
cleanupUserEventList(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  UserEventList* pList)
{
  UserEventInfo* pInfo;
  UserEventInfo* pNext;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pList->lock);
  pInfo = pList->pHead;
  while (NULL != pInfo) {
    pNext = pInfo->pNext;
    if (pInfo->pClCtx == pClCtx) {
      removeNode(pList, pInfo);
      dfUserEventPut(pInfo->userEvent);
      dfFree(pInfo);
    }
    pInfo = pNext;
  }
  dfSpinLockPut(&pList->lock, f);
}
#endif

#if DF_HAS_USER_EVENT_HANDLE
/* Removes from the device and frees all user event nodes that are associated with 'pClCtx' */
void
cleanupUserEvents(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx)
{
  UserEventList* pList;
  unsigned int i;

  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    pList = &pDevCtx->fpga[i].interrupt.eventList;
    cleanupUserEventList(pDevCtx, pClCtx, pList);
#if defined(TODO)
    pList = &pDevCtx->fpga[i].alert.eventList;
    cleanupUserEventList(pDevCtx, pClCtx, pList);
#endif
  }
}
#endif

/*
** -----------------------------------------------------------------------------
** Callbacks
** -----------------------------------------------------------------------------
*/

void
fpgaInterruptNotifyCallback(
  CoreNotification* pCoreNotification,
  unsigned int notifyType,
  void* pCallbackContext)
{
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pCallbackContext;
  FpgaInterruptNotification* pNotification = (FpgaInterruptNotification*)pCoreNotification;
  unsigned int targetIndex;
  InterruptWaitList* pWaitList;
#if DF_HAS_USER_EVENT_HANDLE
  UserEventList* pEventList;
#endif

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("fpgaInterruptNotifyCallback: pNotification=%p notifyType=%lu pCallbackContext=%p\n",
    pNotification, (unsigned long)notifyType, pCallbackContext));

  targetIndex = pNotification->targetIndex;
  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;
#if DF_HAS_USER_EVENT_HANDLE
  pEventList = &pDevCtx->fpga[targetIndex].interrupt.eventList;
  notifyAllEvents(pDevCtx, pEventList, pNotification->targetIndex);
#endif
  completeInterruptWaits(pDevCtx, pWaitList, targetIndex, DfIoStatusSuccess);
}

#if defined(TODO)
void
fpgaAlertNotifyCallback(
  CoreNotification* pCoreNotification,
  unsigned int notifyType,
  void* pCallbackContext)
{
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pCallbackContext;
  FpgaAlertNotification* pNotification = (FpgaAlertNotification*)pCoreNotification;
  unsigned int targetIndex;
  NotificationWaitList* pWaitList;
#if DF_HAS_USER_EVENT_HANDLE
  UserEventList* pEventList;
#endif

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("fpgaAlertNotifyCallback: pNotification=%p notifyType=%lu pCallbackContext=%p\n",
    pNotification, (unsigned long)notifyType, pCallbackContext));

  targetIndex = pNotification->targetIndex;
#if DF_HAS_USER_EVENT_HANDLE
  pEventList = &pDevCtx->fpga[targetIndex].alert.eventList;
  notifyAllEvents(pDevCtx, pEventList, pNotification->targetIndex);
#endif
  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;
  completeNotificationWaits(pDevCtx, pWaitList, targetIndex, DfIoStatusSuccess);
}
#endif

static void
interruptWaitCancelCallback(
  DfClientRequest* pRequest,
  DfIoStatus reason)
{
  InterruptWaitTicket* pTicket = (InterruptWaitTicket*)dfRequestGetContext(pRequest);
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  InterruptWaitList* pWaitList;
  unsigned int targetIndex;
  DfSpinLockFlags f;

  targetIndex = pTicket->targetIndex;
  dfAssert(targetIndex < pDevCtx->info.numTargetFpga);
  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;

  f = dfSpinLockGet(&pWaitList->lock);
  dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitCancelCallback: pTicket=%p waitState=%u\n", pTicket, pTicket->waitState));
  pTicket->cancelState = CancelStateNoCallback;
  switch (pTicket->waitState) {
  case WaitStateInList:
    dfListRemove(&pTicket->node);
    pTicket->waitState = WaitStateNotInList;
    if (pTicket->timerState == TimerStateNotScheduled || dfTimerCancel(&pTicket->timer)) {
      /* No timer scheduled, or cancelled timer successfully; can complete request below */
      pTicket->timerState = TimerStateNotScheduled;
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitCancelCallback: No timer or cancelled timer, pTicket=%p\n", pTicket));
    } else {
      /* Failed to cancel timer */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitCancelCallback: Failed to cancel timer, pTicket=%p\n", pTicket));
      return;
    }
    break;

  case WaitStateNotInList:
    if (pTicket->timerState == TimerStateNotScheduled || dfTimerCancel(&pTicket->timer)) {
      /* No timer scheduled, or cancelled timer successfully; can complete request below */
      pTicket->timerState = TimerStateNotScheduled;
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitCancelCallback: No timer or cancelled timer, pTicket=%p\n", pTicket));
    } else {
      /* Failed to cancel timer */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitCancelCallback: Failed to cancel timer, pTicket=%p\n", pTicket));
      return;
    }
    break;

  case WaitStateInitial: /* Should never happen */
  default: /* Should never happen */
    dfSpinLockPut(&pWaitList->lock, f);
    dfAssert(FALSE);
    return;
  }

  completeInterruptWait(pDevCtx, pTicket, reason);
}

static void
interruptWaitTimerCallback(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pContextIgnored)
{
  InterruptWaitTicket* pTicket = (InterruptWaitTicket*)pTimerContext;
  Admxrc2DeviceContext* pDevCtx = pTicket->pDevCtx;
  InterruptWaitList* pWaitList;
  unsigned int targetIndex;
  DfSpinLockFlags f;

  targetIndex = pTicket->targetIndex;
  dfAssert(targetIndex < pDevCtx->info.numTargetFpga);
  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;

  f = dfSpinLockGet(&pWaitList->lock);
  dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitTimerCallback: pTicket=%p waitState=%u cancelState=%u timerState=%u\n",
    pTicket, pTicket->waitState, pTicket->cancelState, pTicket->timerState));
  pTicket->timerState = TimerStateNotScheduled;
  switch (pTicket->waitState) {
  case WaitStateInList:
    dfListRemove(&pTicket->node);
    pTicket->waitState = WaitStateNotInList;
    if (pTicket->cancelState == CancelStateNoCallback || dfRequestClearCallback(pTicket->pRequest)) {
      /* No callback, or successfully removed cancel callback; we can complete the request. */
      pTicket->cancelState = CancelStateNoCallback;
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitTimerCallback: No callback or removed cancel callback, pTicket=%p\n", pTicket));
    } else {
      /* Cancel callback must be running; allow it to complete the request. */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitTimerCallback: Failed to remove cancel callback, pTicket=%p\n", pTicket));
      return;
    }
    break;

  case WaitStateNotInList:
    if (pTicket->cancelState == CancelStateNoCallback || dfRequestClearCallback(pTicket->pRequest)) {
      /* No callback, or successfully removed cancel callback; we can complete the request. */
      pTicket->cancelState = CancelStateNoCallback;
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitTimerCallback: No callback or removed cancel callback, pTicket=%p\n", pTicket));
    } else {
      /* Cancel callback must be running; allow it to complete the request. */
      dfSpinLockPut(&pWaitList->lock, f);
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("interruptWaitTimerCallback: Failed to remove cancel callback, pTicket=%p\n", pTicket));
      return;
    }
    break;

  case WaitStateInitial: /* Should never happen */
  default: /* Should never happen */
    dfSpinLockPut(&pWaitList->lock, f);
    dfAssert(FALSE);
    return;
  }

  completeInterruptWait(pDevCtx, pTicket, DfIoStatusError(ADMXRC2_TIMEOUT));
}

/*
** -----------------------------------------------------------------------------
** IOCTL handlers
** -----------------------------------------------------------------------------
*/

DfIoStatus
ioctlCancelWaitForInterrupt(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADMXRC2_CANCELWAITFORINTERRUPT* pIoctl = (IOCTLS_ADMXRC2_CANCELWAITFORINTERRUPT*)pBuffer;
  uint32_t targetIndex;
  InterruptWaitList* pWaitList;
  DfList completeList;
  DfSpinLockFlags f;
  unsigned int count;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }

  targetIndex = pIoctl->in.targetIndex;

  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;

  f = dfSpinLockGet(&pWaitList->lock);
  dfListConfiscate(&completeList, &pWaitList->header);
  dfSpinLockPut(&pWaitList->lock, f);

  count = completeInterruptWaitsWork(pDevCtx, pWaitList, &completeList, targetIndex, DfIoStatusError(ADMXRC2_CANCELLED));

  return count ? DfIoStatusSuccess : DfIoStatusError(ADMXRC2_FAILED);
}

DfIoStatus
ioctlWaitForInterrupt(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  DfClientRequest* pRequest,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  const uint32_t legalFlagsMask = ADMXRC2_INTWAIT_TIMEOUT;
#endif
  IOCTLS_ADMXRC2_WAITFORINTERRUPT* pIoctl = (IOCTLS_ADMXRC2_WAITFORINTERRUPT*)pBuffer;
  uint32_t targetIndex, flags;
  InterruptWaitList* pWaitList;
  InterruptWaitTicket* pTicket;
  DfSpinLockFlags f;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  targetIndex = pIoctl->in.targetIndex;
  flags = pIoctl->in.flags;
  if (targetIndex >= pDevCtx->info.numTargetFpga) {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }
#if defined(ADMXRC2_CHECK_FLAGS)
  if (flags & ~legalFlagsMask) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif
  pWaitList = &pDevCtx->fpga[targetIndex].interrupt.waitList;

  pTicket = dfPoolAlloc(&pDevCtx->interruptWait.ticketPool, InterruptWaitTicket);
  if (NULL == pTicket) {
    return DfIoStatusError(ADMXRC2_NO_MEMORY);
  }
  pTicket->pDevCtx = pDevCtx;
  pTicket->pRequest = pRequest;
  pTicket->waitState = WaitStateInitial;
  pTicket->cancelState = CancelStateInitial;
  pTicket->timerState = TimerStateInitial;
  pTicket->targetIndex = targetIndex;
  if (flags & ADMXRC2_INTWAIT_TIMEOUT) {
    /* TO DO - ensure that arithmetic for conversion from microseconds to ticks cannot overflow */
    pTicket->timeoutTicks = dfTimeGet() + dfMicrosecondsToTime(pIoctl->in.timeoutUs);
    pTicket->pIoctl = pIoctl;
    dfTimerInit(&pTicket->timer, pTicket);
  } else {
    pTicket->pIoctl = NULL;
  }
  dfRequestSetContext(pRequest, pTicket);

  f = dfSpinLockGet(&pWaitList->lock);
  if (pDevCtx->fpga[targetIndex].interrupt.bInterruptFlag) {
    /* The interrupt flag is set, so clear it and return immediately. */
    pDevCtx->fpga[targetIndex].interrupt.bInterruptFlag = FALSE;
    dfSpinLockPut(&pWaitList->lock, f);
    dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlWaitForInterrupt: Completing request, pTicket=%p\n", pTicket));
    dfPoolFree(&pDevCtx->interruptWait.ticketPool, pTicket);
    if (flags & ADMXRC2_INTWAIT_TIMEOUT) {
      dfTimerUninit(&pTicket->timer);
    }
    pIoctl->out.timeoutUs = pIoctl->in.timeoutUs;
    return DfIoStatusSuccess;
  }
  if (!dfRequestSetCallback(pRequest, interruptWaitCancelCallback)) {
    dfSpinLockPut(&pWaitList->lock, f);
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("ioctlWaitForInterrupt: Completing request due to cancellation, pTicket=%p\n", pTicket));
    dfPoolFree(&pDevCtx->interruptWait.ticketPool, pTicket);
    dfTimerUninit(&pTicket->timer);
    return DfIoStatusError(ADMXRC2_CANCELLED);
  }
  pTicket->cancelState = CancelStateHasCallback;
  if (flags & ADMXRC2_INTWAIT_TIMEOUT) {
    pTicket->timerState = TimerStateScheduled;
    dfDebugPrint(DEBUGLEVEL_NORMAL, ("ioctlWaitForInterrupt: Scheduling timer, pTicket=%p\n", pTicket));
    dfTimerSchedule(&pTicket->timer, pTicket->timeoutTicks, interruptWaitTimerCallback, NULL);
  } else {
    pTicket->timerState = TimerStateNotScheduled;
  }
  dfListAddToTail(&pWaitList->header, &pTicket->node);
  pTicket->waitState = WaitStateInList;
  dfSpinLockPut(&pWaitList->lock, f);

  dfRequestPending(pRequest);
  return DfIoStatusDeferred;
}

#if DF_HAS_USER_EVENT_HANDLE
DfIoStatus
ioctlRegisterEvent(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADMXRC2_REGISTEREVENT* pIoctl = (IOCTLS_ADMXRC2_REGISTEREVENT*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  DfSpinLockFlags f;
  unsigned int notification, targetIndex;
  user_event_handle_t hEvent;
  DfUserEvent userEvent;
  UserEventInfo* pInfo;
  UserEventList* pList;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }

  notification = pIoctl->in.notification;
  hEvent = pIoctl->in.hEvent;

  /* Get the event list corresponding to 'notification' from the client context */
  if (notification >= ADMXRC2_EVENT_FPGAINTERRUPT(0) && notification <= ADMXRC2_EVENT_FPGAINTERRUPT(0xffU)) {
    targetIndex = notification - ADMXRC2_EVENT_FPGAINTERRUPT(0);
    if (targetIndex >= pDevCtx->info.numTargetFpga) {
      return DfIoStatusError(ADMXRC2_INVALID_INDEX);
    }
    pList = &pDevCtx->fpga[targetIndex].interrupt.eventList;
#if defined(TODO)
  } else if (notification >= ADMXRC2_EVENT_FPGAALERT(0) && notification <= ADMXRC2_EVENT_FPGAALERT(0xffU)) {
    targetIndex = notification - ADMXRC2_EVENT_FPGAALERT(0);
    if (targetIndex >= pDevCtx->info.numTargetFpga) {
      return DfIoStatusError(ADMXRC2_INVALID_INDEX);
    }
    pList = &pDevCtx->fpga[targetIndex].alert.eventList;
#endif
  } else {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  /* Convert the user-mode event handle to a kernel user-mode event */
  if (!dfUserEventGet(hEvent, &userEvent)) {
    return DfIoStatusError(ADMXRC2_INVALID_HANDLE);
  }

  /* Create a user-mode event information node */
  pInfo = (UserEventInfo*)dfMalloc(sizeof(UserEventInfo));
  if (NULL == pInfo) {
    dfUserEventPut(userEvent);
    return DfIoStatusError(ADMXRC2_NO_MEMORY);
  }
  pInfo->pClCtx = pClCtx;
  pInfo->hUserEvent = hEvent;
  pInfo->userEvent = userEvent;
  pInfo->refCount = 1; /* Initial reference count */
  pInfo->bRemoving = FALSE;

  /* Add node to user-mode event list for client context */
  f = dfSpinLockGet(&pList->lock);
  insertNodeAtTail(pList, pInfo);
  dfSpinLockPut(&pList->lock, f);

  return status;
}
#endif

#if DF_HAS_USER_EVENT_HANDLE && DF_NEED_THUNK
DfIoStatus
ioctlRegisterEventThunk(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_REGISTEREVENT* pIoctl32 = (IOCTLS32_ADMXRC2_REGISTEREVENT*)pBuffer;
  IOCTLS_ADMXRC2_REGISTEREVENT ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.notification = pIoctl32->in.notification;
  ioctl.in.hEvent = dfThunkHandle(pIoctl32->in.hEvent);
  return ioctlRegisterEvent(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in));
}
#endif

#if DF_HAS_USER_EVENT_HANDLE
DfIoStatus
ioctlUnregisterEvent(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADMXRC2_UNREGISTEREVENT* pIoctl = (IOCTLS_ADMXRC2_UNREGISTEREVENT*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  DfSpinLockFlags f;
  unsigned int notification, targetIndex;
  user_event_handle_t hEvent;
  UserEventList* pList;
  UserEventInfo* pInfo;
  UserEventInfo* pNext;
  boolean_t bFreeIt = FALSE;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }

  notification = pIoctl->in.notification;
  hEvent = pIoctl->in.hEvent;

  /* Get the user-mode event list corresponding to 'notification' from the client context */
  if (notification >= ADMXRC2_EVENT_FPGAINTERRUPT(0) && notification <= ADMXRC2_EVENT_FPGAINTERRUPT(0xffU)) {
    targetIndex = notification - ADMXRC2_EVENT_FPGAINTERRUPT(0);
    if (targetIndex >= pDevCtx->info.numTargetFpga) {
      return DfIoStatusError(ADMXRC2_INVALID_INDEX);
    }
    pList = &pDevCtx->fpga[targetIndex].interrupt.eventList;
#if defined(TODO)
  } else if (notification >= ADMXRC2_EVENT_FPGAALERT(0) && notification <= ADMXRC2_EVENT_FPGAALERT(0xffU)) {
    targetIndex = notification - ADMXRC2_EVENT_FPGAALERT(0);
    if (targetIndex >= pDevCtx->info.numTargetFpga) {
      return DfIoStatusError(ADMXRC2_INVALID_INDEX);
    }
    pList = &pDevCtx->fpga[targetIndex].alert.eventList;
#endif
  } else {
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);
  }

  /* Search the user-mode event list for a node matching 'hEvent', and remove it from the list if found */
  f = dfSpinLockGet(&pList->lock);
  pInfo = pList->pHead;
  while (NULL != pInfo) {
    pNext = pInfo->pNext;
    if (pInfo->pClCtx == pClCtx && pInfo->hUserEvent == hEvent && !pInfo->bRemoving) {
      /* Found the node; decrement its reference count */
      pInfo->refCount--;
      pInfo->bRemoving = TRUE;
      if (0 == pInfo->refCount) {
        /* Node's reference count has returned to 0, so remove it from the list */
        removeNode(pList, pInfo);
        bFreeIt = TRUE;
      } else {
        /* Must be about to be signalled in notifyAllEvents() */
      }
      break;
    }
    pInfo = pNext;
  }
  dfSpinLockPut(&pList->lock, f);

  /* Nothing was found in the list; hence 'hEvent' must be considered invalid */
  if (NULL == pInfo) {
    return DfIoStatusError(ADMXRC2_INVALID_HANDLE);
  }

  if (bFreeIt) {
    /* Found a match whose reference count went to 0; release the kernel user-mode event & free the list node */
    dfUserEventPut(pInfo->userEvent);
    dfFree(pInfo);
  }

  return status;
}
#endif

#if DF_HAS_USER_EVENT_HANDLE && DF_NEED_THUNK
DfIoStatus
ioctlUnregisterEventThunk(
	Admxrc2DeviceContext* pDevCtx,
  Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_UNREGISTEREVENT* pIoctl32 = (IOCTLS32_ADMXRC2_UNREGISTEREVENT*)pBuffer;
  IOCTLS_ADMXRC2_UNREGISTEREVENT ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.notification = pIoctl32->in.notification;
  ioctl.in.hEvent = dfThunkHandle(pIoctl32->in.hEvent);
  return ioctlUnregisterEvent(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in));
}
#endif
