/*
** File: prodtest.c  
** Project: ADB3 core driver
** Purpose: OS-independent IOCTL handlers to support production testing.
**
** (C) Copyright Alpha Data 2011
*/

#include <df.h>
#include "device.h"
#include "prodtest.h"
#include <adb3/iprogt.h>

DfIoStatus
ioctlGetProdTestInfo(
	Adb3CoreDeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADB3_GETPRODTESTINFO* pIoctl = (IOCTLS_ADB3_GETPRODTESTINFO*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  pIoctl->out.serialNumber = pDevCtx->info.prodTest.serialNumber;
  pIoctl->out.cpldRevision = pDevCtx->info.prodTest.cpldRevision;
  pIoctl->out.pcbRevision = pDevCtx->info.prodTest.pcbRevision;
  pIoctl->out.bridgeRevision = pDevCtx->pDevObj->hardwareId.variant.pci.revision;
  pIoctl->out.bridgeDate = pDevCtx->info.prodTest.bridgeDate;
  pIoctl->out.bridgeTime = pDevCtx->info.prodTest.bridgeTime;

  return status;
}

DfIoStatus
ioctlGetUniqueId(
	Adb3CoreDeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADB3_GETUNIQUEID* pIoctl = (IOCTLS_ADB3_GETUNIQUEID*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  pIoctl->out.model = pDevCtx->info.model;
  pIoctl->out.serialNumber = pDevCtx->info.prodTest.serialNumber;

  return status;
}

static boolean_t
serviceInterrupts(
	Adb3CoreDeviceContext* pDevCtx)
{
  return pDevCtx->methods.pIsr(&pDevCtx->interrupt.object, pDevCtx);
}

DfIoStatus
ioctlServiceIrq(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreClientContext* pClCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADB3_SERVICEIRQ* pIoctl = (IOCTLS_ADB3_SERVICEIRQ*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }
  if (!pClCtx->bPrivileged) {
    return DfIoStatusError(ADB3_ACCESS_DENIED);
  }
  if (!pDevCtx->interrupt.bPolledMode) {
    return DfIoStatusError(ADB3_WRONG_MODE);
  }

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  if (pDevCtx->interrupt.bConnected) {
    serviceInterrupts(pDevCtx);
  }

  return status;
}

static DfIoStatus
mapIprogResult(
  IprogResult result)
{
  switch (result) {
  case IprogResultOk:
    return DfIoStatusSuccess;

  case IprogResultBadFpgaIndex:
    return DfIoStatusError(ADB3_INVALID_INDEX);

  case IprogResultBadDelay:
    return DfIoStatusError(ADB3_INVALID_PARAMETER);

  case IprogResultBadAddress:
    return DfIoStatusError(ADB3_INVALID_REGION);

  case IprogResultScheduled:
  case IprogResultNotScheduled:
    return DfIoStatusError(ADB3_WRONG_MODE);

  case IprogResultUnknownError:
  default:
    return DfIoStatusError(ADB3_UNEXPECTED_ERROR);
  }
}

DfIoStatus
ioctlAbortIprog(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADB3_ABORTIPROG* pIoctl = (IOCTLS_ADB3_ABORTIPROG*)pBuffer;
  IprogResult result;
  uint32_t targetIndex;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!pClCtx->bPrivileged) {
    return DfIoStatusError(ADB3_ACCESS_DENIED);
  }

  if (NULL == pDevCtx->methods.pIprogControl) {
    return DfIoStatusError(ADB3_NOT_SUPPORTED);
  }

  targetIndex = pIoctl->in.targetIndex;
  result = pDevCtx->methods.pIprogControl(pDevCtx, targetIndex, IprogControlOpAbort, 0, 0);

  return mapIprogResult(result);
}

extern DfIoStatus
ioctlScheduleIprog(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  static const uint32_t mask = ADB3_IPROG_FROMNOW | ADB3_IPROG_ONSTOP;
  IOCTLS_ADB3_SCHEDULEIPROG* pIoctl = (IOCTLS_ADB3_SCHEDULEIPROG*)pBuffer;
  IprogResult result;
  uint64_t address;
  uint32_t targetIndex, flags, delayMs;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!pClCtx->bPrivileged) {
    return DfIoStatusError(ADB3_ACCESS_DENIED);
  }

  if (NULL == pDevCtx->methods.pIprogControl) {
    return DfIoStatusError(ADB3_NOT_SUPPORTED);
  }

  targetIndex = pIoctl->in.targetIndex;
  flags = pIoctl->in.flags;
  address = pIoctl->in.address;
  delayMs = pIoctl->in.delayMs;
  switch (flags & mask) {
  case ADB3_IPROG_FROMNOW:
    result = pDevCtx->methods.pIprogControl(pDevCtx, targetIndex, IprogControlOpScheduleFromNow, address, delayMs);
    break;

  case ADB3_IPROG_ONSTOP:
    result = pDevCtx->methods.pIprogControl(pDevCtx, targetIndex, IprogControlOpScheduleOnStop, address, delayMs);
    break;

  default:
    return DfIoStatusError(ADB3_INVALID_FLAG);
  }

  return mapIprogResult(result);
}

extern DfIoStatus
ioctlStatusIprog(
	Adb3CoreDeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADB3_STATUSIPROG* pIoctl = (IOCTLS_ADB3_STATUSIPROG*)pBuffer;
  IprogScheduleState state;
  uint64_t address;
  uint32_t targetIndex, delayMs;
  IprogResult result;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  if (NULL == pDevCtx->methods.pIprogStatus) {
    return DfIoStatusError(ADB3_NOT_SUPPORTED);
  }

  targetIndex = pIoctl->in.targetIndex;
  result = pDevCtx->methods.pIprogStatus(pDevCtx, targetIndex, &state, &address, &delayMs);
  if (result == IprogResultOk) {
    switch (state) {
    case IprogScheduleStateFromNow:
      pIoctl->out.address = address;
      pIoctl->out.flags = ADB3_IPROG_FROMNOW;
      pIoctl->out.delayMs = delayMs;
      break;

    case IprogScheduleStateOnStop:
      pIoctl->out.address = address;
      pIoctl->out.flags = ADB3_IPROG_ONSTOP;
      pIoctl->out.delayMs = delayMs;
      break;

    case IprogScheduleStateIdle:
    default:
      pIoctl->out.address = 0U;
      pIoctl->out.flags = 0U;
      pIoctl->out.delayMs = 0U;
      break;
    }
  } else {
    pIoctl->out.address = 0U;
    pIoctl->out.flags = 0U;
    pIoctl->out.delayMs = 0U;
  }

  return mapIprogResult(result);
}

#define AVR2_COMMAND_BUFFER_SIZE (576)

static Avr2DeviceContext*
getAvr2Context(
	Adb3CoreDeviceContext* pDevCtx)
{
  switch (pDevCtx->info.model) {
  case CoreModelAdmxrcku1:
    return &pDevCtx->model.admxrcku1.avr2.context;

  case CoreModelAdmpcie8v3:
    return &pDevCtx->model.admpcie8v3.avr2.context;

  case CoreModelAdmpcie8k5:
    return &pDevCtx->model.admpcie8k5.avr2.context;

  default:
    return NULL;
  }
}

extern DfIoStatus
ioctlCommandAvr2(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADB3_COMMANDAVR2* pIoctl = (IOCTLS_ADB3_COMMANDAVR2*)pBuffer;
  uint32_t /* flags, */ timeoutUs, commandLength, responseLength;
  size_t actualResponseLength;
  const void* pCommandUser;
  void* pResponseUser;
  uint8_t buffer[AVR2_COMMAND_BUFFER_SIZE];
  Avr2DeviceContext* pAvr2Ctx;
  Avr2TransStatus result;
  DfIoStatus status = DfIoStatusSuccess;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  if (!pClCtx->bPrivileged) {
    return DfIoStatusError(ADB3_ACCESS_DENIED);
  }

  /* flags = pIoctl->in.flags; */
  timeoutUs = pIoctl->in.timeoutUs;
  commandLength = pIoctl->in.commandLength;
  pCommandUser = pIoctl->in.pCommand;
  responseLength = pIoctl->in.responseLength;
  pResponseUser = pIoctl->in.pResponse;

  if (commandLength > AVR2_COMMAND_BUFFER_SIZE || responseLength > AVR2_COMMAND_BUFFER_SIZE) {
    return DfIoStatusError(ADB3_REGION_TOO_LARGE);
  }

  /* Don't leak information to user mode */
  dfZeroMemory(buffer, responseLength);

  pAvr2Ctx = getAvr2Context(pDevCtx);
  if (NULL == pAvr2Ctx) {
    return DfIoStatusError(ADB3_NOT_SUPPORTED);
  }

  DF_USER_SPACE_BEGIN(TAG1);

  dfUserBufferValidate(pCommandUser, commandLength, FALSE, TAG1);
  dfCopyUK(buffer, pCommandUser, commandLength, TAG1);

  DF_USER_SPACE_EXCEPT(TAG1);

  status = DfIoStatusError(ADB3_INVALID_BUFFER);

  DF_USER_SPACE_END(TAG1);

  if (DfIoStatusIsError(status)) {
    return status;
  }

  result = avr2TransactionSync(pAvr2Ctx, buffer, commandLength, buffer, responseLength, timeoutUs, &actualResponseLength);
  switch (result) {
    case Avr2TransStatusSuccess:
      break;

    case Avr2TransStatusQueueTimeout:
    case Avr2TransStatusRXTimeout:
    case Avr2TransStatusTXTimeout:
      return DfIoStatusError(ADB3_TIMEOUT);

    default:
      return DfIoStatusError(ADB3_UNEXPECTED_ERROR);
  }

  if (actualResponseLength >= 0xFFFFFFFFU) {
    actualResponseLength = 0xFFFFFFFFU; /* Don't allow uint32_t overflow */
  }
  pIoctl->out.actualResponseLength = (uint32_t)actualResponseLength; /* Cast safe because of check immediately above */

  DF_USER_SPACE_BEGIN(TAG2);

  dfUserBufferValidate(pResponseUser, responseLength, TRUE, TAG2);
  dfCopyKU(pResponseUser, buffer, responseLength, TAG2);

  DF_USER_SPACE_EXCEPT(TAG2);

  status = DfIoStatusError(ADB3_INVALID_BUFFER);

  DF_USER_SPACE_END(TAG2);

  if (DfIoStatusIsError(status)) {
    return status;
  }

  return DfIoStatusSuccess;
}

#if DF_NEED_THUNK
extern DfIoStatus
ioctlCommandAvr2Thunk(
	Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS32_ADB3_COMMANDAVR2* pIoctl32 = (IOCTLS32_ADB3_COMMANDAVR2*)pBuffer;
  IOCTLS_ADB3_COMMANDAVR2 ioctl;
  DfIoStatus status;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize || sizeof(pIoctl32->out) != outSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.timeoutUs = pIoctl32->in.timeoutUs;
  ioctl.in.commandLength = pIoctl32->in.commandLength;
  ioctl.in.pCommand = dfThunkPtr(pIoctl32->in.pCommand);
  ioctl.in.responseLength = pIoctl32->in.responseLength;
  ioctl.in.pResponse = dfThunkPtr(pIoctl32->in.pResponse);
  status = ioctlCommandAvr2(pDevCtx, pClCtx, &ioctl, sizeof(ioctl.in), sizeof(ioctl.out));
  if (status == DfIoStatusSuccess) {
    pIoctl32->out.actualResponseLength = ioctl.out.actualResponseLength;
  }
  return status;
}
#endif

extern DfIoStatus
ioctlGetAvr2Status(
	Adb3CoreDeviceContext* pDevCtx,
  void* pBuffer,
  unsigned int outSize)
{
  IOCTLS_ADB3_GETAVR2STATUS* pIoctl = (IOCTLS_ADB3_GETAVR2STATUS*)pBuffer;
  Avr2DeviceContext* pAvr2Ctx;
  Avr2Status avr2Status;

  if (NULL == pIoctl || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }

  pAvr2Ctx = getAvr2Context(pDevCtx);
  if (NULL == pAvr2Ctx) {
    return DfIoStatusError(ADB3_NOT_SUPPORTED);
  }

  avr2GetStatus(pAvr2Ctx, &avr2Status);

  dfZeroMemory(&pIoctl->out, sizeof(pIoctl->out));
  pIoctl->out.asUint32 = avr2Status.asUint32;

  return DfIoStatusSuccess;
}
