/*
** File: model_adpexrc4fx.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADPE-XRC-4FX, including function to bootstrap
**          device context, specific to:
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "adb1_common.h"
#include "adpexrc4fx.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics307_common.h"
#include "ics8430_61_common.h"
#include "model_boot.h"

static struct {
  unsigned int regOffset; /* LM87 register offset */
  boolean_t bSigned;      /* TRUE => register value is int8_t; otherwise uint8_t */
  int32_t factor;         /* Fixed point, where 0x00100000 => 1, 0x00080000 => 0.5, 0xFFF80000 => -0.5 etc. */
} g_lm87Scaling[] = {
  { 0x28U, FALSE, 10276,   /* 0.0098 */ },
  { 0x29U, FALSE, 10276,   /* 0.0098 */ },
  { 0x20U, FALSE, 13631    /* 0.0130 */ },
  { 0x22U, FALSE, 18036,   /* 0.0172 */ },
  { 0x23U, FALSE, 27263,   /* 0.0260 */ },
  { 0x24U, FALSE, 65536,   /* 0.0625 */ },
  { 0x25U, FALSE, 14785,   /* 0.0141 */ },
  { 0x27U, TRUE,  1048576, /* 1.0000 */ },
  { 0x26U, TRUE,  1048576, /* 1.0000 */ }
};

static CoreSensorInfo g_lm87SensorInfo[] = {
  /* LM87 sensors */
  { "1.2V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x28, FAN1/AIN1 reading */
  { "1.8V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x29, FAN2/AIN2 reading */
  { "2.5V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x20, +2.5V / ext temp. 2 reading */
  { "3.3V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x22, +Vcc reading */
  { "5V supply rail",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x23, 5V reading */
  { "PCIe 12V supply rail",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x24, +12V reading */
  { "Front panel I/O power rail",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x25, Vccp2 reading */
  { "LM87 internal temperature",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* LM87 register 0x27, Int temp reading */
  { "Target FPGA ext. temp.",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* LM87 register 0x26, Ext temp. 1 reading */
};

static unsigned int
initSensorInfo(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex)
{
  unsigned int i, n = 0;

  dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_lm87SensorInfo) <= MAX_NUM_SENSOR);
  for (i = 0; i < DF_ARRAY_LENGTH(g_lm87SensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
    pDevCtx->info.sensor[sensorIndex] = g_lm87SensorInfo[i];
    n++;
  }
  return n; 
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int lm87SensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  int16_t val16;
  int32_t factor, value;

  dfAssert(DF_ARRAY_LENGTH(g_lm87SensorInfo) == DF_ARRAY_LENGTH(g_lm87Scaling));

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

  if (lm87SensorIndex < DF_ARRAY_LENGTH(g_lm87SensorInfo)) {
    /* LM87 sensors */
    if (g_lm87Scaling[lm87SensorIndex].bSigned) {
      val16 = (int16_t)(int8_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    } else {
      val16 = (int16_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    }
    factor = g_lm87Scaling[lm87SensorIndex].factor;
    value = (val16 * factor) / 16;
    pReading->doubleValue.intPart = (int32_t)(int16_t)((value >> 16) & 0xFFFFU);
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }

  return CoreSensorInvalidIndex;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdpexrc4fxRev1* pVpd = &pDevCtx->info.vpd.pData->adpexrc4fx.rev1;
  VpdAdpexrc4fxRev1 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
  case 1:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->rev0.version = vpdVersion;
  pVpd->rev0.boardType = dfLe16ToCpu(vpdRaw.rev0.boardType);
  pVpd->rev0.modifications = dfLe32ToCpu(vpdRaw.rev0.modifications);
  pVpd->rev0.stepping = dfLe8ToCpu(vpdRaw.rev0.stepping);
  pVpd->rev0.scd = dfLe32ToCpu(vpdRaw.rev0.scd);
  pVpd->rev0.lclkRef = dfLe32ToCpu(vpdRaw.rev0.lclkRef);
  pVpd->rev0.mclkRef = dfLe32ToCpu(vpdRaw.rev0.mclkRef);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.rev0.sdram); i++) {
    pVpd->rev0.sdram[i].order = dfLe8ToCpu(vpdRaw.rev0.sdram[i].order);
  }
  pVpd->rev0.cardId = dfLe32ToCpu(vpdRaw.rev0.cardId);
  pVpd->rev0.serialNumber = dfLe32ToCpu(vpdRaw.rev0.serialNumber);
  pVpd->rev0.fpgaType = dfLe16ToCpu(vpdRaw.rev0.fpgaType);
  pVpd->rev0.pcbRev = dfLe8ToCpu(vpdRaw.rev0.pcbRev);
  pVpd->rev0.cpldRev = (uint8_t)(0x10U + pDevCtx->pDevObj->hardwareId.variant.pci.revision);
  if (vpdVersion >= 1) {
    pVpd->ssram.order = dfLe8ToCpu(vpdRaw.ssram.order);
  } else {
    pVpd->ssram.order = 18;
  }

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onIcs307ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIgnored,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, ics307MapProgramStatus(status));
}

static void
onIcs8430_61ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIgnored,
  Ics8430_61ProgramRequest* pRequest,
  Ics8430_61ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 1, ics8430_61MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return ics307MapProgramStatus(ics307Program(pDevCtx, &pDevCtx->model.adpexrc4fx.ics307.context, &pDevCtx->model.adpexrc4fx.ics307.programming, pClockWord, onIcs307ProgramDone));

  case 1:
    return ics8430_61MapProgramStatus(ics8430_61Program(pDevCtx, &pDevCtx->model.adpexrc4fx.ics8430_61.context, &pDevCtx->model.adpexrc4fx.ics8430_61.programming, pClockWord, onIcs8430_61ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdpexrc4fxRev1* pVpd = &pDevCtx->info.vpd.pData->adpexrc4fx.rev1;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return ics307MapClockWordStatus(clockWordIcs307(flags, pVpd->rev0.lclkRef, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord));

  case 1:
    return ics8430_61MapClockWordStatus(clockWordIcs8430_61(flags, pVpd->rev0.mclkRef, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl, flpage;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x2000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

	flpage = (uint32_t)location & 0x1fe0000U;
	dfPciMemWrite32(hModel, &pModelRegs->flpage, flpage);
	(void)dfPciMemRead32(hModel, &pModelRegs->flpage);

	if (bWrite) {
		/* Select mode 0, set fl_a, drive fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x7U << 24) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Set fl_d */
		dfPciMemWrite32(hModel, &pModelRegs->fldata, *pValue);
		(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

		/* Assert fl_we_n */
		flctl |= (0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_we_n */
		flctl &= ~(0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x7U << 24);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);
	} else {
		/* Select mode 0, set fl_a, Drive fl_a, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x3U << 25) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Assert fl_oe_n */
		flctl |= (0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Read fl_d */
		*pValue = (uint16_t)dfPciMemRead32(hModel, &pModelRegs->fldata);

		/* Deassert fl_oe_n */
		flctl &= ~(0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x3U << 25);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

    dfDebugPrint(9, ("flash: *pValue=0x%lx\n", (unsigned long)*pValue));
	}
}

static void
flashReset(
  Adb3CoreDeviceContext* pDevCtx)
{
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl;

  dfDebugPrint(9, ("flashReset: entered\n"));

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_rst_n, assert fl_rst_n */
	flctl = (0x1U << 27) | (0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Deassert fl_rst_n */
	flctl &= ~(0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Stop driving fl_rst_n */
	flctl &= ~(0x1U << 27);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADPEXRC4FX_FPCTL_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone fpctl=0x%08lx *pbValue=%s\n", val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADPEXRC4FX_FPCTL_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.adpexrc4fx.shadow.fpctl;
    if (*pbValue) {
      val32 = (uint32_t)(val32 | ADPEXRC4FX_FPCTL_PROG);
    } else {
      val32 = (uint32_t)(val32 & ~ADPEXRC4FX_FPCTL_PROG);
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpctl, val32);
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->fpsmap[0], val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->fpsmap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdpexrc4fxRev1* pVpd = &pDevCtx->info.vpd.pData->adpexrc4fx.rev1;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdpexrc4fx;

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.adpexrc4fx.shadow.ictl = ADPEXRC4FX_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.adpexrc4fx.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.adpexrc4fx.shadow.fpctl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl, pDevCtx->model.adpexrc4fx.shadow.fpctl);

      /* Reset Flash and identify it */
      flashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADPEXRC4FX_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADPEXRC4FX_FLASH_TARGET_AREA_START;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize ICS307 stuff */
      if (!ics307Init(
        pDevCtx,
        &pDevCtx->model.adpexrc4fx.ics307.context,
        pVpd->rev0.lclkRef,
        Ics307InterfaceTypeXpl,
        FALSE,
        &pDevCtx->hardware.model.pAdpexrc4fx->lclk,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS307 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Initialize ICS843034-01 synthesizer stuff */
      if (!ics8430_61Init(
        pDevCtx,
        &pDevCtx->model.adpexrc4fx.ics8430_61.context,
        pVpd->rev0.mclkRef,
        &pDevCtx->hardware.model.pAdpexrc4fx->mclk,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS843034-01 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[1].currentFrequency = fOut;
      }

      /* Initialize information about sensors */
      (void)initSensorInfo(pDevCtx, 0U);

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdpexrc4fx;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS8430-61 stuff */
      ics8430_61Uninit(pDevCtx, &pDevCtx->model.adpexrc4fx.ics8430_61.context);

      /* Deinitialize ICS307 stuff */
      ics307Uninit(pDevCtx, &pDevCtx->model.adpexrc4fx.ics307.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int channel, n;
  uint32_t pistat, istat;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts, and mask the value with the interrupts we want */
  pistat = dfPciMemRead32(hBridge, &pRegs->pistat) & pDevCtx->hardware.shadow.bridge.adb1.pictl;
  dfDebugPrint(8, ("isr: pistat=0x%08lx\n", (unsigned long)pistat));

  /* Clear the interrupts that we saw */
  dfPciMemWrite32(hBridge, &pRegs->pistat, pistat);

  if (pistat & ADB1_PISTAT_IRQMODEL) {
    dfDebugPrint(9, ("isr: MODEL interrupt\n"));
    istat = dfPciMemRead32(hModel, &pModelRegs->istat) & pDevCtx->model.adpexrc4fx.shadow.ictl;
    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->istat, istat);
    if (istat & ADPEXRC4FX_ISTAT_IRQFPGA) {
      dfDebugPrint(9, ("isr: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (pistat & ADB1_PISTAT_IRQDMA(channel)) {
      dfDebugPrint(9, ("isr: DMA channel %u interrupt\n", channel));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, channel);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return pistat ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap Bridge registers\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap Model registers\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("mapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, sizeof(Adb1BridgeRegisters), &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map Bridge registers\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped Bridge regs @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map Model registers\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped Model regs @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Ics8430_61ProgramStatus ics8430_61Status;
  Ics307ProgramStatus ics307Status;
  ModelRegsAdpexrc4fx* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdpexrc4fx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICS307 state */
      ics307SaveState(pDevCtx, &pDevCtx->model.adpexrc4fx.ics307.context);

      /* Save ICS8430-61 state */
      ics8430_61SaveState(pDevCtx, &pDevCtx->model.adpexrc4fx.ics8430_61.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      flashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.adpexrc4fx.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS8430-61 state */
      ics8430_61Status = ics8430_61RestoreState(pDevCtx, &pDevCtx->model.adpexrc4fx.ics8430_61.context);
      if (Ics8430_61ProgramStatusSuccess != ics8430_61Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS8430-61 state, status=%u\n", ics8430_61Status));
      }

      /* Restore ICS307 state */
      ics307Status = ics307RestoreState(pDevCtx, &pDevCtx->model.adpexrc4fx.ics307.context);
      if (Ics307ProgramStatusSuccess != ics307Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS307 state, status=%u\n", ics307Status));
      }
    }
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  CoreFlashStatus flashStatus;

  if (CHECK_BOUNDS(address, length, pDevCtx->info.flash[0].totalSize)) {
    return CoreVpdInvalidRegion;
  }
	if (bWrite) {
    flashStatus = flashWrite(pDevCtx, 0, address, length, pData);
	} else {
		flashStatus = flashRead(pDevCtx, 0, address, length, pData);
	}
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

static CoreVpdStatus
syncVpd(
	Adb3CoreDeviceContext* pDevCtx)
{
  CoreFlashStatus flashStatus;

  flashStatus = flashSync(pDevCtx, 0);
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Adb1BridgeRegisters)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x%lx bytes\n",
      (unsigned long)sizeof(Adb1BridgeRegisters)));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < sizeof(ModelRegsAdpexrc4fx)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x%lx bytes\n",
      (unsigned long)sizeof(ModelRegsAdpexrc4fx)));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("validateBusResources: OK\n"));
  
  return TRUE;
}

void
adb3CoreBootstrapAdpexrc4fx(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdpexrc4fx: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pSyncVpd = syncVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = ADPEXRC4FX_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1U;
  pDevCtx->info.bootstrap.numSensor = ADPEXRC4FX_NUM_LM87_SENSOR;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADPEXRC4FX_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADPEXRC4FX_VPD_OFFSET_IN_FLASH;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdpexrc4fxRev1);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
