/*
** File: model_admxrc2.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-II-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2009-2011
*/

#include <df.h>

#include "admxrc2.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "icd2061_common.h"
#include "model_boot.h"
#include "pci9xxx.h"
#include "pci9xxx_common.h"
#include "pci9xxx_93x6.h"

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData);

static void
onIcd2061ProgramDone(
  Icd2061DeviceContext* pV6ClkSynthCtx,
  Icd2061ProgramRequest* pRequest,
  unsigned int index,
  Icd2061ProgramStatus status,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;

  onClockProgramDone(pDevCtx, index, icd2061MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  if (clockIndex >= 2) {
    return ClockProgramInvalidIndex;
  }

  return icd2061MapProgramStatus(icd2061Program(&pDevCtx->model.admxrc2.icd2061.context, &pDevCtx->model.admxrc2.icd2061.programming[clockIndex], clockIndex, pClockWord, onIcd2061ProgramDone, pDevCtx));
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  uint32_t refFrequency;
  VpdAdmxrc2Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc2.rev0;
  
  if (pVpd->refClock == 0) {
    refFrequency = 16666667U;
  } else {
    refFrequency = 14318181U;
  }
  return icd2061MapClockWordStatus(icd2061ClockWord(clockIndex, flags, refFrequency, frequencyReq, frequencyMin, frequencyMax, pClockWord));
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t page = (uint8_t)(location >> 21);
  uint8_t* pFlash;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x10000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;
  pFlash = ((uint8_t*)pDevCtx->hardware.model.pAdmxrc2) + 0x200000U;

  /* Set Flash page register */
  dfPciMemWrite8(hModel, &pModelRegs->flpage, (uint8_t)(page & 0x7fU));
  (void)dfPciMemRead8(hModel, &pModelRegs->flpage);

  location &= 0x1fffffU;
  if (bWrite) {
    dfPciMemWrite8(hModel, pFlash + location, (uint8_t)*pValue);
    (void)dfPciMemRead8(hModel, &pModelRegs->flpage);
  } else {
    *pValue = (uint32_t)dfPciMemRead8(hModel, pFlash + location);
  }
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t val8;

  dfDebugPrint(9, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val8 = dfPciMemRead8(hModel, &pModelRegs->fcon);
    *pbValue = (val8 & ADMXRC2_FCON_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val8 = dfPciMemRead8(hModel, &pModelRegs->fcon);
    *pbValue = (val8 & ADMXRC2_FCON_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val8 = pDevCtx->model.admxrc2.shadow.fcon;
    if (*pbValue) {
      val8 = (uint8_t)(val8 | ADMXRC2_FCON_PROG);
    } else {
      val8 = (uint8_t)(val8 & ~ADMXRC2_FCON_PROG);
    }
    dfDebugPrint(9, ("fpgaControlCommonPlxBased: opCode=SetProg val8=0x%lx *pbValue=%s\n", (unsigned long)val8, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite8(hModel, &pModelRegs->fcon, val8);
    (void)dfPciMemRead8(hModel, &pModelRegs->fcon);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->smap[0], val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->smap[0], val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->smap[0], val8);
    }
    
    (void)dfPciMemRead8(hModel, &pModelRegs->fcon); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->smap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  CoreVpdStatus status;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  VpdAdmxrc2Rev0* pVpd, vpdRaw;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxrc2;

      /* Initialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, TRUE);

      /* Ensure that all interrupts are disabled in PCI9656 */
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaster = PLX_INTCSR_PCI_IE | PLX_INTCSR_LOC_IE | PLX_INTCSR_DMA0_IE | PLX_INTCSR_DMA1_IE;
      pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMaskMaster = PLX_INTCSR_LOC_INT | PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT;
      dfPciMemWrite32(hBridge, &pRegs->INTCSR, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->INTCSR);
      dfPciMemWrite16(hBridge, &pRegs->DMACSR, 0x0808U);
      (void)dfPciMemRead16(hBridge, &pRegs->DMACSR);

      /* Enable target FPGA interrupt in CPLD */
      pDevCtx->model.admxrc2.shadow.ienab = 0x1U;
      dfPciMemWrite8(hModel, &pModelRegs->imset, 0xffU);
      dfPciMemWrite8(hModel, &pModelRegs->imclr, pDevCtx->model.admxrc2.shadow.ienab);
      dfPciMemWrite8(hModel, &pModelRegs->icon, 0xffU);
      (void)dfPciMemRead8(hModel, &pModelRegs->imset);

      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      status = readWriteVpd(pDevCtx, FALSE, ADMXRC2_EEPROM_VPD_OFFSET, sizeof(vpdRaw), &vpdRaw);
      if (CoreVpdSuccess == status) {
        pDevCtx->info.vpd.bValid = TRUE;
        pVpd = &pDevCtx->info.vpd.pData->admxrc2.rev0;
        pVpd->model = dfLe16ToCpu(vpdRaw.model);
        pVpd->modifications = dfLe16ToCpu(vpdRaw.modifications);
        pVpd->refClock = dfLe16ToCpu(vpdRaw.refClock);
        pVpd->flashType = dfLe16ToCpu(vpdRaw.flashType);
        pVpd->zbt01 = dfLe8ToCpu(vpdRaw.zbt01);
        pVpd->zbt23 = dfLe8ToCpu(vpdRaw.zbt23);
        pVpd->zbt45 = dfLe8ToCpu(vpdRaw.zbt45);
        pVpd->cardId = dfLe16ToCpu(vpdRaw.cardId);
        pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
        pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
        pVpd->cpldRev = dfLe8ToCpu(vpdRaw.cpldRev);
        pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to read VPD from EEPROM, status=0x%lx\n", (unsigned long)status));
      }

      /* Initialize ICD2061 stuff */
      icd2061Init(&pDevCtx->model.admxrc2.icd2061.context, &pDevCtx->hardware.model.pAdmxrc2->ccon, pDevCtx->hardware.hModel);
      /*
      ** TO DO - it would be nice to somehow get initial frequencies, i.e.
      **
      ** pDevCtx->clockGenerator[0].currentFrequency = <something>;
      ** pDevCtx->clockGenerator[1].currentFrequency = <something>;
      **
      ** At power-on, initial frequencies are 33 MHz and 25 MHz for LCLK (0) and MCLK (1)
      ** respectively. But the clock generators are not reset by PCI reset, so cannot
      ** simply assume these frequencies at driver startup :-(
      **
      ** For the time being, initial frequencies reported by driver are 0.
      */

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->LAS0BA);
      ls0r = dfPciMemRead32(hBridge, &pRegs->LAS0RR);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->LAS1BA);
      ls1r = dfPciMemRead32(hBridge, &pRegs->LAS1RR);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdmxrc2;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      icd2061Uninit(&pDevCtx->model.admxrc2.icd2061.context);
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Pci9656Registers* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intcsr, clearMask;
  uint8_t istat;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pPci9656;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts */
  intcsr = dfPciMemRead32(hBridge, &pRegs->INTCSR);
  
  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  intcsr &= pDevCtx->hardware.shadow.bridge.pci9xxx.intcsrMask;
  /* Clear the interrupts we saw */
  dfPciMemWrite32(hBridge, &pRegs->INTCSR, intcsr | pDevCtx->hardware.shadow.bridge.pci9xxx.intcsr);

  if (intcsr & PLX_INTCSR_LOC_INT) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    istat = (uint8_t)(dfPciMemRead8(hModel, &pModelRegs->icon) & pDevCtx->model.admxrc2.shadow.ienab);
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite8(hModel, &pModelRegs->icon, istat);

    /* Take action for the model-specific interrupts that we saw. */
    if (istat & ADMXRC2_ICON_FPGA) {
      dfDebugPrint(9, ("isr: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  if (intcsr & (PLX_INTCSR_DMA0_INT | PLX_INTCSR_DMA1_INT)) {
    clearMask = 0x0U;
    if (intcsr & PLX_INTCSR_DMA0_INT) {
      dfDebugPrint(9, ("isr: DMA0 interrupt\n"));
      clearMask |= 0x0008U;
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
      dfDebugPrint(9, ("isr: DMA1 interrupt\n"));
      clearMask |= 0x0800U;
    }
    dfPciMemWrite16(hBridge, &pRegs->DMACSR, (uint16_t)(clearMask | pDevCtx->hardware.shadow.bridge.pci9xxx.dmacsr));
    if (intcsr & PLX_INTCSR_DMA0_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[0].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 0);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[0].dpc, (void*)(uintptr_t)0);
    }
    if (intcsr & PLX_INTCSR_DMA1_INT) {
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[1].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, 1);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[1].dpc, (void*)(uintptr_t)1);
    }
  }

  return intcsr ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap PCI9656\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap CPLD\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** mapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("mapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, sizeof(Pci9656Registers), &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map PCI9656\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped PCI9656 @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map CPLD\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped CPLD @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  ModelRegsAdmxrc2* pModelRegs;
  DfMemoryHandle hModel;
  CoreClockStatus clockStatus;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdmxrc2;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICD2061 clock generator state */
      icd2061SaveState(&pDevCtx->model.admxrc2.icd2061.context);
    } else { /* Powering up (from D3) */
      /* Reinitialize misc. registers in PCI9656 */
      pci9xxxInitRegs(pDevCtx, TRUE);

      /* Reenable target FPGA interrupt in CPLD */
      pDevCtx->model.admxrc2.shadow.ienab = 0x1U;
      dfPciMemWrite8(hModel, &pModelRegs->imset, 0xffU);
      dfPciMemWrite8(hModel, &pModelRegs->imclr, pDevCtx->model.admxrc2.shadow.ienab);
      dfPciMemWrite8(hModel, &pModelRegs->icon, 0xffU);
      (void)dfPciMemRead8(hModel, &pModelRegs->imset);

      /* Restore ICD2061 clock generator state */
      clockStatus = (CoreClockStatus)icd2061RestoreState(&pDevCtx->model.admxrc2.icd2061.context);
      if (CoreClockSuccess != clockStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICD2061 state, status=%u\n", clockStatus));
      }
    }
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    return write93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  } else {
    return read93x6Buffer(pDevCtx, TRUE, addressLength9356, instructionLength9356, (uint32_t)address, (uint32_t)length, pData);
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Pci9656Registers)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x%lx bytes\n", (unsigned long)sizeof(Pci9656Registers)));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < 0x400000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("validateBusResources: OK\n"));
  
  return TRUE;
}

void
adb3CoreBootstrapAdmxrc2(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc2: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = pci9xxxDmaList;
  pDevCtx->methods.pDmaTransfer = pci9xxxDmaTransfer;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pEnableInterrupts = pci9xxxEnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = pci9xxxWindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = PCI9656_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = eepromSize9356;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc2Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 1;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 16;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
