/*
** File: model_admpcie8k5.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADM-PCIE-8K5, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2016
**
** SI5338 output to clock index mapping:
**
** Output#   Signal        Clock index
** ==================================
**    0      GTH_CLK_0        0
**    1      GTH_CLK_1        1
**    2      MEM_CLK_{0,1}    2
**    3      GTH_CLK_2        3
**
** All four clocks are user-programmable.
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admpcie8k5.h"
#include "admxrc6tx_common.h"
#include "coreclock.h"
#include "coreif.h"
#include "coredma.h"
#include "device.h"
#include "ddr3_spd_common.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "icap_common.h"
#include "avr2_common.h"
#include "avr2_clock_common.h"
#include "avr2_flash.h"
#include "avr_sensor.h"

/* TO DO - error values */
static CoreSensorInfo g_avrVoltageInfo[] = {
  { "12V rail (edge conn.)",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_0 */
  { "3.3V rail (edge conn.)",     0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_1 */
  { "3.3V Aux rail (edge conn.)", 0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_2 */
  { "2.5V power",                 0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_3 */
  { "1.8V MGT power",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_4 */
  { "1.2V DDR4 SDRAM power",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_5 */
  { "1.2V MGT AVTT",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_6 */
  { "1.0V MGT AVCC",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_7 */
  { "0.95V FPGA core power",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_8 */
  { "0.6V DDR4 SDRAM VTT",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* AVR ADC_9 */
};

static CoreSensorInfo g_avrCurrentInfo[] = {
  { "12V current (edge conn.)",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }, /* AVR ADC_10 */
  { "3.3V current (edge conn.)",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }, /* AVR ADC_11 */
  { "1.8V MGT current",           0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }, /* AVR ADC_12 */
  { "2.5V current",               0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }  /* AVR ADC_13 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrTempInfo[] = {
  { "uC internal temperature",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Microcontroller internal temp */
  { "FPGA temperature",            0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* Temperature sensor near middle of board */
};

/* TO DO - error values */
static CoreSensorInfo g_avrMiscInfo[] = {
  { "Total powered on time",       0, { 0, 16384 }, CoreSensorDataDouble, CoreSensorUnitS,    0 }, /* Number of seconds spent powered on */
  { "Power cycle count",           0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* Number of elapsed events */
};

/* These correspond to 'g_avrVoltageInfo' above, to convert the raw ADC values into Volt */
static AvrVoltageScaling g_avrVoltageScaling[] = {
  {  8U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_0 */
  { 10U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_1 */
  { 12U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_2 */
  { 14U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_3 */
  { 16U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_4 */
  { 18U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_5 */
  { 20U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_6 */
  { 22U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_7 */
  { 24U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_8 */
  { 26U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_9 */
};

static AvrCurrentScaling g_avrCurrentScaling[] = {
  { 28U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_10 */
  { 30U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_11 */
  { 32U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_12 */
  { 34U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }  /* AVR ADC_13 */
};

/* These correspond to 'g_avrTempInfo' above, to convert the raw ADC values into deg. C */
static AvrTempScaling g_avrTempScaling[] = {
  { 36U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 38U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ }
};

static AvrMiscScaling g_avrMiscScaling = {
  { 0U },
  { 4U, FALSE }
};

static void
initSensorInfo(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0;

  dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfo, sizeof(g_avrVoltageInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrCurrentInfo, sizeof(g_avrCurrentInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrCurrentInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrTempInfo, sizeof(g_avrTempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrMiscInfo, sizeof(g_avrMiscInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
}

static boolean_t
createFakeVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmpcie8k5* pVpd = &pDevCtx->info.vpd.pData->admpcie8k5;
  unsigned int bank;

  DF_ZERO_OBJECT(pVpd);
  pVpd->rev0.version = 0;
  pVpd->rev0.length = sizeof(VpdAdmpcie8k5Rev0);
  pVpd->rev0.partNumber = 1319U;
  pVpd->rev0.serial = 99U;
  pVpd->rev0.pcbRev = 0x0U;
  pVpd->rev0.si5338Ref = 25000000U;
  pVpd->rev0.fabricClkFreq = 200000000U;
  pVpd->rev0.emcclkFreq = 100000000U;
  for (bank = 0; bank < DF_ARRAY_LENGTH(pVpd->rev0.sdram); bank++) {
    pVpd->rev0.sdram[bank].order = 30; /* 8 GiB per bank */
    pVpd->rev0.sdram[bank].speed = 12000;
    pVpd->rev0.sdram[bank].ranks = 0;
  }
  pVpd->rev0.fpga.device = 326U; /* KU115 */
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->rev0.fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->rev0.fpga.speedGrade), "1");
  dfStrCpy((char*)pVpd->rev0.fpga.stepping, DF_ARRAY_LENGTH(pVpd->rev0.fpga.stepping), "");
  pVpd->rev0.fpga.tempGrade = ADMPCIE8K5_VPD_TEMPGRADE_COMMERCIAL;
  pVpd->rev0.checksum = 0U;

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  CoreVpdStatus status;
  VpdAdmpcie8k5* pVpd = &pDevCtx->info.vpd.pData->admpcie8k5;
  VpdAdmpcie8k5Rev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  unsigned int bank;

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdVersion), &vpdVersion);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD version field\n"));
    return FALSE;
  }
  vpdVersion = dfLe16ToCpu(vpdVersion);
  status = admxrc6txReadAvrVpdData(pDevCtx, 2, sizeof(vpdLength), &vpdLength);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD length field\n"));
    return FALSE;
  }
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  admxrc6txVerifyAvrVpdChecksum(pDevCtx, vpdLength);

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdRaw), &vpdRaw);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to block read VPD\n"));
    return FALSE;
  }
  DF_ZERO_OBJECT(pVpd);
  pVpd->rev0.version = vpdVersion;
  pVpd->rev0.length = vpdLength;
  pVpd->rev0.partNumber = dfLe32ToCpu(vpdRaw.partNumber);
  pVpd->rev0.serial = dfLe32ToCpu(vpdRaw.serial);
  pVpd->rev0.pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->rev0.si5338Ref = dfLe32ToCpu(vpdRaw.si5338Ref);
  pVpd->rev0.fabricClkFreq = dfLe32ToCpu(vpdRaw.fabricClkFreq);
  pVpd->rev0.emcclkFreq = dfLe32ToCpu(vpdRaw.emcclkFreq);
  for (bank = 0; bank < DF_ARRAY_LENGTH(pVpd->rev0.sdram); bank++) {
    pVpd->rev0.sdram[bank].order = dfLe8ToCpu(vpdRaw.sdram[bank].order);
    pVpd->rev0.sdram[bank].speed = dfLe16ToCpu(vpdRaw.sdram[bank].speed);
    pVpd->rev0.sdram[bank].ranks = dfLe8ToCpu(vpdRaw.sdram[bank].ranks);
  }
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->rev0.fpga.stepping, DF_ARRAY_LENGTH(pVpd->rev0.fpga.stepping), (const char*)vpdRaw.fpga.stepping);
  dfStrCpy((char*)pVpd->rev0.fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->rev0.fpga.speedGrade), (const char*)vpdRaw.fpga.speedGrade);
  pVpd->rev0.fpga.device = dfLe16ToCpu(vpdRaw.fpga.device);
  pVpd->rev0.fpga.tempGrade = dfLe8ToCpu(vpdRaw.fpga.tempGrade);
  pVpd->rev0.checksum = dfLe8ToCpu(vpdRaw.checksum);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE; /* Success */
}

static void
enableAvr2RxNotEmptyInt(
  Avr2DeviceContext* pAvr2Ctx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvr2Ctx, Adb3CoreDeviceContext, model.admpcie8k5.avr2.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admpcie8k5.shadow.intCtl |= ADMPCIE8K5_V_INTCTL_AVR2RXNE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcie8k5.shadow.intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

static void
enableAvr2TxEmptyInt(
  Avr2DeviceContext* pAvr2Ctx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvr2Ctx, Adb3CoreDeviceContext, model.admpcie8k5.avr2.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admpcie8k5.shadow.intCtl |= ADMPCIE8K5_V_INTCTL_AVR2TXE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcie8k5.shadow.intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.admpcie8k5.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcie8k5.shadow.intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static void
onAvr2ClockProgramDone(
  Avr2ClockContext* pAvrClockCtx,
  Avr2ClockRequest* pRequest,
  Avr2ClockStatus status,
  unsigned int clockIndex,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;

  onClockProgramDone(pDevCtx, clockIndex, avr2ClockMapStatus(status));
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockProgramInvalidIndex;
  }

  return avr2ClockMapStatus(avr2ClockSetFrequency(&pDevCtx->model.admpcie8k5.avr2Clock.context, &pDevCtx->model.admpcie8k5.avr2Clock.request[clockIndex], clockIndex, pClockWord, onAvr2ClockProgramDone, (void*)pDevCtx));
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  /* Only frequency field of clock word is used */
  pClockWord->frequency = frequencyReq;
  return ClockWordSuccess;
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  /* ADM-PCIE-8K5 cannot be fully reconfigured */
  return FpgaControlNotSupported;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }
  return FpgaSelectMapNotSupported;
}

static void
dmaEngineInit(
  Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  /* Initialize DMA enables */
  dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

  /* Initialize DMA engine registers */
  adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t fOutAvr2Clock[ADMPCIE8K5_NUM_AVR2_CLOCK];
  unsigned int i;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmpcie8k5;

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.admpcie8k5.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcie8k5.shadow.intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Initialize DMA engine stuff */
      dmaEngineInit(pDevCtx);

      /* Initialize ICAP interface */
      pDevCtx->model.admpcie8k5.icap.bPresent = adb3HasIcapInterface(pDevCtx);
      if (pDevCtx->model.admpcie8k5.icap.bPresent) {
        icapInit(&pDevCtx->model.admpcie8k5.icap.context, hModel, &pModelRegs->icap, ADMPCIE8K5_IPROG_DELAYCNT_FREQ_HZ);
      } else {
        /* Ensure that IPROG methods are not called */
        pDevCtx->methods.pIprogControl = NULL;
        pDevCtx->methods.pIprogStatus = NULL;
      }

      /* Read in VPD */
      if (!readVpd(pDevCtx)) {
        createFakeVpd(pDevCtx);
      }

      /* Identify Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0x0U;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize information needed for production test */
      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admpcie8k5.rev0.serial;
      pDevCtx->info.prodTest.cpldRevision = 0U;
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admpcie8k5.rev0.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      initSensorInfo(pDevCtx);
      break;

    case 1:
      /* Initialize the AVR2 uC interface */
      if (avr2Init(
        &pDevCtx->model.admpcie8k5.avr2.context,
        &pDevCtx->interrupt.object,
        &pDevCtx->hardware.model.pAdmxrc6tx->model2.avr.curr.ctlStat,
        pDevCtx->hardware.hModel,
        ADMPCIE8K5_AVR_FLASH_SIZE,
        ADMPCIE8K5_AVR_FLASH_PAGE_SIZE_LOG2,
        enableAvr2RxNotEmptyInt,
        enableAvr2TxEmptyInt))
      {
        pDevCtx->model.admpcie8k5.bAvr2Initialized = TRUE;

        /* Initialize the AVR2 clock programming context */
        if (avr2ClockInit(
          &pDevCtx->model.admpcie8k5.avr2Clock.context,
          &pDevCtx->model.admpcie8k5.avr2.context,
          DF_ARRAY_LENGTH(fOutAvr2Clock),
          fOutAvr2Clock))
        {
          pDevCtx->model.admpcie8k5.bAvr2ClockInitialized = TRUE;
          dfDebugPrint(2, ("initHardware: AVR2 clock initial frequencies are { %lu, %lu, %lu, %lu } Hz\n",
            (unsigned long)fOutAvr2Clock[0], (unsigned long)fOutAvr2Clock[1], (unsigned long)fOutAvr2Clock[2], (unsigned long)fOutAvr2Clock[3]));
          pDevCtx->clockGenerator[0].currentFrequency = fOutAvr2Clock[0];
          pDevCtx->clockGenerator[1].currentFrequency = fOutAvr2Clock[1];
          pDevCtx->clockGenerator[2].currentFrequency = fOutAvr2Clock[2];
          pDevCtx->clockGenerator[3].currentFrequency = fOutAvr2Clock[3];
        }
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      if (pDevCtx->model.admpcie8k5.icap.bPresent) {
        /* Uninitialize ICAP interface */
        icapUninit(&pDevCtx->model.admpcie8k5.icap.context, pDevCtx->hardware.bCanTouch);
      }

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Deinitialize the AVR2 clock programming context */
      if (pDevCtx->model.admpcie8k5.bAvr2ClockInitialized) {
        avr2ClockUninit(&pDevCtx->model.admpcie8k5.avr2Clock.context);
        pDevCtx->model.admpcie8k5.bAvr2ClockInitialized = FALSE;
      }

      /* Deinitialize the AVR2 interface */
      if (pDevCtx->model.admpcie8k5.bAvr2Initialized) {
        avr2Uninit(&pDevCtx->model.admpcie8k5.avr2.context);
        pDevCtx->model.admpcie8k5.bAvr2Initialized = FALSE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static IprogResult
iprogCtl(
	Adb3CoreDeviceContext* pDevCtx,
  uint32_t targetIndex,
  IprogControlOp op,
  uint64_t address,
  uint32_t delayMs)
{
  if (targetIndex != 0U) {
    return IprogResultBadFpgaIndex; 
  }

  switch (op) {
  case IprogControlOpScheduleFromNow:
    if (address > 0xFFFFFFFFU) {
      return IprogResultBadAddress; 
    }
    return iprogScheduleFromNow(&pDevCtx->model.admpcie8k5.icap.context, (uint32_t)address, delayMs);
  
  case IprogControlOpScheduleOnStop:
    if (address > 0xFFFFFFFFU) {
      return IprogResultBadAddress; 
    }
    return iprogScheduleOnStop(&pDevCtx->model.admpcie8k5.icap.context, (uint32_t)address, delayMs);

  case IprogControlOpAbort:
    return iprogAbort(&pDevCtx->model.admpcie8k5.icap.context);

  default:
    return IprogResultUnknownError;
  }
}

static IprogResult
iprogStat(
	Adb3CoreDeviceContext* pDevCtx,
  uint32_t targetIndex,
  IprogScheduleState* pState,
  uint64_t* pAddress,
  uint32_t* pDelayMs)
{
  uint32_t cfgaddrValue;

  if (targetIndex != 0) {
    return IprogResultBadFpgaIndex; 
  }

  iprogStatus(&pDevCtx->model.admpcie8k5.icap.context, pState, &cfgaddrValue, pDelayMs);
  *pAddress = cfgaddrValue;

  return IprogResultOk;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admpcie8k5.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADMPCIE8K5_V_INTSTAT_LEVELBITS) {
      if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
        /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
        intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      if (intStat & ADMPCIE8K5_V_INTSTAT_AVR2RXNE) {
        /* Disable RX FIFO not empty interrupt. */
        intCtl &= ~ADMPCIE8K5_V_INTSTAT_AVR2RXNE;
      }
      if (intStat & ADMPCIE8K5_V_INTSTAT_AVR2TXE) {
        /* Disable TX FIFO empty interrupt. */
        intCtl &= ~ADMPCIE8K5_V_INTSTAT_AVR2TXE;
      }
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.admpcie8k5.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMPCIE8K5_V_INTSTAT_AVR2RXNE) {
      dfDebugPrint(9, ("isr: AVR2 RX FIFO not empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admpcie8k5.avr2.context.transport.rx.dpc, NULL);
    }
    if (intStat & ADMPCIE8K5_V_INTSTAT_AVR2TXE) {
      dfDebugPrint(9, ("isr: AVR2 TX FIFO empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admpcie8k5.avr2.context.transport.tx.dpc, NULL);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  Avr2ClockStatus avr2ClkStatus;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      if (pDevCtx->model.admpcie8k5.icap.bPresent) {
        /* Save ICAP interface state */
        icapSaveState(&pDevCtx->model.admpcie8k5.icap.context);
      }

      /* Save AVR clock programming state */
      avr2ClockSaveState(&pDevCtx->model.admpcie8k5.avr2Clock.context);
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.admpcie8k5.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcie8k5.shadow.intCtl | ADMPCIE8K5_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA engine stuff */
      dmaEngineInit(pDevCtx);

      /* Reset Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);

      /* Restore AVR clock programming state */
      avr2ClkStatus = avr2ClockRestoreState(&pDevCtx->model.admpcie8k5.avr2Clock.context);
      if (Avr2ClockStatusSuccess != avr2ClkStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore AVR2 clock programming state, status=%u\n", avr2ClkStatus));
      }

      if (pDevCtx->model.admpcie8k5.icap.bPresent) {
        /* Restore ICAP interface state */
        icapRestoreState(&pDevCtx->model.admpcie8k5.icap.context);
      }
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = avrSensorVoltageRead(pDevCtx, index, g_avrVoltageScaling, DF_ARRAY_LENGTH(g_avrVoltageScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageScaling);
  }

  status = avrSensorCurrentRead(pDevCtx, index, g_avrCurrentScaling, DF_ARRAY_LENGTH(g_avrCurrentScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrCurrentScaling);
  }

  status = avrSensorTempRead(pDevCtx, index, g_avrTempScaling, DF_ARRAY_LENGTH(g_avrTempScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  }

  status = avrSensorMiscRead(pDevCtx, index, &g_avrMiscScaling, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  }

  return status;
}

static CoreVpdStatus
mapAvr2FlashStatus(
  Avr2FlashStatus status)
{
  switch (status) {
  case Avr2FlashSuccess:
    return CoreVpdSuccess;

  case Avr2FlashTimeout:
    return CoreVpdTimeout;

  case Avr2FlashNotServiceMode:
    return CoreVpdNotPresent;

  case Avr2FlashInvalidRegion:
    return CoreVpdInvalidRegion;

  case Avr2FlashGeneralFailure:
    return CoreVpdGeneralFailure;

  default:
    if (status >= Avr2FlashHardwareFailure && status < Avr2FlashHardwareFailure + 0x100U) {
      return CoreVpdGeneralFailure;
    }
    return CoreVpdGeneralFailure;
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  Avr2FlashStatus status;

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    /* Casts are safe because of above bounds check */
    status = avr2FlashWrite(pDevCtx, &pDevCtx->model.admpcie8k5.avr2.context, (uint32_t)address, (uint32_t)length, pData);
  } else {
    /* Casts are safe because of above bounds check */
    status = avr2FlashRead(pDevCtx, &pDevCtx->model.admpcie8k5.avr2.context, (uint32_t)address, (uint32_t)length, pData);
  }
  return mapAvr2FlashStatus(status);
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

static void
bootstrap(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfAssert(DF_OFFSET_OF(VpdAdmpcie8k5Rev0, checksum) == sizeof(VpdAdmpcie8k5Rev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = admxrc6txFlashSingleBank;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIprogControl = iprogCtl; /* Assume that we have ICAP interface; will nullify later if necessary */
  pDevCtx->methods.pIprogStatus = iprogStat; /* Assume that we have ICAP interface; will nullify later if necessary */
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;

  pDevCtx->info.bootstrap.numClockGen = ADMPCIE8K5_NUM_AVR2_CLOCK;
  pDevCtx->info.bootstrap.numDmaChannel = ADMPCIE8K5_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 0U;
  pDevCtx->info.bootstrap.numSensor = 0U;
  pDevCtx->info.bootstrap.numSensor +=
    (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfo) +
    (unsigned int)DF_ARRAY_LENGTH(g_avrCurrentInfo) +
    (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo) +
    (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = ADMPCIE8K5_AVR_FLASH_SIZE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmpcie8k5);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}

void
adb3CoreBootstrapAdmpcie8k5(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmpcie8k5: entered\n"));
  bootstrap(pDevCtx);
}
