/*
** File: identify.c  
** Project: ADB3 core driver
** Purpose: Identify device given Hardware ID, and set the bootstrap method of
**          the device context appropriately for that Hardware ID.
**
** (C) Copyright Alpha Data 2009-2014
*/

#include <df.h>

#include "device.h"
#include "device_defs.h"
#include "identify.h"
#include "model_boot.h"

#if !defined(ADB3_CUSTOM_ONLY)

# if defined(ADMXRC_DRIVER)
static boolean_t
identifyPci9056(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyPci9056: entered\n"));

	switch (pId->variant.pci.subDevice) {
	case PCI_SUBDEVICE_PCI9056_ADMPCIE6S1:
		dfDebugPrint(0, ("identifyPci9056: identified ADM-PCIE-6S1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcie6s1;
    *pbMatch = TRUE;
		return TRUE;

  default:
		dfDebugPrint(0, ("*** identifyPci9056: unknown PCI9056 device\n"));
		return FALSE;
	}
}
# endif

# if defined(ADMXRC_DRIVER)
static boolean_t
identifyPci9080(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyPci9080: entered\n"));

	switch (pId->variant.pci.subDevice) {
	case PCI_SUBDEVICE_PCI9080_ADMXRC:
		dfDebugPrint(0, ("identifyPci9080: identified ADM-XRC / ADM-XRC-P\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc;
    *pbMatch = TRUE;
		return TRUE;

	case PCI_SUBDEVICE_PCI9080_ADMXRC2L:
		dfDebugPrint(0, ("identifyPci9080: identified ADM-XRC-II-Lite\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc2l;
    *pbMatch = TRUE;
		return TRUE;

  default:
		dfDebugPrint(0, ("*** identifyPci9080: unknown PCI9656 device\n"));
		return FALSE;
	}
}
# endif

# if defined(ADMXRC_DRIVER)
static boolean_t
identifyPci9656(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyPci9656: entered\n"));

	switch (pId->variant.pci.subDevice) {
	case PCI_SUBDEVICE_PCI9656_ADMXRC2:
		dfDebugPrint(0, ("identifyPci9656: identified ADM-XRC-II\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc2;
    *pbMatch = TRUE;
		return TRUE;

	case PCI_SUBDEVICE_PCI9656_ADPDRC2:
		dfDebugPrint(0, ("identifyPci9656: identified ADP-DRC-II\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpdrc2;
    *pbMatch = TRUE;
		return TRUE;

	case PCI_SUBDEVICE_PCI9656_ADPWRC2:
		dfDebugPrint(0, ("identifyPci9656: identified ADP-WRC-II\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpwrc2;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_PCI9656_ADMXRC4LX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-4LX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc4;
    *pbMatch = TRUE;
    return TRUE;

  case PCI_SUBDEVICE_PCI9656_ADMXRC4SX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-4SX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc4;
    *pbMatch = TRUE;
    return TRUE;

  default:
		dfDebugPrint(0, ("*** identifyPci9656: unknown PCI9656 device\n"));
		return FALSE;
	}
}
# endif

# if defined(ADMXRC_DRIVER)
static boolean_t
identifyPlxTech(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyPlxTech: entered\n"));

	switch (pId->variant.pci.device) {
	case PCI_DEVICE_PLXTECH_PCI9056:
		dfDebugPrint(1, ("identifyPlxTech: identified PCI9056\n"));
    return identifyPci9056(pDevCtx, pId, pbMatch);

	case PCI_DEVICE_PLXTECH_PCI9080:
		dfDebugPrint(1, ("identifyPlxTech: identified PCI9080\n"));
    return identifyPci9080(pDevCtx, pId, pbMatch);

	case PCI_DEVICE_PLXTECH_PCI9656:
		dfDebugPrint(1, ("identifyPlxTech: identified PCI9656\n"));
    return identifyPci9656(pDevCtx, pId, pbMatch);

  default:
		break;
	}

	dfDebugPrint(0, ("***  identifyPlxTech: failed to recognize device\n"));

	return FALSE;
}
# endif

#if defined(ADB3_DRIVER)
static boolean_t
identifyAdb3(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyAdb3: entered\n"));

	switch (pId->variant.pci.subDevice) {
	case PCI_SUBDEVICE_ADB3_ADPEXRC5T:
		dfDebugPrint(0, ("identifyAdb3: identified ADPe-XRC-5T\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc5t;
    *pbMatch = TRUE;
		return TRUE;

	case PCI_SUBDEVICE_ADB3_ADMXRC6TL_LX50T:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6TL (5VLX50T bridge)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tl;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TL_FX50T:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6TL (5VFX70T bridge)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tl;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6T1:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6T1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6t1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6T1_BERT:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6T1 (BERT configuration)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6t1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TGE:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6TGE\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tge;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TGEL:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6TGEL\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tge;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TADV8:
  case PCI_SUBDEVICE_ADB3_ADMXRC6TADV8_REV2:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6T-ADV8\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tadv8;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6T:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6t;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6T_BERT:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T (BERT configuration)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6t;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6TL:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T-L\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6t;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6TL_BERT:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T-L (BERT configuration)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6t;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6TADV_C:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T-ADV (Controller)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6tadv;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADPEXRC6TADV_T:
		dfDebugPrint(0, ("identifyAdb3: identified ADPE-XRC-6T-ADV (Target)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc6tadv;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TDA1:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6T-DA1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tda1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC6TDA1_BERT:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-6T-DA1 (BERT configuration)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc6tda1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC7K1_REV1:
  case PCI_SUBDEVICE_ADB3_ADMXRC7K1_REV2:
		dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-7K1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc7k1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC7V1:
  case PCI_SUBDEVICE_ADB3_ADMXRC7V1_PASS2:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-7V1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc7v1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMVPX37V2:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-VPX3-7V2\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmvpx37v2;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC7Z1:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-7Z1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc7z1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRC7Z2:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-7Z2\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc7z2;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRCKU1:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-KU1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrcku1;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRCKU1_P5:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-KU1-P5\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrcku1P5;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMXRCKU1_P6:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-XRC-KU1-P6\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrcku1P6;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMPCIE7V3:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-PCIE-7V3\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcie7v3;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMPCIEKU3:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-PCIE-KU3\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcieku3;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMPCIEKU3_2ND:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-PCIE-KU3 2nd PCIe Endpoint\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcieku3;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMPCIE8V3:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-PCIE-8V3\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcie8v3;
    *pbMatch = TRUE;
		return TRUE;

  case PCI_SUBDEVICE_ADB3_ADMPCIE8K5:
    dfDebugPrint(0, ("identifyAdb3: identified ADM-PCIE-8K5\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmpcie8k5;
    *pbMatch = TRUE;
		return TRUE;

  default:
		dfDebugPrint(0, ("identifyAdb3: identified generic ADB3 device\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapGenericAdb3;
    *pbMatch = TRUE;
		return TRUE;
	}
}
#endif

static boolean_t
identifyAlphaData(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyAlphaData: entered\n"));

	switch (pId->variant.pci.device) {
#if defined(ADMXRC_DRIVER)
	case PCI_DEVICE_ADATA_ADMXPL:
    switch (pId->variant.pci.subDevice) {
    case PCI_SUBDEVICE_ADATA_ADMXPL_PCI:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADM-XPL (PCI)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxpl;
      *pbMatch = TRUE;
      return TRUE;

    case PCI_SUBDEVICE_ADATA_ADMXPL_PCIX:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADM-XPL (PCI-X)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxpl;
      *pbMatch = TRUE;
      return TRUE;

    default:
      break;
    }
    break;

	case PCI_DEVICE_ADATA_ADMXP:
    switch (pId->variant.pci.subDevice) {
    case PCI_SUBDEVICE_ADATA_ADMXP_PCI:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADM-XP (PCI)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxp;
      *pbMatch = TRUE;
      return TRUE;

    case PCI_SUBDEVICE_ADATA_ADMXP_PCIX:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADM-XP (PCI-X)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxp;
      *pbMatch = TRUE;
      return TRUE;

    default:
      break;
    }
    break;

	case PCI_DEVICE_ADATA_ADPXPI:
    switch (pId->variant.pci.subDevice) {
    case PCI_SUBDEVICE_ADATA_ADPXPI_PCI:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADP-XPI (PCI)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpxpi;
      *pbMatch = TRUE;
      return TRUE;

    case PCI_SUBDEVICE_ADATA_ADPXPI_PCIX:
		  dfDebugPrint(0, ("identifyAlphaData: identified ADP-XPI (PCI-X)\n"));
		  pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpxpi;
      *pbMatch = TRUE;
      return TRUE;

    default:
      break;
    }
    break;

	case PCI_DEVICE_ADATA_ADMXRC4FX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-4FX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc4fx;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADPEXRC4FX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADPE-XRC-4FX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdpexrc4fx;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5LX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5LX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5lx;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T1:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t1;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T1_VPDBRAM:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T1 (VPD in BRAM)\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t1;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T2:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T2\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t2;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T2ADV:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T2-ADV\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t2;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T2ADV6:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T2-ADV6\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t2;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5T2ADVCC1:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T2-ADV-CC1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5t2;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5TDA1:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5T-DA1\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5tda1;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMXRC5LXA:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5LXA\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5lxa;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADCPXRC4LX:
		dfDebugPrint(0, ("identifyAlphaData: identified ADCP-XRC-4LX\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdcpxrc4lx;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADMAMC5A2:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-AMC-5A2\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmamc5a2;
    *pbMatch = TRUE;
    return TRUE;

	case PCI_DEVICE_ADATA_ADCBBP:
		dfDebugPrint(0, ("identifyAlphaData: identified ADC-BBP\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdcbbp;
    *pbMatch = TRUE;
    return TRUE;

  case PCI_DEVICE_ADATA_ADMXRC5TZ:
		dfDebugPrint(0, ("identifyAlphaData: identified ADM-XRC-5TZ\n"));
		pDevCtx->methods.pModelBootstrap = adb3CoreBootstrapAdmxrc5tz;
    *pbMatch = TRUE;
    return TRUE;
#endif

#if defined(ADB3_DRIVER)
	case PCI_DEVICE_ADATA_ADB3:
		dfDebugPrint(1, ("identifyAlphaData: identified ADB3\n"));
    return identifyAdb3(pDevCtx, pId, pbMatch);
#endif

	default:
		break;
	}

	dfDebugPrint(0, ("***  identifyAlphaData: failed to recognize device\n"));

	return FALSE;
}
#endif

void
identifyDevice(
	Adb3CoreDeviceContext* pDevCtx,
	DfHardwareId* pId,
  boolean_t* pbMatch)
{
	dfDebugPrint(1, ("identifyDevice: entered\n"));

	if (pId->busType == DfBusPci) {
    dfDebugPrint(0, ("identifyDevice: PCI, vid=0x%04x did=0x%04x svid=0x%04x sdid=0x%04x rev=0x%02x\n",
      (unsigned int)pId->variant.pci.vendor, (unsigned int)pId->variant.pci.device,
      (unsigned int)pId->variant.pci.subVendor, (unsigned int)pId->variant.pci.subDevice,
      (unsigned int)pId->variant.pci.revision));
  }

#if defined(ADB3_CUSTOM)
  if (identifyDeviceCustom(pDevCtx, pId, pbMatch)) {
    return;
  }
#endif

#if !defined(ADB3_CUSTOM_ONLY)
	if (pId->busType == DfBusPci) {
    switch (pId->variant.pci.vendor) {
    case PCI_VENDOR_ADATA:
    	dfDebugPrint(1, ("identifyDevice: vendor is ALPHA DATA\n"));
			identifyAlphaData(pDevCtx, pId, pbMatch);
      break;

# if defined(ADMXRC_DRIVER)
    case PCI_VENDOR_PLXTECH:
    	dfDebugPrint(1, ("identifyDevice: vendor is PLX Tech.\n"));
			identifyPlxTech(pDevCtx, pId, pbMatch);
      break;
# endif

    default:
    	/* Vendor not recognized */
      break;
    }
	}
#endif
}
