/*
** File: ics843034_01.c  
** Project: ADB3 core driver
** Purpose: Functions for programming the ICS843034-01 clock generator.
**
** (C) Copyright Alpha Data 2011
*/

#include <df.h>

#include "ics843034_01_common.h"

/* N divide values */
static const uint32_t g_nVal[8] = {
  1, 2, 3, 4, 5, 6, 8, 16
};

static const uint32_t g_pDiv = 1; /* P divide is 1 because P pin is floating */

static const uint32_t g_fVcoMin = 490000000U; /* Minimum allowed VCO frequency in Hz, from datasheet */
static const uint32_t g_fVcoMax = 640000000U; /* Maximum allowed VCO frequency in Hz, from datasheet */

/*
** Bits 0-16 of the returned Clock Word are the serial clock programming word
** for the ICS843034_01, but bit 17 is the reference clock select, which is
** driven to the SEL pins of the ICS843034-01 by ADB1-XRC4FX.
*/
Ics843034_01ClockWordStatus
clockWordIcs843034_01(
  uint32_t flags,
	uint32_t xtalFrequency1,
	uint32_t xtalFrequency2,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
	uint32_t T = 0x0; /* Make TEST output low */
	uint32_t xtal, bestXtal = 0; /* init. to avoid compiler whine */
	uint32_t mMin, mMax;
	uint32_t m, bestM = 0, n, nIdx, bestNIdx  = 0; /* init. to avoid compiler whine */
	uint32_t minFrequency, maxFrequency;
	uint32_t refFrequency;
	uint32_t f, bestFrequency = 0; /* init. to avoid compiler whine */
	uint32_t error, bestError = (uint32_t)-1;

	dfDebugPrint(3, ("clockWordIcs843034_01: xtalFrequency1=%lu xtalFrequency2=%lu reqFrequency=" DF_FMT_U64 "\n",
    (unsigned long)xtalFrequency1, (unsigned long)xtalFrequency2, (unsigned long long)frequencyReq));

  if (xtalFrequency2 == 0 || xtalFrequency1 == 0) {
		return Ics843034_01ClockWordStatusGeneralFailure;
  }

	for (xtal = 0; xtal < 2; xtal++) {
		refFrequency = xtal ? xtalFrequency2 : xtalFrequency1;
		mMin = (g_fVcoMin * g_pDiv + refFrequency - 1) / refFrequency;
		mMax = g_fVcoMax * g_pDiv / refFrequency;
		minFrequency = refFrequency * mMin / (16 * g_pDiv) - 1000;
		maxFrequency = refFrequency * mMax / g_pDiv + 1000;
		if (frequencyReq < minFrequency || frequencyReq > maxFrequency) {
			continue;
		}
		for (m = mMin; m <= mMax; m++) {
			for (nIdx = 0; nIdx < 8; nIdx++) {
				n = g_nVal[nIdx];
				f = (uint32_t)refFrequency * m / (n * g_pDiv);
        if ((flags & CoreClockMinimumFrequency) && f < frequencyMin) {
          continue;
        }
        if ((flags & CoreClockMaximumFrequency) && f > frequencyMax) {
          continue;
        }
				if (f > (uint32_t)frequencyReq) {
					error = f - (uint32_t)frequencyReq;
				} else {
					error = (uint32_t)frequencyReq - f;
				}
				if (error < bestError) {
					bestError = error;
					bestFrequency = f;
					bestM = m;
					bestNIdx = nIdx;
					bestXtal = xtal;
				}
			}
		}
	}

	if (bestM == 0) {
		/* Failed to find a suitable value */
		return Ics843034_01ClockWordStatusOutOfRange;
	}

	pClockWord->frequency = bestFrequency;
	pClockWord->d0 = (bestXtal << 17) | (T << 15) | (bestNIdx << 12) | (bestNIdx << 9) | (bestM << 0);

	dfDebugPrint(3, ("clockWordIcs843034_01: bestFrequency=%lu clockWord=0x%08lx\n", (unsigned long)bestFrequency, (unsigned long)pClockWord->d0));

	return Ics843034_01ClockWordStatusSuccess;
}

boolean_t
ics843034_01Init(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx,
  unsigned int refClk1,
  unsigned int refClk2,
  uint32_t* pReg,
  DfMemoryHandle hReg,
  uint32_t* pFOut)
{
  uint32_t val32;
  uint32_t fRef, fVco, fOut;
  uint32_t sel, m, mMin, mMax, nIdx, n;

  dfTimerInit(&pIcs843034_01Ctx->timer, pDevCtx);
  pIcs843034_01Ctx->pReg = pReg;
  pIcs843034_01Ctx->hReg = hReg;
  pIcs843034_01Ctx->save.bLastValid = FALSE;

  /* Figure out current output frequency of ICS843034-01 */
	val32 = dfPciMemRead32(pIcs843034_01Ctx->hReg, pIcs843034_01Ctx->pReg);
  sel = val32 & (0x1U << 30);
  m = val32 & 0x1FFU;
  fRef = sel ? refClk2 : refClk1;
  if (0 == fRef) {
  	dfDebugPrint(0, ("*** ics843034_01Init: reference clock frequency is invalid, sel=%lu refClk1=%lu refClk2=%lu\n",
      (unsigned long)sel, (unsigned long)refClk1, (unsigned long)refClk2));
    return FALSE;
  }
  dfDebugPrint(1, ("ics843034_01Init: sel=u m=%u fRef=%lu\n", (unsigned int)sel, (unsigned int)m, (unsigned long)fRef));
  nIdx = (val32 >> 9) & 0x7U;
  n = g_nVal[nIdx];
	mMin = (g_fVcoMin * g_pDiv + fRef - 1) / fRef;
	mMax = g_fVcoMax * g_pDiv / fRef;
  if (m < mMin || m > mMax) {
    dfDebugPrint(0, ("*** ics843034_01Init: m=%u, outside allowed range [%u, %u]\n",
      (unsigned int)m, (unsigned int)mMin, (unsigned int)mMax));
  }
  fVco = fRef * m / g_pDiv;
  fOut = fVco / n;
  dfDebugPrint(1, ("ics843034_01Init: fVco=%lu fOut=%lu\n", (unsigned long)fVco, (unsigned long)fOut));
  *pFOut = fOut;
  return TRUE;
}

void
ics843034_01Uninit(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx)
{
  dfTimerUninit(&pIcs843034_01Ctx->timer);
}

static void
onCheckIcs843034_01(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  Ics843034_01ProgramRequest* pRequest = (Ics843034_01ProgramRequest*)pCallbackContext;
  Ics843034_01DeviceContext* pIcs843034_01Ctx = pRequest->pIcs843034_01Ctx;
	uint32_t val32;
  Ics843034_01ProgramStatus status = Ics843034_01ProgramStatusSuccess;

	val32 = dfPciMemRead32(pIcs843034_01Ctx->hReg, pIcs843034_01Ctx->pReg);
  dfDebugPrint(5, ("onCheckIcs843034_01: mclkCtl=0x%lx\n", (unsigned long)val32));	
	if (!(val32 & 0x80000000U)) {
		status = Ics843034_01ProgramStatusHardwareError;
	}
  if (Ics843034_01ProgramStatusSuccess == status) {
    pIcs843034_01Ctx->save.bLastValid = TRUE;
    pIcs843034_01Ctx->save.last = pRequest->clockWord;
  }
  pRequest->pCallback(pDevCtx, pIcs843034_01Ctx, pRequest, status);
}

Ics843034_01ProgramStatus
ics843034_01Program(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx,
  Ics843034_01ProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  Ics843034_01ProgramCallback* pCallback)
{
	uint32_t* pReg;
  DfMemoryHandle hReg;
  uint32_t clockWord;

  pRequest->pIcs843034_01Ctx = pIcs843034_01Ctx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  clockWord = pClockWord->d0;
  pReg = pIcs843034_01Ctx->pReg;
  hReg = pIcs843034_01Ctx->hReg;
  dfPciMemWrite32(hReg, pReg, (0x1U << 31) | ((clockWord & 0x20000U) << 13) | (clockWord & 0x1ffffU));
  dfPciMemRead32(hReg, pReg);
  dfTimerSchedule(&pIcs843034_01Ctx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckIcs843034_01, pRequest);
  return Ics843034_01ProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pV6ClkSynthCtx,
  Ics843034_01ProgramRequest* pRequest,
  Ics843034_01ProgramStatus status)
{
  Ics843034_01ProgramRequestSync* pRequestSync = (Ics843034_01ProgramRequestSync*)pRequest;

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

Ics843034_01ProgramStatus
ics843034_01ProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx,
  const CoreClockWord* pClockWord)
{
  Ics843034_01ProgramRequestSync req;
  Ics843034_01ProgramStatus status;

  dfEventInit(&req.ev);
  status = ics843034_01Program(pDevCtx, pIcs843034_01Ctx, &req.request, pClockWord, onProgramDone);
  if (Ics843034_01ProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
ics843034_01SaveState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx)
{
  /* Nothing to do yet */
}

Ics843034_01ProgramStatus
ics843034_01RestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pIcs843034_01Ctx)
{
  Ics843034_01ProgramStatus status = Ics843034_01ProgramStatusSuccess;
 
  /* If the ICS843034-01 had been programmed before power-down, reprogram it with the same thing */
  if (pIcs843034_01Ctx->save.bLastValid) {
    status = ics843034_01ProgramSync(pDevCtx, pIcs843034_01Ctx, &pIcs843034_01Ctx->save.last);
  }

  return status;
}

ClockProgramStatus
ics843034_01MapProgramStatus(
  Ics843034_01ProgramStatus status)
{
  switch (status) {
  case Ics843034_01ProgramStatusSuccess:
    return ClockProgramSuccess;

  case Ics843034_01ProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case Ics843034_01ProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
ics843034_01MapClockWordStatus(
  Ics843034_01ClockWordStatus status)
{
  switch (status) {
  case Ics843034_01ClockWordStatusSuccess:
    return ClockWordSuccess;

  case Ics843034_01ClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case Ics843034_01ClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
