/*
** File: icap.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing ICAP interface that is present on certain
**          Bridgeless models. Includes IPROG reconfiguration functionality.
**
** (C) Copyright Alpha Data 2015
**
** 1. Value for cfgdelay register calculated as:
**
**     cfgdelay = ceil((t * R) / (65536 * 1000))
**
**   where
**     t is delayMs (in milliseconds)
**     R is delayCntFreqHz (delay counter clock frequency)
**
**   and the inverse is
**
**     t = floor(cfgdelay * (65536 * 1000) / R)
**
*/

#include "device.h"
#include "icap.h"
#include "icap_common.h"

static void
abortIPROGOnHardware(
  IcapContext* pIcapContext)
{
  DfMemoryHandle hIcapRegs = pIcapContext->hIcapRegs;
  IcapRegs* pIcapRegs = pIcapContext->pIcapRegs;

  dfPciMemWrite32(hIcapRegs, &pIcapRegs->ctl, pIcapContext->shadow.ctl | ICAP_REG_CTL_V_IPROG_ABORT);
  dfPciMemRead32(hIcapRegs, &pIcapRegs->ctl);
}

static void
scheduleIPROGOnHardware(
  IcapContext* pIcapContext)
{
  DfMemoryHandle hIcapRegs = pIcapContext->hIcapRegs;
  IcapRegs* pIcapRegs = pIcapContext->pIcapRegs;

  /* Set ICAP WBSTAR register value & cfgdelay register value */
  dfPciMemWrite32(hIcapRegs, &pIcapRegs->cfgaddr, pIcapContext->schedule.cfgaddrValue);
  dfPciMemWrite32(hIcapRegs, &pIcapRegs->cfgdelay, pIcapContext->schedule.cfgdelayValue);
  dfPciMemRead32(hIcapRegs, &pIcapRegs->cfgdelay);

  /* Initiate countdown */
  dfPciMemWrite32(hIcapRegs, &pIcapRegs->ctl, pIcapContext->shadow.ctl | ICAP_REG_CTL_V_IPROG_INIT);
  dfPciMemRead32(hIcapRegs, &pIcapRegs->ctl);
}

static void
readIPROGStatus(
  IcapContext* pIcapContext,
  uint32_t* pDelayCnt,
  uint32_t* pStatus)
{
  DfMemoryHandle hIcapRegs = pIcapContext->hIcapRegs;
  IcapRegs* pIcapRegs = pIcapContext->pIcapRegs;

  *pDelayCnt = dfPciMemRead32(hIcapRegs, &pIcapRegs->delaycnt);
  *pStatus = dfPciMemRead32(hIcapRegs, &pIcapRegs->ctl);
}

void
icapInit(
  IcapContext* pIcapContext,
  DfMemoryHandle hIcapRegs,
  IcapRegs* pIcapRegs,
  uint32_t delayCntFreqHz)
{
  dfSpinLockInit(&pIcapContext->lock);
  pIcapContext->pIcapRegs = pIcapRegs;
  pIcapContext->hIcapRegs = hIcapRegs;
  pIcapContext->state = IprogScheduleStateIdle;
  pIcapContext->delayCntFreqHz = delayCntFreqHz;
  pIcapContext->shadow.ctl = dfPciMemRead32(pIcapContext->hIcapRegs, &pIcapRegs->ctl) & ICAP_REG_CTL_V_FIFOMODE;
  /* Ensure that there is no pending reconfiguration */
  abortIPROGOnHardware(pIcapContext);
}

void
icapUninit(
  IcapContext* pIcapContext,
  boolean_t bCanTouchHardware)
{
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pIcapContext->lock);
  if (pIcapContext->state == IprogScheduleStateOnStop) {
    if (bCanTouchHardware) {
      scheduleIPROGOnHardware(pIcapContext);
    }
    pIcapContext->state = IprogScheduleStateFromNow;
  }
  dfSpinLockPut(&pIcapContext->lock, f);
}

extern void
icapSaveState(
  IcapContext* pIcapContext)
{
  /* Nothing to do */
}

extern void
icapRestoreState(
  IcapContext* pIcapContext)
{
  switch (pIcapContext->state) {
  case IprogScheduleStateOnStop:
    /* Nothing to do */
    break;

  case IprogScheduleStateIdle:
    /* Nothing to do */
    break;

  case IprogScheduleStateFromNow:
    /* Restoring scheduled IPROG when the scheduling type is "From Now" is not supported, so go back to idle */
    pIcapContext->state = IprogScheduleStateIdle;
    break;

  default:
    /* Invalid state, so go to idle. */
    dfDebugPrint(0, ("*** iprogRestoreState: in invalid state, going to idle.\n"));
    pIcapContext->state = IprogScheduleStateIdle;
    break;
  }
}

extern IprogResult
iprogAbort(
  IcapContext* pIcapContext)
{
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pIcapContext->lock);
  switch (pIcapContext->state) {
  case IprogScheduleStateFromNow:
    abortIPROGOnHardware(pIcapContext);
    pIcapContext->state = IprogScheduleStateIdle;
    break;

  case IprogScheduleStateOnStop:
    pIcapContext->state = IprogScheduleStateIdle;
    break;

  case IprogScheduleStateIdle:
  default:
    dfSpinLockPut(&pIcapContext->lock, f);
    return IprogResultNotScheduled;
  }

  dfSpinLockPut(&pIcapContext->lock, f);
  return IprogResultOk;
}

extern IprogResult
iprogScheduleFromNow(
  IcapContext* pIcapContext,
  uint32_t wbstarValue,
  uint32_t delayMs)
{
  static const uint32_t divisor = 65536U * 1000U;
  static const uint32_t maxDelayMs = 1000000U * 1000U; /* Max delay is 1 million seconds */
  IprogResult result = IprogResultOk;
  uint32_t cfgdelayValue;
  DfSpinLockFlags f;

  if (delayMs > maxDelayMs) {
    return IprogResultBadDelay;
  }

  /* See comments in header of this file for explanation */
  cfgdelayValue = (uint32_t)dfDivide64By32((uint64_t)delayMs * (uint64_t)pIcapContext->delayCntFreqHz + (divisor - 1U), divisor);

  f = dfSpinLockGet(&pIcapContext->lock);
  switch (pIcapContext->state) {
  case IprogScheduleStateFromNow:
  case IprogScheduleStateOnStop:
  default:
    result = IprogResultScheduled;
    break;

  case IprogScheduleStateIdle:
    pIcapContext->schedule.delayMs = delayMs;
    pIcapContext->schedule.cfgaddrValue = wbstarValue;
    pIcapContext->schedule.cfgdelayValue = cfgdelayValue;
    scheduleIPROGOnHardware(pIcapContext);
    pIcapContext->state = IprogScheduleStateFromNow;
    break;
  }
  dfSpinLockPut(&pIcapContext->lock, f);
  return result;
}

extern IprogResult
iprogScheduleOnStop(
  IcapContext* pIcapContext,
  uint32_t wbstarValue,
  uint32_t delayMs)
{
  static const uint32_t divisor = 65536U * 1000U;
  static const uint32_t maxDelayMs = 1000000U * 1000U; /* Max delay is 1 million seconds */
  IprogResult result = IprogResultOk;
  uint32_t cfgdelayValue;
  DfSpinLockFlags f;

  if (delayMs > maxDelayMs) {
    return IprogResultBadDelay;
  }

  /* See comments in header of this file for explanation */
  cfgdelayValue = (uint32_t)dfDivide64By32((uint64_t)delayMs * (uint64_t)pIcapContext->delayCntFreqHz + (divisor - 1U), divisor);

  f = dfSpinLockGet(&pIcapContext->lock);
  switch (pIcapContext->state) {
  case IprogScheduleStateFromNow:
  case IprogScheduleStateOnStop:
  default:
    result = IprogResultScheduled;
    break;

  case IprogScheduleStateIdle:
    pIcapContext->schedule.delayMs = delayMs;
    pIcapContext->schedule.cfgaddrValue = wbstarValue;
    pIcapContext->schedule.cfgdelayValue = cfgdelayValue;
    pIcapContext->state = IprogScheduleStateOnStop;
    break;
  }
  dfSpinLockPut(&pIcapContext->lock, f);

  return result;
}

extern void
iprogStatus(
  IcapContext* pIcapContext,
  IprogScheduleState* pState,
  uint32_t* pWbstarValue,
  uint32_t* pDelayMs)
{
  static const uint32_t factor = 65536U * 1000U;
  DfSpinLockFlags f;
  uint32_t status, delaycnt;
  IprogScheduleState state;
  uint32_t wbstarValue, elapsedMs, origDelayMs, delayMs;

  f = dfSpinLockGet(&pIcapContext->lock);
  switch (pIcapContext->state) {
  case IprogScheduleStateFromNow:
    readIPROGStatus(pIcapContext, &delaycnt, &status);
    wbstarValue = pIcapContext->schedule.cfgaddrValue;
    origDelayMs = pIcapContext->schedule.delayMs;
    dfSpinLockPut(&pIcapContext->lock, f);
    state = IprogScheduleStateFromNow;
    if (status & ICAP_REG_STAT_V_RECONFIGURING) {
      delayMs = 0U;
    } else {
      /* See comments in header of this file for explanation */
      elapsedMs = (uint32_t)dfDivide64By32((uint64_t)delaycnt * factor, pIcapContext->delayCntFreqHz);
      if (elapsedMs > origDelayMs) {
        delayMs = 0U;
      } else {
        delayMs = origDelayMs - elapsedMs;
      }
    }
    break;

  case IprogScheduleStateOnStop:
    wbstarValue = pIcapContext->schedule.cfgaddrValue;
    delayMs = pIcapContext->schedule.delayMs;
    dfSpinLockPut(&pIcapContext->lock, f);
    state = IprogScheduleStateOnStop;
    break;

  case IprogScheduleStateIdle:
  default:
    dfSpinLockPut(&pIcapContext->lock, f);
    state = IprogScheduleStateIdle;
    wbstarValue = 0;
    delayMs = 0;
    break;
  }

  *pState = state;
  *pWbstarValue = wbstarValue;
  *pDelayMs = delayMs;
}
