/*
** File: ddr3_spd.h  
** Project: ADB3 core driver
** Purpose: Definitions for SPD ROM format of DDR3 SDRAM DIMMs.
**
** (C) Copyright Alpha Data 2013
*/

#ifndef _ADATA_CORE_DDR3_SPD_H
#define _ADATA_CORE_DDR3_SPD_H

#include <df.h>

/* DDR3 SDRAM DIMM SPD ROM data structure, as per JEDEC Standard No. 21-C Annex K */
#pragma pack(1)
typedef struct _Ddr3SdramSpd {
  /* Field                   Offset    Usage [note] */
  uint8_t size;           /* 0         Number of Serial PD Bytes Written / SPD Device Size / CRC Coverage [1, 2] */
  uint8_t revision;       /* 1         SPD Revision */
  uint8_t dramType;       /* 2         Key Byte / DRAM Device Type */
  uint8_t moduleType;     /* 3         Key Byte / Module Type */
  uint8_t densityBanks;   /* 4         SDRAM Density and Banks [3] */
  uint8_t addressing;     /* 5         SDRAM Addressing [3] */
  uint8_t nomVdd;         /* 6         Module Nominal Voltage, VDD */
  uint8_t organization;   /* 7         Module Organization [3] */
  uint8_t busWidth;       /* 8         Module Memory Bus Width */
  uint8_t ftb;            /* 9         Fine Timebase (FTB) Dividend / Divisor */
  uint8_t mtbDividend;    /* 10        Medium Timebase (MTB) Dividend */
  uint8_t mtbDivisor;     /* 11        Medium Timebase (MTB) Divisor */
  uint8_t tCKmin;         /* 12        SDRAM Minimum Cycle Time (tCKmin) [3] */
  uint8_t _reserved1;     /* 13        (Reserved) */
  uint8_t casLatLSB;      /* 14        CAS Latencies Supported, Least Significant Byte [3] */
  uint8_t casLatMSB;      /* 15        CAS Latencies Supported, Most Significant Byte [3] */
  uint8_t tAAmin;         /* 16        Minimum CAS Latency Time (tAAmin) */
  uint8_t tWRmin;         /* 17        Minimum Write Recovery Time (tWRmin) [3] */
  uint8_t tRCDmin;        /* 18        Minimum RAS# to CAS# Delay Time (tRCDmin) [3] */
  uint8_t tRRDmin;        /* 19        Minimum Row Active to Row Active Delay Time (tRRDmin) 3 */
  uint8_t tRPmin;         /* 20        Minimum Row Precharge Delay Time (tRPmin) [3] */
  uint8_t tRAStRCUpper;   /* 21        Upper Nibbles for tRAS and tRC [3] */
  uint8_t tRASminLSB;     /* 22        Minimum Active to Precharge Delay Time (tRASmin), Least Significant Byte [3] */
  uint8_t tRCminLSB;      /* 23        Minimum Active to Active/Refresh Delay Time (tRCmin), Least Significant Byte [3] */
  uint8_t tRFCminLSB;     /* 24        Minimum Refresh Recovery Delay Time (tRFCmin), Least Significant Byte [3] */
  uint8_t tRFCminMSB;     /* 25        Minimum Refresh Recovery Delay Time (tRFCmin), Most Significant Byte [3] */
  uint8_t tWTRmin;        /* 26        Minimum Internal Write to Read Command Delay Time (tWTRmin) [3] */
  uint8_t tRTPmin;        /* 27        Minimum Internal Read to Precharge Command Delay Time (tRTPmin) [3] */
  uint8_t tFAWUpper;      /* 28        Upper Nibble for tFAW [3] */
  uint8_t tFAWmin;        /* 29        Minimum Four Activate Window Delay Time (tFAWmin) [3] */
  uint8_t optFeat;        /* 30        SDRAM Optional Features [3] */
  uint8_t thermRef;       /* 31        SDRAM Thermal and Refresh Options [3] */
  uint8_t thermSens;      /* 32        Module Thermal Sensor */
  uint8_t deviceType;     /* 33        SDRAM Device Type */
  int8_t  tCKminOffset;   /* 34        Fine Offset for SDRAM Minimum Cycle Time (tCKmin) */
  int8_t  tAAminOffset;   /* 35        Fine Offset for Minimum CAS Latency Time (tAAmin) */
  int8_t  tRCDminOffset;  /* 36        Fine Offset for Minimum RAS# to CAS# Delay Time (tRCDmin) */
  int8_t  tRPminOffset;   /* 37        Fine Offset for Minimum Row Precharge Delay Time (tRPmin) */
  int8_t  tRCminOffset;   /* 38        Fine Offset for Minimum Active to Active/Refresh Delay Time (tRCmin) */
  uint8_t _reserved2[21]; /* 39 ~ 59   Reserved, General Section */
  uint8_t _typeSpec[57];  /* 60 ~ 116  Module Type Specific Section, Indexed by Key Byte [3] */
  uint8_t  modJedecId[2]; /* 117 ~ 118 Module ID: Module Manufacturers JEDEC ID Code */
  uint8_t  manufLoc;      /* 119       Module ID: Module Manufacturing Location */
  uint8_t  manufDate[2];  /* 120 ~ 121 Module ID: Module Manufacturing Date */
  uint8_t  modSerial[4];  /* 122 ~ 125 Module ID: Module Serial Number */
  uint8_t  crc[2];        /* 126 ~ 127 Cyclical Redundancy Code */
  uint8_t  partNum[18];   /* 128 ~ 145 Module Part Number [4] */
  uint8_t  modRev[2];     /* 146 ~ 147 Module Revision Code [4] */
  uint8_t  manufCode[2];  /* 148 ~ 149 DRAM Manufacturers JEDEC ID Code [4] */
  uint8_t  manufData[16]; /* 150 ~ 175 Manufacturers Specific Data [4] */
  uint8_t  customer[80];  /* 176 ~ 255 Open for customer use */
  /* Note 1. Number of SPD bytes written will typically be programmed as 128 or 176 bytes. */
  /*      2. Size of SPD device will typically be programmed as 256 bytes. */
  /*      3. From DDR3 SDRAM datasheet. */
  /*      4. These are optional, in accordance with the JEDEC spec. */
} Ddr3SdramSpd;
#pragma pack()

/* Size (minimum?) of a ROM that contains DDR3 SPD information */
#define DDR3_SPD_ROM_SIZE (0x100U)

/* DDR3 SPD I2C Device Type Identifier Codes (DTICs) */
#define DDR3_DTIC_SPD_ROM     (0xAU) /* SPD PROM */
#define DDR3_DTIC_TEMP_SENSOR (0x3U) /* Temperature sensor */
#define DDR3_DTIC_SWP         (0x6U) /* Software write-protect */

/* DDR3 SPD I2C PROM slot number */
#define DDR3_I2C_SLOT_SPD_ROM(x) ((DDR3_DTIC_SPD_ROM << 3) | ((x) & 0x7U))

#endif
