/*
** File: coredma.c  
** Project: ADB3 core driver
** Purpose: Implements functions for DMA transfers in Core Interface.
**
** (C) Copyright Alpha Data 2009-2011, 2015
*/

#include <df.h>

#include "resource.h"
#include "coredma.h"
#include "coredmabus.h"

#define DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_CANCEL (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_NORMAL (6) /* Level of debug messages relating to normal operation */

#if defined(BUILD_DEBUGTS)
#include <admxrc2/debugtst.h>
#endif

typedef enum _MainState {
  MainIdle = 0,
  MainInitial = 1,
  MainRunning = 2,
  MainCompleted = 3,
  MainCancelled = 4
} MainState;

typedef enum _SetupState {
  SetupIdle = 0,
  SetupRunning = 1
} SetupState;

typedef enum _TransferState {
  TransferIdle = 0,
  TransferRunning = 1
} TransferState;

typedef enum _TeardownState {
  TeardownIdle = 0,
  TeardownRunning = 1
} TeardownState;

/* Forward declarations */

static void
dmaMapperCallback(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex,
  DfDmaMapperNode* pTable,
  unsigned int tableLength,
  DfDmaMapperPosition* pProgress,
  void* pContext);

/* Low-level functions */

#if defined(BUILD_DEBUGTS)
static void
dmaLogEvent(
  DmaChannelContext* pDmaContext,
  uint8_t channelIndex,
  uint8_t eventType,
  uint16_t chunkIndex,
  uint32_t parameter,
  uint64_t timestamp)
{
#pragma pack(1)
  typedef union _Overlay {
    struct {
      uint8_t  facility;
      uint8_t  eventType;
      uint16_t chunkIndex;
      uint32_t parameter;
    } asFields;
    uint64_t asUint64;
  } Metadata;
#pragma pack()
  DfTimestampBuffer* pTSBuf = &pDmaContext->timestampBuffer;
  Metadata metadata;

  metadata.asFields.facility = ADMXRC2_DEBUGTS_FACILITY_DMA(channelIndex);
  metadata.asFields.eventType = eventType;
  metadata.asFields.chunkIndex = chunkIndex;
  metadata.asFields.parameter = parameter;

  dfTimestampBufferLog(pTSBuf, metadata.asUint64, timestamp);
}
#endif

/* Called while holding lock */
static void
dmaStartOnHardware(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channelIndex)
{
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  unsigned int tableIndex = pDmaContext->curr.state.transfer.tableIndex;
  DmaMappingInfo* pMapInfo = &pDmaContext->curr.table[tableIndex];
  DmaDescriptors* pDescriptors = &pDmaContext->descriptor[tableIndex];

#ifdef ADB3_DBG_IRQ_COUNT
  pDevCtx->interrupt.dma[channelIndex].expected++;
#endif
  pDevCtx->methods.pDmaList(
    pDevCtx,
    pDmaContext->curr.parameters.user.bWriteToLocal,
    pDmaContext->curr.parameters.user.bFixedLocalAddress,
    pMapInfo->pTable,
    pMapInfo->length,
    &pMapInfo->position,
    pDescriptors->buffer.pGeneric,
    pDevCtx->dma.mapLength,
    pDescriptors->busAddress);
  dfDmaBufferSync(pDescriptors->pDmaBuffer, !pDmaContext->curr.parameters.user.bWriteToLocal, 0, pDevCtx->info.bootstrap.dmaDescriptorSize * pMapInfo->length);

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    unsigned int chunkIndex = pDmaContext->curr.state.transfer.chunkIndex;
    uint64_t timestamp = dfTimestampGet();

    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMAHWSTART,
      (uint16_t)chunkIndex,
      0,
      timestamp);
  }
#endif

  pDevCtx->methods.pDmaTransfer(pDevCtx, channelIndex, TRUE, pDescriptors->buffer.pGeneric, pDescriptors->busAddress);
}

static void
dmaStopOnHardware(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channelIndex)
{
  pDevCtx->methods.pDmaTransfer(pDevCtx, channelIndex, FALSE, NULL, 0);
}

static boolean_t
dmaSetup(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channelIndex)
{
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  unsigned int tableIndex = pDmaContext->curr.state.setup.tableIndex;
  unsigned int mapIndex = (channelIndex << 1) | tableIndex;

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    uint64_t timestamp = dfTimestampGet();
    unsigned int chunkIndex = pDmaContext->curr.state.setup.chunkIndex;

    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMASETUPSTART,
      (uint16_t)chunkIndex,
      (uint32_t)pDevCtx->dma.mapLength,
      timestamp);
  }
#endif

  return dfDmaMapperSetupThread(
    pDevCtx->pDevObj,
    mapIndex,
    pDmaContext->curr.parameters.user.pBufferDescription,
    dmaMapperCallback,
    pDevCtx,
    pDmaContext->curr.parameters.user.bWriteToLocal,
    pDmaContext->curr.parameters.user.bFixedLocalAddress,
    &pDmaContext->curr.slice.user,
    1,
    &pDmaContext->curr.position);
}

static void
dmaTeardown(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channelIndex)
{
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  unsigned int tableIndex = pDmaContext->curr.state.teardown.tableIndex;
#if defined(BUILD_DEBUGTS)
  unsigned int chunkIndex = pDmaContext->curr.state.teardown.chunkIndex;
#endif

  pDmaContext->curr.state.teardown.tableIndex = tableIndex ^ 0x1U;

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    uint64_t timestamp = dfTimestampGet();

    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMATEARDOWNSTART,
      (uint16_t)chunkIndex,
      0,
      timestamp);
  }
#endif

  dfDmaMapperTeardown(pDevCtx->pDevObj, (channelIndex << 1) | tableIndex);

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    uint64_t timestamp = dfTimestampGet();

    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMATEARDOWNFINISH,
      (uint16_t)chunkIndex,
      0,
      timestamp);
    pDmaContext->curr.state.teardown.chunkIndex = chunkIndex + 1;
  }
#endif
}

static void
dmaCleanup(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channelIndex)
{
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];

  while (pDmaContext->curr.state.setup.count) {
    dmaTeardown(pDevCtx, channelIndex);
    pDmaContext->curr.state.setup.count--;
  }
  pDmaContext->curr.state.setup.count = 0;
  pDmaContext->curr.state.setup.tableIndex = 0;
  pDmaContext->curr.state.transfer.count = 0;
  pDmaContext->curr.state.transfer.tableIndex = 0;
  pDmaContext->curr.state.teardown.count = 0;
  pDmaContext->curr.state.teardown.tableIndex = 0;
}

static boolean_t
bCancelling(
  DmaChannelContext* pDmaContext)
{
  return pDmaContext->curr.state.bCancel;
}

static boolean_t
bCanCancel(
  DmaChannelContext* pDmaContext)
{
  return (pDmaContext->curr.state.setup.state == SetupIdle && pDmaContext->curr.state.transfer.state == TransferIdle && pDmaContext->curr.state.teardown.state == TeardownIdle) ? TRUE : FALSE;
}

static boolean_t
bCanMap(
  DmaChannelContext* pDmaContext)
{
  return (pDmaContext->curr.state.setup.state == SetupIdle && pDmaContext->curr.state.setup.count < 2 && pDmaContext->curr.position.sliceIndex != 1) ? TRUE : FALSE;
}

static boolean_t
bCanTransfer(
  DmaChannelContext* pDmaContext)
{
  return (pDmaContext->curr.state.transfer.state == TransferIdle && pDmaContext->curr.state.transfer.count > 0) ? TRUE : FALSE;
}

static boolean_t
bCanUnmap(
  DmaChannelContext* pDmaContext)
{
  return (pDmaContext->curr.state.teardown.state == TeardownIdle && pDmaContext->curr.state.teardown.count > 0) ? TRUE : FALSE;
}

static boolean_t
bTransferFinished(
  DmaChannelContext* pDmaContext)
{
  unsigned int tableIndex = pDmaContext->curr.state.transfer.tableIndex;
  DmaMappingInfo* pMapInfo = &pDmaContext->curr.table[tableIndex];

  return (pMapInfo->position.index >= pMapInfo->length) ? TRUE : FALSE;
}

static boolean_t
bOperationFinished(
  DmaChannelContext* pDmaContext)
{
  return (pDmaContext->curr.position.sliceIndex == 1 /* TO DO - support more than one slice */ && pDmaContext->curr.state.setup.count == 0 && pDmaContext->curr.state.teardown.count == 0) ? TRUE : FALSE;
}

/* Callbacks */

static void
dmaMapperCallback(
  DfDeviceObject* pDevObj,
  unsigned int mapIndex,
  DfDmaMapperNode* pTable,
  unsigned int tableLength,
  DfDmaMapperPosition* pProgress,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
  unsigned int channelIndex = mapIndex >> 1;
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  unsigned int tableIndex = pDmaContext->curr.state.setup.tableIndex;
  DmaMappingInfo* pMapInfo = &pDmaContext->curr.table[tableIndex];
  CoreDmaStatus status;
  DfSpinLockFlags f;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: pDevCtx=%p mapIndex=%lu pTable=%p tableLength=%lu\n",
    (void*)pDevCtx, (unsigned long)mapIndex, (void*)pTable, (unsigned long)tableLength));

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    uint64_t timestamp = dfTimestampGet();
    unsigned int chunkIndex = pDmaContext->curr.state.setup.chunkIndex;
    unsigned int i;
    size_t bytesMapped = 0;

    for (i = 0; i < tableLength; i++) {
      bytesMapped += pTable[i].length;
    }
    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMASETUPFINISH,
      (uint16_t)chunkIndex,
      (uint32_t)bytesMapped,
      timestamp);
    pDmaContext->curr.state.setup.chunkIndex = chunkIndex + 1;
  }
#endif

  pMapInfo->pTable = pTable;
  pMapInfo->length = tableLength;
  pMapInfo->position.index = 0U;
  pMapInfo->position.offset = 0U;
  pDmaContext->curr.state.setup.tableIndex = tableIndex ^ 0x1;
  f = dfSpinLockGet(&pDmaContext->lock);
  if (tableLength == 0) {
    if (!pDmaContext->curr.state.bCancel) {
      pDmaContext->curr.status = CoreDmaMappingFailure;
      pDmaContext->curr.state.bCancel = TRUE;
    }
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** dmaMapperCallback: DMA mapping failed (no pages mapped), channelIndex=%lu\n", (unsigned long)channelIndex));
  }
  pDmaContext->curr.position = *pProgress;
  pDmaContext->curr.state.setup.state = SetupIdle;
  pDmaContext->curr.state.setup.count++;
  dfAssert(pDmaContext->curr.state.setup.count <= 2);
  pDmaContext->curr.state.transfer.count++;
  dfAssert(pDmaContext->curr.state.transfer.count <= 2);
  while (1) {
    if (bCancelling(pDmaContext)) {
      /* The DMA transfer is being cancelled */
      if (pDmaContext->curr.state.setup.state == SetupIdle && pDmaContext->curr.state.transfer.state == TransferIdle && pDmaContext->curr.state.teardown.state == TeardownIdle) {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaMapperCallback: cancelled, invoking callback, channelIndex=%lu\n", (unsigned long)channelIndex));
        pDmaContext->curr.state.main = MainCancelled;
        pDmaContext->curr.state.bCancel = FALSE;
        status = pDmaContext->curr.status;
        dfSpinLockPut(&pDmaContext->lock, f);
        dmaCleanup(pDevCtx, channelIndex);
        pDmaContext->curr.pCallback(pDmaContext->curr.pTicket, pDmaContext->curr.pCallbackContext, status);
        break;
      } else {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaMapperCallback: cancelled, waiting for all idle, channelIndex=%lu\n", (unsigned long)channelIndex));
        dfSpinLockPut(&pDmaContext->lock, f);
        break;
      }
    }
    if (bCanTransfer(pDmaContext)) {
      /* Start DMA transfer on hardware */
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: starting DMA transfer on hardware, channelIndex=%lu\n", (unsigned long)channelIndex));
      pDmaContext->curr.state.transfer.state = TransferRunning;
      pDmaContext->curr.type = DmaRequestTypeUser;
      dmaStartOnHardware(pDevCtx, channelIndex);
    } else {
      if (bCanMap(pDmaContext)) {
        /* Set up DMA transfer */
        dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: setting up DMA transfer, channelIndex=%lu\n", (unsigned long)channelIndex));
        pDmaContext->curr.state.setup.state = SetupRunning;
        dfSpinLockPut(&pDmaContext->lock, f);
        dmaSetup(pDevCtx, channelIndex);
        f = dfSpinLockGet(&pDmaContext->lock);
      } else {
        if (bCanUnmap(pDmaContext)) {
          /* Tear down DMA transfer */
          dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: tearing down DMA transfer, channelIndex=%lu\n", (unsigned long)channelIndex));
          pDmaContext->curr.state.teardown.state = TeardownRunning;
          dfSpinLockPut(&pDmaContext->lock, f);
          dmaTeardown(pDevCtx, channelIndex);
          f = dfSpinLockGet(&pDmaContext->lock);
          pDmaContext->curr.state.setup.count--;
          dfAssert(pDmaContext->curr.state.setup.count <= 2);
          pDmaContext->curr.state.teardown.count--;
          dfAssert(pDmaContext->curr.state.teardown.count <= 2);
          pDmaContext->curr.state.teardown.state = TeardownIdle;
          if (bOperationFinished(pDmaContext)) {
            /* The DMA transfer is finished */
            dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: DMA transfer finished, channelIndex=%lu\n", (unsigned long)channelIndex));
            pDmaContext->curr.state.main = MainCompleted;
            dfAssert(pDmaContext->curr.state.setup.state == SetupIdle);
            dfAssert(pDmaContext->curr.state.setup.count == 0);
            dfAssert(pDmaContext->curr.state.transfer.state == TransferIdle);
            dfAssert(pDmaContext->curr.state.transfer.count == 0);
            dfAssert(pDmaContext->curr.state.teardown.state == TeardownIdle);
            dfAssert(pDmaContext->curr.state.teardown.count == 0);
            dfSpinLockPut(&pDmaContext->lock, f);
            pDmaContext->curr.pCallback(pDmaContext->curr.pTicket, pDmaContext->curr.pCallbackContext, CoreDmaSuccess);
            break;
          }
        } else {
          /* Nothing more to do */
          dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaMapperCallback: nothing more to do, channelIndex=%lu\n", (unsigned long)channelIndex));
          dfSpinLockPut(&pDmaContext->lock, f);
          break;
        }
      }
    }
  }
}

/* Will be called holding the lock for a DMA channel */
static void
dmaCompletedUser(
  Adb3CoreDeviceContext* pDevCtx,
  DmaChannelContext* pDmaContext,
  unsigned int channelIndex,
  DfSpinLockFlags f)
{
  CoreDmaStatus status;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: pDevCtx=%p pDmaContext=%p channelIndex=%lu f=0x%lx\n",
    (void*)pDevCtx, (void*)pDmaContext, (unsigned long)channelIndex, (unsigned long)f));

  if (bTransferFinished(pDmaContext)) {
    pDmaContext->curr.state.transfer.tableIndex = pDmaContext->curr.state.transfer.tableIndex ^ 0x1;
    pDmaContext->curr.state.transfer.count--;
    dfAssert(pDmaContext->curr.state.transfer.count <= 2);
    pDmaContext->curr.state.teardown.count++;
    dfAssert(pDmaContext->curr.state.teardown.count <= 2);
  }
  while (1) {
    if (bCancelling(pDmaContext)) {
      /* The DMA transfer is being cancelled */
      if (bCanCancel(pDmaContext)) {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaCompletedUser: cancelled, invoking callback, channelIndex=%lu\n", (unsigned long)channelIndex));
        pDmaContext->curr.state.main = MainCancelled;
        pDmaContext->curr.state.bCancel = FALSE;
        status = pDmaContext->curr.status;
        dfSpinLockPut(&pDmaContext->lock, f);
        dmaCleanup(pDevCtx, channelIndex);
        pDmaContext->curr.pCallback(pDmaContext->curr.pTicket, pDmaContext->curr.pCallbackContext, status);
        break;
      } else {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("dmaCompletedUser: cancelled, waiting for all idle, channelIndex=%lu\n", (unsigned long)channelIndex));
        dfSpinLockPut(&pDmaContext->lock, f);
        break;
      }
    }
    if (bCanTransfer(pDmaContext)) {
      /* Start DMA transfer on hardware */
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: starting DMA transfer on hardware, channelIndex=%lu\n", (unsigned long)channelIndex));
      pDmaContext->curr.state.transfer.state = TransferRunning;
      dmaStartOnHardware(pDevCtx, channelIndex);
    } else {
      if (bCanMap(pDmaContext)) {
        /* Setup DMA transfer */
        dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: setting up DMA transfer, channelIndex=%lu\n", (unsigned long)channelIndex));
        pDmaContext->curr.state.setup.state = SetupRunning;
        dfSpinLockPut(&pDmaContext->lock, f);
        dmaSetup(pDevCtx, channelIndex);
        f = dfSpinLockGet(&pDmaContext->lock);
      } else {
        if (bCanUnmap(pDmaContext)) {
          /* Tear down DMA transfer */
          dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: tearing down DMA transfer, channelIndex=%lu\n", (unsigned long)channelIndex));
          pDmaContext->curr.state.teardown.state = TeardownRunning;
          dfSpinLockPut(&pDmaContext->lock, f);
          dmaTeardown(pDevCtx, channelIndex);
          f = dfSpinLockGet(&pDmaContext->lock);
          pDmaContext->curr.state.setup.count--;
          dfAssert(pDmaContext->curr.state.setup.count <= 2);
          pDmaContext->curr.state.teardown.count--;
          dfAssert(pDmaContext->curr.state.teardown.count <= 2);
          pDmaContext->curr.state.teardown.state = TeardownIdle;
          if (bOperationFinished(pDmaContext)) {
            /* The DMA transfer is finished */
            dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: DMA transfer finished, channelIndex=%lu\n", (unsigned long)channelIndex));
            dfAssert(pDmaContext->curr.state.transfer.count == 0);
            pDmaContext->curr.state.main = MainCompleted;
            dfAssert(pDmaContext->curr.state.setup.state == SetupIdle);
            dfAssert(pDmaContext->curr.state.setup.count == 0);
            dfAssert(pDmaContext->curr.state.transfer.state == TransferIdle);
            dfAssert(pDmaContext->curr.state.transfer.count == 0);
            dfAssert(pDmaContext->curr.state.teardown.state == TeardownIdle);
            dfAssert(pDmaContext->curr.state.teardown.count == 0);
            dfSpinLockPut(&pDmaContext->lock, f);
            pDmaContext->curr.pCallback(pDmaContext->curr.pTicket, pDmaContext->curr.pCallbackContext, CoreDmaSuccess);
            break;
          }
        } else {
          /* Nothing more to do */
          dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedUser: nothing more to do, channelIndex=%lu\n", (unsigned long)channelIndex));
          dfSpinLockPut(&pDmaContext->lock, f);
          break;
        }
      }
    }
  }
}

void
dmaCompletedRoutine(
  DfDpc* pDpc,
  void* pContext, /* channel */
  void* pArg)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
  unsigned int channelIndex = (unsigned int)(uintptr_t)pArg;
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  DfSpinLockFlags f;

#if defined(BUILD_DEBUGTS)
  if (dfTimestampBufferEnabled(&pDmaContext->timestampBuffer)) {
    unsigned int chunkIndex = pDmaContext->curr.state.transfer.chunkIndex;
    uint64_t timestamp = dfTimestampGet();

    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMADPC,
      (uint16_t)chunkIndex,
      0,
      timestamp);
    pDmaContext->curr.state.transfer.chunkIndex = chunkIndex + 1;

    /* Log the timestamp that the ISR saved */
    dmaLogEvent(
      pDmaContext,
      (uint8_t)channelIndex,
      ADMXRC2_DEBUGTS_EVENT_DMAHWFINISH,
      (uint16_t)chunkIndex,
      0,
      pDmaContext->isrTimestamp);
    pDmaContext->curr.state.transfer.chunkIndex = chunkIndex + 1;
  }
#endif

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("dmaCompletedRoutine: pDevCtx=%p channelIndex=%lu\n", (void*)pDevCtx, (unsigned long)channelIndex));

  f = dfSpinLockGet(&pDmaContext->lock);
  if (pDmaContext->curr.state.transfer.state == TransferIdle) {
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("+++ dmaCompletedRoutine: spurious call, pDevCtx=%p channelIndex=%lu\n", (void*)pDevCtx, (unsigned long)channelIndex));
    return;
  }
  pDmaContext->curr.state.transfer.state = TransferIdle;

  switch (pDmaContext->curr.type) {
  case DmaRequestTypeUser:
    dmaCompletedUser(pDevCtx, pDmaContext, channelIndex, f);
    return;

  case DmaRequestTypeBus:
    dmaCompletedBus(pDevCtx, pDmaContext, channelIndex, f);
    return;

  default:
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("** dmaCompletedRoutine: unknown type of DMA request, type=%u\n", pDmaContext->curr.type));
    return;
  }
}

/*
** Handle CoreDmaCancel
*/

CoreDmaStatus
coreDmaCancel(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int channelIndex)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  CoreDmaStatus status;
  DfSpinLockFlags f;

  if (channelIndex >= pDevCtx->info.bootstrap.numDmaChannel) {
    return CoreDmaInvalidChannel;
  }

  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_DMAENGINE, channelIndex)) {
    return CoreDmaInvalidTicket;
  }

  f = dfSpinLockGet(&pDmaContext->lock);
  if (pDmaContext->curr.state.bCancel) {
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** coreDmaCancel: already cancelling\n"));
    return CoreDmaSuccess;
  }
  pDmaContext->curr.state.bCancel = TRUE;
  pDmaContext->curr.status = CoreDmaCancelled;
  switch (pDmaContext->curr.state.main) {
  case MainIdle:
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** coreDmaCancel: in Idle state\n"));
    break;

  case MainInitial:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("coreDmaCancel: in Initial state\n"));
    dfSpinLockPut(&pDmaContext->lock, f);
    break;

  case MainRunning:
    if (bCanCancel(pDmaContext)) {
      dfDebugPrint(DEBUGLEVEL_CANCEL, ("coreDmaCancel: in Running state, cleaning up & invoking callback\n"));
      pDmaContext->curr.state.main = MainIdle;
      status = pDmaContext->curr.status;
      dfSpinLockPut(&pDmaContext->lock, f);
      dmaCleanup(pDevCtx, channelIndex);
      pDmaContext->curr.pCallback(pDmaContext->curr.pTicket, pDmaContext->curr.pCallbackContext, status);
    } else {
      if (pDmaContext->curr.state.transfer.state == TransferRunning) {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("coreDmaCancel: in Running state, aborting DMA on hardware\n"));
        dmaStopOnHardware(pDevCtx, channelIndex);
      } else {
        dfDebugPrint(DEBUGLEVEL_CANCEL, ("coreDmaCancel: in Running state, not aborting DMA on hardware\n"));
      }
      dfSpinLockPut(&pDmaContext->lock, f);
    }
    break;

  case MainCompleted:
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("*** coreDmaCancel: in Completed state\n"));
    break;

  case MainCancelled:
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** coreDmaCancel: in Cancelled state\n"));
    break;

  default:
    dfSpinLockPut(&pDmaContext->lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** coreDmaCancel: in INVALID state (%u)\n", pDmaContext->curr.state.main));
    break;
  }

  return CoreDmaSuccess;
}

/*
** Handle CoreDmaInitialize
*/

CoreDmaStatus
coreDmaInitialize(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int channelIndex,
  const CoreDmaParameters* pParameters,
  CoreDmaCallback* pCallback,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];

  if (channelIndex >= pDevCtx->info.bootstrap.numDmaChannel) {
    return CoreDmaInvalidChannel;
  }
  if (pParameters->bWriteToLocal) {
    if (pDevCtx->info.bootstrap.dmaRestrictions[channelIndex].bNoHostToFpga) {
      return CoreDmaDirectionNotSupported;
    }
  } else {
    if (pDevCtx->info.bootstrap.dmaRestrictions[channelIndex].bNoFpgaToHost) {
      return CoreDmaDirectionNotSupported;
    }
  }

  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_DMAENGINE, channelIndex)) {
    return CoreDmaInvalidTicket;
  }

  dfAssert(pDmaContext->curr.state.main != MainRunning);

  pDmaContext->curr.pTicket = pTicket;
  pDmaContext->curr.parameters.user = *pParameters;
  pDmaContext->curr.status = CoreDmaSuccess;
  pDmaContext->curr.pCallback = pCallback;
  pDmaContext->curr.pCallbackContext = pCallbackContext;
  pDmaContext->curr.slice.user.length = pParameters->region.length;
  pDmaContext->curr.slice.user.offset = pParameters->region.offset;
  pDmaContext->curr.slice.user.localAddress = pParameters->region.localAddress;
  pDmaContext->curr.position.sliceIndex = 0;
  pDmaContext->curr.position.slicePosition = 0;

  pDmaContext->curr.state.bCancel = FALSE;
  pDmaContext->curr.state.main = MainInitial;
  pDmaContext->curr.state.setup.tableIndex = 0;
  pDmaContext->curr.state.transfer.tableIndex = 0;
  pDmaContext->curr.state.teardown.tableIndex = 0;
#if defined(BUILD_DEBUGTS)
  pDmaContext->curr.state.setup.chunkIndex = 0;
  pDmaContext->curr.state.transfer.chunkIndex = 0;
  pDmaContext->curr.state.teardown.chunkIndex = 0;
#endif

  return CoreDmaSuccess;
}

/*
** Handle CoreDmaTransfer
*/

void
coreDmaStart(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int channelIndex)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DmaChannelContext* pDmaContext = &pDevCtx->dma.channel[channelIndex];
  CoreDmaStatus status;
  DfSpinLockFlags f;

  if (channelIndex >= pDevCtx->info.bootstrap.numDmaChannel) {
    pDmaContext->curr.pCallback(pTicket, pDmaContext->curr.pCallbackContext, CoreDmaInvalidChannel);
  }

  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_DMAENGINE, channelIndex)) {
    pDmaContext->curr.pCallback(pTicket, pDmaContext->curr.pCallbackContext, CoreDmaInvalidTicket);
  }

  f = dfSpinLockGet(&pDmaContext->lock);
  dfAssert(pDmaContext->curr.state.main == MainInitial);
  dfAssert(pDmaContext->curr.state.setup.state == SetupIdle);
  dfAssert(pDmaContext->curr.state.setup.count == 0);
  dfAssert(pDmaContext->curr.state.transfer.state == TransferIdle);
  dfAssert(pDmaContext->curr.state.transfer.count == 0);
  dfAssert(pDmaContext->curr.state.teardown.state == TeardownIdle);
  dfAssert(pDmaContext->curr.state.teardown.count == 0);
  if (pDmaContext->curr.state.bCancel) {
    /* We were cancelled */
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("coreDmaStart: cancelled, channelIndex=%lu\n", (unsigned long)channelIndex));
    pDmaContext->curr.state.bCancel = FALSE;
    pDmaContext->curr.state.main = MainIdle;
    status = pDmaContext->curr.status;
    dfSpinLockPut(&pDmaContext->lock, f);
    pDmaContext->curr.pCallback(pTicket, pDmaContext->curr.pCallbackContext, status);
  } else {
    dfDebugPrint(DEBUGLEVEL_NORMAL, ("coreDmaStart: starting setup, channelIndex=%lu\n", (unsigned long)channelIndex));
    pDmaContext->curr.state.main = MainRunning;
    pDmaContext->curr.state.setup.state = SetupRunning;
    dfSpinLockPut(&pDmaContext->lock, f);
    dmaSetup(pDevCtx, channelIndex);
  }
}
