/*
** File: coreclock.c  
** Project: ADB3 core driver
** Purpose: Implements functions for clock generators in Core Interface.
**
** (C) Copyright Alpha Data 2009-2010
*/

#include <df.h>

#include "resource.h"
#include "coreclock.h"

/*
** Handle CoreClockGetFreq
*/

CoreClockStatus
coreClockGetFreq(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int clockIndex,
  uint64_t* pActualFrequency)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return CoreClockInvalidIndex;
  }

  if (!checkTicketValid(pDevCtx, pTicket, CORE_RESOURCE_CLOCKGEN, clockIndex)) {
    return CoreClockInvalidTicket;
  }

  *pActualFrequency = pDevCtx->clockGenerator[clockIndex].currentFrequency;

  return CoreClockSuccess;
}

/*
** Handle CoreClockProgram
*/

CoreClockStatus
coreClockProgram(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord,
  CoreClockProgCallback* pCallback,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  ClockGeneratorContext* pDevClockCtx;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return CoreClockInvalidIndex;
  }
  pDevClockCtx = &pDevCtx->clockGenerator[clockIndex];
  pDevClockCtx->program.pCallback = pCallback;
  pDevClockCtx->program.pCallbackTicket = pTicket;
  pDevClockCtx->program.pCallbackContext = pContext;
  pDevClockCtx->program.word = *pClockWord;
  return (CoreClockStatus)pDevCtx->methods.pClockProgram(pDevCtx, clockIndex, pClockWord);
}

void
onClockProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  ClockProgramStatus status)
{
  ClockGeneratorContext* pDevClockCtx;

  pDevClockCtx = &pDevCtx->clockGenerator[clockIndex];
  switch (status) {
  case ClockProgramSuccess:
    pDevClockCtx->currentFrequency = pDevClockCtx->program.word.frequency;
    break;

  case ClockProgramOutOfRange:
  case ClockProgramInvalidIndex:
    break;

  case ClockProgramHardwareError:
  case ClockProgramGeneralFailure:
  default:
    pDevClockCtx->currentFrequency = 0;
    break;
  }
  pDevClockCtx->program.pCallback(pDevClockCtx->program.pCallbackTicket, pDevClockCtx->program.pCallbackContext, (CoreClockStatus)status);
}

/*
** Handle CoreClockTestFreq
*/

CoreClockStatus
coreClockTestFreq(
  void* pInterfaceContext,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return CoreClockInvalidIndex;
  }

  return (CoreClockStatus)pDevCtx->methods.pClockWord(pDevCtx, clockIndex, flags, frequencyReq, frequencyMin, frequencyMax, pClockWord);
}

unsigned int
coreGetNumClockGen(
  void* pInterfaceContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;

  return pDevCtx->info.bootstrap.numClockGen;
}
