/*
** File: avr_fmc.c  
** Project: ADB3 core driver
** Purpose: Shared code for identifying FMC/XRM attached by I2C to AVR uC.
**
** (C) Copyright Alpha Data 2012
*/

#include <df.h>

#include "avr_fmc.h"

boolean_t
avrFmcIdentify(
	Adb3CoreDeviceContext* pDevCtx,
  boolean_t bIsFmc,
  uint8_t fmcStatus)
{
  CoreIoModuleInfo* pInfo = &pDevCtx->info.ioModule[0];
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  boolean_t bAssumeXrmPresent = FALSE, bPresent, bForce2V5, bRomOk, bLegacyXrm = FALSE;
  uint8_t val8;
  unsigned int i;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  bPresent = (fmcStatus & AVR_FMCSTATUS_V_PRSNT) ? TRUE : FALSE;
  if (bIsFmc) {
    bForce2V5 = FALSE;
  } else {
    bForce2V5 = (fmcStatus & AVR_XRMSTATUS_V_FORCE2V5) ? TRUE : FALSE;
  }
  bRomOk = (fmcStatus & (AVR_FMCSTATUS_V_I2CFAIL | AVR_FMCSTATUS_V_CSUMERR | AVR_FMCSTATUS_V_NODATA)) ? FALSE : TRUE;

  /* This driver parameter can be used to override presence detection for XRMs that don't drive the presence pin correctly and don't have FRU ROM */
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "AssumeXrmPresent", &bAssumeXrmPresent);
  if (bAssumeXrmPresent) {
    bPresent = TRUE;
  }

  if (bPresent) {
    if (!bRomOk) {
      dfDebugPrint(2, ("+++ avrFmcIdentify: FRU ROM not detected or in error state - assuming legacy XRM, fmcStatus=0x%02x( %s%s%s)\n",
        (unsigned int)fmcStatus,
        (fmcStatus & AVR_FMCSTATUS_V_I2CFAIL) ? "I2CFAIL " : "",
        (fmcStatus & AVR_FMCSTATUS_V_CSUMERR) ? "CSUMERR " : "",
        (fmcStatus & AVR_FMCSTATUS_V_NODATA) ? "NODATA " : " "));
      bLegacyXrm = TRUE;
    }
  } else {
    dfDebugPrint(2, ("avrFmcIdentify: XRM not present\n"));
  }

  pInfo->bIsFmc = bIsFmc;
  pInfo->bPresent = bPresent;
  pInfo->bLegacyXrm = bLegacyXrm;
  pInfo->bForce2v5 = bForce2V5;
  if (bPresent && bRomOk) {
    val8 = dfPciMemRead8(hModel, &pModelRegs->xrmBuf[0]);
    if (0x1U != val8) {
      dfDebugPrint(2, ("+++ avrFmcIdentify: FRU ROM invalid first byte, expected=0x%02x actual=0x%02x\n", (unsigned int)0x1U, (unsigned int)val8));
    }
    pInfo->pVpdBuffer = dfMalloc(ADMXRC6TX_FRU_ROM_SIZE);
    if (NULL == pInfo->pVpdBuffer) {
      dfDebugPrint(0, ("*** avrFmcIdentify: Failed to allocate FRU ROM buffer, size %p bytes\n", (void*)(uintptr_t)ADMXRC6TX_FRU_ROM_SIZE));
      return FALSE;
    }
    pInfo->vpdLength = ADMXRC6TX_FRU_ROM_SIZE;
    for (i = 0; i < ADMXRC6TX_FRU_ROM_SIZE; i++) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->xrmBuf[i]);
      pInfo->pVpdBuffer[i] = val8;
    }
  }

  return TRUE;
}

