/*
** File: avr_common.h  
** Project: ADB3 core driver
** Purpose: Exports functions for accessing the AVR uC interface
**
** (C) Copyright Alpha Data 2011
*/

#ifndef _ADATA_CORE_AVR_COMMON_H
#define _ADATA_CORE_AVR_COMMON_H

#include "avr.h"
#include "serializer.h"

struct _Adb3CoreDeviceContext;
struct _AvrDeviceContext;
typedef struct _AvrDeviceContext AvrDeviceContext;
struct _AvrRequest;
typedef struct _AvrRequest AvrRequest;
struct _AvrTransRequest;
typedef struct _AvrTransRequest AvrTransRequest;

/*
** ----------------------------------------------------------------
** AVR uC access byte transport layer *
** ----------------------------------------------------------------
*/

typedef enum _AvrRequestStatus {
  AvrRequestStatusSuccess   = 0,
  AvrRequestStatusCancelled = 1
} AvrRequestStatus;

typedef void AvrCallback(AvrRequest* pRequest, AvrRequestStatus status, uint8_t data, void* pContext);

typedef enum _AvrRequestState {
  AvrRequestStateQueued   = 0, /* Queued */
  AvrRequestStateActive   = 1, /* Waiting for transmission / reception */
  AvrRequestStateComplete = 2  /* Finished; callback is executing or has executed */
} AvrRequestState;

struct _AvrRequest { /* Client must not touch */
  SerializerRequest serRequest;
  AvrDeviceContext* pAvrCtx;
  AvrCallback* pCallback;
  void* pContext;
  uint8_t data;
  boolean_t bWrite; /* TRUE for transmitting to AVR, FALSE for receiving */
  AvrRequestState state;
  boolean_t bCancel;
};

/* Must be called in thread context */
extern AvrRequestStatus
avrByteReadSync(
  AvrDeviceContext* pAvrCtx,
  uint8_t* pVal8);

/* Must be called in thread context */
extern AvrRequestStatus
avrByteWriteSync(
  AvrDeviceContext* pAvrCtx,
  uint8_t val8);

extern void
avrByteRead(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest,
  AvrCallback* pCallback,
  void* pContext);

extern void
avrByteWrite(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest,
  uint8_t data,
  AvrCallback* pCallback,
  void* pContext);

extern boolean_t
avrByteCancel(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest);

/*
** ----------------------------------------------------------------
** AVR uC access transaction layer
** ----------------------------------------------------------------
*/

typedef enum _AvrTransStatus {
  AvrTransStatusSuccess = 0,
  AvrTransStatusTimeout = 1
} AvrTransStatus;

typedef void AvrTransCallback(AvrTransRequest* pRequest, AvrTransStatus status, void* pContext);

typedef enum _AvrTransactionState {
  AvrTransactionStateQueued   = 0, /* Queued */
  AvrTransactionStateCommand  = 1, /* Sending the command */
  AvrTransactionStateResponse = 2, /* Receiving the response */
  AvrTransactionStateComplete = 3  /* Finished; callback is executing or has executed */
} AvrTransactionState;

struct _AvrTransRequest {  /* Client must not touch */
  SerializerRequest serRequest;
  AvrDeviceContext* pAvrCtx;
  AvrTransCallback* pCallback;
  void* pContext;
  const uint8_t* pCommand; /* Command to uC */
  size_t commandLength;    /* Command length, in bytes */
  uint8_t* pResponse;      /* Filled in with response from uC */
  size_t responseLength;   /* Response length, in bytes */
  unsigned int timeout;    /* In microseconds */
  AvrTransactionState state;
  boolean_t bTimeout;
};

/* Must be called in thread context */
extern AvrTransStatus
avrTransactionSync(
  AvrDeviceContext* pAvrCtx,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout); /* In microseconds */

extern void
avrTransaction(
  AvrDeviceContext* pAvrCtx,
  AvrTransRequest* pRequest,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout, /* In microseconds */
  AvrTransCallback* pCallback,
  void* pContext);

/*
** ----------------------------------------------------------------
** AVR uC access device context
** ----------------------------------------------------------------
*/

typedef void AvrRxIntEnableCallback(AvrDeviceContext* pAvrCtx);
typedef void AvrTxIntEnableCallback(AvrDeviceContext* pAvrCtx);

struct _AvrDeviceContext {
  uint32_t flashSize;                        /* Size of AVR uC Flash memory */
  uint32_t flashPageSize;                    /* Size of a page AVR uC Flash memory */
  uint32_t ctlStat;                          /* Shadow of AVR interface ctl/status register */
  uint32_t* pCtlStat;                        /* Pointer to AVR interface ctl/status register */
  DfMemoryHandle hCtlStat;                   /* Handle to AVR interface ctl/status register */
  DfInterruptObject* pInterruptObject;       /* Interrupt object associated with DPCs */
  struct {
    DfSpinLock lock;                         /* Spinlock that guards this struct */
    struct {
      SerializerQueue serializer;            /* Serializes reception from AVR uC */
      AvrRxIntEnableCallback* pEnableInt;    /* Method called to enable RX interrupt */
      DfDpc dpc;                             /* RX interrupt DPC */
    } rx;
    struct {
      SerializerQueue serializer;            /* Serializes transmission to AVR uC */
      AvrTxIntEnableCallback* pEnableInt;    /* Method called to enable TX interrupt */
      DfDpc dpc;                             /* TX interrupt DPC */
    } tx;
  } transport; /* byte transport layer */
  struct {
    DfSpinLock lock;                         /* Spinlock that guards this struct */
    SerializerQueue serializer;              /* Serializes transactions with AVR uC */
    DfTimer timer;                           /* Timeout timer */
    AvrRequest avrRequest;                   /* Used for sending/receiving each byte of a transaction */
    size_t position;                         /* Byte index into command/response, for current transaction */
  } transaction; /* transaction layer */
  struct {
    uint8_t* pMergeBuffer;                   /* Used in AVR flash writes */
  } vpd;
};

/* Must be called in thread context */
extern boolean_t
avrInit(
  AvrDeviceContext* pAvrCtx,
  DfInterruptObject* pInterruptObject,
  uint32_t* pCtlStat,
  DfMemoryHandle hCtlStat,
  uint32_t flashSize,                       /* uC Flash memory size */
  uint32_t flashPageSize,                   /* uC Flash memory page size */
  AvrRxIntEnableCallback* pEnableRxInt,  /* Function that enables AVR RX interrupt */
  AvrTxIntEnableCallback* pEnableTxInt); /* Function that enables AVR TX interrupt */

/* Must be called in thread context */
extern void
avrUninit(
  AvrDeviceContext* pAvrCtx);

extern boolean_t
avrIsServiceMode(
  AvrDeviceContext* pAvrCtx);

#endif
