/*******************************************************************************
 *	BQueue.h	Queue Classes
 *	T.Barnaby,	BEAM Ltd,	2014-07-23
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 *
 * Simple thread safe queue.
 */
#ifndef BQueue_h
#define BQueue_h	1

#include <BTypes.h>
#include <BError.h>
#include <BList.h>
#include <BMutex.h>
#include <BCondInt.h>

/// Queue class
template <class T> class BQueue : private BList<T>{
public:
			BQueue(BUInt size);
			~BQueue();
		
	void		clear();							///< Clear the queue

	BUInt		writeAvailable() const;
	BError		write(const T& v,  BTimeout timeout = BTimeoutForever);		///< Append an item onto the queue

	BUInt		readAvailable() const;
	BError		read(T& v,  BTimeout timeout = BTimeoutForever);		///< Get an item from the queue

private:
	BMutex		olock;
	BUInt		osize;
	BCondInt	onumber;
};

// BQueueInt class
typedef BQueue<BInt32>	BQueueInt;

// BQueue class implementation
template <class T> BQueue<T>::BQueue(BUInt size){
	osize = size;
}

template <class T> BQueue<T>::~BQueue(){
}

template <class T> void BQueue<T>::clear(){
	olock.lock();
	BList<T>::clear();
	olock.unlock();
}

template <class T> BUInt BQueue<T>::writeAvailable() const{
	return osize - onumber.value();
}

template <class T> BUInt BQueue<T>::readAvailable() const{
	return onumber.value();
}

template <class T> BError BQueue<T>::write(const T& v,  BTimeout timeout){
	BError		err;

	if(!onumber.waitLessThanOrEqual(osize - 1, 0, timeout))
		return err.set(ErrorTimeout, "Timeout");

	olock.lock();
	BList<T>::queueAdd(v);
	olock.unlock();
	onumber.increment();
		
	return err;
}

template <class T> BError BQueue<T>::read(T& v,  BTimeout timeout){
	BError		err;

	if(!onumber.waitMoreThanOrEqual(1, 1, timeout))
		return err.set(ErrorTimeout, "Timeout");

	olock.lock();
	v = BList<T>::queueGet();
	olock.unlock();
		
	return err;
}

#endif
