/*******************************************************************************
 *	BEndian.h	Byte swap fubctions
 *			T.Barnaby,	BEAM Ltd,	2009-12-18
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 */
#ifndef BEndian_H
#define BEndian_H	1

#include <BTypes.h>

#if TARGET_win32
#define __BYTE_ORDER __LITTLE_ENDIAN
uint16_t __bswap_16(uint16_t v){
	uint16_t	r;
	const char*	sp = (const char*)&v;
	char*		dp = (char*)&r;

	dp[1] = sp[0];
	dp[0] = sp[1];

	return r;
}
uint32_t __bswap_32(uint32_t v){
	uint32_t	r;
	const char*	sp = (const char*)&v;
	char*		dp = (char*)&r;

	dp[3] = sp[0];
	dp[2] = sp[1];
	dp[1] = sp[2];
	dp[0] = sp[3];

	return r;
}
uint64_t __bswap_64(uint64_t v){
	uint64_t	r;
	const char*	sp = (const char*)&v;
	char*		dp = (char*)&r;

	dp[7] = sp[0];
	dp[6] = sp[1];
	dp[5] = sp[2];
	dp[4] = sp[3];
	dp[3] = sp[4];
	dp[2] = sp[5];
	dp[1] = sp[6];
	dp[0] = sp[7];

	return r;
}
#else
#include <byteswap.h>
#endif


#ifndef htobe16
# if __BYTE_ORDER == __LITTLE_ENDIAN
#  define htobe16(x) __bswap_16 (x)
#  define htole16(x) (x)
#  define be16toh(x) __bswap_16 (x)
#  define le16toh(x) (x)

#  define htobe32(x) __bswap_32 (x)
#  define htole32(x) (x)
#  define be32toh(x) __bswap_32 (x)
#  define le32toh(x) (x)

#  define htobe64(x) __bswap_64 (x)
#  define htole64(x) (x)
#  define be64toh(x) __bswap_64 (x)
#  define le64toh(x) (x)
# else
#  define htobe16(x) (x)
#  define htole16(x) __bswap_16 (x)
#  define be16toh(x) (x)
#  define le16toh(x) __bswap_16 (x)

#  define htobe32(x) (x)
#  define htole32(x) __bswap_32 (x)
#  define be32toh(x) (x)
#  define le32toh(x) __bswap_32 (x)

#  define htobe64(x) (x)
#  define htole64(x) __bswap_64 (x)
#  define be64toh(x) (x)
#  define le64toh(x) __bswap_64 (x)
# endif
#endif

inline void bswap_p8(const void* s, void* d){
	const char*	sp = (const char*)s;
	char*		dp = (char*)d;

	*dp = *sp;
}

inline void bswap_p16(const void* s, void* d){
	const char*	sp = (const char*)s;
	char*		dp = (char*)d;

	dp[1] = sp[0];
	dp[0] = sp[1];
}

inline void bswap_p32(const void* s, void* d){
	const char*	sp = (const char*)s;
	char*		dp = (char*)d;

	dp[3] = sp[0];
	dp[2] = sp[1];
	dp[1] = sp[2];
	dp[0] = sp[3];
}

inline void bswap_p64(const void* s, void* d){
	const char*	sp = (const char*)s;
	char*		dp = (char*)d;

	dp[7] = sp[0];
	dp[6] = sp[1];
	dp[5] = sp[2];
	dp[4] = sp[3];
	dp[3] = sp[4];
	dp[2] = sp[5];
	dp[1] = sp[6];
	dp[0] = sp[7];
}

void bswap_copy(int swap, const void* src, void* dst, BUInt32 nBytes, const char* swapType);


inline uint16_t htole(uint16_t v){
	return htole16(v);
}

inline int16_t htole(int16_t v){
	return htole16(v);
}

inline uint32_t htole(uint32_t v){
	return htole32(v);
}

inline int32_t htole(int32_t v){
	return htole32(v);
}

inline uint64_t htole(uint64_t v){
	return htole64(v);
}

inline int64_t htole(int64_t v){
	return htole64(v);
}

inline double htole(double v){
	union {
		int64_t	i;
		double	d;
	} t;

	t.d = v;
	t.i = htole64(t.i);
	
	return t.d;
}

inline float htole(float v){
	union {
		int32_t	i;
		float	d;
	} t;

	t.d = v;
	t.i = htole32(t.i);
	
	return t.d;
}

inline uint16_t htobe(uint16_t v){
	return htobe16(v);
}

inline int16_t htobe(int16_t v){
	return htobe16(v);
}

inline uint32_t htobe(uint32_t v){
	return htobe32(v);
}

inline int32_t htobe(int32_t v){
	return htobe32(v);
}

inline uint64_t htobe(uint64_t v){
	return htobe64(v);
}

inline int64_t htobe(int64_t v){
	return htobe64(v);
}

inline double htobe(double v){
	union {
		int64_t	i;
		double	d;
	} t;

	t.d = v;
	t.i = htobe64(t.i);
	
	return t.d;
}

inline float htobe(float v){
	union {
		int32_t	i;
		float	d;
	} t;

	t.d = v;
	t.i = htobe32(t.i);
	
	return t.d;
}


inline uint16_t letoh(uint16_t v){
	return le16toh(v);
}

inline int16_t letoh(int16_t v){
	return le16toh(v);
}

inline uint32_t letoh(uint32_t v){
	return le32toh(v);
}

inline int32_t letoh(int32_t v){
	return le32toh(v);
}

inline uint64_t letoh(uint64_t v){
	return le64toh(v);
}

inline int64_t letoh(int64_t v){
	return le64toh(v);
}

inline double letoh(double v){
	union {
		int64_t	i;
		double	d;
	} t;

	t.d = v;
	t.i = le64toh(t.i);
	
	return t.d;
}

inline float letoh(float v){
	union {
		int32_t	i;
		float	d;
	} t;

	t.d = v;
	t.i = le32toh(t.i);
	
	return t.d;
}


inline uint16_t betoh(uint16_t v){
	return be16toh(v);
}

inline int16_t betoh(int16_t v){
	return be16toh(v);
}

inline uint32_t betoh(uint32_t v){
	return be32toh(v);
}

inline int32_t betoh(int32_t v){
	return be32toh(v);
}

inline uint64_t betoh(uint64_t v){
	return be64toh(v);
}

inline int64_t betoh(int64_t v){
	return be64toh(v);
}

inline double betoh(double v){
	union {
		int64_t	i;
		double	d;
	} t;

	t.d = v;
	t.i = be64toh(t.i);
	
	return t.d;
}

inline float betoh(float v){
	union {
		int32_t	i;
		float	d;
	} t;

	t.d = v;
	t.i = be32toh(t.i);
	
	return t.d;
}

#endif
