/*
** File: model_admxrc2l.c
** Project: ADMXRC2 module driver
** Purpose: ADM-XRC-II-Lite specific code, including the function to
**          bootstrap the device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include <core/pci9xxx.h>
#include <core/admxrc2l.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdmxrc2lRev0* pVpd = (VpdAdmxrc2lRev0*)pVpdBuffer;
  boolean_t bFitted;
  uint16_t type, order;

  if (bankIndex >= 4) {
    return ADMXRC2_INVALID_INDEX;
  }
  type = (uint16_t)((pVpd->zbt0123 >> (bankIndex * 4)) & 0x3U);
  if (type != 0) {
    order = (uint16_t)(((pVpd->zbt0123 >> (bankIndex * 4 + 2)) & 0x3U) + 19U);
    bFitted = TRUE;
  } else {
    order = 0;
    bFitted = FALSE;
  }
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.width = 18U;
  pIoctl->out.typeMask = 0;
  if (type & 0x1U) {
    pIoctl->out.typeMask |= ADMXRC2_RAM_ZBTFT;
  }
  if (type & 0x2U) {
    pIoctl->out.typeMask |= ADMXRC2_RAM_ZBTP;
  }
  pIoctl->out.bFitted = bFitted;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdmxrc2lRev0* pVpd = (VpdAdmxrc2lRev0*)pVpdBuffer;
  unsigned int i;
  uint16_t type;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = (uint32_t)pVpd->serialLow | ((uint32_t)pVpd->serialHigh << 16);
  pIoctl->out.numMemoryBank = 4;
  pIoctl->out.memoryBankFitted = 0;
  for (i = 0; i < 4; i++) {
    type = (uint16_t)((pVpd->zbt0123 >> (i * 4)) & 0x3U);
    if (type != 0) {
      pIoctl->out.memoryBankFitted |= 0x1U << i;
    }
  }
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdmxrc2lRev0* pVpd = (VpdAdmxrc2lRev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FG456");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = (uint32_t)-1;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdmxrc2l(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdmxrc2l: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 2U;
  pDevCtx->configureDmaAddress = 0x800008U; /* TODO - should make this dynamic, i.e. based on current value of LAS1BA */
  pDevCtx->configureDmaMode =
    PCI9656_DMAMODE_8BIT | PCI9656_DMAMODE_USE_READY | PCI9656_DMAMODE_USE_BTERM |
    PCI9656_DMAMODE_ENABLE_BURST | PCI9656_DMAMODE_FIXED_LOCAL;
}
