/*
** File: resource.c  
** Project: ADB3 core driver
** Purpose: Implements resource acquisition and release functions in Core Interface.
**
** (C) Copyright Alpha Data 2009-2010
*/

#include <df.h>

#include "resource.h"

typedef enum _State {
  StateInitial = 0,
  StateQueued = 1,
  StateAcquired = 2,
  StateCancelled = 3,
  StateFailed = 4
} State;

static CoreAcquireStatus
resourceValid(
  Adb3CoreDeviceContext* pDevCtx,
  uint8_t code,
  uint8_t index)
{
  switch (code) {
  case CORE_RESOURCE_CLOCKGEN:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      return CoreAcquireSuccess;
    } else {
      if (index >= pDevCtx->info.bootstrap.numClockGen) {
        return CoreAcquireInvalidIndex;
      }
      return CoreAcquireSuccess;
    }

  case CORE_RESOURCE_DMAENGINE:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      return CoreAcquireSuccess;
    } else {
      if (index >= pDevCtx->info.bootstrap.numDmaChannel) {
        return CoreAcquireInvalidIndex;
      }
      return CoreAcquireSuccess;
    }

  case CORE_RESOURCE_EEPROM:
    if (CORE_RESOURCE_INDEX_ANY != index && index >= 1) {
      return CoreAcquireInvalidIndex;
    }
    return CoreAcquireSuccess;

  case CORE_RESOURCE_FLASH:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      return CoreAcquireSuccess;
    } else {
      if (index >= MAX_NUM_FLASH_BANK) {
        return CoreAcquireInvalidIndex;
      }
      return CoreAcquireSuccess;
    }

  case CORE_RESOURCE_TARGETFPGA:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      return CoreAcquireSuccess;
    } else {
      if (index >= pDevCtx->info.bootstrap.numTargetFpga) {
        return CoreAcquireInvalidIndex;
      }
      return CoreAcquireSuccess;
    }

  case CORE_RESOURCE_SYSMON:
    if (CORE_RESOURCE_INDEX_ANY != index && index >= 1) {
      return CoreAcquireInvalidIndex;
    }
    return CoreAcquireSuccess;

  case CORE_RESOURCE_VPD:
    if (CORE_RESOURCE_INDEX_ANY != index && index >= 1) {
      return CoreAcquireInvalidIndex;
    }
    return CoreAcquireSuccess;
    
  default:
    return CoreAcquireInvalid;
  }
}

/*
** Caller must hold pDevCtx->resource.lock
**
** Returns TRUE if available, FALSE otherwise.
*/
static boolean_t
resourceAvailable(
  Adb3CoreDeviceContext* pDevCtx,
  CoreResourceDesc* pDesc)
{
  uint8_t i;
  uint8_t code, index;

  code = pDesc->code;
  index = pDesc->index;

  switch (code) {
  case CORE_RESOURCE_CLOCKGEN:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      if (pDevCtx->resource.clockGen.queue.users < pDevCtx->info.bootstrap.numClockGen) {
        for (i = 0; i < pDevCtx->info.bootstrap.numClockGen; i++) {
          if (0 == pDevCtx->resource.clockGen.users[i]) {
            pDesc->actualIndex = i;
            return TRUE;
          }
        }
      }
      return FALSE;
    } else {
      pDesc->actualIndex = index;
      return (0 == pDevCtx->resource.clockGen.users[index]) ? TRUE : FALSE;
    }

  case CORE_RESOURCE_DMAENGINE:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      if (pDevCtx->resource.dmaEngine.queue.users < pDevCtx->info.bootstrap.numDmaChannel) {
        for (i = 0; i < pDevCtx->info.bootstrap.numDmaChannel; i++) {
          if (0 == pDevCtx->resource.dmaEngine.users[i]) {
            pDesc->actualIndex = i;
            return TRUE;
          }
        }
      }
      return FALSE;
    } else {
      pDesc->actualIndex = index;
      return (0 == pDevCtx->resource.dmaEngine.users[index]) ? TRUE : FALSE;
    }

  case CORE_RESOURCE_EEPROM:
    pDesc->actualIndex = 0;
    return (0 == pDevCtx->resource.eeprom.users[index]) ? TRUE : FALSE;

  case CORE_RESOURCE_FLASH:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      if (pDevCtx->resource.flash.queue.users < MAX_NUM_FLASH_BANK) {
        for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
          if (0 == pDevCtx->resource.flash.users[i]) {
            pDesc->actualIndex = i;
            return TRUE;
          }
        }
      }
      return FALSE;
    } else {
      pDesc->actualIndex = index;
      return (0 == pDevCtx->resource.flash.users[index]) ? TRUE : FALSE;
    }

  case CORE_RESOURCE_TARGETFPGA:
    if (CORE_RESOURCE_INDEX_ANY == index) {
      if (pDevCtx->resource.targetFpga.queue.users < pDevCtx->info.bootstrap.numTargetFpga) {
        for (i = 0; i < pDevCtx->info.bootstrap.numTargetFpga; i++) {
          if (0 == pDevCtx->resource.targetFpga.users[i]) {
            pDesc->actualIndex = i;
            return TRUE;
          }
        }
      }
      return FALSE;
    } else {
      pDesc->actualIndex = index;
      return (0 == pDevCtx->resource.targetFpga.users[index]) ? TRUE : FALSE;
    }

  case CORE_RESOURCE_SYSMON:
    pDesc->actualIndex = 0;
    return (0 == pDevCtx->resource.sysMon.users[index]) ? TRUE : FALSE;

  case CORE_RESOURCE_VPD:
    if (pDevCtx->info.bootstrap.bVpdInFlash) {
      if (CORE_RESOURCE_INDEX_ANY == index) {
        if (pDevCtx->resource.flash.queue.users < MAX_NUM_FLASH_BANK) {
          for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
            if (0 == pDevCtx->resource.flash.users[i]) {
              pDesc->actualIndex = i;
              return TRUE;
            }
          }
        }
        return FALSE;
      } else {
        pDesc->actualIndex = index;
        return (0 == pDevCtx->resource.flash.users[index]) ? TRUE : FALSE;
      }
    } else {
      pDesc->actualIndex = 0;
      return (0 == pDevCtx->resource.eeprom.users[0]) ? TRUE : FALSE;
    }
    
  default:
    /* Should never get here */
    dfAssert(FALSE);
    return FALSE;
  }
}

/*
** Caller must hold pDevCtx->resource.lock
**
** Returns TRUE if all resources for specified ticket are available
*/
static boolean_t
resourcesAvailable(
  Adb3CoreDeviceContext* pDevCtx,
  CoreTicket* pTicket)
{
  CoreResourceDesc* pDesc;
  unsigned int i, count;

  pDesc = &pTicket->resources[0];
  count = pTicket->count;
  for (i = 0; i < count; i++) {
    if (!resourceAvailable(pDevCtx, pDesc)) {
      return FALSE;
    }
    pDesc++;
  }
  return TRUE;
}

/* Caller must hold pDevCtx->resource.lock */
static void
takeResources(
  Adb3CoreDeviceContext* pDevCtx,
  CoreTicket* pTicket)
{
  CoreResourceDesc* pDesc;
  unsigned int i, count;
  uint8_t code, index;

  pDesc = &pTicket->resources[0];
  count = pTicket->count;
  for (i = 0; i < count; i++) {
    code = pDesc->code;
    index = pDesc->actualIndex;
    switch (code) {
    case CORE_RESOURCE_CLOCKGEN:
      pDevCtx->resource.clockGen.queue.users++;
      pDevCtx->resource.clockGen.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.clockGen.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_DMAENGINE:
      pDevCtx->resource.dmaEngine.queue.users++;
      pDevCtx->resource.dmaEngine.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.dmaEngine.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_EEPROM:
      pDevCtx->resource.eeprom.queue.users++;
      pDevCtx->resource.eeprom.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.eeprom.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_FLASH:
      pDevCtx->resource.flash.queue.users++;
      pDevCtx->resource.flash.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.flash.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_TARGETFPGA:
      pDevCtx->resource.targetFpga.queue.users++;
      pDevCtx->resource.targetFpga.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.targetFpga.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_SYSMON:
      pDevCtx->resource.sysMon.queue.users++;
      pDevCtx->resource.sysMon.users[index]++;
#if defined(DF_DBG_BUILD)
      pDevCtx->resource.sysMon.pCurrentTicket[index] = pTicket;
#endif
      break;

    case CORE_RESOURCE_VPD:
      if (pDevCtx->info.bootstrap.bVpdInFlash) {
        pDevCtx->resource.flash.queue.users++;
        pDevCtx->resource.flash.users[index]++;
#if defined(DF_DBG_BUILD)
        pDevCtx->resource.flash.pCurrentTicket[index] = pTicket;
#endif
      } else {
        pDevCtx->resource.eeprom.queue.users++;
        pDevCtx->resource.eeprom.users[index]++;
#if defined(DF_DBG_BUILD)
        pDevCtx->resource.eeprom.pCurrentTicket[index] = pTicket;
#endif
      }
      break;

    default:
      dfBugCheck(("invalid resource code 0x%lx", (unsigned long)code));
      break;
    }
    pDesc++;
  }
}

/* Caller must hold pDevCtx->resource.lock */
static void
queueTicketForResource(
  ResourceQueue* pQ,
  CoreResourceDesc* pDesc)
{
  pDesc->system.pPrev = pQ->pTail;
  pDesc->system.pNext = NULL;
  pDesc->system.pQueue = pQ;
  if (NULL == pQ->pTail) {
    pQ->pTail = pQ->pHead = pDesc;
  } else {
    pQ->pTail->system.pNext = pDesc;
    pQ->pTail = pDesc;
  }
  pQ->length++;
}

/* Caller must hold pDevCtx->resource.lock */
static void
queueTicketForResources(
  Adb3CoreDeviceContext* pDevCtx,
  CoreTicket* pTicket)
{
  CoreResourceDesc* pDesc;
  unsigned int i, count;
  uint8_t code, index;

  pDesc = &pTicket->resources[0];
  count = pTicket->count;
  for (i = 0; i < count; i++) {
    pDesc->system.pTicket = pTicket;
    code = pDesc->code;
    index = pDesc->index;
    switch (code) {
    case CORE_RESOURCE_CLOCKGEN:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = MAX_NUM_CLOCK_GENERATOR;
      }
      queueTicketForResource(&pDevCtx->resource.clockGen.queue, pDesc);
      break;

    case CORE_RESOURCE_DMAENGINE:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = MAX_NUM_DMA_ENGINE;
      }
      queueTicketForResource(&pDevCtx->resource.dmaEngine.queue, pDesc);
      break;

    case CORE_RESOURCE_EEPROM:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = 0;
      }
      queueTicketForResource(&pDevCtx->resource.eeprom.queue, pDesc);
      break;

    case CORE_RESOURCE_FLASH:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = 0;
      }
      queueTicketForResource(&pDevCtx->resource.flash.queue, pDesc);
      break;

    case CORE_RESOURCE_TARGETFPGA:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = MAX_NUM_TARGET_FPGA;
      }
      queueTicketForResource(&pDevCtx->resource.targetFpga.queue, pDesc);
      break;

    case CORE_RESOURCE_SYSMON:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = 0;
      }
      queueTicketForResource(&pDevCtx->resource.sysMon.queue, pDesc);
      break;

    case CORE_RESOURCE_VPD:
      if (index == CORE_RESOURCE_INDEX_ANY) {
        index = 0;
      }
      if (pDevCtx->info.bootstrap.bVpdInFlash) {
        queueTicketForResource(&pDevCtx->resource.flash.queue, pDesc);
      } else {
        queueTicketForResource(&pDevCtx->resource.eeprom.queue, pDesc);
      }
      break;

    default:
      dfBugCheck(("invalid resource code 0x%lx", (unsigned long)code));
      break;
    }
  }
}

/* Caller must hold pDevCtx->resource.lock */
static void
dequeueTicket(
  Adb3CoreDeviceContext* pDevCtx,
  CoreTicket* pTicket)
{
  unsigned int i, count;
  CoreResourceDesc* pDesc;
  ResourceQueue* pQ;

  count = pTicket->count;
  pDesc = &pTicket->resources[0];
  for (i = 0; i < count; i++) {
    pQ = pDesc->system.pQueue;
    if (pDesc->system.pNext == NULL) {
      pQ->pTail = pDesc->system.pPrev;
    } else {
      pDesc->system.pNext->system.pPrev = pDesc->system.pPrev;
    }
    if (pDesc->system.pPrev == NULL) {
      pQ->pHead = pDesc->system.pNext;
    } else {
      pDesc->system.pPrev->system.pNext = pDesc->system.pNext;
    }
    pQ->length--;
    pDesc++;
  }
}

/* Caller must NOT hold pDevCtx->resource.lock */
static CoreTicket*
releaseAndCheckResourceQueue(
  Adb3CoreDeviceContext* pDevCtx,
  ResourceQueue* pQ,
  unsigned int* pUsers,
#if defined(DF_DBG_BUILD)
  CoreTicket** pCurrentTicket,
#endif
  unsigned int index)
{
  DfSpinLockFlags f;
  CoreTicket* pTicket = NULL;
  CoreResourceDesc* pDesc;

  f = dfSpinLockGet(&pDevCtx->resource.lock);
  pQ->users--;
  pUsers[index]--;
#if defined(DF_DBG_BUILD)
  pCurrentTicket[index] = NULL;
#endif
  if (0 == pUsers[index]) {
    pDesc = pQ->pHead;
    while (NULL != pDesc) {
      pTicket = pDesc->system.pTicket;
      if (resourcesAvailable(pDevCtx, pTicket)) {
        pTicket->system.state = StateAcquired;
        dequeueTicket(pDevCtx, pTicket);
        takeResources(pDevCtx, pTicket);
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        return pTicket;
      }
      pDesc = pDesc->system.pNext;
    }
  }
  dfSpinLockPut(&pDevCtx->resource.lock, f);

  return NULL;
}

/* Caller must NOT hold pDevCtx->resource.lock */
static CoreTicket*
releaseAndCheckResource(
  Adb3CoreDeviceContext* pDevCtx,
  uint32_t code,
  uint32_t index)
{
  ResourceQueue* pQ;
  unsigned int* pUsers;
#if defined(DF_DBG_BUILD)
  CoreTicket** pCurrentTicket;
#endif

  switch (code) {
  case CORE_RESOURCE_CLOCKGEN:
    pQ = &pDevCtx->resource.clockGen.queue;
    pUsers = pDevCtx->resource.clockGen.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.clockGen.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_DMAENGINE:
    pQ = &pDevCtx->resource.dmaEngine.queue;
    pUsers = pDevCtx->resource.dmaEngine.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.dmaEngine.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_EEPROM:
    pQ = &pDevCtx->resource.eeprom.queue;
    pUsers = pDevCtx->resource.eeprom.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.eeprom.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_FLASH:
    pQ = &pDevCtx->resource.flash.queue;
    pUsers = pDevCtx->resource.flash.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.flash.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_TARGETFPGA:
    pQ = &pDevCtx->resource.targetFpga.queue;
    pUsers = pDevCtx->resource.targetFpga.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.targetFpga.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_SYSMON:
    pQ = &pDevCtx->resource.sysMon.queue;
    pUsers = pDevCtx->resource.sysMon.users;
#if defined(DF_DBG_BUILD)
    pCurrentTicket = pDevCtx->resource.sysMon.pCurrentTicket;
#endif
    break;

  case CORE_RESOURCE_VPD:
    if (pDevCtx->info.bootstrap.bVpdInFlash) {
      pQ = &pDevCtx->resource.flash.queue;
      pUsers = pDevCtx->resource.flash.users;
#if defined(DF_DBG_BUILD)
      pCurrentTicket = pDevCtx->resource.flash.pCurrentTicket;
#endif
    } else {
      pQ = &pDevCtx->resource.eeprom.queue;
      pUsers = pDevCtx->resource.eeprom.users;
#if defined(DF_DBG_BUILD)
      pCurrentTicket = pDevCtx->resource.eeprom.pCurrentTicket;
#endif
    }
    break;

  default:
    dfBugCheck(("releaseAndCheckResource: invalid resource code 0x%lx", (unsigned long)code));
    return NULL;
  }

#if defined(DF_DBG_BUILD)
  return releaseAndCheckResourceQueue(pDevCtx, pQ, pUsers, pCurrentTicket, index);
#else
  return releaseAndCheckResourceQueue(pDevCtx, pQ, pUsers, index);
#endif
}

/*
** ======================================================================================
** Exported in Core Interface
** ======================================================================================
*/

void
coreAcquireAsync(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  uint32_t flags,
  CoreAcquireCallback* pCallback)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfSpinLockFlags f;

  pTicket->system.bSync = FALSE;
  pTicket->system.variant.pCallback = pCallback;

  f = dfSpinLockGet(&pDevCtx->resource.lock);
  if (pTicket->system.bCancel) {
    pTicket->system.state = StateCancelled;
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    pCallback(pDevCtx, pTicket, CoreAcquireCancelled);
  } else {
    if (resourcesAvailable(pDevCtx, pTicket)) {
      /* This ticket can proceed */
      pTicket->system.state = StateAcquired;
      takeResources(pDevCtx, pTicket);
      dfSpinLockPut(&pDevCtx->resource.lock, f);
      pCallback(pDevCtx, pTicket, CoreAcquireSuccess);
      return;
    } else {
      if (flags & CORE_ACQUIRE_NOWAIT) {
        /* This ticket must fail because it cannot proceed but NOWAIT was specificed */
        pTicket->system.state = StateFailed;
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        pCallback(pDevCtx, pTicket, CoreAcquireMustWait);
        return;
      } else {
        /* This ticket cannot yet proceed */
        pTicket->system.state = StateQueued;
        queueTicketForResources(pDevCtx, pTicket);
        dfSpinLockPut(&pDevCtx->resource.lock, f);
      }
    }
  }
}

CoreAcquireStatus
coreAcquireAsyncPoll(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  uint32_t flags,
  CoreAcquireCallback* pCallback)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfSpinLockFlags f;

  pTicket->system.bSync = FALSE;
  pTicket->system.variant.pCallback = pCallback;

  f = dfSpinLockGet(&pDevCtx->resource.lock);
  if (pTicket->system.bCancel) {
    pTicket->system.state = StateCancelled;
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    return CoreAcquireCancelled;
  } else {
    if (resourcesAvailable(pDevCtx, pTicket)) {
      /* This ticket can proceed */
      pTicket->system.state = StateAcquired;
      takeResources(pDevCtx, pTicket);
      dfSpinLockPut(&pDevCtx->resource.lock, f);
      return CoreAcquireSuccess;
    } else {
      if (flags & CORE_ACQUIRE_NOWAIT) {
        /* This ticket must fail because it cannot proceed but NOWAIT was specificed */
        pTicket->system.state = StateFailed;
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        return CoreAcquireMustWait;
      } else {
        /* This ticket cannot yet proceed */
        pTicket->system.state = StateQueued;
        queueTicketForResources(pDevCtx, pTicket);
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        return CoreAcquireWaiting;
      }
    }
  }
}

/* Returns TRUE if acquisition was cancellable at time of call */
boolean_t
coreAcquireCancel(
  void* pInterfaceContext,
  CoreTicket* pTicket)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pDevCtx->resource.lock);
  switch (pTicket->system.state) {
  case StateInitial:
    pTicket->system.bCancel = TRUE;
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    return FALSE;

  case StateQueued:
    pTicket->system.state = StateCancelled;
    dequeueTicket(pDevCtx, pTicket);
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    if (pTicket->system.bSync) {
      dfEventSignal(pTicket->system.variant.pEvent);
    } else {
      pTicket->system.variant.pCallback(pDevCtx, pTicket, CoreAcquireCancelled);
    }
    return TRUE;

  case StateAcquired:
  case StateFailed:
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    return FALSE;

  case StateCancelled:
  default:
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    dfAssert(FALSE);
    return FALSE;
  }
}

CoreAcquireStatus
coreAcquireInitialize(
  void* pInterfaceContext,
  CoreTicket* pTicket)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  unsigned int i, count;
  CoreAcquireStatus status;

  pTicket->system.bCancel = FALSE;
#if DF_DBG_BUILD
  pTicket->system.bResourcesValid = FALSE;
#endif
  pTicket->system.state = StateInitial;
  count = pTicket->count;
  for (i = 0; i < count; i++) {
    status = resourceValid(pDevCtx, pTicket->resources[i].code, pTicket->resources[i].index);
    if (status != CoreAcquireSuccess) {
      return status;
    }
  }
#if DF_DBG_BUILD
  pTicket->system.bResourcesValid = TRUE;
#endif
  return CoreAcquireSuccess;
}

void
coreRelease(
  void* pInterfaceContext,
  CoreTicket* pTicket)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  CoreTicket* pNextTicket;
  unsigned int i, count;
  CoreResourceDesc* pDesc;

  dfAssert(pTicket->system.state == StateAcquired);

  pDesc = &pTicket->resources[0];
  count = pTicket->count;
  for (i = 0; i < count; i++) {
    pNextTicket = releaseAndCheckResource(pDevCtx, pDesc->code, pDesc->actualIndex);
    if (NULL != pNextTicket) {
      if (pNextTicket->system.bSync) {
        dfEventSignal(pNextTicket->system.variant.pEvent);
      } else {
        pNextTicket->system.variant.pCallback(pDevCtx, pNextTicket, CoreAcquireSuccess);
      }
    }
    pDesc++;
  }
}

CoreAcquireStatus
coreAcquireSync(
  void* pInterfaceContext,
  CoreTicket* pTicket,
  uint32_t flags)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfSpinLockFlags f;
  DfEvent ev;
  CoreAcquireStatus status = CoreAcquireSuccess;
  boolean_t bInterrupted;

  pTicket->system.bSync = TRUE;
  dfEventInit(&ev);
  pTicket->system.variant.pEvent = &ev;

  f = dfSpinLockGet(&pDevCtx->resource.lock);
  if (pTicket->system.bCancel) {
    pTicket->system.state = StateCancelled;
    dfSpinLockPut(&pDevCtx->resource.lock, f);
    status = CoreAcquireCancelled;
  } else {
    if (resourcesAvailable(pDevCtx, pTicket)) {
      /* This ticket can proceed */
      pTicket->system.state = StateAcquired;
      takeResources(pDevCtx, pTicket);
      dfSpinLockPut(&pDevCtx->resource.lock, f);
    } else {
      if (flags & CORE_ACQUIRE_NOWAIT) {
        /* This ticket must fail because it cannot proceed but NOWAIT was specificed */
        pTicket->system.state = StateFailed;
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        status = CoreAcquireMustWait;
      } else {
        /* This ticket cannot yet proceed */
        pTicket->system.state = StateQueued;
        queueTicketForResources(pDevCtx, pTicket);
        dfSpinLockPut(&pDevCtx->resource.lock, f);
        if (flags & CORE_ACQUIRE_SYNC_UNINTERRUPTIBLE) {
          bInterrupted = FALSE;
          dfEventWait(&ev);
        } else {
          bInterrupted = dfEventWaitInterruptible(&ev);
        }
        if (bInterrupted) {
          f = dfSpinLockGet(&pDevCtx->resource.lock);
          dfAssert (pTicket->system.state == StateAcquired || pTicket->system.state == StateQueued);
          pTicket->system.state = StateCancelled;
          if (pTicket->system.state == StateAcquired) {
            dfSpinLockPut(&pDevCtx->resource.lock, f);
            coreRelease(pDevCtx, pTicket);
            status = CoreAcquireCancelled;
          } else {
            dequeueTicket(pDevCtx, pTicket);
            dfSpinLockPut(&pDevCtx->resource.lock, f);
            status = CoreAcquireCancelled;
          }
        } else {
          /* Wait completed */
        }
      }
    }
  }
  dfEventUninit(&ev);
  return status;
}

#if defined(DF_DBG_BUILD)

boolean_t
checkTicketValid(
  Adb3CoreDeviceContext* pDevCtx,
  CoreTicket* pTicket,
  uint8_t code,
  unsigned int index)
{
  boolean_t bValid = FALSE;

  if (pTicket->system.state != StateAcquired) {
    return FALSE;
  }

  switch (code) {
  case CORE_RESOURCE_CLOCKGEN:
    if (index >= MAX_NUM_CLOCK_GENERATOR) {
      return FALSE;
    }
    if (pDevCtx->resource.clockGen.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_DMAENGINE:
    if (index >= MAX_NUM_DMA_ENGINE) {
      return FALSE;
    }
    if (pDevCtx->resource.dmaEngine.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_EEPROM:
    if (index >= 1) {
      return FALSE;
    }
    if (pDevCtx->resource.eeprom.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_FLASH:
    if (index >= MAX_NUM_FLASH_BANK) {
      return FALSE;
    }
    if (pDevCtx->resource.flash.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_TARGETFPGA:
    if (index >= MAX_NUM_TARGET_FPGA) {
      return FALSE;
    }
    if (pDevCtx->resource.targetFpga.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_SYSMON:
    if (index >= 1) {
      return FALSE;
    }
    if (pDevCtx->resource.sysMon.pCurrentTicket[index] == pTicket) {
      bValid = TRUE;
    }
    break;

  case CORE_RESOURCE_VPD:
    if (index >= 1) {
      return FALSE;
    }
    if (pDevCtx->info.bootstrap.bVpdInFlash) {
      if (pDevCtx->resource.flash.pCurrentTicket[index] == pTicket) {
        bValid = TRUE;
      }
    } else {
      if (pDevCtx->resource.eeprom.pCurrentTicket[index] == pTicket) {
        bValid = TRUE;
      }
    }
    break;

  default:
    dfBugCheck(("checkTicketValid: invalid resource code 0x%lx", (unsigned long)code));
    return FALSE;
  }

  return bValid;
}

#endif
