/*
** File: model_admxrc7z1.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADM-XRC-7Z1/-7Z2, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admxrc7z1.h"
#include "coreif.h"
#include "coredma.h"
#include "device.h"
#include "model_boot.h"

static boolean_t
createFakeVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxrc7z1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc7z1.rev0;

  DF_ZERO_OBJECT(pVpd);
  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  /* VPD access not implemented yet */
  return FALSE;
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  /* TO DO - implement this when VPD is implemented in -7Z1 ADB3 core */
  return FpgaControlInvalidIndex;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  /* TO DO - implement this when VPD is implemented in -7Z1 ADB3 core */
  return FpgaSelectMapInvalidIndex;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMXRC7Z2) {
        pDevCtx->info.model = CoreModelAdmxrc7z2;
      } else {
        pDevCtx->info.model = CoreModelAdmxrc7z1;
      }

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc7z1.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7z1.shadow.intCtl | ADMXRC7Z1_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Initialize information needed for production test */
      if (NULL != pDevCtx->info.vpd.pData) {
        /* Read in VPD */
        if (!readVpd(pDevCtx)) {
          createFakeVpd(pDevCtx);
        }

        pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admxrc7z1.rev0.serial;
        pDevCtx->info.prodTest.cpldRevision = pDevCtx->info.vpd.pData->admxrc7z1.rev0.cpldRev;
        pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admxrc7z1.rev0.pcbRev;
      }
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;
      break;

    case 1:
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admxrc7z1.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Nothing to do yet */
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc7z1.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7z1.shadow.intCtl | ADMXRC7Z1_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);
    }
  }
}

static CoreVpdStatus
readWriteVpd(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bWrite,
	size_t address,
	size_t length,
	void* pData)
{
  /* TO DO - implement VPD read/write when ADB3 core VPD functionality exists */
  return CoreVpdGeneralFailure;
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

static void
bootstrap(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfAssert(DF_OFFSET_OF(VpdAdmxrc7z1Rev0, checksum) == sizeof(VpdAdmxrc7z1Rev0) - 1);

  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;

  pDevCtx->info.bootstrap.numClockGen = 0U;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXRC7Z1_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 0U;
  pDevCtx->info.bootstrap.numSensor = 0U;
  pDevCtx->info.bootstrap.numTargetFpga = 1U;
  pDevCtx->info.bootstrap.numWindow = 4U;
  pDevCtx->info.bootstrap.numFlashBank = 0U;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
#if 1
  pDevCtx->info.bootstrap.vpdLength = ADMXRC7Z1_AVR_FLASH_SIZE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc7z1Rev0);
#else
  pDevCtx->info.bootstrap.vpdLength = 0;
  pDevCtx->info.bootstrap.vpdOffset = 0;
#endif
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}

void
adb3CoreBootstrapAdmxrc7z1(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc7z1: entered\n"));
  bootstrap(pDevCtx);
}

void
adb3CoreBootstrapAdmxrc7z2(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc7z2: entered\n"));
  bootstrap(pDevCtx);
}
