/*
** File: model_admxrc6tl.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-6TL-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2011
**
** NOTES
**
** 1. Organization of Flash device U32 (all offsets in bytes):
**
**  0x0000000 - 0x07FFFFF  Alternate ADB3 PCI-E to OCP Bridge bitstream
**  0x0800000 - 0x0FFFFFF  Default ADB3 PCI-E to OCP Bridge bitstream
**  0x1000000 - 0x10FFFFF  Alpha Data VPD region
**  0x1100000 - 0x11FFFFF  Customer VPD region
**  0x1200000 - 0x28FFFFF  Target FPGA bitstream
**  0x2900000 - 0x3FFFFFF  Failsafe target FPGA bitstream
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admxrc6tl.h"
#include "admxrc6tx_common.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"
#include "v5clksynth_common.h"

static boolean_t
readVpd(
	Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxrc6tlRev1* pVpd = &pDevCtx->info.vpd.pData->admxrc6tl.rev1;
  union {
    VpdAdmxrc6tlRev0 rev0;
    VpdAdmxrc6tlRev1 rev1;
  } vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset + 2, sizeof(vpdLength), &vpdLength);
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw.rev0), &vpdRaw.rev0);
    break;

  case 1:
    flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw.rev1), &vpdRaw.rev1);
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    /* Assume rev1 */
    flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw.rev1), &vpdRaw.rev1);
    break;
  }

  adb3VerifyChecksum(pDevCtx, vpdLength);

  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->length = vpdLength;
  pVpd->serial = dfLe32ToCpu(vpdRaw.rev0.serial);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.rev0.modifications);
  pVpd->lclkRef = dfLe32ToCpu(vpdRaw.rev0.lclkRef);
  for (i = 0; i < 4; i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.rev0.sdram[i].order);
    pVpd->sdram[i].speed = dfLe16ToCpu(vpdRaw.rev0.sdram[i].speed);
    pVpd->sdram[i].link = dfLe16ToCpu(vpdRaw.rev0.sdram[i].link);
  }
  pVpd->fpga.device = dfLe16ToCpu(vpdRaw.rev0.fpga.device);
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->fpga.speedGrade), (const char*)vpdRaw.rev0.fpga.speedGrade);
  dfStrCpy((char*)pVpd->fpga.stepping, DF_ARRAY_LENGTH(pVpd->fpga.stepping), (const char*)vpdRaw.rev0.fpga.stepping);
  pVpd->fpga.tempGrade = dfLe8ToCpu(vpdRaw.rev0.fpga.tempGrade);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.rev0.pcbRev);
  pVpd->cpldRev = dfLe8ToCpu(vpdRaw.rev0.cpldRev);
  pVpd->checksum = dfLe8ToCpu(vpdRaw.rev0.checksum);
  if (vpdVersion >= 1) {
    pVpd->mgtRef = dfLe32ToCpu(vpdRaw.rev1.mgtRef);
  } else {
    pVpd->mgtRef = 125000000U;
  }
  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.admxrc6tl.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tl.shadow.intCtl | ADMXRC6TL_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static void
onV5ClkSynthProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V5ClkSynthDeviceContext* pV5ClkSynthCtx,
  V5ClkSynthProgramRequest* pRequest,
  V5ClkSynthProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, v5ClkSynthMapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return v5ClkSynthMapProgramStatus(v5ClkSynthProgram(pDevCtx, &pDevCtx->model.admxrc6tl.v5ClkSynth.context, &pDevCtx->model.admxrc6tl.v5ClkSynth.programming, pClockWord, onV5ClkSynthProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
	Adb3CoreDeviceContext* pDevCtx,
	unsigned int clockIndex,
  uint32_t flags,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
	CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return v5ClkSynthMapClockWordStatus(v5ClkSynthClockWord(flags, pDevCtx->model.admxrc6tl.v5ClkSynth.refClkFrequency, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;
  boolean_t bOk;

  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    bOk = TRUE;
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    if (!(val32 & 0x4U)) {
      dfDebugPrint(9, ("fpgaControl: fpgaStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    bOk = TRUE;
    /* Check that MPTL is up */
    val32 = dfPciMemRead32(hModel, &pModelRegs->mptlStat);
    if ((val32 & 0x7U) != 0x7U) {
      dfDebugPrint(9, ("fpgaControl: mptlStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    *pbValue = (val32 & 0x2U) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.admxrc6tl.shadow.fpgaCtl;
    if (*pbValue) {
      val32 |= 0x1U;
    } else {
      val32 &= ~0x1U;
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, val32);
    dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

	if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
			val8 = *pBuffer8++;
			dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
    }

    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
			dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->selectMap[0], val32);
      length32 -= 4;
    }

    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
		while (length--) {
			val8 = *pBuffer8++;
			dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
		}

		dfPciMemRead32(hModel, &pModelRegs->fpgaCtl); /* Ensure above writes have propagated to destination */
	} else {
		while (length--) {
			val8 = dfPciMemRead8(hModel, &pModelRegs->selectMap[0]);
			*pBuffer8++ = val8;
		}
	}

  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;
  uint32_t fOut;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxrc6tl;

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc6tl.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tl.shadow.intCtl | ADMXRC6TL_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Initialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.admxrc6tl.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Initialize FPGACTL register for 8-bit SelectMap port */
      pDevCtx->model.admxrc6tl.shadow.fpgaCtl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.admxrc6tl.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Identify Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0x1200000U;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize I2C stuff */
      i2cInit(&pDevCtx->model.admxrc6tl.i2c.context, NULL, admxrc6txI2cStart, admxrc6txI2cPoll, pDevCtx);

      /* Identify any XRM module fitted */
      admxrc6txI2cXrmIdentify(pDevCtx, &pDevCtx->model.admxrc6tl.i2c.context);

      /* Divide LCLK reference clock by 2 because on the ADM-XRC-6TL, a CLKDIV value of 2 is used in the DCMs that generate LCLK */
      pDevCtx->model.admxrc6tl.v5ClkSynth.refClkFrequency = pDevCtx->info.vpd.pData->admxrc6tl.rev1.lclkRef / 2;
      /* Initialize V5 clock synthesizer stuff */
      if (!v5ClkSynthInit(pDevCtx, &pDevCtx->model.admxrc6tl.v5ClkSynth.context, pDevCtx->model.admxrc6tl.v5ClkSynth.refClkFrequency, &fOut)) {
        dfDebugPrint(0, ("*** initHardware: failed to initialize V5 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admxrc6tl.rev1.serial;
      pDevCtx->info.prodTest.cpldRevision = pDevCtx->info.vpd.pData->admxrc6tl.rev1.cpldRev;
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admxrc6tl.rev1.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].localBase = 0;
      pDevCtx->info.window[0].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].localBase = 0;
      pDevCtx->info.window[1].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[3].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[3].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[3].localBase = 0;
      pDevCtx->info.window[3].localSize = 0;
      pDevCtx->info.window[3].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[3].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[3].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[3].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;

      /* Initialize sensor information structures */
      i = 0;
      i += admxrc6txInitSensorInfoLm87(pDevCtx, i);
      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision >= 0x04U) {
        i += admxrc6txInitSensorInfoXsmon(pDevCtx, i, FALSE);
      }
      if (pDevCtx->info.bootstrap.bIncludeDiagSensors) {
        i += admxrc6txInitSensorInfoDiag(pDevCtx, i);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize V5 clock synthesizer stuff */
      v5ClkSynthUninit(pDevCtx, &pDevCtx->model.admxrc6tl.v5ClkSynth.context);

      /* Uninitialize I2C stuff */
      i2cUninit(&pDevCtx->model.admxrc6tl.i2c.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admxrc6tl.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
      intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.admxrc6tl.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADMXRC6TL_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.admxrc6tl.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
		if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
		}
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  V5ClkSynthProgramStatus v5ClkSynthStatus;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save V5 clock synthesizer state */
      v5ClkSynthSaveState(pDevCtx, &pDevCtx->model.admxrc6tl.v5ClkSynth.context);
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc6tl.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tl.shadow.intCtl | ADMXRC6TL_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Reinitialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Restore FPGACTL register for 8-bit SelectMap port */
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.admxrc6tl.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Reset Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);

      /* Restore V5 clock synthesizer state */
      v5ClkSynthStatus = v5ClkSynthRestoreState(pDevCtx, &pDevCtx->model.admxrc6tl.v5ClkSynth.context);
      if (V5ClkSynthProgramStatusSuccess != v5ClkSynthStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore V5 clock synthesizer state, status=%u\n", v5ClkSynthStatus));
      }
    }
  }
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

void
adb3CoreBootstrapAdmxrc6tl(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc6tl: entered\n"));

  dfAssert(DF_OFFSET_OF(VpdAdmxrc6tlRev0, checksum) == sizeof(VpdAdmxrc6tlRev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = admxrc6txFlashSingleBank;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = admxrc6txReadSensor;
  pDevCtx->methods.pReadWriteVpd = admxrc6txReadWriteVpd;
  pDevCtx->methods.pSyncVpd = admxrc6txSyncVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb3WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 1;
  pDevCtx->info.bootstrap.numDmaChannel = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMXRC6TL_FX50T) ? 2 : 1;
  pDevCtx->info.bootstrap.numSensor = ADMXRC6TX_NUM_LM87_SENSOR;
  if (pDevCtx->pDevObj->hardwareId.variant.pci.revision >= 0x04U) {
    /* Revision 1.4 firmware has Xilinx system monitor sensor registers */
    pDevCtx->info.bootstrap.numSensor += ADMXRC6TX_NUM_XSMON_SENSOR;
  }
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "IncludeDiagSensors", &pDevCtx->info.bootstrap.bIncludeDiagSensors);
  if (pDevCtx->info.bootstrap.bIncludeDiagSensors) {
    pDevCtx->info.bootstrap.numSensor += g_admxrc6txNumDiagSensor;
  }
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = 0x200000U;
  pDevCtx->info.bootstrap.vpdOffset = 0x1000000U;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc6tlRev1);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = (pDevCtx->pDevObj->hardwareId.variant.pci.revision <= 0x01U) ? TRUE : FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}
