/*
** File: model_admxrc5tda1.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-5T-DA1-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "adb1_common.h"
#include "adb1v4v5_common.h"
#include "admxrc5tda1.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics843034_01_common.h"
#include "model_boot.h"
#include "v4clksynth_common.h"

static struct {
  unsigned int regOffset; /* LM87 register offset */
  boolean_t bSigned;      /* TRUE => register value is int8_t; otherwise uint8_t */
  int32_t factor;         /* Fixed point, where 0x00100000 => 1, 0x00080000 => 0.5, 0xFFF80000 => -0.5 etc. */
} g_lm87Scaling[] = {
  { 0x21U, FALSE, 14785,   /* 0.0141 */ },
  { 0x28U, FALSE, 10276,   /* 0.0098 */ },
  { 0x29U, FALSE, 10276,   /* 0.0098 */ },
  { 0x25U, FALSE, 14785,   /* 0.0141 */ },
  { 0x20U, FALSE, 13631    /* 0.0130 */ },
  { 0x22U, FALSE, 18036,   /* 0.0172 */ },
  { 0x23U, FALSE, 27263,   /* 0.0260 */ },
  { 0x24U, FALSE, 65536,   /* 0.0625 */ },
  { 0x27U, TRUE,  1048576, /* 1.0000 */ },
  { 0x26U, TRUE,  1048576, /* 1.0000 */ }
};

static CoreSensorInfo g_lm87SensorInfo[] = {
  /* LM87 sensors */
  { "1V supply rail",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x21, Vccp1 reading */
  { "1.2V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x28, FAN1/AIN1 reading */
  { "1.8V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x29, FAN2/AIN2 reading */
  { "2.1V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x25, Vccp2 reading */
  { "2.5V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x20, +2.5V / ext temp. 2 reading */
  { "3.3V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x22, +Vcc reading */
  { "5V supply rail",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x23, 5V reading */
  { "Rear panel I/O power rail",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x24, +12V reading */
  { "LM87 internal temperature",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* LM87 register 0x27, Int temp reading */
  { "Target FPGA ext. temp.",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* LM87 register 0x26, Ext temp. 1 reading */
};

static unsigned int
initSensorInfo(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex)
{
  unsigned int i, n = 0;

  dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_lm87SensorInfo) <= MAX_NUM_SENSOR);
  for (i = 0; i < DF_ARRAY_LENGTH(g_lm87SensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
    pDevCtx->info.sensor[sensorIndex] = g_lm87SensorInfo[i];
    n++;
  }
  return n; 
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int lm87SensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
  int16_t val16;
  int32_t factor, value;

  dfAssert(DF_ARRAY_LENGTH(g_lm87SensorInfo) == DF_ARRAY_LENGTH(g_lm87Scaling));

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  if (lm87SensorIndex < DF_ARRAY_LENGTH(g_lm87SensorInfo)) {
    /* LM87 sensors */
    if (g_lm87Scaling[lm87SensorIndex].bSigned) {
      val16 = (int16_t)(int8_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    } else {
      val16 = (int16_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    }
    factor = g_lm87Scaling[lm87SensorIndex].factor;
    value = (val16 * factor) / 16;
    pReading->doubleValue.intPart = (int32_t)(int16_t)((value >> 16) & 0xFFFFU);
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }

  return CoreSensorInvalidIndex;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxrc5tda1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5tda1.rev0;
  VpdAdmxrc5tda1Rev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->boardType = dfLe16ToCpu(vpdRaw.boardType);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->stepping = dfLe8ToCpu(vpdRaw.stepping);
  pVpd->tempGrade = dfLe8ToCpu(vpdRaw.tempGrade);
  pVpd->scd = dfLe32ToCpu(vpdRaw.scd);
  pVpd->lclkRef = dfLe32ToCpu(vpdRaw.lclkRef);
  pVpd->mclkARef = dfLe32ToCpu(vpdRaw.mclkARef);
  pVpd->mclkBRef = dfLe32ToCpu(vpdRaw.mclkBRef);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
  }
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.ssram); i++) {
    pVpd->ssram[i].order = dfLe8ToCpu(vpdRaw.ssram[i].order);
  }
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->cpldRev = (uint8_t)(0x10U + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onV4ClkSynthProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  V4ClkSynthProgramRequest* pRequest,
  V4ClkSynthProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, v4ClkSynthMapProgramStatus(status));
}

static void
onIcs843034_01ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics843034_01DeviceContext* pV4ClkSynthCtx,
  Ics843034_01ProgramRequest* pRequest,
  Ics843034_01ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 1, ics843034_01MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return v4ClkSynthMapProgramStatus(v4ClkSynthProgram(pDevCtx, &pDevCtx->model.admxrc5tda1.v4ClkSynth.context, &pDevCtx->model.admxrc5tda1.v4ClkSynth.programming, pClockWord, onV4ClkSynthProgramDone));

  case 1:
    return ics843034_01MapProgramStatus(ics843034_01Program(pDevCtx, &pDevCtx->model.admxrc5tda1.ics843034_01.context, &pDevCtx->model.admxrc5tda1.ics843034_01.programming, pClockWord, onIcs843034_01ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdmxrc5tda1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5tda1.rev0;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return v4ClkSynthMapClockWordStatus(v4ClkSynthClockWord(flags, pVpd->lclkRef, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  case 1:
    return ics843034_01MapClockWordStatus(clockWordIcs843034_01(flags, pVpd->mclkARef, pVpd->mclkARef, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}


static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  return adb1V4V5FpgaControl(pDevCtx, index, opCode, &pDevCtx->model.admxrc5tda1.shadow.fpctl, pbValue);
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdmxrc5tda1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5tda1.rev0;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxrc5tda1;

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.admxrc5tda1.shadow.ictl = ADB1V4V5_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc5tda1.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.admxrc5tda1.shadow.fpctl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl, pDevCtx->model.admxrc5tda1.shadow.fpctl);

      /* Reset Flash and identify it */
      adb1V4V5FlashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADMXRC5TDA1_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADMXRC5TDA1_FLASH_TARGET_AREA_START;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize clock synthesizer stuff */
      if (!v4ClkSynthInit(pDevCtx, &pDevCtx->model.admxrc5tda1.v4ClkSynth.context, pVpd->lclkRef, &fOut)) {
        dfDebugPrint(0, ("*** initHardware: failed to initialize V5 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Initialize ICS843034-01 synthesizer stuff */
      if (!ics843034_01Init(
        pDevCtx,
        &pDevCtx->model.admxrc5tda1.ics843034_01.context,
        pVpd->mclkARef,
        pVpd->mclkBRef,
        &pDevCtx->hardware.model.pAdb1V4V5->clocking.others.mclkctl,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS843034-01 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[1].currentFrequency = fOut;
      }

      /* Initialize information about sensors */
      (void)initSensorInfo(pDevCtx, 0U);

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdb1V4V5;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS843034-01 stuff */
      ics843034_01Uninit(pDevCtx, &pDevCtx->model.admxrc5tda1.ics843034_01.context);

      /* Deinitialize clock synthesizer stuff */
      v4ClkSynthUninit(pDevCtx, &pDevCtx->model.admxrc5tda1.v4ClkSynth.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;

  return adb1V4V5HandleInterrupt(pDevCtx, pDevCtx->hardware.shadow.bridge.adb1.pictl, pDevCtx->model.admxrc5tda1.shadow.ictl);
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Ics843034_01ProgramStatus ics843034_01Status;
  V4ClkSynthProgramStatus v4ClkSynthStatus;
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save V4 clock synthesizer state */
      v4ClkSynthSaveState(pDevCtx, &pDevCtx->model.admxrc5tda1.v4ClkSynth.context);

      /* Save ICS843034-01 state */
      ics843034_01SaveState(pDevCtx, &pDevCtx->model.admxrc5tda1.ics843034_01.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      adb1V4V5FlashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc5tda1.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS843034-01 state */
      ics843034_01Status = ics843034_01RestoreState(pDevCtx, &pDevCtx->model.admxrc5tda1.ics843034_01.context);
      if (Ics843034_01ProgramStatusSuccess != ics843034_01Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS843034-01 state, status=%u\n", ics843034_01Status));
      }

      /* Restore V4 clock synthesizer state */
      v4ClkSynthStatus = v4ClkSynthRestoreState(pDevCtx, &pDevCtx->model.admxrc5tda1.v4ClkSynth.context);
      if (V4ClkSynthProgramStatusSuccess != v4ClkSynthStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore V5 clock synthesizer state, status=%u\n", v4ClkSynthStatus));
      }
    }
  }
}

void
adb3CoreBootstrapAdmxrc5tda1(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc5tda1: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = adb1V4V5Flash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = adb1V4V5FpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = adb1V4V5MapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = adb1V4V5ReadWriteVpd;
  pDevCtx->methods.pSyncVpd = adb1V4V5SyncVpd;
  pDevCtx->methods.pValidateBusResources = adb1V4V5ValidateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXRC5TDA1_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 0;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADMXRC5TDA1_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADMXRC5TDA1_VPD_OFFSET_IN_FLASH;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc5tda1Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
