/*
** File: model_adcbbp.c  
** Project: ADB3 core driver
** Purpose: Code, including function to bootstrap device context, specific to ADC-BBP.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include "adb1_common.h"
#include "adb1v4v5_common.h"
#include "adcbbp.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "model_boot.h"
#include "v4clksynth_common.h"

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdcbbpRev0* pVpd = &pDevCtx->info.vpd.pData->adcbbp.rev0;
  VpdAdcbbpRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.stepping); i++) {
    pVpd->stepping[i] = dfLe8ToCpu(vpdRaw.stepping[i]);
  }
  pVpd->tempGrade = dfLe8ToCpu(vpdRaw.tempGrade);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.stepping); i++) {
    pVpd->scd[i] = dfLe32ToCpu(vpdRaw.scd[i]);
  }
  pVpd->lclkRef = dfLe32ToCpu(vpdRaw.lclkRef);
  pVpd->mclkAFreq = dfLe32ToCpu(vpdRaw.mclkAFreq);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.zbt0); i++) {
    pVpd->zbt0[i].order = dfLe8ToCpu(vpdRaw.zbt0[i].order);
    pVpd->zbt0[i].type = dfLe8ToCpu(vpdRaw.zbt0[i].type);
  }
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.zbt1); i++) {
    pVpd->zbt1[i].order = dfLe8ToCpu(vpdRaw.zbt1[i].order);
    pVpd->zbt1[i].type = dfLe8ToCpu(vpdRaw.zbt1[i].type);
  }
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.fpgaType); i++) {
    pVpd->fpgaType[i] = dfLe32ToCpu(vpdRaw.fpgaType[i]);
  }
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->logicRev = (uint8_t)(0x10U + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onV4ClkSynthProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V4ClkSynthDeviceContext* pV4ClkSynthCtx,
  V4ClkSynthProgramRequest* pRequest,
  V4ClkSynthProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, v4ClkSynthMapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return v4ClkSynthMapProgramStatus(v4ClkSynthProgram(pDevCtx, &pDevCtx->model.adcbbp.v4ClkSynth.context, &pDevCtx->model.adcbbp.v4ClkSynth.programming, pClockWord, onV4ClkSynthProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdcbbpRev0* pVpd = &pDevCtx->info.vpd.pData->adcbbp.rev0;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return v4ClkSynthMapClockWordStatus(v4ClkSynthClockWord(flags, pVpd->lclkRef, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static void
flashReset(
  Adb3CoreDeviceContext* pDevCtx)
{
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl;

  dfDebugPrint(9, ("flashReset: entered\n"));

  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_rst_n, assert fl_rst_n */
	flctl = (0x1U << 27) | (0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Deassert fl_rst_n */
	flctl &= ~(0x1U << 31);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	/* Stop driving fl_rst_n */
	flctl &= ~(0x1U << 27);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
	(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl, flpage;

  dfDebugPrint(9, ("flash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x4000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

	flpage = (uint32_t)location & 0x3fe0000U;
	dfPciMemWrite32(hModel, &pModelRegs->flpage, flpage);
	(void)dfPciMemRead32(hModel, &pModelRegs->flpage);

	if (bWrite) {
		/* Select mode 0, set fl_a, drive fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x7U << 24) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Set fl_d */
		dfPciMemWrite32(hModel, &pModelRegs->fldata, (uint16_t)*pValue);
		(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

		/* Assert fl_we_n */
		flctl |= (0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_we_n */
		flctl &= ~(0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x7U << 24);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);
	} else {
		/* Select mode 0, set fl_a, Drive fl_a, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
		flctl = ((uint32_t)location & 0x1fffeU) | (0x3U << 25) | (0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Assert fl_oe_n */
		flctl |= (0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Read fl_d */
		*pValue = (uint16_t)dfPciMemRead32(hModel, &pModelRegs->fldata);

		/* Deassert fl_oe_n */
		flctl &= ~(0x1U << 29);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Deassert fl_ce_n */
		flctl &= ~(0x1U << 28);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		/* Stop driving fl_a, fl_ce_n, fl_oe_n, fl_we_n */
		flctl &= ~(0x3U << 25);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
		(void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		dfDebugPrint(9, ("flash: *pValue=0x%lx\n", (unsigned long)*pValue));
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  dfDebugPrint(9, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (index >= 2) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    if (index == 0) {
      val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl0);
    } else {
      val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl1);
    }
    *pbValue = (val32 & ADCBBP_FPCTLx_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone fpctl=0x%08lx *pbValue=%s\n", val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    if (index == 0) {
      val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl0);
    } else {
      val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl1);
    }
    *pbValue = (val32 & ADCBBP_FPCTLx_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = (index == 0) ? pDevCtx->model.adcbbp.shadow.fpctl[0] : pDevCtx->model.adcbbp.shadow.fpctl[1];
    if (*pbValue) {
      val32 = (uint32_t)(val32 | ADCBBP_FPCTLx_PROG);
    } else {
      val32 = (uint32_t)(val32 & ~ADCBBP_FPCTLx_PROG);
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    if (index == 0) {
      dfPciMemWrite32(hModel, &pModelRegs->fpctl0, val32);
      (void)dfPciMemRead32(hModel, &pModelRegs->fpctl0);
    } else {
      dfPciMemWrite32(hModel, &pModelRegs->fpctl1, val32);
      (void)dfPciMemRead32(hModel, &pModelRegs->fpctl1);
    }
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;
  void* pSelectMapReg;

  dfDebugPrint(9, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (index >= 2) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  pSelectMapReg = (index == 0) ? &pModelRegs->fpsmap0[0] : &pModelRegs->fpsmap1[0];

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, (uint8_t*)pSelectMapReg, val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)pSelectMapReg, val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, (uint8_t*)pSelectMapReg, val8);
    }
    
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl0); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, (uint8_t*)pSelectMapReg);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdcbbpRev0* pVpd = &pDevCtx->info.vpd.pData->adcbbp.rev0;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdcbbp;

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.adcbbp.shadow.ictl = ADCBBP_ICTL_ENFPGA(0) | ADCBBP_ICTL_ENFPGA(1);
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.adcbbp.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.adcbbp.shadow.fpctl[0] = 0U;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl0, pDevCtx->model.adcbbp.shadow.fpctl[0]);
      pDevCtx->model.adcbbp.shadow.fpctl[1] = 0U;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl1, pDevCtx->model.adcbbp.shadow.fpctl[1]);

      /* Reset Flash and identify it */
      flashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADCBBP_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADCBBP_FLASH_TARGET_AREA_START;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize clock synthesizer stuff */
      if (!v4ClkSynthInit(pDevCtx, &pDevCtx->model.adcbbp.v4ClkSynth.context, pVpd->lclkRef, &fOut)) {
        dfDebugPrint(0, ("*** initHardware: failed to initialize V5 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdcbbp;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize clock synthesizer stuff */
      v4ClkSynthUninit(pDevCtx, &pDevCtx->model.adcbbp.v4ClkSynth.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int channel, n;
  uint32_t pistat, istat;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts, and mask the value with the interrupts we want */
  pistat = dfPciMemRead32(hBridge, &pRegs->pistat) & pDevCtx->hardware.shadow.bridge.adb1.pictl;
  dfDebugPrint(8, ("isr: pistat=0x%08lx\n", (unsigned long)pistat));

  /* Clear the interrupts that we saw */
  dfPciMemWrite32(hBridge, &pRegs->pistat, pistat);

  if (pistat & ADB1_PISTAT_IRQMODEL) {
    dfDebugPrint(9, ("isr: MODEL interrupt\n"));
    istat = dfPciMemRead32(hModel, &pModelRegs->istat) & pDevCtx->model.adcbbp.shadow.ictl;
    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->istat, istat);
    if (istat & ADCBBP_ISTAT_IRQFPGA(0)) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (istat & ADCBBP_ISTAT_IRQFPGA(1)) {
      dfDebugPrint(9, ("isr: target FPGA 1 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[1].dpc, (void*)(uintptr_t)1);
    }
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (pistat & ADB1_PISTAT_IRQDMA(channel)) {
      dfDebugPrint(9, ("isr: DMA channel %u interrupt\n", channel));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, channel);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return pistat ? TRUE : FALSE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  V4ClkSynthProgramStatus v4ClkSynthStatus;
  ModelRegsAdcbbp* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdcbbp;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save V4 clock synthesizer state */
      v4ClkSynthSaveState(pDevCtx, &pDevCtx->model.adcbbp.v4ClkSynth.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      flashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.adcbbp.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore V4 clock synthesizer state */
      v4ClkSynthStatus = v4ClkSynthRestoreState(pDevCtx, &pDevCtx->model.adcbbp.v4ClkSynth.context);
      if (V4ClkSynthProgramStatusSuccess != v4ClkSynthStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore V5 clock synthesizer state, status=%u\n", v4ClkSynthStatus));
      }
    }
  }
}

static boolean_t
validateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Adb1BridgeRegisters)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x100 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x800000U) {
    dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x800000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < sizeof(ModelRegsAdcbbp)) {
    dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("validateBusResources: OK\n"));
  
  return TRUE;
}

void
adb3CoreBootstrapAdcbbp(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdcbbp: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = flash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = adb1V4V5MapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = adb1V4V5ReadWriteVpd;
  pDevCtx->methods.pSyncVpd = adb1V4V5SyncVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = ADCBBP_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1U;
  pDevCtx->info.bootstrap.numSensor = 0U; /* TO DO */
  pDevCtx->info.bootstrap.numTargetFpga = 2;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADCBBP_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADCBBP_VPD_OFFSET_IN_FLASH;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdcbbpRev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
