/*
** File: avr2.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing AVR2 microcontroller interface.
**
** (C) Copyright Alpha Data 2015
**
** TO DO: Flush RX FIFO when a timeout occurs.
*/

#include "device.h"
#include "avr2.h"
#include "avr2_common.h"

#define DEBUGLEVEL_INIT    (2) /* Level of debug messages relating to initialization / uninitialization */
#define DEBUGLEVEL_CANCEL  (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_ERROR   (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_WARNING (1) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_NORMAL  (6) /* Level of debug messages relating to normal operation */

#define AVR2_FACILITY_USER   (0x03)
#define AVR2_COMMAND_VERSION (0x83)

/* AVR uC "Query firmware version" */
static const uint8_t g_versionQuery[] = { AVR2_FACILITY_USER, AVR2_COMMAND_VERSION };

/*
** ----------------------------------------------------------------
** AVR uC access byte transport layer *
** ----------------------------------------------------------------
*/

typedef struct _Avr2RequestSync {
  Avr2Request request;
  DfEvent* pEvent;
  Avr2RequestStatus status;
} Avr2RequestSync;

static void
avr2RxSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContextIgnored)
{
  Avr2Request* pAvr2Request = DF_CONTAINER_OF(pRequest, Avr2Request, serRequest);
  Avr2DeviceContext* pAvr2Ctx = pAvr2Request->pAvr2Ctx;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvr2Ctx->transport.lock);
  if (pAvr2Request->bCancel) {
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    serializerPut(&pAvr2Ctx->transport.rx.serializer);
    pAvr2Request->pCallback(pAvr2Request, Avr2RequestStatusCancelled, 0, (Avr2DataFlag)0, pAvr2Request->pContext);
  } else {
    pAvr2Request->state = Avr2RequestStateActive;
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    /* Enable AVR2 RX not empty interrupt */
    pAvr2Ctx->transport.rx.pEnableInt(pAvr2Ctx);
  }
}

static void
avr2TxSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContext)
{
  Avr2Request* pAvr2Request = DF_CONTAINER_OF(pRequest, Avr2Request, serRequest);
  Avr2DeviceContext* pAvr2Ctx = pAvr2Request->pAvr2Ctx;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvr2Ctx->transport.lock);
  if (pAvr2Request->bCancel) {
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    serializerPut(&pAvr2Ctx->transport.tx.serializer);
    pAvr2Request->pCallback(pAvr2Request, Avr2RequestStatusCancelled, 0, (Avr2DataFlag)0, pAvr2Request->pContext);
  } else {
    pAvr2Request->state = Avr2RequestStateActive;
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    /* Enable AVR2 TX empty interrupt */
    pAvr2Ctx->transport.tx.pEnableInt(pAvr2Ctx);
  }
}

static void
avr2CallbackSync(
  Avr2Request* pRequest,
  Avr2RequestStatus status,
  uint8_t data,
  Avr2DataFlag dataFlag,
  void* pContext /* ignored */)
{
  Avr2RequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, Avr2RequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(pRequestSync->pEvent);
}

static void
avr2OnRxNotEmpty(
  DfDpc* pDpc,
  void* pContextIgnored,
  void* pArgIgnored)
{
  Avr2DeviceContext* pAvr2Ctx = DF_CONTAINER_OF(pDpc, Avr2DeviceContext, transport.rx.dpc);
  Avr2Request* pRequest = NULL;
  SerializerRequest* pSerRequest;
  uint32_t val32;
  uint8_t data8;
  Avr2DataFlag dataFlag;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvr2Ctx->transport.lock);
  pSerRequest = serializerGetHead(&pAvr2Ctx->transport.rx.serializer);
  if (NULL != pSerRequest) {
    pRequest = DF_CONTAINER_OF(pSerRequest, Avr2Request, serRequest);
  }
  if (NULL != pRequest && !pRequest->bWrite) {
    dfAssert(pRequest->state == Avr2RequestStateActive);
    val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
    if (!(val32 & AVR2_STAT_V_RX_EMPTY)) {
      /* Extract the byte received */
      data8 = (uint8_t)AVR2_STAT_E_DATA(val32);
      dataFlag = Avr2DataFlagNone;
      if (val32 & AVR2_STAT_V_SOH) {
        dataFlag = Avr2DataFlagSOH;
      } else if (val32 & AVR2_STAT_V_EOT) {
        dataFlag = Avr2DataFlagEOT;
      }
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2OnRxNotEmpty: data8=0x%x('%c') flag=%u\n",
        (unsigned int)data8, isprint(data8) ? data8 : '.',
        (unsigned int)dataFlag));
      /* Advance the RX FIFO */
      dfPciMemWrite32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat, AVR2_CTL_V_ADVANCE);
      val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
      pRequest->state = Avr2RequestStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
      serializerPut(&pAvr2Ctx->transport.rx.serializer);
      pRequest->pCallback(pRequest, Avr2RequestStatusSuccess, data8, dataFlag, pRequest->pContext);
    } else {
      dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
      /* Shouldn't get here, but if we do, enable AVR2 RX not empty interrupt in order to try again */
      pAvr2Ctx->transport.rx.pEnableInt(pAvr2Ctx);
    }
  } else {
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("+++ avr2OnRxNotEmpty: nothing to do\n"));
  }
}

static void
avr2OnTxEmpty(
  DfDpc* pDpc,
  void* pContextIgnored,
  void* pArgIgnored)
{
  Avr2DeviceContext* pAvr2Ctx = DF_CONTAINER_OF(pDpc, Avr2DeviceContext, transport.tx.dpc);
  Avr2Request* pRequest = NULL;
  SerializerRequest* pSerRequest;
  uint32_t val32;
  uint8_t data8;
  Avr2DataFlag dataFlag;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvr2Ctx->transport.lock);
  pSerRequest = serializerGetHead(&pAvr2Ctx->transport.tx.serializer);
  if (NULL != pSerRequest) {
    pRequest = DF_CONTAINER_OF(pSerRequest, Avr2Request, serRequest);
  }
  if (NULL != pRequest && pRequest->bWrite) {
    dfAssert(pRequest->state == Avr2RequestStateActive);
    val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
    if (val32 & AVR2_STAT_V_TX_EMPTY) {
      /* Transmit the byte */
      data8 = pRequest->data;
      dataFlag = pRequest->dataFlag;
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2OnTxEmpty: data8=0x%x('%c') flag=%u\n",
        (unsigned int)data8, isprint(data8) ? data8 : '.', (unsigned int)dataFlag));
      val32 = AVR2_CTL_V_WRITE | AVR2_CTL_E_DATA(data8);
      switch (dataFlag) {
      case Avr2DataFlagSOH:
        val32 |= AVR2_CTL_V_SOH;
        break;

      case Avr2DataFlagEOT:
        val32 |= AVR2_CTL_V_EOT;
        break;

      case Avr2DataFlagNone:
        break;
      }
      dfPciMemWrite32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat, val32);
      val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
      pRequest->state = Avr2RequestStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
      serializerPut(&pAvr2Ctx->transport.tx.serializer);
      pRequest->pCallback(pRequest, Avr2RequestStatusSuccess, 0, (Avr2DataFlag)0, pRequest->pContext);
    } else {
      dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
      /* Shouldn't get here, but if we do, enable AVR2 TX empty interrupt in order to try again */
      pAvr2Ctx->transport.tx.pEnableInt(pAvr2Ctx);
    }
  } else {
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    dfDebugPrint(DEBUGLEVEL_WARNING, ("+++ avr2OnTxEmpty: nothing to do\n"));
  }
}

void
avr2ByteRead(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest,
  Avr2Callback* pCallback,
  void* pContext)
{
  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2ByteRead: pRequest=%p pCallback=%p\n", pRequest, pCallback));

  pRequest->pAvr2Ctx = pAvr2Ctx;
  pRequest->bWrite = FALSE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->state = Avr2RequestStateQueued;
  pRequest->bCancel = FALSE;
  serializerGet(&pAvr2Ctx->transport.rx.serializer, &pRequest->serRequest, avr2RxSerializerAcquired, NULL);
}

void
avr2ByteWrite(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest,
  uint8_t data,
  Avr2DataFlag dataFlag,
  Avr2Callback* pCallback,
  void* pContext)
{
  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avr2ByteWrite: pRequest=%p data=0x%02x pCallback=%p\n", pRequest, data, pCallback));

  pRequest->pAvr2Ctx = pAvr2Ctx;
  pRequest->data = data;
  pRequest->dataFlag = dataFlag;
  pRequest->bWrite = TRUE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->state = Avr2RequestStateQueued;
  pRequest->bCancel = FALSE;
  serializerGet(&pAvr2Ctx->transport.tx.serializer, &pRequest->serRequest, avr2TxSerializerAcquired, NULL);
}

boolean_t
avr2ByteCancel(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest)
{
  DfSpinLockFlags f;
  boolean_t bCancelled = FALSE;

  dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2ByteCancel: pRequest=%p\n", pRequest));

  f = dfSpinLockGet(&pAvr2Ctx->transport.lock);
  pRequest->bCancel = TRUE;
  switch (pRequest->state) {
  case Avr2RequestStateQueued:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2ByteCancel: cancelling AVR2 request in Queued state\n"));
    if (pRequest->bWrite) {
      bCancelled = serializerCancel(&pAvr2Ctx->transport.tx.serializer, &pRequest->serRequest);
    } else {
      bCancelled = serializerCancel(&pAvr2Ctx->transport.rx.serializer, &pRequest->serRequest);
    }
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    break;

  case Avr2RequestStateActive:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2ByteCancel: cancelling AVR2 request in Active state\n"));
    if (pRequest->bWrite) {
      serializerPut(&pAvr2Ctx->transport.tx.serializer);
    } else {
      serializerPut(&pAvr2Ctx->transport.rx.serializer);
    }
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    bCancelled = TRUE;
    break;

  case Avr2RequestStateComplete:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2ByteCancel: cancelling AVR2 request in Complete state\n"));
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    break;

  default:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("*** avr2ByteCancel: cancelling AVR2 request in unknown state (%u)\n", pRequest->state));
    dfSpinLockPut(&pAvr2Ctx->transport.lock, f);
    dfAssert(FALSE);
    break;
  }

  return bCancelled;
}

Avr2RequestStatus
avr2ByteReadSync(
  Avr2DeviceContext* pAvr2Ctx,
  uint8_t* pData,
  Avr2DataFlag* pDataFlag)
{
  Avr2RequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avr2ByteRead(pAvr2Ctx, &request.request, avr2CallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  *pData = request.request.data;
  *pDataFlag = request.request.dataFlag;
  return request.status;
}

Avr2RequestStatus
avr2ByteWriteSync(
  Avr2DeviceContext* pAvr2Ctx,
  uint8_t data,
  Avr2DataFlag dataFlag)
{
  Avr2RequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avr2ByteWrite(pAvr2Ctx, &request.request, data, dataFlag, avr2CallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  return request.status;
}

/*
** ----------------------------------------------------------------
** AVR2 uC access transaction layer
** ----------------------------------------------------------------
*/

typedef struct _Avr2TransRequestSync {
  Avr2TransRequest request;
  DfEvent* pEvent;
  Avr2TransStatus status;
  size_t actualResponseLength;
} Avr2TransRequestSync;

#if 0
static void
avr2TransByteCallback(
  Avr2Request* pAvr2Request,
  Avr2RequestStatus reqStatus,
  uint8_t data,
  Avr2DataFlag dataFlag,
  void* pContext)
{
  Avr2TransRequest* pRequest = (Avr2TransRequest*)pContext;
  Avr2DeviceContext* pAvr2Ctx = pRequest->pAvr2Ctx;
  size_t position;
  DfSpinLockFlags f;
  unsigned int timeout;
  Avr2TransStatus status = Avr2TransStatusSuccess;
  boolean_t bInvoke = FALSE;
  boolean_t bFramingError = FALSE;
  boolean_t bWrongResponse = FALSE;

  f = dfSpinLockGet(&pAvr2Ctx->transaction.lock);
  position = pAvr2Ctx->transaction.position;
  switch (pRequest->state) {
  case Avr2TransactionStateCommand:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = Avr2TransactionStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      status = Avr2TransStatusTXTimeout;
      bInvoke = TRUE;
    } else {
      dfAssert(reqStatus == Avr2RequestStatusSuccess);
      position++;
      pAvr2Ctx->transaction.position = position;
      if (position >= pRequest->commandLength) {
        if (pRequest->responseLength) {
          /* Finished sending command, start receiving response */
          pAvr2Ctx->transaction.position = 0;
          pRequest->state = Avr2TransactionStateResponse;
          avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        } else {
          /* Finished sending command, and no response expected */
          pRequest->state = Avr2TransactionStateComplete;
          timeout = pRequest->timeout;
          if (0 == timeout || dfTimerCancel(&pAvr2Ctx->transaction.timer)) {
            /* Successfully cancelled the timer function; we can invoke the callback */
            dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
            bInvoke = TRUE;
          } else {
            /* Let the timer function invoke the callback */
            dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
          }
        }
      } else {
        /* Send next byte of command command */
        Avr2DataFlag dataFlag = (position + 1 == pRequest->commandLength) ? Avr2DataFlagEOT : Avr2DataFlagNone;
        avr2ByteWrite(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, pRequest->pCommand[position], dataFlag, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      }
    }
    break;

  case Avr2TransactionStateResponse:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = Avr2TransactionStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      status = Avr2TransStatusRXTimeout;
      bInvoke = TRUE;
    } else {
      /* Do framing checks. */
      if (dataFlag == Avr2DataFlagSOH) {
        if (0 != position) {
          /* Framing error; did not expect to get SOH. */
          bFramingError = TRUE;
        }
      } else if (dataFlag == Avr2DataFlagEOT) {
        if (position + 1 != pRequest->responseLength) {
          /* Framing error; did not expect to get EOT. */
          bFramingError = TRUE;
        }
      } else {
        if (position == 0) {
          /* Framing error; expected SOH but did not get it. */
          bFramingError = TRUE;
        } else if (position + 1 == pRequest->responseLength) {
          /* Framing error; expected EOT but did not get it. */
          bFramingError = TRUE;
        }
      }
      
      /* Check that response code is the expected one. */
      if (position == 0) {
        if (data != pRequest->pCommand[0]) {
          /* Wrong response code first byte. */
          bWrongResponse = TRUE;
        }
      } else if (position == 1) {
        uint8_t expected = (uint8_t)(pRequest->pCommand[1] ^ 0x80U);
        if (data != expected) {
          /* Wrong response code second byte. */
          bWrongResponse = TRUE;
        }
      }

      if (bFramingError || bWrongResponse) {
        /* Go back to looking for SOH. */
        pAvr2Ctx->transaction.position = 0;
        avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        break;
      }

      /* Store the byte of the response */
      pRequest->pResponse[position] = data;
      position++;
      pAvr2Ctx->transaction.position = position;
      if (position >= pRequest->responseLength) {
        /* Finished receiving response */
        pRequest->state = Avr2TransactionStateComplete;
        timeout = pRequest->timeout;
        if (0 == timeout || dfTimerCancel(&pAvr2Ctx->transaction.timer)) {
          /* Successfully cancelled the timer function; we can invoke the callback */
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
          bInvoke = TRUE;
        } else {
          /* Let the timer function invoke the callback */
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        }
      } else {
        /* Receive next byte of response */
        avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      }
    }
    break;

  case Avr2TransactionStateQueued:
  case Avr2TransactionStateComplete:
  default:
    /* Should never get here */
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    dfAssert(FALSE);
    break;
  }

  if (bInvoke) {
    serializerPut(&pAvr2Ctx->transaction.serializer);
    pRequest->pCallback(pRequest, status, pAvr2Ctx->transaction.position, pRequest->pContext);
  }
}
#else
static void
avr2TransByteCallback(
  Avr2Request* pAvr2Request,
  Avr2RequestStatus reqStatus,
  uint8_t data,
  Avr2DataFlag dataFlag,
  void* pContext)
{
  Avr2TransRequest* pRequest = (Avr2TransRequest*)pContext;
  Avr2DeviceContext* pAvr2Ctx = pRequest->pAvr2Ctx;
  size_t position;
  DfSpinLockFlags f;
  unsigned int timeout;
  Avr2TransStatus status = Avr2TransStatusSuccess;
  boolean_t bInvoke = FALSE;
  boolean_t bFramingError = FALSE;
  boolean_t bWrongResponse = FALSE;

  f = dfSpinLockGet(&pAvr2Ctx->transaction.lock);
  position = pAvr2Ctx->transaction.position;
  switch (pRequest->state) {
  case Avr2TransactionStateCommand:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = Avr2TransactionStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      status = Avr2TransStatusTXTimeout;
      bInvoke = TRUE;
    } else {
      dfAssert(reqStatus == Avr2RequestStatusSuccess);
      position++;
      pAvr2Ctx->transaction.position = position;
      if (position >= pRequest->commandLength) {
        if (pRequest->responseLength) {
          /* Finished sending command, start receiving response */
          pAvr2Ctx->transaction.position = 0;
          pRequest->state = Avr2TransactionStateResponse;
          avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        } else {
          /* Finished sending command, and no response expected */
          pRequest->state = Avr2TransactionStateComplete;
          timeout = pRequest->timeout;
          if (0 == timeout || dfTimerCancel(&pAvr2Ctx->transaction.timer)) {
            /* Successfully cancelled the timer function; we can invoke the callback */
            dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
            bInvoke = TRUE;
          } else {
            /* Let the timer function invoke the callback */
            dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
          }
        }
      } else {
        /* Send next byte of command command */
        Avr2DataFlag dataFlag = (position + 1 == pRequest->commandLength) ? Avr2DataFlagEOT : Avr2DataFlagNone;
        avr2ByteWrite(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, pRequest->pCommand[position], dataFlag, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      }
    }
    break;

  case Avr2TransactionStateResponse:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = Avr2TransactionStateComplete;
      dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      status = Avr2TransStatusRXTimeout;
      bInvoke = TRUE;
    } else {
      /* Do framing checks. */
      if (position == 0) {
        if (dataFlag != Avr2DataFlagSOH) {
          /* Expect SOH but haven't got SOH; throw away this byte. */
          bFramingError = TRUE;
        }
      } else {
        if (dataFlag == Avr2DataFlagSOH) {
          /* Got SOH, so set position back to 0 */
          position = 0;
        }
      }
      /* Check that response code is the expected one. */
      if (position == 0) {
        if (data != pRequest->pCommand[0]) {
          /* Wrong response code first byte. */
          bWrongResponse = TRUE;
        }
      } else if (position == 1) {
        uint8_t expected = (uint8_t)(pRequest->pCommand[1] ^ 0x80U);
        if (data != expected) {
          /* Wrong response code second byte. */
          bWrongResponse = TRUE;
        }
      }

      if (bFramingError || bWrongResponse) {
        /* Go back to looking for SOH. */
        pAvr2Ctx->transaction.position = 0;
        avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        break;
      }

      /* Store the byte of the response */
      if (position < pRequest->responseLength) {
        pRequest->pResponse[position] = data;
      }
      position++;
      pAvr2Ctx->transaction.position = position;
      if (dataFlag == Avr2DataFlagEOT) {
        /* Finished receiving response */
        pRequest->state = Avr2TransactionStateComplete;
        timeout = pRequest->timeout;
        if (0 == timeout || dfTimerCancel(&pAvr2Ctx->transaction.timer)) {
          /* Successfully cancelled the timer function; we can invoke the callback */
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
          bInvoke = TRUE;
        } else {
          /* Let the timer function invoke the callback */
          dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
        }
      } else {
        /* Receive next byte of response */
        avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
        dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
      }
    }
    break;

  case Avr2TransactionStateQueued:
  case Avr2TransactionStateComplete:
  default:
    /* Should never get here */
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    dfAssert(FALSE);
    break;
  }

  if (bInvoke) {
    serializerPut(&pAvr2Ctx->transaction.serializer);
    pRequest->pCallback(pRequest, status, pAvr2Ctx->transaction.position, pRequest->pContext);
  }
}
#endif

static void
avr2TransactionTimeout(
  DfTimer* pTimer,
  void* pTimerContextIgnored,
  void* pCallbackContext)
{
  Avr2TransRequest* pRequest = (Avr2TransRequest*)pCallbackContext;
  Avr2DeviceContext* pAvr2Ctx = pRequest->pAvr2Ctx;
  DfSpinLockFlags f;
  boolean_t bInvoke = FALSE;
  Avr2TransStatus status = Avr2TransStatusSuccess;

  dfDebugPrint(DEBUGLEVEL_ERROR, ("avr2TransactionTimeout: timed out, cancelling AVR2 transaction\n"));

  f = dfSpinLockGet(&pAvr2Ctx->transaction.lock);
  pRequest->bTimeout = TRUE;
  switch (pRequest->state) {
  case Avr2TransactionStateQueued:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2TransactionTimeout: cancelling AVR2 transaction in Queued state\n"));
    bInvoke = serializerCancel(&pAvr2Ctx->transaction.serializer, &pRequest->serRequest);
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    status = Avr2TransStatusQueueTimeout;
    break;

  case Avr2TransactionStateCommand:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2TransactionTimeout: cancelling AVR2 transaction in Command state\n"));
    bInvoke = avr2ByteCancel(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request);
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    status = Avr2TransStatusTXTimeout;
    break;

  case Avr2TransactionStateResponse:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2TransactionTimeout: cancelling AVR2 transaction in Response state\n"));
    bInvoke = avr2ByteCancel(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request);
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    status = Avr2TransStatusRXTimeout;
    break;

  case Avr2TransactionStateComplete:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avr2TransactionTimeout: cancelling AVR2 transaction in Complete state\n"));
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    bInvoke = TRUE;
    status = Avr2TransStatusRXTimeout;
    break;

  default:
    /* Should never get here */
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avr2TransactionTimeout: cancelling AVR2 transaction in unknown state (%u)\n", pRequest->state));
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    break;
  }

  if (bInvoke) {
    serializerPut(&pAvr2Ctx->transaction.serializer);
    pRequest->pCallback(pRequest, status, pAvr2Ctx->transaction.position, pRequest->pContext);
  }
}

static void
avr2TransactionSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pSerRequest,
  void* pContextIgnored)
{
  Avr2TransRequest* pRequest = DF_CONTAINER_OF(pSerRequest, Avr2TransRequest, serRequest);
  Avr2DeviceContext* pAvr2Ctx = pRequest->pAvr2Ctx;
  DfSpinLockFlags f;
  unsigned int timeout;

  pAvr2Ctx->transaction.position = 0;
  f = dfSpinLockGet(&pAvr2Ctx->transaction.lock);
  if (pRequest->bTimeout) {
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
    serializerPut(pQueue);
    pRequest->pCallback(pRequest, Avr2TransStatusTXTimeout, pAvr2Ctx->transaction.position, pRequest->pContext);
  } else {
    if (pRequest->commandLength) {
      pRequest->state = Avr2TransactionStateCommand;
      avr2ByteWrite(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, pRequest->pCommand[pAvr2Ctx->transaction.position], Avr2DataFlagSOH, avr2TransByteCallback, pRequest);
    } else {
      pRequest->state = Avr2TransactionStateResponse;
      avr2ByteRead(pAvr2Ctx, &pAvr2Ctx->transaction.avr2Request, avr2TransByteCallback, pRequest);
    }
    timeout = pRequest->timeout;
    if (timeout) {
      dfTimerSchedule(&pAvr2Ctx->transaction.timer, dfTimeGet() + dfMicrosecondsToTime(timeout), avr2TransactionTimeout, pRequest);
    }
    dfSpinLockPut(&pAvr2Ctx->transaction.lock, f);
  }
}

static void
avr2TransactionCallbackSync(
  Avr2TransRequest* pRequest,
  Avr2TransStatus status,
  size_t actualResponseLength,
  void* pContextIgnored)
{
  Avr2TransRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, Avr2TransRequestSync, request);

  pRequestSync->actualResponseLength = actualResponseLength;
  pRequestSync->status = status;
  dfEventSignal(pRequestSync->pEvent);
}

Avr2TransStatus
avr2TransactionSync(
  Avr2DeviceContext* pAvr2Ctx,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout,
  size_t* pActualResponseLength)
{
  Avr2TransRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avr2Transaction(pAvr2Ctx, &request.request, pCommand, commandLength, pResponse, responseLength, timeout, avr2TransactionCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  if (NULL != pActualResponseLength) {
    *pActualResponseLength = request.actualResponseLength;
  }
  return request.status;
}

void
avr2Transaction(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2TransRequest* pRequest,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout,
  Avr2TransCallback* pCallback,
  void* pContext)
{
  pRequest->pAvr2Ctx = pAvr2Ctx;
  pRequest->pCommand = pCommand;
  pRequest->commandLength = commandLength;
  pRequest->pResponse = pResponse;
  pRequest->responseLength = responseLength;
  pRequest->timeout = timeout;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->bTimeout = FALSE;
  pRequest->state = Avr2TransactionStateQueued;
  serializerGet(&pAvr2Ctx->transaction.serializer, &pRequest->serRequest, avr2TransactionSerializerAcquired, NULL);
}

/*
** ----------------------------------------------------------------
** AVR2 uC access device context
** ----------------------------------------------------------------
*/

boolean_t
avr2Init(
  Avr2DeviceContext* pAvr2Ctx,
  DfInterruptObject* pInterruptObject,
  uint32_t* pCtlStat,
  DfMemoryHandle hCtlStat,
  uint32_t flashSize,
  uint8_t flashPageOrder,
  Avr2RxIntEnableCallback* pEnableRxInt,
  Avr2TxIntEnableCallback* pEnableTxInt)
{
  static const unsigned int maxFlushCount = 10000U; /* max. number of RX FIFO reads before giving up flushing */
  static const unsigned int versionTimeout = 200000U; /* in microseconds */
  uint8_t versionResponse[10];
  Avr2ResponseVersion version;
  Avr2TransStatus status;
  uint32_t val32;
  unsigned int count;
  size_t mergeBufferSize;
  uint32_t flashPageSize = 1 << flashPageOrder;

  dfAssert(flashSize <= 0x20000U);
  dfAssert(flashPageSize <= flashSize);

  /* Cannot fail */

  DF_ZERO_OBJECT(pAvr2Ctx);
  pAvr2Ctx->pInterruptObject = pInterruptObject;
  pAvr2Ctx->pCtlStat = pCtlStat;
  pAvr2Ctx->hCtlStat = hCtlStat;
  pAvr2Ctx->flashSize = flashSize;
  pAvr2Ctx->flashPageOrder = flashPageOrder;
  pAvr2Ctx->flashPageSize = flashPageSize;
  dfSpinLockInit(&pAvr2Ctx->transport.lock);
  serializerQueueInit(&pAvr2Ctx->transport.rx.serializer);
  pAvr2Ctx->transport.rx.pEnableInt = pEnableRxInt;
  dfDpcInit(&pAvr2Ctx->transport.rx.dpc, pInterruptObject, avr2OnRxNotEmpty, pAvr2Ctx);
  serializerQueueInit(&pAvr2Ctx->transport.tx.serializer);
  pAvr2Ctx->transport.tx.pEnableInt = pEnableTxInt;
  dfDpcInit(&pAvr2Ctx->transport.tx.dpc, pInterruptObject, avr2OnTxEmpty, pAvr2Ctx);
  dfSpinLockInit(&pAvr2Ctx->transaction.lock);
  serializerQueueInit(&pAvr2Ctx->transaction.serializer);
  dfTimerInit(&pAvr2Ctx->transaction.timer, pAvr2Ctx);

  /* From here on can fail */

  /* Establish AVR2 ctl/status register shadow */
  val32 = dfPciMemRead32(hCtlStat, pCtlStat);
  if (val32 & AVR2_STAT_V_SERVICEMODE) {
    dfDebugPrint(DEBUGLEVEL_WARNING, ("avr2Init: in service mode, ctlStat=0x%08lx\n", (unsigned long)val32));
  }

  mergeBufferSize = flashPageSize + 5U; /* 5 bytes is enough for AVR2 Flash commands */
  pAvr2Ctx->vpd.pMergeBuffer = dfMalloc(mergeBufferSize);
  if (NULL == pAvr2Ctx->vpd.pMergeBuffer) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avr2Init: failed to allocate Flash merge buffer of %u(0x%x) bytes\n",
      (unsigned int)mergeBufferSize, (unsigned int)mergeBufferSize));
    goto failed;
  }
  dfDebugPrint(DEBUGLEVEL_INIT, ("avr2Init: allocated Flash merge buffer of %u(0x%x) bytes\n",
    (unsigned int)mergeBufferSize, (unsigned int)mergeBufferSize));

  /* Flush any preexisting data out of the RX FIFO */
  count = 0;
  val32 = dfPciMemRead32(hCtlStat, pCtlStat);
  while (!(val32 & AVR2_STAT_V_RX_EMPTY)) {
    dfPciMemWrite32(hCtlStat, pCtlStat, AVR2_CTL_V_ADVANCE);
    val32 = dfPciMemRead32(hCtlStat, pCtlStat);
    count++;
    if (count == maxFlushCount) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avr2Init: RX FIFO still not empty after %u reads - assuming broken\n", count));
      goto failed;
    }
  };
  if (count) {
    dfDebugPrint(DEBUGLEVEL_WARNING, ("+++ avr2Init: flushed %u character(s) out of RX FIFO\n", count));
  }

  /* Get AVR uC firmware version */
  status = avr2TransactionSync(pAvr2Ctx, g_versionQuery, DF_ARRAY_LENGTH(g_versionQuery), versionResponse, sizeof(versionResponse), versionTimeout, NULL);
  if (Avr2TransStatusSuccess != status) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avr2Init: failed to get AVR2 uC firmware version\n"));
  } else {
    dfCopyKK(&version, &versionResponse[2], sizeof(version));
    dfDebugPrint(0, ("avr2Init: AVR firmware version %u.%u.%u.%u\n",
      (unsigned int)dfLe16ToCpu(version.nMajorRelease),
      (unsigned int)dfLe16ToCpu(version.nMinorRelease),
      (unsigned int)dfLe16ToCpu(version.nMajorBuild),
      (unsigned int)dfLe16ToCpu(version.nMinorBuild)));
  }

  /* Success */
  return TRUE;

failed: 
  avr2Uninit(pAvr2Ctx);
  return FALSE;
}

void
avr2Uninit(
  Avr2DeviceContext* pAvr2Ctx)
{
  if (NULL != pAvr2Ctx->vpd.pMergeBuffer) {
    dfFree(pAvr2Ctx->vpd.pMergeBuffer);
    pAvr2Ctx->vpd.pMergeBuffer = NULL;
  }
  dfDpcUninit(&pAvr2Ctx->transport.rx.dpc);
  dfDpcUninit(&pAvr2Ctx->transport.tx.dpc);
  dfTimerUninit(&pAvr2Ctx->transaction.timer);
}

boolean_t
avr2IsServiceMode(
  Avr2DeviceContext* pAvr2Ctx)
{
  uint32_t val32;

  val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
  return (val32 & AVR2_STAT_V_SERVICEMODE) ? TRUE : FALSE;
}

extern void
avr2GetStatus(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Status* pAvr2Status)
{
  uint32_t val32;

  dfZeroMemory(pAvr2Status, sizeof(*pAvr2Status));

  /* CTL/STAT register */
  val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat);
  if (val32 & AVR2_STAT_V_SERVICEMODE) {
    pAvr2Status->flags.serviceMode = 1U;
  }
  if (val32 & AVR2_STAT_V_TX_ERROR) {
    pAvr2Status->flags.txFramingError = 1U;
  }

  /* STAT2 register */
  val32 = dfPciMemRead32(pAvr2Ctx->hCtlStat, pAvr2Ctx->pCtlStat + 1);
  if (val32 & AVR2_STAT2_V_RX_ERROR) {
    pAvr2Status->flags.rxProtocolError = 1U;
  }
}
