/*
** File: admxpl_common.c  
** Project: ADB3 core driver
** Purpose: ADB1 common routines for:
**   o ADM-XPL
**   o ADM-XP
**   o ADP-XPI
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "device.h"
#include "flash.h"
#include "admxpl_common.h"
#include "coredma.h"

void
admxplFlash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl, fldata;

  dfDebugPrint(9, ("admxplFlash: bankIndex=%lu bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    (unsigned long)bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0 || location >= 0x2000000U) {
    return;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_a, fl_sel_n, fl_oe_n, fl_we_n, assert fl_sel_n */
	flctl = ((uint32_t)location & 0x1FFFFFFU) | (0x1fU << 25);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
  (void)dfPciMemRead32(hModel, &pModelRegs->flctl);

	if (bWrite) {
		/* Drive fl_d (data to Flash) */
		fldata = (*pValue & 0xFFU) | 0x100U;
		dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Assert fl_we_n (write enable for Flash) */
		flctl |= (0x1U << 31);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Deassert fl_we_n */
		flctl &= ~(0x1U << 31);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Stop driving fl_d (data to Flash) */
		fldata &= ~0x100U;
		dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
	} else {
		/* Assert fl_oe_n (output enable for Flash) */
		flctl |= (0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
		/* Read fl_d (data from Flash) */
		*pValue = (uint8_t)dfPciMemRead32(hModel, &pModelRegs->fldata);
		/* Deassert fl_oe_n */
		flctl &= ~(0x1U << 30);
		dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flctl);

		dfDebugPrint(9, ("admxplFlash: *pValue=0x%lx\n", (unsigned long)*pValue));
	}

	/* Stop driving fl_a, fl_sel_n, fl_oe_n, fl_we_n, assert fl_sel_n */
	flctl &= ~(0x1fU << 25);
	dfPciMemWrite32(hModel, &pModelRegs->flctl, flctl);
  (void)dfPciMemRead32(hModel, &pModelRegs->flctl);
}

void
admxplFlashReset(
  Adb3CoreDeviceContext* pDevCtx)
{
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t fldata;

  dfDebugPrint(9, ("admxplFlashReset: entered\n"));

  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

	/* Drive fl_rst_n, assert fl_rst_n */
	fldata = (0x3U << 30);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

	/* Deassert fl_rst_n */
	fldata = (0x1U << 30);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

	/* Stop driving fl_rst_n */
	fldata = (0x0U << 30);
	dfPciMemWrite32(hModel, &pModelRegs->fldata, fldata);
	(void)dfPciMemRead32(hModel, &pModelRegs->fldata);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

FpgaControlStatus
admxplFpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  uint32_t* pFpctlShadow,
  boolean_t* pbValue)
{
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  dfDebugPrint(9, ("admxplFpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXPL_FPCTL_DONE) ? TRUE : FALSE;
    dfDebugPrint(9, ("admxplFpgaControl: opCode=GetDone fpctl=0x%08lx *pbValue=%s\n", val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    *pbValue = TRUE;
    dfDebugPrint(9, ("admxplFpgaControl: opCode=GetLinkStat *pbValue=%s\n", *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpctl);
    *pbValue = (val32 & ADMXPL_FPCTL_INIT) ? TRUE : FALSE;
    dfDebugPrint(9, ("admxplFpgaControl: opCode=GetInit fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = *pFpctlShadow;
    if (*pbValue) {
      val32 = (uint32_t)(val32 | ADMXPL_FPCTL_PROG);
    } else {
      val32 = (uint32_t)(val32 & ~ADMXPL_FPCTL_PROG);
    }
    dfDebugPrint(9, ("admxplFpgaControl: opCode=SetProg fpctl=0x%08lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpctl, val32);
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

boolean_t
admxplHandleInterrupt(
  Adb3CoreDeviceContext* pDevCtx,
  uint32_t gmask,  /* Bitmask of enabled generic interrupts */
  uint32_t mmask)  /* Bitmask of model-specific interrupts */
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int channel, n;
  uint32_t pistat, istat;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts, and mask the value with the interrupts we want */
  pistat = dfPciMemRead32(hBridge, &pRegs->pistat) & gmask;
  dfDebugPrint(8, ("admxplHandleInterrupt: pistat=0x%08lx\n", (unsigned long)pistat));

  /* Clear the interrupts that we saw */
  dfPciMemWrite32(hBridge, &pRegs->pistat, pistat);

  if (pistat & ADB1_PISTAT_IRQMODEL) {
    dfDebugPrint(9, ("admxplHandleInterrupt: MODEL interrupt\n"));
    istat = dfPciMemRead32(hModel, &pModelRegs->istat) & mmask;
    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->istat, istat);
    if (istat & ADMXPL_ISTAT_IRQFPGA) {
      dfDebugPrint(9, ("admxplHandleInterrupt: target FPGA interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (pistat & ADB1_PISTAT_IRQDMA(channel)) {
      dfDebugPrint(9, ("admxplHandleInterrupt: DMA channel %u interrupt\n", channel));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
#if defined(BUILD_DEBUGTS)
      dmaLogIsrTimestamp(pDevCtx, channel);
#endif
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return pistat ? TRUE : FALSE;
}

boolean_t
admxplMapBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated,
  boolean_t bUnmap)
{
  dfDebugPrint(2, ("admxplMapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));
  
  if (bUnmap) {
    if (NULL != pDevCtx->hardware.bridge.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
        dfDebugPrint(0, ("*** admxplMapBusResources: failed to unmap Bridge regs\n"));
      }
    }
    if (NULL != pDevCtx->hardware.model.pGeneric) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
        dfDebugPrint(0, ("*** admxplMapBusResources: failed to unmap Model regs\n"));
      }
    }
    if (NULL != pDevCtx->hardware.pBridged) {
      if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
        dfDebugPrint(0, ("*** admxplMapBusResources: failed to unmap FPGA space\n"));
      }
    }
    dfDebugPrint(2, ("admxplMapBusResources: unmapped\n"));
  } else {
    pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
    if (NULL == pDevCtx->hardware.bridge.pGeneric) {
      dfDebugPrint(0, ("*** admxplMapBusResources: failed to map Bridge regs\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxplMapBusResources: mapped Bridge regs @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
    }
    pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hModel);
    if (NULL == pDevCtx->hardware.model.pGeneric) {
      dfDebugPrint(0, ("*** admxplMapBusResources: failed to map Model regs\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxplMapBusResources: mapped Model regs @ 0x%08lx_%08lx to %p\n",
      dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.model.pGeneric));
    }
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hBridged);
    if (NULL == pDevCtx->hardware.pBridged) {
      dfDebugPrint(0, ("*** admxplMapBusResources: failed to map FPGA space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("admxplMapBusResources: mapped FPGA space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.pBridged));
    }
    dfDebugPrint(2, ("admxplMapBusResources: mapped OK\n"));
  }
  
  return TRUE;
}

FpgaSelectMapStatus
admxplFpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(9, ("admxplFpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->fpsmap[0], val32);
      length32 -= 4;
    }
    
    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->fpsmap[0], val8);
    }
    
    (void)dfPciMemRead32(hModel, &pModelRegs->fpctl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->fpsmap[0]);
      *pBuffer8++ = val8;
    }
  }
  
  return FpgaSelectMapSuccess;
}

boolean_t
admxplValidateBusResources(
  Adb3CoreDeviceContext* pDevCtx,
  DfBusResources* pRaw,
  DfBusResources* pTranslated)
{
  if (pTranslated->variant.pci.numMemBar < 3) {
    dfDebugPrint(0, ("*** admxplValidateBusResources: less than 3 memory BARs\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[0].size < sizeof(Adb1BridgeRegisters)) {
    dfDebugPrint(0, ("*** admxplValidateBusResources: BAR0 smaller than 0x%lx bytes\n", (unsigned long)sizeof(Adb1BridgeRegisters)));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[1].size < 0x400000U) {
    dfDebugPrint(0, ("*** admxplValidateBusResources: BAR1 smaller than 0x400000 bytes\n"));
    return FALSE;
  }
  if (pTranslated->variant.pci.memBar[2].size < sizeof(ModelRegsAdmxpl)) {
    dfDebugPrint(0, ("*** admxplValidateBusResources: BAR2 smaller than 0x%lx bytes\n", (unsigned long)sizeof(ModelRegsAdmxpl)));
    return FALSE;
  }
  if (pTranslated->variant.pci.numIrq != 1) {
    dfDebugPrint(0, ("*** admxplValidateBusResources: not exactly 1 interrupt\n"));
    return FALSE;
  }
  
  dfDebugPrint(2, ("admxplValidateBusResources: OK\n"));
  
  return TRUE;
}

CoreVpdStatus
admxplReadWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  CoreFlashStatus flashStatus;

  if (CHECK_BOUNDS(address, length, pDevCtx->info.flash[0].totalSize)) {
    return CoreVpdInvalidRegion;
  }
	if (bWrite) {
    flashStatus = flashWrite(pDevCtx, 0, address, length, pData);
	} else {
		flashStatus = flashRead(pDevCtx, 0, address, length, pData);
	}
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

CoreVpdStatus
admxplSyncVpd(
	Adb3CoreDeviceContext* pDevCtx)
{
  CoreFlashStatus flashStatus;

  flashStatus = flashSync(pDevCtx, 0);
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}
