/*
** File: adb1_common.c  
** Project: ADB3 core driver
** Purpose: ADB1-specific code.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "adb1_common.h"

#define DMA_DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to DMA driver logic errors */
#define DMA_DEBUGLEVEL_CANCEL (1) /* Level of debug messages relating to DMA cancellation / cleanup */
#define DMA_DEBUGLEVEL_NORMAL (7) /* Level of debug messages relating to DMA normal operation */

#define SPACE_DEBUGLEVEL_NORMAL (6)

void
adb1DumpDmaDescriptors(
  Adb3CoreDeviceContext* pDevCtx,
  Adb1DmaDescriptor* pDescriptors,
  uint64_t descBus,
  unsigned int count)
{
  Adb1DmaDescriptor* pCurrent;
  unsigned int i;

  if (sizeof(void*) == 4) {
    dfDebugPrint(0, ("  ### DescVirt   DescBus             [ PA                LA       SIZ      ND                ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx     %08lx_08lx %08lx %08lx %08lx_%08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pah), (unsigned long)dfLe32ToCpu(pCurrent->pal),
        (unsigned long)dfLe32ToCpu(pCurrent->la), (unsigned long)dfLe32ToCpu(pCurrent->siz),
        (unsigned long)dfLe32ToCpu(pCurrent->ndh), (unsigned long)dfLe32ToCpu(pCurrent->ndl)));
      descBus += sizeof(Adb1DmaDescriptor);
    }
  } else {
    dfDebugPrint(0, ("  ### DescVirt           DescBus             [ PA                LA       SIZ      ND                ]\n"));
    for (i = 0; i < count; i++) {
      pCurrent = &pDescriptors[i];
      dfDebugPrint(0, ("  %03lu %p 0x%08lx_%08lx     %08lx %08lx %08lx %08lx %08lx_%08lx\n",
        (unsigned long)i, (void*)pCurrent, dfSplitUint64(descBus),
        (unsigned long)dfLe32ToCpu(pCurrent->pah), (unsigned long)dfLe32ToCpu(pCurrent->pal),
        (unsigned long)dfLe32ToCpu(pCurrent->la), (unsigned long)dfLe32ToCpu(pCurrent->siz),
        (unsigned long)dfLe32ToCpu(pCurrent->ndh), (unsigned long)dfLe32ToCpu(pCurrent->ndl)));
      descBus += sizeof(Adb1DmaDescriptor);
    }
  }
}

void
adb1DumpDmaRegisters(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel)
{
  Adb1DmaRegisters* pDmaRegs = &pDevCtx->hardware.bridge.pAdb1->dma[channel];
  uint32_t pal, pah, la, siz, ndl, ndh, cfg, ctl;

  pal = pDmaRegs->pal;
  pah = pDmaRegs->pah;
  la = pDmaRegs->la;
  siz = pDmaRegs->siz;
  ndl = pDmaRegs->ndl;
  ndh = pDmaRegs->ndh;
  cfg = pDmaRegs->cfg;
  ctl = pDmaRegs->ctl;

  dfDebugPrint(0, ("  channel=%u PAH_PAL=0x%08lx_0%08lx LA=0x%08lx SIZ=0x%08lx NDH_NDL=0x%08lx_%08lx CFG=0x%08lx CTL=0x%08lx\n",
    channel,
    (unsigned long)pah, (unsigned long)pal, (unsigned long)la, (unsigned long)siz,
    (unsigned long)ndh, (unsigned long)ndl, (unsigned long)cfg, (unsigned long)ctl));
}

void
adb1DmaList(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWriteToDevice,
  boolean_t bFixedLocal,
  DfDmaMapperNode* pTable,
  unsigned int tableLength,
  DmaNodeTablePosition* pTablePosition,
  void* pDescriptorBuffer,
  unsigned int maxDescriptor,
  uint64_t descriptorBusAddress)
{
  static const uint32_t ndlFlags = ADB1_DMA_NDL_PCISPACE;
  Adb1DmaDescriptor* pDescriptor = (Adb1DmaDescriptor*)pDescriptorBuffer;
  DfDmaMapperNode* pTableTmp;
  uint64_t busAddress, nextBusAddress, tableBusAddress, localAddress = 0U;
  unsigned int i;
  uint32_t sizFlags;
  unsigned int numDesc = 0;
  unsigned int index;
  size_t offset, remaining;

  dfDebugPrint(DMA_DEBUGLEVEL_NORMAL, ("adb1DmaList: entered, pDevCtx=%p bWriteToDevice=%s bFixedLocal=%s pTable=%p tableLength=%u *pTablePosition={%u, 0x%08lx_%08lx} pDescriptorBuffer=%p maxDescriptor=%u descriptorBusAddrss=0x%08lx_%08lx\n",
    (void*)pDevCtx, bWriteToDevice ? "TRUE" : "FALSE", bFixedLocal ? "TRUE" : "FALSE", (void*)pTable, tableLength, pTablePosition->index, dfSplitUint64((uint64_t)pTablePosition->offset), pDescriptorBuffer, maxDescriptor, dfSplitUint64(descriptorBusAddress)));

  sizFlags = bWriteToDevice ? ADB1_DMA_SIZ_PCITOLOCAL : ADB1_DMA_SIZ_LOCALTOPCI;
  busAddress = descriptorBusAddress;
  index = pTablePosition->index;
  offset = pTablePosition->offset;
  pTableTmp = pTable + pTablePosition->index;
  for (i = index; i < tableLength && numDesc < maxDescriptor; i++, numDesc++) {
    nextBusAddress = busAddress + sizeof(Adb1DmaDescriptor);
    tableBusAddress = pTableTmp->busAddress + offset;
    localAddress = pTableTmp->localAddress;
    if (!bFixedLocal) {
      localAddress += offset;
    }
    remaining = pTableTmp->length - offset;
    pDescriptor->pal = dfCpuToLe32((uint32_t)tableBusAddress);
    pDescriptor->pah = 0U;
    pDescriptor->la = dfCpuToLe32((uint32_t)localAddress);
    pDescriptor->siz = dfCpuToLe32((uint32_t)remaining | sizFlags);
    pDescriptor->ndl = dfCpuToLe32((uint32_t)nextBusAddress | ndlFlags);
    pDescriptor->ndh = 0U;
    pTableTmp++;
    busAddress = nextBusAddress;
    offset = 0;
    pDescriptor++;
  }
  pDescriptor--;
  pDescriptor->ndl = dfCpuToLe32(ndlFlags | ADB1_DMA_NDL_ENDCHAIN);

  /* Cast is safe as pTableTmp cannot be much greater than pTable */
  pTablePosition->index = (unsigned int)(pTableTmp - pTable);
  pTablePosition->offset = offset;

#if DF_DBG_BUILD
  if (g_dfDebugLevel >= DMA_DEBUGLEVEL_NORMAL) {
    dfDebugPrint(0, ("adb1DmaList: dump of DMA engine linked list\n"));
    adb1DumpDmaDescriptors(pDevCtx, (Adb1DmaDescriptor*)pDescriptorBuffer, descriptorBusAddress, numDesc);
  }
#endif
}

void
adb1DmaTransfer(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int channel,
  boolean_t bStart,
  void* pDescriptorBuffer,       /* Valid iff 'bStart' is TRUE */
  uint64_t descriptorBusAddress) /* Valid iff 'bStart' is TRUE */
{
  static const uint32_t specifiableModeBits =
    ADB1_DMA_CFG_USE_READY | ADB1_DMA_CFG_USE_BTERM | ADB1_DMA_CFG_ENABLE_BURST | ADB1_DMA_CFG_USE_EOT |
    ADB1_DMA_CFG_FIXED_LOCAL | ADB1_DMA_CFG_DEMAND | ADB1_DMA_CFG_WIDTHMASK;
  static const uint32_t alwaysModeBits =
    ADB1_DMA_CFG_CHAINED | ADB1_DMA_CFG_INTERRUPT_PCI | ADB1_DMA_CFG_DONE_INTERRUPT;
  const unsigned int maxAbortPoll = 20;
  Adb1BridgeRegisters* pRegs;
  Adb1DmaRegisters* pDmaRegs;
  DfMemoryHandle hBridge;
  uint32_t mode, dxctl, val32;
  unsigned int i;

  dfDebugPrint(DMA_DEBUGLEVEL_NORMAL, ("adb1DmaTransfer: entered, pDevCtx=%p channel=%lu bStart=%s pDescriptorBuffer=%p busAddress=0x%08lx_%08lx\n",
    (void*)pDevCtx, (unsigned long)channel, bStart ? "TRUE" : "FALSE", pDescriptorBuffer, dfSplitUint64(descriptorBusAddress)));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pDmaRegs = &pRegs->dma[channel];

  if (bStart) {
    mode = (pDevCtx->dma.channel[channel].curr.parameters.user.modeFlags & specifiableModeBits) | alwaysModeBits;
    /* Set current transfer mode */
    dfPciMemWrite32(hBridge, &pDmaRegs->cfg, mode);
    /* Set the descriptor pointer & flags */
    dfPciMemWrite32(hBridge, &pDmaRegs->ndl, (uint32_t)descriptorBusAddress | ADB1_DMA_NDL_PCISPACE);
    dfPciMemRead32(hBridge, &pDmaRegs->ndl);
    
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= DMA_DEBUGLEVEL_NORMAL) {
      dfDebugPrint(0, ("adb1DmaTransfer: DMA channel %u registers before starting on hardware:\n", channel));
      adb1DumpDmaRegisters(pDevCtx, channel);
    }
#endif

    /* Start the DMA transfer */
    dfPciMemWrite32(hBridge, &pDmaRegs->ctl, pDevCtx->hardware.shadow.bridge.adb1.dmaCtl[channel] | ADB1_DMA_CTL_GO);
    (void)dfPciMemRead32(hBridge, &pDmaRegs->ctl); 

#if DF_DBG_BUILD
    if (g_dfDebugLevel >= DMA_DEBUGLEVEL_NORMAL) {
      dfDebugPrint(0, ("adb1DmaTransfer: DMA channel %u registers after starting on hardware:\n", channel));
      adb1DumpDmaRegisters(pDevCtx, channel);
    }
#endif
  } else {
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= DMA_DEBUGLEVEL_CANCEL) {
      dfDebugPrint(0, ("adb1DmaTransfer: DMA channel %u registers before attempt to abort:\n", channel));
      adb1DumpDmaRegisters(pDevCtx, channel);
    }
#endif

    /* First, disable the DMA channel */
		dxctl = pDevCtx->hardware.shadow.bridge.adb1.dmaCtl[channel];
		dxctl &= ~ADB1_DMA_CTL_ENABLE;
    dfPciMemWrite32(hBridge, &pDmaRegs->ctl, dxctl);

		/* Wait for the Bridge to pause the DMA */
    for (i = 0; i < maxAbortPoll; i++) {
      val32 = dfPciMemRead32(hBridge, &pDmaRegs->ctl);
      if (!(val32 & ADB1_DMA_CTL_ENABLED)) {
        break;
      }
    }
    if (maxAbortPoll == i) {
      /* No! Not much we can do, though */
      dfDebugPrint(DMA_DEBUGLEVEL_ERROR, ("*** adb1DmaTransfer: disable failed, channel=%lu\n", (unsigned long)channel));
      goto reenable;
    }

    /* Now check if DMA channel is active or not */
    val32 = dfPciMemRead32(hBridge, &pDmaRegs->ctl);
    if (val32 & ADB1_DMA_CTL_IDLE) {
      /* DMA is not active so must have already finished */
      dfDebugPrint(DMA_DEBUGLEVEL_CANCEL, ("adb1DmaTransfer: nothing to abort, channel=%lu\n", (unsigned long)channel));
    } else {
      /* DMA is active - abort it */
      dfPciMemWrite32(hBridge, &pDmaRegs->ctl, dxctl | ADB1_DMA_CTL_ABORT);

      /* Poll for Bridge aborting the DMA */
      for (i = 0; i < maxAbortPoll; i++) {
        val32 = dfPciMemRead32(hBridge, &pDmaRegs->ctl);
        if (val32 & ADB1_DMA_CTL_IDLE) {
          dfDebugPrint(DMA_DEBUGLEVEL_CANCEL, ("adb1DmaTransfer: aborted, channel=%lu\n", (unsigned long)channel));
          break;
        }
      }
      if (maxAbortPoll == i) {
        /* No! Not much we can do, though */
        dfDebugPrint(0, ("*** adb1DmaTransfer: abort failed, channel=%lu\n", (unsigned long)channel));
      }
    }

reenable:
    /* Reenable the DMA channel */
		dxctl |= ADB1_DMA_CTL_ENABLE;
    dfPciMemWrite32(hBridge, &pDmaRegs->ctl, dxctl);
    (void)dfPciMemRead32(hBridge, &pDmaRegs->ctl);
    
#if DF_DBG_BUILD
    if (g_dfDebugLevel >= DMA_DEBUGLEVEL_CANCEL) {
      dfDebugPrint(0, ("adb1DmaTransfer: DMA channel %lu registers after attempt to abort:\n", (unsigned long)channel));
      adb1DumpDmaRegisters(pDevCtx, channel);
    }
#endif
  }
}

void
adb1EnableDmaEngines(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnable)
{
  Adb1DmaRegisters* pDmaRegs;
  DfMemoryHandle hBridge;
  unsigned int channel;

  hBridge = pDevCtx->hardware.hBridge;

  for (channel = 0; channel < pDevCtx->info.bootstrap.numDmaChannel; channel++) {
    pDmaRegs = &pDevCtx->hardware.bridge.pAdb1->dma[channel];
    pDevCtx->hardware.shadow.bridge.adb1.dmaCtl[channel] = bEnable ? ADB1_DMA_CTL_ENABLE : 0;
    dfPciMemWrite32(hBridge, &pDmaRegs->ctl, pDevCtx->hardware.shadow.bridge.adb1.dmaCtl[channel]);
    (void)dfPciMemRead32(hBridge, &pDmaRegs->ctl); 
  }
}

void
adb1EnableInterrupts(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bEnable)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  uint32_t pictl;
  
  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;

  dfDebugPrint(1, ("enableInterrupts: entered bEnable=%s\n", bEnable ? "TRUE" : "FALSE"));
  
  if (bEnable) {
    pDevCtx->hardware.shadow.bridge.adb1.pictl = pDevCtx->hardware.shadow.bridge.adb1.pictlMaster;
    dfPciMemWrite32(hBridge, &pRegs->pictl, pDevCtx->hardware.shadow.bridge.adb1.pictl);
    pictl = dfPciMemRead32(hBridge, &pRegs->pictl);
    dfDebugPrint(2, ("enableInterrupts: interrupts enabled, PICTL=0x%08lx\n", (unsigned long)pictl));
  } else {
    pDevCtx->hardware.shadow.bridge.adb1.pictl = 0U;
    dfPciMemWrite32(hBridge, &pRegs->pictl, pDevCtx->hardware.shadow.bridge.adb1.pictl);
    pictl = dfPciMemRead32(hBridge, &pRegs->pictl);
    dfDebugPrint(2, ("enableInterrupts: interrupts disabled, PICTL=0x%08lx\n", (unsigned long)pictl));
  }
}

static CoreWindowConfigStatus
adb1GetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  CoreWindowConfig* pConfig)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t lsxb, lsxr, lsxcfg, mask;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;

  switch (windowIndex) {
  case 0:
  case 1:
    f = dfSpinLockGet(&pDevCtx->info.window[windowIndex].lock);
    lsxb = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0b : &pRegs->ls1b);
    lsxr = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0r : &pRegs->ls1r);
    lsxcfg = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0cfg : &pRegs->ls1cfg);
    dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);

    mask = lsxr & 0xFFFFFFF0U;
    pConfig->base = lsxb & mask;

    switch (lsxcfg & 0x3U) {
    case 0:
      pConfig->width = COREWINDOWCONFIG_WIDTH_8BIT;
      break;

    case 1:
      pConfig->width = COREWINDOWCONFIG_WIDTH_16BIT;
      break;

    case 2:
      pConfig->width = COREWINDOWCONFIG_WIDTH_32BIT;
      break;

    case 3:
      pConfig->width = COREWINDOWCONFIG_WIDTH_64BIT;
      break;
    }

    if (lsxcfg & (0x1U << 5)) {
      if (((lsxcfg >> 2) & 0x7U) > 0x2U) {
        pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_MAX;
      } else {
        pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_DEFAULT;
      }
    } else {
      /* Prefetch disabled */
      pConfig->prefetch = COREWINDOWCONFIG_PREFETCH_MIN;
    }

    if (lsxcfg & (0x1U << 6)) {
      pConfig->burst = COREWINDOWCONFIG_BURST_ENABLE;
    } else {
      pConfig->burst = COREWINDOWCONFIG_BURST_DISABLE;
    }
    break;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

static CoreWindowConfigStatus
adb1SetWindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  uint32_t flags,
  const CoreWindowConfig* pConfig)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  DfSpinLockFlags f;
  uint32_t lsxb, lsxr, lsxcfg, mask, val32;

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;

  switch (windowIndex) {
  case 0:
    f = dfSpinLockGet(&pDevCtx->info.window[windowIndex].lock);
    lsxb = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0b : &pRegs->ls1b);
    lsxr = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0r : &pRegs->ls1r);
    lsxcfg = dfPciMemRead32(hBridge, (windowIndex == 0) ? &pRegs->ls0cfg : &pRegs->ls1cfg);
    if (flags & COREWINDOWCONFIG_SET_BASE) {
      mask = lsxr & 0xFFFFFFF0U;
      if (pConfig->base & ~(uint64_t)mask) {
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
      lsxb = (lsxb & ~mask) | ((uint32_t)pConfig->base & mask);
    }
    if (flags & COREWINDOWCONFIG_SET_WIDTH) {
      switch (pConfig->width) {
      case COREWINDOWCONFIG_WIDTH_DEFAULT:
      case COREWINDOWCONFIG_WIDTH_32BIT:
        lsxcfg = (lsxcfg & ~0x3U) | (0x2U << 0);
        break;

      case COREWINDOWCONFIG_WIDTH_64BIT:
        lsxcfg = (lsxcfg & ~0x3U) | (0x3U << 0);
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    if (flags & COREWINDOWCONFIG_SET_PREFETCH) {
      switch (pConfig->prefetch) {
      case COREWINDOWCONFIG_PREFETCH_MIN:
        lsxcfg = (lsxcfg & ~0x3CU); /* Disable prefetch */
        break;

      case COREWINDOWCONFIG_PREFETCH_DEFAULT:
      case COREWINDOWCONFIG_PREFETCH_NORMAL:
        lsxcfg = (lsxcfg & ~0x3CU) | (0x1U << 5) | (0x2U << 2);
        break;

      case COREWINDOWCONFIG_PREFETCH_MAX:
        lsxcfg = (lsxcfg & ~0x3CU) | (0x1U << 5) | (0x4U << 2);
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    if (flags & COREWINDOWCONFIG_SET_BURST) {
      switch (pConfig->prefetch) {
      case COREWINDOWCONFIG_BURST_DEFAULT:
      case COREWINDOWCONFIG_BURST_ENABLE:
        lsxcfg = lsxcfg | (0x1U << 6); /* Enable bursting */
        break;

      case COREWINDOWCONFIG_BURST_DISABLE:
        lsxcfg = lsxcfg & ~(0x1U << 6); /* Disable bursting */
        break;

      default:
        dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
        return CoreWindowConfigInvalidValue;
      }
    }
    /* Update registers only after we know requested values are valid */
    if (flags & COREWINDOWCONFIG_SET_BASE) {
      dfPciMemWrite32(hBridge, (windowIndex == 0) ? &pRegs->ls0b : &pRegs->ls1b, lsxb);
      val32 = dfPciMemRead32(hBridge, &pRegs->ls0b);
      dfDebugPrint(SPACE_DEBUGLEVEL_NORMAL, ("adb1SetWindowConfig: new LS%uB=0x%08lx\n", windowIndex, (unsigned long)lsxb));
      pDevCtx->info.window[windowIndex].localBase = pConfig->base;
    }
    if (flags & (COREWINDOWCONFIG_SET_WIDTH | COREWINDOWCONFIG_SET_PREFETCH | COREWINDOWCONFIG_SET_BURST)) {
      dfPciMemWrite32(hBridge, (windowIndex == 0) ? &pRegs->ls0cfg : &pRegs->ls1cfg, lsxcfg);
      val32 = dfPciMemRead32(hBridge, &pRegs->ls0cfg);
      dfDebugPrint(SPACE_DEBUGLEVEL_NORMAL, ("adb1SetWindowConfig: new LS%uCFG=0x%08lx\n", windowIndex, (unsigned long)lsxcfg));
    }
    dfSpinLockPut(&pDevCtx->info.window[windowIndex].lock, f);
    break;

  case 1:
    return CoreWindowConfigNotSupported;

  default:
    return CoreWindowConfigInvalidIndex;
  }

  return CoreWindowConfigSuccess;
}

CoreWindowConfigStatus
adb1WindowConfig(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int windowIndex,
  boolean_t bConfigure,
  uint32_t flags,
  CoreWindowConfig* pConfig)
{
  if (bConfigure) {
    return adb1SetWindowConfig(pDevCtx, windowIndex, flags, pConfig);
  } else {
    return adb1GetWindowConfig(pDevCtx, windowIndex, pConfig);
  }
}
