/*
** File: admxrc2_platform.c  
** Project: ADMXRC2 API module library
** Purpose: Implements Linux-specific functions for ADMXRC2 API module library
**
** (C) Copyright Alpha Data 2013
*/

#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <unistd.h>

#include "admxrc2_platform.h"
#include <ioctl_dflinux.h>

#define ARRAY_LENGTH(x) (sizeof(x) / sizeof((x)[0]))

static const char g_deviceName[] = "admxrc2i";

ADMXRC2_STATUS
admxrc2OpenDevice(
  unsigned int index,
  bool_t bPassive,
  ADMXRC2_HANDLE* phAdmxrc2)
{
  char filename[32];
  int hDevice;

  snprintf(filename, ARRAY_LENGTH(filename), "/dev/%s%u", g_deviceName, index);
  hDevice = open(filename, bPassive ? O_RDONLY : O_RDWR);
  if (hDevice < 0) {
    switch (errno) {
    case EACCES:
      return ADMXRC2_ACCESS_DENIED;

    case ENXIO:
    case ENOENT:
      return ADMXRC2_CARD_NOT_FOUND;

    default:
      fprintf(stderr, "admxrc2OpenDevice: errno=%d (%s)\n", errno, strerror(errno));
      return ADMXRC2_UNKNOWN_ERROR;
    }
  }
  *phAdmxrc2 = hDevice;
  return ADMXRC2_SUCCESS;
}

ADMXRC2_STATUS
admxrc2CloseDevice(
  ADMXRC2_HANDLE hAdmxrc2)
{
  ADMXRC2_STATUS status = ADMXRC2_SUCCESS;

  admxrc2UnmapAllRegions(hAdmxrc2);

  /* Close the device handle. This prevents client code from using the handle to do further operations (such as mmap) */
  if (close(hAdmxrc2) < 0) {
    switch (errno) {
    case EBADF:
      status = ADMXRC2_INVALID_HANDLE;
      break;

    default:
      fprintf(stderr, "admxrc2CloseDevice: errno=%d (%s)\n", errno, strerror(errno));
      status = ADMXRC2_UNKNOWN_ERROR;
      break;
    }
  }

  if (ADMXRC2_SUCCESS != status) {
    return status;
  }

  admxrc2UnmapAllRegions(hAdmxrc2);

  return ADMXRC2_SUCCESS;
}

#define MAX_IOCTL_SIZE (64)
#define IOCTL_SIZE_IN_SHIFT  (0)
#define IOCTL_SIZE_OUT_SHIFT (6)

ADMXRC2_STATUS
_admxrc2Ioctl(
ADMXRC2_HANDLE hAdmxrc2,
  unsigned int code,
  void* pIoctl,
  unsigned int sizeIn,
  unsigned int sizeOut)
{
  static const int ioctlType = 150;
  int ioctlCode, err;

  if (sizeIn >= MAX_IOCTL_SIZE || sizeOut >= MAX_IOCTL_SIZE) {
    fprintf(stderr, "admxrc2Ioctl: sizeIn (%u) or sizeOut (%u) greater than 64\n", sizeIn, sizeOut);
    return ADMXRC2_UNKNOWN_ERROR;
  }
  ioctlCode = _IOC(_IOC_READ | _IOC_WRITE, ioctlType, (int)code, ((int)sizeIn << IOCTL_SIZE_IN_SHIFT) | ((int)sizeOut << IOCTL_SIZE_OUT_SHIFT));
  err = ioctl(hAdmxrc2, ioctlCode, pIoctl);
  if (0 == err) {
    return ADMXRC2_SUCCESS;
  } else {
    if (err < 0) {
      switch (errno) {
      case EBADF:
        return ADMXRC2_INVALID_HANDLE;

      case ENOMEM:
        return ADMXRC2_NO_MEMORY;

      case EINPROGRESS:
        fprintf(stderr, "admxrc2Ioctl: EINPROGRESS returned for blocking operation\n");
        return ADMXRC2_PENDING;

      case ECANCELED:
        return ADMXRC2_CANCELLED;

      case ENOTTY:
      case EFAULT:
      case EINVAL:
      default:
        fprintf(stderr, "admxrc2Ioctl: errno=%d (%s)\n", errno, strerror(errno));
        return ADMXRC2_UNKNOWN_ERROR;
      }
    } else {
      return (ADMXRC2_STATUS)err;
    }
  }
}
