/*
* Title:	tmsStress.cpp
* Author:	M.Thomas Beam Ltd
* Date:		2007-03-15
* Contents:	tms stress testing program
*
*/

#include <iostream>
#include <stdio.h>
#include <getopt.h>
#include <TmsD.h>
#include <TmsC.h>
#include <BTimer.h>
#include <BFile.h>
#include "tmsSink.h"

using namespace Tms;
using namespace std;

static const UInt32	tmsStateNum = 16;
static const UInt32	tmsPickupNum = 40;
static const int	MEGABYTE = 1024 * 1024;			// A Megabyte
static long		MAX_BYTES_PER_CYCLE = 1024 * 1024 * 100;	// Max Bytes per sink per cycle
static BString   	CHANNELS = "1,16,31";			// Default Channels



// Initialise and test the TMS system
BError tmsInit(TmsControl& tmsControl){
	BError			err;
	ConfigInfo		configInfo;
	BIter			i;
	UInt32			c;
	UInt32			m;
	UInt32			pn;
	UInt32			pc;
	BList<BError>		errorList;
	
	
	// Initialise TMS system
	if(err = tmsControl.init()){
		return err.set(1, BString("Error: initialising TMS: ") + err.getString());
	}

	// Configure TMS system - logical mapping of pickups

	configInfo.puReferences.resize(64);
	c = 1;
	for(m = 1; m < 4; m++){
		for(pn = 1; pn < 6; pn++){
			for(pc = 1; pc < 4; pc++){
				configInfo.puReferences[c++] = PuChannel(m, pn, pc);
			}
		}
	}
	if(err = tmsControl.configure(configInfo)){
		return err.set(1, BString("Error: configuring TMS: ") + err.getString());
	}
	return err;
}


BError	printVersions(TmsControl& tmsControl,TmsProcess& tmsProcess) {
	BError		err;
	BString		controlVersion;
	BString		processVersion;

	if(err = tmsControl.getVersion(controlVersion)){
		return err;
	}
	if(err = tmsProcess.getVersion(processVersion)){
		return err;
	}
	cout << "TmsControl Version: " << controlVersion << "\n";
	cout << "TmsProcess Version: " << processVersion << "\n";
	return err;
}


BError	printStatus(TmsControl& tmsControl) {
	BError			err;
	BIter			i;
	BList<NameValue>	status;

	if(err = tmsControl.getStatus(status)){
		return err;
	}
	for (status.start(i);! status.isEnd(i); status.next(i)) {
		cout << status[i].name << ":" << status[i].value << "\n";
	}
	return err;
}

BError	printStatistics(TmsControl& tmsControl) {
	BError			err;
	BIter			i;
	BList<NameValue>	status;

	if(err = tmsControl.getStatistics(status)){
		return err;
	}
	for (status.start(i);! status.isEnd(i); status.next(i)) {
		cout << status[i].name << ":" << status[i].value << "\n";
	}
	return err;
}

BError	printCycle(TmsProcess& tmsProcess) {
	BError		err;
	UInt32 		cycleNumber;
	BString		cycleType;


	if(err = tmsProcess.getCycleInfo(cycleNumber,cycleType)){
		return err;
	}
	cout << "Cycle Number: " << cycleNumber << "\n";
	return err;
}

void mtTest() {
	BError	err;
	BString	ohost = "tms3";
	int n;

	for (n =0;n < 20;n++) {
		TmsProcess*	p = new TmsProcess;
	
		if(err = p->connectService(BString("//") + ohost + "/tmsProcess")){
			cerr << "Error: cannot connect to tmsProcess: " << err.getString() << "\n";
		}
		printf("Contected (%d)\n",n);
	}
}

BError	benchMark(BString hostName,BString fname,double packetSize,BString channels,long maxBytes) {
	BError		err;
	double		totalRate;
	double 		totalPackets;
	double 		totalBytes;
	double 		overRuns;
	
	double 		deltaBytes;
	double 		deltaPackets;
	double 		deltaRate;
	double		deltaOverRuns;
	double		bytesPerCycle = maxBytes;
	
	DataInfo	dataInfo; 
	BIter		i;
	
	BList<TmsSink*>	sinks;
	BList<UInt32>	chanList;
	BList<BString>	l;
	BList<BString>	r;

	if (fname != "") {
		unlink(fname);
	}
	
	l = channels.getTokenList(",");
	for (l.start(i);! l.isEnd(i);l.next(i)) {
		r = l[i].getTokenList(":");
			if (r.number() > 1) {
				int	start,end,n;
				start 	= r[0].retInt();
				end   	= r[1].retInt();		
				n 	= start;
			
				while (n <= end) {
					chanList.append(n++);
				}
			}
			else {
				chanList.append(r[0].retInt());
			}			
	}		

	dataInfo.cyclePeriod	= 2;
	dataInfo.startTime	= 0;
	dataInfo.orbitNumber	= 0;
	dataInfo.bunchNumber	= 0;
	dataInfo.function	= 0;
	dataInfo.argument	= 0;
	dataInfo.numValues	= (UInt32)packetSize;	
		

	int n = 0;
	for (chanList.start(i);! chanList.isEnd(i);chanList.next(i)) {
		TmsSink*	s = new TmsSink(hostName,n);
		sinks.append(s);
		dataInfo.channel = chanList[i];
		s->setupRequest(dataInfo);
		s->setRate(bytesPerCycle);
		s->connect();
		n++;		
	}

	for (sinks.start(i);! sinks.isEnd(i);sinks.next(i)) {
		TmsSink*	s = sinks[i];

		s->startTimer();
		s->start();

		TmsSinkStats	stats = s->getStats();

		printf("Data sink (%d) channel (%d) Packet Size (%8.4lf) MBytes. Maximum per cycle (%5.2lf) MByte\n",
			s->oid,
			stats.dataInfo.channel,
			(double)((stats.dataInfo.numValues * 8) + 8)/MEGABYTE,
			bytesPerCycle / MEGABYTE
			);
	}

	n = 0;	
	while (1) {
		totalPackets = 0;
		totalRate = 0;
		totalBytes = 0;
		overRuns = 0;
		deltaBytes = 0;
		deltaPackets = 0;
		deltaRate = 0;
		deltaOverRuns = 0;

		sleep(5);
		for (sinks.start(i);! sinks.isEnd(i);sinks.next(i)) {
			TmsSinkStats	stats;
			stats = sinks[i]->getStats();
			totalPackets 	+= stats.packets;
			totalRate 	+= stats.transferRate;
			totalBytes 	+= stats.bytes;
			overRuns	+= stats.overRuns;
			
			deltaBytes 	+= stats.deltaBytes;
			deltaPackets 	+= stats.deltaPackets;
			deltaRate 	+= stats.deltaRate;
			deltaOverRuns 	+= stats.deltaOverRuns;
#ifndef ZAP
			printf("Sink (%d), Bytes (%4.0lf)\n",sinks[i]->oid,stats.deltaBytes);
#endif			

		}

		// Error reporting

		for (sinks.start(i);! sinks.isEnd(i);sinks.next(i)) {
			if (err = sinks[i]->getLastError()) {
				BString	s = err.getString();
				
				if (s.compareWild("*Broken pipe*")) {
					printf("(%s)\n",err.getString().retStr());
					exit(1);
				}	
				else {				
					printf("(%d) (%s)\n",sinks[i]->oid,err.getString().retStr());
				}
			}	
		}
		printf("Rate/s ( %5.1lf MB, %3.0lf Pkts, %4.0lf Ovr) Total ( %5.3lf MB/s, %7.3lf MB, %7.0lf Pkts, %4.0lf Ovr)\n",
			deltaRate,
			deltaPackets,
			deltaOverRuns,
			totalRate,
			totalBytes/MEGABYTE,
			totalPackets,
			overRuns
			);
	
		if (fname != "") {
			BFile f;
			f.open(fname,"a");
			f.printf("%d %5.3lf\n",n += 5,deltaRate);		// Time and Data Rate
			f.close();
		}	
	}
}

BError	testServer(TmsControl& tmsControl) {
	BError			err;
	BIter			i;
	BList<BError>		errList;

	if(err = tmsControl.test(errList)){
		return err;
	}
	for (errList.start(i);! errList.isEnd(i); errList.next(i)) {
		cout << errList[i].getString() << "\n";
	}
	return err;
}

// Set up the state tables, this will need to set the raw data for a particular machine
// Cycle. This includes low level timing information used in the FPGA phase Tables for
// timing the data capture and configuring the Phase locked loops.
BError	tmsSetStateTables(BArray<PuStateTable>& stateTable){
	BError			err;
	UInt32			i;
	UInt32			p;

	stateTable.resize(tmsStateNum);

	for(i = 0; i < tmsStateNum; i++){
		stateTable[i].state = 0;
		stateTable[i].numBunches = 4;
		stateTable[i].phaseTable.resize(512);
		for(p = 0; p < 512; p++){
			stateTable[i].phaseTable[p] = p;
		}
	}
	
	return err;
}

// Form Client access to configure the state tables and read some data
BError tmsTest1(TmsProcess& tmsProcess){
	BError			err;
#ifdef ZAP
	CycleParam		cParam;
	UInt32			i;
	DataInfo		dataInfo;
	Data			data;
	UInt32 		cycleNumber;

	// Configure TMS system for next and subsequent cycles
	cParam.cycleNumber		= 1;
	cParam.cycleType		= 0;
	cParam.pllInitialFrequency	= 0;
	cParam.pllInitialFrequencyDelay	= 0;
	cParam.pllGain			= 0;
	cParam.pllDdsMinimum		= 0;
	cParam.pllDdsMaximum		= 0;

	// The Phase delay params
	cParam.frefPhaseDelay.resize(tmsPickupNum);
	for(i = 0; i < tmsPickupNum; i++){
		cParam.frefPhaseDelay[i] = 0;
	}

	// The State table Params
	if(err = tmsSetStateTables(cParam.stateTable)){
		return err;
	}
	
	if(err = tmsProcess.setControlInfo(cParam)){
		return err.set(1, BString("Error: Setting Control Info: ") + err.getString());
	}

#endif
	return err;
}

void usage(void) {
	cerr << "Usage:\ttmxControl [options] hostname\n";
	cerr << "\t--help             - Help on command line parameters\n";
	cerr << "\t--debug 0x1122     - Set Debug mask\n";
	cerr << "\t--version          - Display Version numbers\n";
	cerr << "\t--init             - Initialise TMS system\n";
	cerr << "\t--status           - Status of TMS system\n";
	cerr << "\t--stats            - Statistics\n";
	cerr << "\t--cycle            - Current Cycle number\n";
	cerr << "\t--test             - Perform internal test\n";
	cerr << "\t--file <fname>     - Filename for output results (suitable for gnuplot)\n";
	cerr << "\t--mttest           - Perform internal test\n";
	cerr << "\t--benchmark <psize>- Benchmark transfer rate(psize is number of data items per packet) ctrl-C to quit\n";
	cerr << "\t--channels <list,[s:e],>-  Channels to sink data from (Comma seperated list. Range using :) ctrl-C to quit\n";
	cerr << "\t--configure <file> - Configure taking channel information from the given file\n";
	cerr << "\n";
}

static struct option options[] = {
		{ "?",			0, NULL, 0 },
		{ "h",			0, NULL, 0 },
		{ "help",		0, NULL, 0 },
		{ "channels",		1, NULL, 0 },
		{ "debug",		1, NULL, 0 },
		{ "benchmark",		1, NULL, 0 },
		{ "version",		0, NULL, 0 },
		{ "init",		0, NULL, 0 },
		{ "configure",		1, NULL, 0 },
		{ "status",		0, NULL, 0 },
		{ "stats",		0, NULL, 0 },
		{ "test",		0, NULL, 0 },
		{ "mttest",		0, NULL, 0 },
		{ "file",		1, NULL, 0 },
		{ "maxbytes",		1, NULL, 0 },
		{ "cycle",		0, NULL, 0 },
		{ 0,0,0,0 }
};

int main(int argc, char** argv){
	BError		err;
	TmsControl	tmsControl("tmsControl");
	TmsProcess	tmsProcess("tmsProcess");
	int		c;
	int		optIndex = 0;
	BString		s;
	BString		channels = CHANNELS;
	BString		fname;
	bool		debug = false;
	bool		version = false;
	bool		init = false;
	bool		status = false;
	bool		stats = false;
	bool		cycle = false;
	bool		benchmark = false;
	bool		test = false;
	bool		mttest = false;
	long		packetSize = 1024;
	long		maxbytes = MAX_BYTES_PER_CYCLE;

	BString		configFileName;
	BString		hostName;

	while((c = getopt_long_only(argc, argv, "", options, &optIndex)) == 0){
		s = options[optIndex].name;
		if(s == "help" || s == "h" || s == "?"){
			usage();
			return 1;
		}
		else if(s == "debug"){
			debug = strtol(optarg, 0, 0);
		}
		else if(s == "benchmark"){
			benchmark = true;
			packetSize = strtol(optarg,0,0);
		}
		else if(s == "channels"){
			channels = optarg;
		}
		else if(s == "maxbytes"){
			maxbytes = strtol(optarg,0,0);
		}
		else if(s == "init"){
			init = true;
		}
		else if(s == "version"){
			version = true;
		}
		else if(s == "status"){
			status = true;
		}
		else if(s == "stats"){
			stats = true;
		}
		else if(s == "test"){
			test = true;
		}
		else if(s == "cycle"){
			cycle = true;
		}
		else if(s == "mttest"){
			mttest = true;
		}
		else if(s == "file"){
			fname = optarg;
		}
		else if(s == "configure"){
			configFileName = optarg;
		}
	}
	if(optind == argc){
		usage();
		return 1;
	}
	hostName = argv[optind++];

	// Connect to TMS control objects

	if(err = tmsControl.connectService(BString("//") + hostName + "/tmsControl")){
		cerr << "Error: cannot connect to tmsControl: " << err.getString() << "\n";
		return 1;
	}
	if(err = tmsProcess.connectService(BString("//") + hostName + "/tmsProcess")){
		cerr << "Error: cannot connect to tmsProcess: " << err.getString() << "\n";
		return 1;
	}

	if(version && (err = printVersions(tmsControl,tmsProcess))){	
		cerr << "Error: geting Versions: " << err.getString() << "\n";
		return 1;
	}	

	if(init && (err = tmsControl.init())){
		cerr << "Error: initialising TMS: " << err.getString() + "\n";
		return 1;
	}

	if(status && (err = printStatus(tmsControl))){
		cerr << "Error: Unable to get TmsControl status: " << err.getString() + "\n";
		return 1;
	}
	
	if(stats && (err = printStatistics(tmsControl))){
		cerr << "Error: Unable to get TmsControl statistics: " << err.getString() + "\n";
		return 1;
	}
	if(cycle && (err = printCycle(tmsProcess))){
		cerr << "Error: Unable to get TmsProcess Cycle: " << err.getString() + "\n";
		return 1;
	}

	if(benchmark && (err = benchMark(hostName,fname,packetSize,channels,maxbytes))){
		cerr << "Error: Benchmarking Err: " << err.getString() + "\n";
		return 1;
	}
	if(mttest){
		mtTest();
		return 1;
	}
	if(test && (err = testServer(tmsControl))){
		cerr << "Error: Testing Err: " << err.getString() + "\n";
		return 1;
	}

	if(configFileName != ""){
		// Configure TMS system
	}
	
	return 0;
}
